#include "config.h"

#ifdef USE_XOPEN_SOURCE
#define _XOPEN_SOURCE 
#endif

#include <pthread.h>
#include <sys/types.h>
#include <pwd.h>
#include <errno.h>
#include <sys/time.h>
#include <glib.h>
#include "manager.h"
#include <libgnome/libgnome.h>
#include <string.h>
#include <gpilot-userinfo.h>
#include <gpilot-structures.h>
#include <unistd.h>
#include <signal.h>
#include <pi-source.h>
#include <pi-dlp.h>
#include <pi-version.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <pwd.h>

#include "orbit_daemon_glue.h"
#include "gpilot-gui.h"
#include <errno.h>

/* Set to true when the config should be reloaded */
gboolean reread_config;


gint device_equal_by_io(GPilotDevice *,GIOChannel *);
gboolean device_in(GIOChannel *,
		   GIOCondition ,
		   GPilotContext *);
gboolean device_err(GIOChannel *,
		    GIOCondition ,
		    GPilotContext *);
void monitor_channel(GPilotDevice *,GPilotContext *);
void remove_pid_file(void);

gint 
device_equal_by_io(GPilotDevice *dev,GIOChannel *io) 
{
	return !(dev->io==io);
}

static void
remove_device(GPilotContext *context,GPilotDevice *device) 
{
	GList *l;

	g_message("Removing %s",device->name);

	l = g_list_find(context->devices,device);
	if (l != NULL) {
		gpilot_device_free(l->data);
		g_list_remove(context->devices,l);
	} else {
		g_message("%s not found",device->name);
	}
}

static void 
pilot_set_baud_rate(GPilotDevice *device) 
{
	static gchar rate_buf[128];
	g_snprintf(rate_buf,128,"PILOTRATE=%d", device->speed);
	g_message("setting %s",rate_buf);
	putenv(rate_buf);
}

/*
  Sets *error to 1 on fatal error on the device, 2 on other errors , 0 otherwise.
 */
static int 
pilot_connect(GPilotDevice *device,int *error) 
{
	struct pi_sockaddr addr;
	int sd;
	int ret;
    
	pilot_set_baud_rate(device);

	if (!(sd = pi_socket(PI_AF_SLP, PI_SOCK_STREAM, PI_PF_PADP))) {
		g_warning("pi_socket: %s",strerror(errno));
		if (error) *error = 1;
		return -1;
	}
	
	addr.pi_family = PI_AF_SLP;
	strcpy(addr.pi_device,device->port);

	ret = pi_bind(sd, (struct sockaddr*)&addr, sizeof(addr));
	if(ret != 0) {
		g_warning(_("Unable to bind to pilot"));
		if (error) *error = 1;
		return 0;
	}

	ret = pi_listen(sd,1);
	if(ret != 0) {
		g_warning("pi_listen: %s", strerror(errno));
		if (error) *error = 2;
		return 0;
	}

	sd = pi_accept_to(sd, NULL,0,1000); /* set timeout to 1 second */
	if(sd == -1) {
		g_warning("pi_accept: %s", strerror(errno));
		if (error) *error = 2;
		return 0;
	}
	if (error) *error = 0;
	return sd;
}

static void pilot_disconnect(int sd)
{
	dlp_EndOfSync(sd, 0);
	pi_close(sd);
}

static void write_sync_stamp(GPilotPilot *pilot,
			     int pfd,
			     struct PilotUser *pu,
			     guint32 last_sync_pc, 
			     time_t t)
{
	gchar prefix[256];

	pu->lastSyncPC=last_sync_pc;
	pu->lastSyncDate=t;

	g_snprintf(prefix,255,"/gnome-pilot.d/gpilotd/Pilot%d/",pilot->number);
	gnome_config_push_prefix(prefix);
	gnome_config_private_set_int("sync_date",t);
	gnome_config_pop_prefix();
	gnome_config_sync();

	dlp_WriteUserInfo(pfd,pu);
}

/** pilot lookup methods **/


/**************************/

/*
 * If there are events for the cradle, this executes them,
 * closes the connection and returns.
 * Returns TRUE if connection should be closed afterwards, FALSE
 * is sync should continue
 */
static gboolean 
do_cradle_events(int pfd,
		 GPilotContext *context,
		 struct PilotUser *pu,
		 GPilotPilot *pilot,
		 GPilotDevice *device) 
{
	GList *events,*it;
	gboolean ret = TRUE;

	/* elements in events freed by gpc_request_purge calls
	   in orbed_notify_completion */
	events = gpc_queue_load_requests_for_cradle(device->name);

	g_message(_("Cradle %s has %d events"),device->name,g_list_length(events));

	/* if now events, return FALSE */
	if (g_list_length(events)==0) ret = FALSE;

	it = events;
	
	while(it) {
		GPilotRequest *req;
		req = it->data;
		switch(req->type) {
		case GREQ_SET_USERINFO:
			g_message(_("Setting userinfo..."));
			g_snprintf(pu->username,127,"%s",req->parameters.set_userinfo.user_id);
			pu->userID = req->parameters.set_userinfo.pilot_id;
			dlp_WriteUserInfo(pfd,pu);
			if (req->parameters.set_userinfo.continue_sync) ret = FALSE;
			orbed_notify_completion(&req);
			break;
		case GREQ_GET_USERINFO:
			g_message(_("Getting userinfo..."));
			orbed_notify_userinfo(*pu,&req);
			orbed_notify_completion(&req);
			break;
		case GREQ_NEW_USERINFO:
			/* FIXME: this is to set the new and return the old (or something) 
			   g_message("getting & setting userinfo");
			   g_snprintf(pu->username,127,"%s",req->parameters.set_userinfo.user_id);
			   pu->userID = req->parameters.set_userinfo.pilot_id;
			   dlp_WriteUserInfo(pfd,pu);
			   orbed_notify_completion(&req);
			*/
			break;
		default:
			g_warning("%s:%d: *** type = %d",__FILE__,__LINE__,req->type);
			g_assert_not_reached();
			break;
		}

		it = g_list_next(it);
	}

	return ret;
}
/**************************/

/*
  This executes a sync for a pilot.

  If first does some printing to the stdout and some logging to the
  pilot, so the dudes can see what is going on. Afterwards, it does
  the initial synchronization operations (which is handling file
  installs, restores, specific conduits runs). This function (in
  manager.c) returns a boolean, whic may abort the entire
  synchronization.

  If it does not, a function in manager.c will be called depending of
  the default_sync_action setting for the pilot (you know, synchronize
  vs copy to/from blablabla).

 */
static void 
do_sync(int pfd,   
	GPilotContext *context,
	struct PilotUser *pu,
	GPilotPilot *pilot, 
	GPilotDevice *device)
{
	GList *conduit_list, *backup_conduit_list, *file_conduit_list;
	GnomePilotSyncStamp stamp;
	gchar *pilot_name;

	pilot_name = pilot_name_from_id(pu->userID,context);

	gpilot_load_conduits(context,
			     pilot,
			     &conduit_list, 
			     &backup_conduit_list,
			     &file_conduit_list);
	stamp.sync_PC_Id=context->sync_PC_Id;

	g_message(_("HotSync button pressed, synchronizing pilot"));
	g_message(_("Pilot ID is %ld, name is %s, owner is %s"),
		  pu->userID,
		  pilot->name,
		  pu->username);
  
	/* Set a log entry in the pilot */
	{
		gchar hostname[64];
		gpilot_add_log_entry(pfd,"gnome-pilot v.%s\n",VERSION);
		if (gethostname(hostname,63)==0)
			gpilot_add_log_entry(pfd,_("On host %s\n"),hostname);
		else
			gpilot_add_log_entry(pfd,_("On host %d\n"),stamp.sync_PC_Id);
	}

	/* first, run the initial operations, such as single conduit runs,
	   restores etc. If this returns True, continue with normal conduit running,
	   if False, don't proceed */
	if (gpilot_initial_synchronize_operations(pfd,&stamp,pu,
						  conduit_list,
						  backup_conduit_list,
						  file_conduit_list,
						  context)) {

		/* FIXME: We need to actually pass a different structure to these
		   functions containing the pu item and the pilot pointer so that
		   we can store the lastSyncPC and assorted fields in
		   ~/.gnome_private/gnome-pilot.d/gpilotd. We will also need some
		   lock mechanism when we start forking for that file :)*/
		switch(pilot->sync_options.default_sync_action) {
		case GnomePilotConduitSyncTypeSynchronize:
			g_message(_("Synchronizing..."));
			gpilot_synchronize(pfd,&stamp,pu,
					   conduit_list,
					   backup_conduit_list,
					   file_conduit_list,
					   context);
			break;
		case GnomePilotConduitSyncTypeCopyToPilot:
			g_message(_("Copying to pilot..."));
			gpilot_copy_to_pilot(pfd,&stamp,pu,
					     conduit_list,
					     backup_conduit_list,
					     file_conduit_list,
					     context);
			break;
		case GnomePilotConduitSyncTypeCopyFromPilot:
			g_message(_("Copying from pilot..."));
			gpilot_copy_from_pilot(pfd,&stamp,pu,
					       conduit_list,
					       backup_conduit_list,
					       context);
			break;
		case GnomePilotConduitSyncTypeMergeToPilot:
			g_message(_("Merging to pilot..."));
			gpilot_merge_to_pilot(pfd,&stamp,pu,
					      conduit_list,
					      backup_conduit_list,
					      file_conduit_list,
					      context);
			break;
		case GnomePilotConduitSyncTypeMergeFromPilot:
			g_message(_("Merging from pilot..."));
			gpilot_merge_from_pilot(pfd,&stamp,pu,
						conduit_list,
						backup_conduit_list,
						context);
			break;
		case GnomePilotConduitSyncTypeNotSet:
		case GnomePilotConduitSyncTypeCustom:
		default:
			g_message(_("Using conduit settings for sync..."));
			gpilot_sync_default(pfd,&stamp,pu,
					    conduit_list,
					    backup_conduit_list,
					    file_conduit_list,
					    context);
			break;
		}
		g_message(_("Synchronization ended")); 
		gpilot_add_log_entry(pfd,"Synchronization completed");
	} else {
		g_message(_("Synchronization ended early"));
		gpilot_add_log_entry(pfd,"Synchronization terminated");
	}

	write_sync_stamp(pilot,pfd,pu,stamp.sync_PC_Id,time(NULL));
  
	g_free(pilot_name);

	gpilot_unload_conduits(conduit_list);
	gpilot_unload_conduits(backup_conduit_list);
	gpilot_unload_conduits(file_conduit_list);
}

/*
  sync_foreach is the first synchronization entry.

  It first connects to the device on which the signal was detected,
  then it tries to read the user info block from the pilot.

  Hereafter, if there are any events queued for the synchronizing
  cradle, execute them and stop the synchronization (note,
  do_cradle_events returns a bool, if this is FALSE, synchronization
  continues, as some cradle specific events also require a normal sync
  afterwards, eg. the REVIVE call)

  Anyways, if the sync continues, sync_foreach tries to match the
  pilot against the known pilots. If this fails, it should handle it
  intelligently, eg. if the id==0, ask if you want to restore a pilot.

  If the pilot is accepted (dude, there's even a password check!), it
  continues into do_sync, which does all the magic stuff.
*/
   
static gboolean 
sync_foreach(GPilotDevice *device, GPilotContext *context)
{
	GPilotPilot *pilot;
	int pfd;
	int connect_error;
	struct PilotUser pu;

	/* signal(SIGHUP,SIG_DFL); */
	pfd=pilot_connect(device,&connect_error);

	if (!connect_error) {
		/* connect succeeded, try to read the userinfo */
		if(dlp_ReadUserInfo(pfd,&pu) < 0) {
			/* no ? drop connection then */
			g_warning(_("An error occured while getting the pilot's user data"));
			pilot_disconnect(pfd);
		} else {
			/* If there are cradle specific events, handle them and stop */
			if(do_cradle_events(pfd,context,&pu,pilot,device)) {
				g_message(_("Completed events for cradle %s (%s)"),device->name,device->port);
				pilot_disconnect(pfd);
			} else {
				/* No cradle events, validate pilot */
				pilot = gpilot_find_pilot_by_id(pu.userID,context->pilots);
				if(pilot == NULL) {
					/* Pilot is not known */
					g_warning(_("Unknown pilot, no userID/username match %ld"),pu.userID);
					/* FIXME: here, restoring one of the available pilots should be
					   offered to the user. Of course with password prompt if the user
					   has password set */
					pilot_disconnect(pfd);
					gpilot_gui_warning_dialog(_("Unknown pilot - no pilots matches ID %ld\n"
								    "Use gnomecc to set pilot's ID"),pu.userID);
				} else {
					struct stat buf; 
					int ret;
 					/* Pilot is known, make connect notifications */
 					orbed_notify_connect(pilot->name,pu);
					
					ret=stat(pilot->sync_options.basedir, &buf); 
					if(ret < 0 || !( S_ISDIR(buf.st_mode) && (buf.st_mode & (S_IRUSR | S_IWUSR |S_IXUSR))) ) {

						g_message("Invalid basedir: %s", pilot->sync_options.basedir);
						gpilot_gui_warning_dialog(_("The base directory %s is invalid.\n"
									    "Please fix it or use gnomecc to choose another directory."),
									  pilot->sync_options.basedir);	
					} else {

						/* If pilot has password, check against the encrypted version
						   on the pilot */
						if(pilot->passwd) {
							char *pwd;
							pwd = (char*)g_malloc(pu.passwordLength);
							strncpy(pwd,pu.password,pu.passwordLength);
							if(g_strcasecmp(pilot->passwd,(char*)crypt(pwd,pilot->passwd))) {
								pilot_disconnect(pfd);
								orbed_notify_disconnect(pilot->name);
								gpilot_gui_warning_dialog(_("Unknown pilot - no pilots matches ID %ld\n"
											    "Use gnomecc to set pilot's ID"),pu.userID);
							}
							g_free(pwd);
						} 

						do_sync(pfd,context,&pu,pilot,device);
					}
					pilot_disconnect(pfd);
					orbed_notify_disconnect(pilot->name);
				}
			}
		}
	} else {
		if(connect_error==1) return FALSE; /* remove this device */
		else return TRUE;
	}

	return TRUE;
}

gboolean 
device_in(GIOChannel *io_channel,
	  GIOCondition condition,
	  GPilotContext *context) {
	GPilotDevice *device;
	GList *element;
      
	element = g_list_find_custom(context->devices,
				     io_channel,
				     (GCompareFunc)device_equal_by_io);

	if(element==NULL) {
		g_warning("cannot find device for active IO channel");
		return FALSE;
	}
	
	device = element->data;
	if (context->paused) {
		return FALSE; 
	}
	g_message(_("Woke on %s"),device->name);
	return sync_foreach(device,context);
}

gboolean 
device_err(GIOChannel *io_channel,
	   GIOCondition condition,
	   GPilotContext *context) {
	GPilotDevice *device;
	GList *element;
	char *tmp;

	switch(condition) {
	case G_IO_IN: tmp = g_strdup_printf("G_IO_IN"); break;
	case G_IO_OUT : tmp = g_strdup_printf("G_IO_OUT"); break;
	case G_IO_PRI : tmp = g_strdup_printf("G_IO_PRI"); break;
	case G_IO_ERR : tmp = g_strdup_printf("G_IO_ERR"); break;
	case G_IO_HUP : tmp = g_strdup_printf("G_IO_HUP"); break;
	case G_IO_NVAL: tmp = g_strdup_printf("G_IO_NVAL"); break;
	default: tmp = g_strdup_printf("unhandled port error"); break;
	}
	
	element = g_list_find_custom(context->devices,io_channel,(GCompareFunc)device_equal_by_io);

	if(element==NULL) {
		/* We most likely end here if the device has just been removed.
		   Eg. start gpilotd with a monitor on a XCopilot fake serial port,
		   kill xcopilot and watch things blow up as the device fails */
		g_warning("Device error on some device, caught %s",tmp); 
		g_free(tmp);
		return FALSE;
	}
	
	device = element->data;

	gpilot_gui_warning_dialog("Device error on %s (%s)\n"
				  "Caught %s",device->name,device->port,tmp); 
	g_warning("Device error on %s (%s), caught %s",device->name,device->port,tmp);

	remove_device(context,device);
	g_free(tmp);
		
	return FALSE;
}

void monitor_channel(GPilotDevice *dev,GPilotContext *context) {
	dev->in_handle = g_io_add_watch(dev->io,
					G_IO_IN,
					(GIOFunc)device_in,
					(gpointer)context);
	dev->err_handle = g_io_add_watch(dev->io,
					 G_IO_ERR|G_IO_PRI|G_IO_HUP|G_IO_NVAL,
					 (GIOFunc)device_err,
					 (gpointer)context);
	g_message(_("Watching %s (%s)"),dev->name,dev->port);
}

static void 
sig_hup_handler(int dummy)
{
	signal(SIGHUP,sig_hup_handler);
	reread_config=TRUE;
}

static void 
sig_term_handler(int dummy) {
	g_message(_("Exiting (caught SIGTERM)..."));
	remove_pid_file();
	gpilotd_corba_quit();
	exit(0);
}

static void 
sig_int_handler(int dummy) {
	g_message(_("Exiting (caught SIGINT)..."));
	remove_pid_file();
	gpilotd_corba_quit();
	exit(0);
}

/* This deletes the ~/.gpilotd.pid file */
void 
remove_pid_file() {
	gchar *home_directory;
	gchar *pid_file;

	home_directory=g_get_home_dir();
	if(home_directory) {
		pid_file=(gchar *)g_malloc(strlen(home_directory) + 
					   strlen("/.gpilotd.pid") + 1);
		strcpy(pid_file,home_directory);
		strcat(pid_file,"/.gpilotd.pid");
		if(access(pid_file,R_OK|W_OK)==0) {
			unlink(pid_file);
		} 
		g_free(pid_file);
	}
}

/*
  The creates a ~/.gilotd.pid, containing the pid
   of the gpilotd process, used by clients to send
   SIGHUPS
*/
static void 
write_pid_file()
{
	gchar *home_directory;
	gchar *pid_file;
	int fd;
	home_directory=g_get_home_dir();
	if(home_directory) {
		pid_file=(gchar *)g_malloc(strlen(home_directory) +
					   strlen("/.gpilotd.pid") + 1);
		strcpy(pid_file,home_directory);
		strcat(pid_file,"/.gpilotd.pid");
		if((fd=open(pid_file,O_RDWR | O_TRUNC | O_CREAT, 0644)) >= 0) {
			gchar pid[50]; /* If the pid is > 50 digits we have problems :) */
			if(sizeof(pid_t) == sizeof(int)) {
				g_snprintf(pid,50,"%d\n",getpid());
			} else { /* Assume pid_t is sizeof long */
				g_snprintf(pid,50,"%ld\n",(unsigned long)getpid());
			}
			write(fd,pid,strlen(pid));
			close(fd);
		} else {
			g_warning(_("Unable to open or create "
				    "file %s with read/write privs"),pid_file);
		}

		g_free(pid_file);
	} else {
		g_warning(_("Unable to find home directory for uid %d"),geteuid());
	}
}

static void 
wait_for_sync_and_sync(GPilotContext *context) {
	signal(SIGTERM,sig_term_handler);
	signal(SIGINT,sig_int_handler);
	signal(SIGHUP,sig_hup_handler);

	g_list_foreach(context->devices,(GFunc)monitor_channel,context);

	while(1) {
		if(reread_config) {
			g_message(_("Rereading configuration..."));
			gpilot_context_init_user (context);
			reread_config=FALSE;
			g_list_foreach(context->devices,(GFunc)monitor_channel,context);
		}
		g_main_iteration(TRUE);
	}
}

int 
main(int argc, char *argv[])
{
	GPilotContext *context;
	
	bindtextdomain (PACKAGE, GNOMELOCALEDIR);
	textdomain (PACKAGE);

	/* Intro */
	g_message("%s %s starting...",PACKAGE,VERSION);
	g_message("compiled for pilot-link version %d.%d.%d",
		  PILOT_LINK_VERSION,PILOT_LINK_MAJOR,PILOT_LINK_MINOR);

	/* Setup the correct gpilotd.pid file */
	remove_pid_file();
	write_pid_file();

	/* Init corba and context, this call also loads the config into context */
	gpilotd_corba_init(&argc,argv,&context);
	reread_config=FALSE;	

	/* Begin... */
	wait_for_sync_and_sync(context);

	/* It is unlikely that we will end here */
	remove_pid_file();
	gpilotd_corba_quit();
	
	return 0;
}







