/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */
/* Control applet ("capplet") for the gnome-pilot expense conduit,          */
/* based on                                                                 */
/* gpilotd control applet ('capplet') for use with the GNOME control center */

#include <pwd.h>
#include <sys/types.h>
#include <signal.h>
#include <ctype.h>
#include <gnome.h>

#include <config.h>
#include <capplet-widget.h>

#include <libgpilotdCM/gnome-pilot-conduit-management.h>
#include <libgpilotdCM/gnome-pilot-conduit-config.h>
#include <gpilotd/gnome-pilot-client.h>
#include "expense_conduit.h"

/* tell changes callbacks to ignore changes or not */
static gboolean ignore_changes=FALSE;

/* capplet widget */
static GtkWidget *capplet=NULL;

/* host/device/pilot configuration windows */
GtkWidget *cfgOptionsWindow=NULL;
GtkWidget *cfgStateWindow=NULL;
GtkWidget *dialogWindow=NULL;

gboolean activated,org_activation_state;
GnomePilotConduitManagement *conduit;
GnomePilotConduitConfig *conduit_config;

ConduitCfg *origState = NULL;
ConduitCfg *curState = NULL;

static void doTrySettings(GtkWidget *widget, gpointer dummy);
static void doRevertSettings(GtkWidget *widget, gpointer dummy);
static void doSaveSettings(GtkWidget *widget, gpointer dummy);
static void doCancelSettings(GtkWidget *widget, gpointer dummy);

static void readOptionsCfg(GtkWidget *w, ConduitCfg *state);
static void setOptionsCfg(GtkWidget *w, ConduitCfg *state);
static void readStateCfg(GtkWidget *w, ConduitCfg *state);
static void setStateCfg(GtkWidget *w, ConduitCfg *state);

gint pilotId;
CORBA_Environment ev;
static GnomePilotClient *gpc;

#define DATE_OPTIONS_COUNT 4
typedef struct {
        gchar *name;
        char *format;
} DateSetting_t;

static DateSetting_t date_options[] = { { N_("Day/Month/Year"), "%d/%m/%Y"}, 
                                       { N_("Month/Day/Year"), "%m/%d/%Y"}, 
                                       { N_("Since 1970-01-01 (in sec)"), "%s"}, 
                                       { N_("Local format"), "%x"}
};

#define WRITEOUT_OPTIONS_COUNT 2
typedef struct {
        gchar *name;
        enum ExpenseOutputFormat format;
} WriteoutSetting_t;

static WriteoutSetting_t writeout_options[] = { { N_("Simple"), eSimpleFormat},
                                                { N_("Complex"), eComplexFormat} };


static void 
load_configuration(ConduitCfg **c,guint32 pilotId) 
{
	gchar *prefix;
	gchar *tempbuf = NULL;

	g_assert(c!=NULL);
	*c = g_new0(ConduitCfg,1);
	(*c)->child = -1;

	prefix = g_strdup_printf("/gnome-pilot.d/expense-conduit/Pilot_%u/",pilotId);
  
	gnome_config_push_prefix(prefix);
	(*c)->dir = gnome_config_get_string( "dir");
	(*c)->dateFormat = gnome_config_get_string( "date_format=%x");
	(*c)->outputFormat = gnome_config_get_int("output_format=0");
	tempbuf = gnome_config_get_string("dir mode=0700");
	(*c)->dirMode =(mode_t)strtol(tempbuf,NULL,0);
	g_free(tempbuf);
	tempbuf = gnome_config_get_string("file mode=0600");
	(*c)->fileMode =(mode_t)strtol(tempbuf,NULL,0);
	g_free(tempbuf);

	gnome_config_pop_prefix();

	(*c)->pilotId = pilotId;

	/* make a default directory if nothing was defined */
	if((*c)->dir == NULL) {
		gnome_pilot_client_get_pilot_base_dir_by_id(gpc,pilotId,&tempbuf);
		(*c)->dir =g_strdup_printf ("%s/expense", tempbuf);
		g_free(tempbuf);
	}
	g_free(prefix);
}


static void 
save_configuration(ConduitCfg *c) 
{
	gchar *prefix;

	g_assert(c!=NULL);

	prefix = g_strdup_printf("/gnome-pilot.d/expense-conduit/Pilot_%u/",c->pilotId);

	gnome_config_push_prefix(prefix);
	gnome_config_set_string("dir", c->dir);
	gnome_config_set_string("date_format", c->dateFormat);
	gnome_config_set_int("output_format", c->outputFormat);
	g_snprintf(prefix,255,"0%o", c->dirMode);
	gnome_config_set_string("dir mode", prefix);
	g_snprintf(prefix,255,"0%o", c->fileMode);
	gnome_config_set_string("file mode", prefix);

	gnome_config_pop_prefix();

	gnome_config_sync();
	gnome_config_drop_all();
	g_free(prefix);
}

static ConduitCfg*
dupe_configuration(ConduitCfg *c) {
	ConduitCfg *retval;
	g_assert(c!=NULL);
	retval = g_new0(ConduitCfg,1);
	retval->dir = g_strdup(c->dir);
	retval->dateFormat = g_strdup(c->dateFormat);
	retval->outputFormat = c->outputFormat;
	retval->dirMode = c->dirMode;
	retval->fileMode = c->fileMode;

	retval->pilotId = c->pilotId;

	return retval;
}

/** this method frees all data from the conduit config */
static void 
destroy_configuration(ConduitCfg **c) 
{
	g_assert(c!=NULL);
	g_assert(*c!=NULL);
	g_free( (*c)->dir);
	g_free( (*c)->dateFormat);
	g_free(*c);
	*c = NULL;
}


static void
setSettings(ConduitCfg* conduitCfg)
{
       if(activated)
               	gnome_pilot_conduit_config_enable(conduit_config,GnomePilotConduitSyncTypeSynchronize);
        else
                gnome_pilot_conduit_config_disable(conduit_config);
        save_configuration(conduitCfg);
}

static void
doTrySettings(GtkWidget *widget, gpointer dummy)
{
        readStateCfg(cfgStateWindow, curState);
        readOptionsCfg(cfgOptionsWindow, curState);
        setSettings(curState);
}

static void
doSaveSettings(GtkWidget *widget, gpointer dummy)
{
        doTrySettings(widget, curState);
}


static void
doRevertSettings(GtkWidget *widget, gpointer dummy)
{
        activated = org_activation_state;
        destroy_configuration(&curState);
        curState = dupe_configuration(origState);
        setOptionsCfg(cfgOptionsWindow, curState);
        setStateCfg(cfgStateWindow, curState);
        setSettings(curState);
}

static void
doCancelSettings(GtkWidget *widget, gpointer dummy)
{
        setSettings(origState);
}

/* Don't allow any spaces */
static void
insert_ignore_space_cb (GtkEditable    *editable, const gchar    *text,
                        gint len, gint *position, void *data)
{
        gint i;
        gchar *curname;

        curname = gtk_entry_get_text(GTK_ENTRY(editable));
        if (*curname == '\0' && len > 0) {
                if (isspace(text[0])) {
                        gtk_signal_emit_stop_by_name(GTK_OBJECT(editable), "insert_text");
                        return;
                }
        } else {
                for (i=0; i<len; i++) {
                        if (isspace(text[i])) {
                                gtk_signal_emit_stop_by_name(GTK_OBJECT(editable), 
                                                             "insert_text");
                                return;
                        }
                }
        }
}

static void
insert_numeric_cb(GtkEditable    *editable, const gchar    *text,
                  gint len, gint *position, void *data)
{
	gint i;

	for (i=0; i<len; i++) {
		if (!isdigit(text[i])) {
			gtk_signal_emit_stop_by_name(GTK_OBJECT(editable), "insert_text");
			return;
		}
	}
}

/* option menu callback for changing dates */
static void
datesetting_cb(GtkMenuItem *widget, gpointer data)
{
        curState->dateFormat = g_strdup((gchar*)data);
        if(!ignore_changes)
                capplet_widget_state_changed(CAPPLET_WIDGET(capplet), TRUE);
}

/* option menu callback for writing out */
static void
writeoutsetting_cb(GtkMenuItem *widget, gpointer data)
{
        curState->outputFormat = *(enum ExpenseOutputFormat*) data;
        if(!ignore_changes)
                capplet_widget_state_changed(CAPPLET_WIDGET(capplet), TRUE);
}

/* make the ok/try/cancel buttons active */
static void
statechange_cb(GtkEditable    *editable, const gchar    *text,
                 gint            length, gint           *position,
                 void *data)
{
        if (!ignore_changes)
                capplet_widget_state_changed(CAPPLET_WIDGET(capplet), TRUE);
}

	
static void 
about_cb (GtkWidget *widget, gpointer data) 
{
        GtkWidget *about;
        const gchar *authors[] = {_("Patrick Decowski <decowski@mit.edu>"),NULL};
  
        about = gnome_about_new(_("Gpilotd Expense conduit"), VERSION,
                                _("(C) 1999 the Free Software Foundation"),
                                authors,
                                _("Configuration utility for the Expense conduit.\n"
                                  "The Expense conduit is responsible for getting "
                                  "expenses records from the pilot"),
                                _("gnome-unknown.xpm"));
        gtk_widget_show (about);
        
        return;
}

static void toggled_cb(GtkWidget *widget, gpointer data) 
{
        if(!ignore_changes) {
                gtk_widget_set_sensitive(cfgOptionsWindow,GTK_TOGGLE_BUTTON(widget)->active);
                capplet_widget_state_changed(CAPPLET_WIDGET(capplet), TRUE);
        }
}

static GtkWidget
*createStateCfgWindow(void)
{
        GtkWidget *vbox, *table;
        GtkWidget *label;
        GtkWidget *button;

        vbox = gtk_vbox_new(FALSE, GNOME_PAD);

        table = gtk_table_new(2, 2, FALSE);
        gtk_table_set_row_spacings(GTK_TABLE(table), 4);
        gtk_table_set_col_spacings(GTK_TABLE(table), 10);
        gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, GNOME_PAD);

        label = gtk_label_new(_("Enabled"));
        gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1,2);
        
        button = gtk_check_button_new();
        gtk_object_set_data(GTK_OBJECT(vbox), "conduit_on_off", button);
        gtk_signal_connect(GTK_OBJECT(button), "toggled",
                           GTK_SIGNAL_FUNC(toggled_cb),
                           NULL);
        gtk_table_attach_defaults(GTK_TABLE(table), button, 1, 2, 1,2);

        return vbox;
}


static void
setStateCfg(GtkWidget *widget,ConduitCfg *cfg)
{
        GtkWidget *button;

        button = gtk_object_get_data(GTK_OBJECT(widget), "conduit_on_off");

        g_assert(button!=NULL);

        ignore_changes = TRUE;
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button),activated);
        gtk_widget_set_sensitive(cfgOptionsWindow,GTK_TOGGLE_BUTTON(button)->active);
        ignore_changes = FALSE;
}


static void
readStateCfg(GtkWidget *widget,ConduitCfg *cfg)
{
        GtkWidget *button;

        button  = gtk_object_get_data(GTK_OBJECT(widget), "conduit_on_off");
        
        g_assert(button!=NULL);

        activated = GTK_TOGGLE_BUTTON(button)->active;
}

typedef struct _FieldInfo FieldInfo;
struct _FieldInfo
{
	gchar    *name;
	gchar    *label_data;
	gchar    *obj_data;
	gpointer  insert_func;
};


FieldInfo fields[] = { { N_("Expense Directory"), NULL, "ExpenseDir", insert_ignore_space_cb},
                       { N_("Directory Mode"), NULL, "DirMode", insert_numeric_cb},
                       { N_("File Mode"), NULL, "FileMode", insert_numeric_cb}, 
                       { NULL, NULL, NULL, NULL}
};


static GtkWidget
*createCfgWindow(void)
{
	GtkWidget *vbox, *table;
	GtkWidget *entry, *label;
        GtkWidget *menuItem, *optionMenu;
        GtkMenu   *menu;

        int i, count=0, widget_offset;

	vbox = gtk_vbox_new(FALSE, GNOME_PAD);

	table = gtk_table_new(2, 5, FALSE);
	gtk_table_set_row_spacings(GTK_TABLE(table), 4);
	gtk_table_set_col_spacings(GTK_TABLE(table), 10);
	gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, GNOME_PAD);

        /* set the date format */
        label = gtk_label_new(_("Date Format"));
        gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2);

        menu = GTK_MENU(gtk_menu_new());
        for (i = 0; i < DATE_OPTIONS_COUNT; i++) {
                menuItem = gtk_menu_item_new_with_label(_(date_options[i].name));
                gtk_widget_show(menuItem);
                gtk_signal_connect(GTK_OBJECT(menuItem),"activate",
                                   GTK_SIGNAL_FUNC(datesetting_cb), 
                                   date_options[i].format); 
                gtk_menu_append(menu, menuItem);
        }

        optionMenu = gtk_option_menu_new(); 
        gtk_option_menu_set_menu(GTK_OPTION_MENU(optionMenu),GTK_WIDGET(menu));
        gtk_signal_connect(GTK_OBJECT(menu), "selection-done",
                           GTK_SIGNAL_FUNC(statechange_cb),
                           NULL);

        gtk_table_attach_defaults(GTK_TABLE(table), optionMenu, 1, 2, 1, 2);
        gtk_object_set_data(GTK_OBJECT(vbox), "DateFormat", optionMenu);

        /* set the writeout format */
        label = gtk_label_new(_("Output Format"));
        gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 2, 3);

        menu = GTK_MENU(gtk_menu_new());
        for (i = 0; i < WRITEOUT_OPTIONS_COUNT; i++) {
                menuItem = gtk_menu_item_new_with_label(_(writeout_options[i].name));
                gtk_widget_show(menuItem);
                gtk_signal_connect(GTK_OBJECT(menuItem),"activate",
                                   GTK_SIGNAL_FUNC(writeoutsetting_cb), 
                                   &writeout_options[i].format); 
                gtk_menu_append(menu, menuItem);
        }

        optionMenu = gtk_option_menu_new(); 
        gtk_option_menu_set_menu(GTK_OPTION_MENU(optionMenu),GTK_WIDGET(menu));
        gtk_signal_connect(GTK_OBJECT(menu), "selection-done",
                           GTK_SIGNAL_FUNC(statechange_cb),
                           NULL);

        gtk_table_attach_defaults(GTK_TABLE(table), optionMenu, 1, 2, 2, 3);
        gtk_object_set_data(GTK_OBJECT(vbox), "OutputFormat", optionMenu);

        /* ugh, so we have an asymmetry here: above is done in paste&copy fashion 
           and below, we do it nicely with structs and stuff */  

        /* do the dir & modes */

	/* how many fields do we have */
	while(fields[count].name!=0) count++;

        widget_offset = 3;
	for(i = 0; i < count; i++) {
		label = gtk_label_new(_(fields[i].name));
		gtk_table_attach(GTK_TABLE(table), label, 0, 1, 
                                 widget_offset+i, widget_offset+i+1, 0,0,0,0);
		if(fields[i].label_data!=NULL) {
			gtk_object_set_data(GTK_OBJECT(vbox), fields[i].label_data, label);
		}
		entry = gtk_entry_new_with_max_length(128);
		gtk_object_set_data(GTK_OBJECT(vbox), fields[i].obj_data, entry);
		gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 
                                 widget_offset+i, widget_offset+i+1, 0,0,0,0);
		gtk_signal_connect(GTK_OBJECT(entry), "insert_text",
				   GTK_SIGNAL_FUNC(fields[i].insert_func),
				   NULL);
		gtk_signal_connect_after(GTK_OBJECT(entry), "insert_text",
					 GTK_SIGNAL_FUNC(statechange_cb),
					 NULL);
		gtk_signal_connect_after(GTK_OBJECT(entry), "delete_text",
					 GTK_SIGNAL_FUNC(statechange_cb),
					 NULL);
	}
	

	return vbox;
}

static void
setOptionsCfg(GtkWidget *pilotcfg, ConduitCfg *state)
{
	GtkWidget *DateFormat, *OutputFormat, *ExpenseDir, *DirMode, *FileMode;
        gchar buf[8];

        int i;

	DateFormat = gtk_object_get_data(GTK_OBJECT(pilotcfg), "DateFormat");
	OutputFormat = gtk_object_get_data(GTK_OBJECT(pilotcfg), "OutputFormat");
	ExpenseDir = gtk_object_get_data(GTK_OBJECT(pilotcfg), "ExpenseDir");
	DirMode = gtk_object_get_data(GTK_OBJECT(pilotcfg), "DirMode");
	FileMode = gtk_object_get_data(GTK_OBJECT(pilotcfg), "FileMode");

	g_assert(DateFormat != NULL);
	g_assert(OutputFormat != NULL);
	g_assert(ExpenseDir != NULL);
	g_assert(DirMode != NULL);
	g_assert(FileMode != NULL);
                
	ignore_changes = TRUE;

	gtk_entry_set_text(GTK_ENTRY(ExpenseDir), state->dir);
	g_snprintf(buf, 7, "0%o", state->dirMode);
	gtk_entry_set_text(GTK_ENTRY(DirMode),buf);
	g_snprintf(buf, 7, "0%o", state->fileMode);
	gtk_entry_set_text(GTK_ENTRY(FileMode),buf);

        /* find the entry in the option menu. if not found, default to the last */
        for(i = 0; i < DATE_OPTIONS_COUNT && g_strncasecmp(state->dateFormat, date_options[i].format, 20) != 0; i++);
        gtk_option_menu_set_history(GTK_OPTION_MENU(DateFormat), i);

        for(i = 0; i < WRITEOUT_OPTIONS_COUNT && state->outputFormat != writeout_options[i].format; i++);
        gtk_option_menu_set_history(GTK_OPTION_MENU(OutputFormat), i);

	ignore_changes = FALSE;
}

static void
readOptionsCfg(GtkWidget *pilotcfg, ConduitCfg *state)
{
	GtkWidget *ExpenseDir, *DirMode, *FileMode;;

	ExpenseDir = gtk_object_get_data(GTK_OBJECT(pilotcfg), "ExpenseDir");
	DirMode = gtk_object_get_data(GTK_OBJECT(pilotcfg), "DirMode");
	FileMode = gtk_object_get_data(GTK_OBJECT(pilotcfg), "FileMode");

        state->dir = g_strdup(gtk_entry_get_text(GTK_ENTRY(ExpenseDir)));
        state->dirMode = strtol(gtk_entry_get_text(GTK_ENTRY(DirMode)), NULL, 0);
        state->fileMode = strtol(gtk_entry_get_text(GTK_ENTRY(FileMode)), NULL, 0);

        /* state of dateFormat, outputFormat changed in datesetting_cb */
}

static void
pilot_capplet_setup(void)
{
        GtkWidget *frame, *table;

        capplet = capplet_widget_new();

        table = gtk_table_new(1, 2, FALSE);
        gtk_container_border_width(GTK_CONTAINER(table), GNOME_PAD);
        gtk_container_add(GTK_CONTAINER(capplet), table); 

        frame = gtk_frame_new(_("Conduit state"));
        gtk_container_border_width(GTK_CONTAINER(frame), GNOME_PAD_SMALL);
        gtk_table_attach_defaults(GTK_TABLE(table), frame, 0, 1, 0, 1);
        cfgStateWindow = createStateCfgWindow();
        gtk_container_add(GTK_CONTAINER(frame), cfgStateWindow);

        frame = gtk_frame_new(_("Expense options"));
        gtk_container_border_width(GTK_CONTAINER(frame), GNOME_PAD_SMALL);
        gtk_table_attach_defaults(GTK_TABLE(table), frame, 0, 1, 1, 2);
        cfgOptionsWindow = createCfgWindow();
        gtk_container_add(GTK_CONTAINER(frame), cfgOptionsWindow);
        
        gtk_signal_connect(GTK_OBJECT(capplet), "try",
                           GTK_SIGNAL_FUNC(doTrySettings), NULL);
        gtk_signal_connect(GTK_OBJECT(capplet), "revert",
                           GTK_SIGNAL_FUNC(doRevertSettings), NULL);
        gtk_signal_connect(GTK_OBJECT(capplet), "ok",
                           GTK_SIGNAL_FUNC(doSaveSettings), NULL);
        gtk_signal_connect(GTK_OBJECT(capplet), "cancel",
                           GTK_SIGNAL_FUNC(doCancelSettings), NULL);
        gtk_signal_connect(GTK_OBJECT(capplet), "help",
                           GTK_SIGNAL_FUNC(about_cb), NULL);


        setStateCfg(cfgStateWindow, curState);
        setOptionsCfg(cfgOptionsWindow, curState);

        gtk_widget_show_all(capplet);
}

static void 
run_error_dialog(gchar *mesg,...) 
{
        char tmp[80];
        va_list ap;

        va_start(ap,mesg);
        vsnprintf(tmp,79,mesg,ap);
        dialogWindow = gnome_message_box_new(mesg,GNOME_MESSAGE_BOX_ERROR,GNOME_STOCK_BUTTON_OK,NULL);
        gnome_dialog_run_and_close(GNOME_DIALOG(dialogWindow));
        va_end(ap);
}

static gint 
get_pilot_id_from_gpilotd() 
{
	GList *pilots=NULL;
	gint pilot;
	int i,err;
  
	i=0;
	/* we don't worry about leaking here, so pilots isn't freed */
	switch(err = gnome_pilot_client_get_pilots(gpc,&pilots)) {
	case GPILOTD_OK: {
		if(pilots) {
			for(i=0;i<g_list_length(pilots);i++) {
				g_message("pilot %d = \"%s\"",i,(gchar*)g_list_nth(pilots,i)->data); 
			}
			if(i==0) {
				run_error_dialog(_("No pilot configured, please choose the\n'Pilot Link Properties' capplet first."));
				return -1;
			} else {
				gnome_pilot_client_get_pilot_id_by_name(gpc,
									pilots->data,  /* this is the first pilot */
									&pilot);
				if(i>1) {
					g_message("too many pilots...");
					/* need a choose here */
				}
				return pilot;
			}
		} else {
			run_error_dialog(_("No pilot configured, please choose the\n'Pilot Link Properties' capplet first."));
			return -1;
		}    
		break;
	}
	case GPILOTD_ERR_NOT_CONNECTED:
		run_error_dialog(_("Not connected to the gnome-pilot daemon"));
		return -1;
		break;
	default:
		g_warning("gnome_pilot_client_get_pilot_ids(...) = %d",err);
		run_error_dialog(_("An error occured when trying to fetch\npilot list from the gnome-pilot daemon"));
		return -1;
		break;
	}
}

int
main( int argc, char *argv[] )
{
	bindtextdomain (PACKAGE, GNOMELOCALEDIR);
	textdomain (PACKAGE);

        /* we're a capplet */
        gnome_capplet_init ("Expense conduit control applet", NULL, argc, argv, 
                            NULL,
                            0, NULL);

  	gpc = GNOME_PILOT_CLIENT(gnome_pilot_client_new());
	gnome_pilot_client_connect_to_daemon(gpc);
        pilotId = get_pilot_id_from_gpilotd();
        if(pilotId==-1) return -1;

        /* put all code to set things up in here */

        load_configuration(&origState,pilotId);
        curState = dupe_configuration(origState);

        conduit = gnome_pilot_conduit_management_new("gpexpense1",GNOME_PILOT_CONDUIT_MGMT_ID);
        conduit_config = gnome_pilot_conduit_config_new(conduit,pilotId);
        org_activation_state = activated = gnome_pilot_conduit_config_is_enabled(conduit_config,NULL);

        pilot_capplet_setup();

        /* done setting up, now run main loop */
        capplet_gtk_main();
        return 0;
}    
