/* $Id: backup_conduit.c,v 1.26 2000/03/20 14:50:29 eskil Exp $ */

#include <glib.h>
#include <gnome.h>

#include <pi-source.h>
#include <pi-socket.h>
#include <pi-file.h>
#include <pi-dlp.h>
#include <pi-version.h>

#include <sys/stat.h>
#include <sys/types.h>
#include <utime.h>
#include <unistd.h>
#include <pwd.h>
#include <signal.h>
#include <errno.h>

#include <gpilotd/gnome-pilot-conduit-backup.h>
#include "backup_conduit.h"

GnomePilotConduit *conduit_get_gpilot_conduit (guint32 pilotId);
void conduit_destroy_gpilot_conduit (GnomePilotConduit *conduit);

static void 
load_configuration(ConduitCfg **c,guint32 pilotId) 
{
	gchar *prefix;
	gchar **exclude_files;
	gint num_of_exclude_files = 0;
	guint i;

	*c = g_new0(ConduitCfg,1);
	(*c)->child = -1;
	
	prefix = g_strdup_printf("/gnome-pilot.d/backup-conduit/Pilot_%u/",pilotId);
	
	gnome_config_push_prefix(prefix);
	(*c)->backup_dir = gnome_config_get_string("backup_dir");
	(*c)->updated_only = gnome_config_get_bool("updated_only=TRUE");
	(*c)->remove_deleted = gnome_config_get_bool("remove_deleted=FALSE");	

	(*c)->exclude_files = NULL;
	gnome_config_get_vector("exclude_files",
				&num_of_exclude_files,
				&exclude_files);
	if(num_of_exclude_files) {
		for( i = 0; i < num_of_exclude_files ; i++ ) {
			(*c)->exclude_files = g_list_append( (*c)->exclude_files , 
							     g_strdup(exclude_files[i]));
			g_free(exclude_files[i]);
		}
		g_free(exclude_files);
	}
	gnome_config_pop_prefix();
	
	(*c)->files_in_backup = NULL;
	if((*c)->backup_dir != NULL && 
	   mkdir((*c)->backup_dir,(mode_t)0755) < 0) { /* Wow, I never though I would
							  use octal in C :) */
		if(errno != EEXIST) {
			/* YECH! 
			   CONDUIT_CFG(c.gpilotd_methods)->log_error("Cannot open whatever...");
			*/
		}
	}    
	(*c)->pilotId = pilotId;
	g_free(prefix);
}
/** this method frees all data from the conduit config */
static void 
destroy_configuration(ConduitCfg **c) 
{
	g_return_if_fail(c!=NULL);
	g_return_if_fail(*c!=NULL);

	if((*c)->remove_deleted) {
		g_list_free((*c)->files_in_backup);
	}
	
	g_list_foreach((*c)->exclude_files,(GFunc)g_free,NULL);
	g_list_free((*c)->exclude_files);
	g_free((*c)->backup_dir);
	g_free(*c);
	*c = NULL;
}

/* Helper functions */
static void
protect_name(char *d, char *s) {
	while(*s) {
		switch(*s) {
		case '/': *(d++) = '='; *(d++) = '2'; *(d++) = 'F'; break;
		case '=': *(d++) = '='; *(d++) = '3'; *(d++) = 'D'; break;
		case '\x0A': *(d++) = '='; *(d++) = '0'; *(d++) = 'A'; break;
		case '\x0D': *(d++) = '='; *(d++) = '0'; *(d++) = 'D'; break;
#if 0
		case ' ': *(d++) = '='; *(d++) = '2'; *(d++) = '0'; break;
#endif
		default: *(d++) = *s;
		}
		++s;
	}
	*d = '\0';
}

/* Signals */
static gint
gnome_real_pilot_conduit_backup_backup_prc (GnomePilotConduitBackup *conduit,
					    GnomePilotDBInfo *dbinfo, 
					    ConduitCfg *cfg)
{
	g_return_val_if_fail (conduit != NULL, -1);
	g_return_val_if_fail (dbinfo != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PILOT_CONDUIT_BACKUP (conduit), -1);
	return 0;
}
static gint
gnome_real_pilot_conduit_backup_backup_db (GnomePilotConduitBackup *conduit,
					   GnomePilotDBInfo *dbinfo, 
					   ConduitCfg *cfg)
{
	g_return_val_if_fail (conduit != NULL, -1);
	g_return_val_if_fail (dbinfo != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PILOT_CONDUIT_BACKUP (conduit), -1);
	return 0;
}

static gint
gnome_real_pilot_conduit_backup_backup (GnomePilotConduitBackup *conduit,
					GnomePilotDBInfo *dbinfo, 
					ConduitCfg *cfg)
{
	char name[256];      /* FIXME: potentiel overun  -  (UGH, this sux -jrb) */
	struct pi_file *f;
	struct stat statb;
	struct utimbuf times;
	GList *iterator;

	g_return_val_if_fail (conduit != NULL, -1);
	g_return_val_if_fail (dbinfo != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PILOT_CONDUIT_BACKUP (conduit), -1);
	
	/* let's see first if we should actually touch this db */
	for (iterator = cfg->exclude_files; iterator; iterator = iterator->next) {
		if( ! g_strcasecmp(iterator->data, PI_DBINFO (dbinfo)->name) ) {
			g_message("excluded %s",PI_DBINFO (dbinfo)->name);
			return 0;
		}
	}

	if (cfg->backup_dir == NULL) {
		g_warning("backup conduit has no backupdir");
		gnome_pilot_conduit_error(GNOME_PILOT_CONDUIT(conduit),
					  "No backup directory specified");
		return -1;
	} 

	strcpy (name, cfg->backup_dir);
	if (cfg->backup_dir[strlen(cfg->backup_dir)-1] != '/')
		strcat (name,"/");
	protect_name (name + strlen (name), PI_DBINFO (dbinfo)->name);

	if (PI_DBINFO (dbinfo)->flags & dlpDBFlagResource)
		strcat (name,".prc");
	else
		strcat (name,".pdb");

	if (cfg->remove_deleted) {
		GList *link;

		link = g_list_find_custom (cfg->files_in_backup,
					   name,
					   (GCompareFunc) g_strcasecmp);
		cfg->files_in_backup =
			g_list_remove_link (cfg->files_in_backup,
					    link);
	}
	if (cfg->updated_only) {
		if (stat (name, &statb) == 0) {
			if (PI_DBINFO (dbinfo)->modifyDate == statb.st_mtime) {
				g_message(_("%s not modified since last sync"),
					  PI_DBINFO (dbinfo)->name);
#if 0
				gnome_pilot_conduit_message(GNOME_PILOT_CONDUIT(conduit),
							    _("%s not modified since last sync"),
							    PI_DBINFO (dbinfo)->name);
#endif
				
				return 0;
			}
		} 
	}

	PI_DBINFO (dbinfo)->flags &= 0xff;

	g_message(_("Making backup of %s"),PI_DBINFO (dbinfo)->name);
	gnome_pilot_conduit_message(GNOME_PILOT_CONDUIT(conduit),
				    _("Making backup of %s"),PI_DBINFO (dbinfo)->name);

	f = pi_file_create (name, PI_DBINFO (dbinfo));
	if(f==0) {
		g_warning(_("Could not create backup file %s"),name);
		gnome_pilot_conduit_error (GNOME_PILOT_CONDUIT(conduit),
					   _("Could not create backup file %s"),name);
		return -1;
	}

	if(pi_file_retrieve (f, dbinfo->pilot_socket, 0) != 0) {
		/* failed!? */
		pi_file_close (f);
		g_warning(_("Backup of %s failed!"),PI_DBINFO (dbinfo)->name);
		gnome_pilot_conduit_error (GNOME_PILOT_CONDUIT(conduit),
					   _("Backup of %s failed!"),
					   PI_DBINFO (dbinfo)->name);
		return -1;
	} else {
		/* ok */
		pi_file_close (f);
		times.actime = PI_DBINFO (dbinfo)->createDate;
		times.modtime = PI_DBINFO (dbinfo)->modifyDate;
		utime (name, &times);
		return 0;
	}
}

GnomePilotConduit *
conduit_get_gpilot_conduit (guint32 pilotId)
{
	ConduitCfg *cfg;
	GtkObject *retval;

	retval = gnome_pilot_conduit_backup_new ();
	g_assert (retval != NULL);
	load_configuration(&cfg,pilotId);

	gtk_object_set_data(retval,"configuration",cfg);

	gtk_signal_connect (retval, "backup", (GtkSignalFunc) gnome_real_pilot_conduit_backup_backup, cfg);
	gtk_signal_connect (retval, "backup_prc", (GtkSignalFunc) gnome_real_pilot_conduit_backup_backup_prc, cfg);
	gtk_signal_connect (retval, "backup_db", (GtkSignalFunc) gnome_real_pilot_conduit_backup_backup_db, cfg);

	return GNOME_PILOT_CONDUIT (retval);
}

void
conduit_destroy_gpilot_conduit (GnomePilotConduit *conduit)
{
	ConduitCfg *cfg;
	cfg = GET_CFG(conduit);
	destroy_configuration(&cfg);
	gtk_object_destroy (GTK_OBJECT (conduit));
}

