/* Control applet ("capplet") for the gnome-pilot backup conduit,           */
/* based on                                                                 */
/* gpilotd control applet ('capplet') for use with the GNOME control center */


#include <pwd.h>
#include <sys/types.h>
#include <signal.h>
#include <gnome.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <ctype.h>

#include <config.h>
#include <capplet-widget.h>

#include <gpilotd/gnome-pilot-client.h>
#include <libgpilotdCM/gnome-pilot-conduit-management.h>
#include <libgpilotdCM/gnome-pilot-conduit-config.h>
#include "backup_conduit.h"

/* tell changes callbacks to ignore changes or not */
static gboolean ignore_changes = FALSE;

/* capplet widget */
static GtkWidget *capplet = NULL;

/* host/device/pilot configuration windows */
GtkWidget *cfgOptionsWindow = NULL;
GtkWidget *cfgStateWindow = NULL;
GtkWidget *dialogWindow = NULL;

gboolean activated,org_activation_state;
GnomePilotConduitManagement *conduit;
GnomePilotConduitConfig *conduit_config;

ConduitCfg *origState = NULL;
ConduitCfg *curState = NULL;


static void doTrySettings(GtkWidget *widget, gpointer whatever);
static void doRevertSettings(GtkWidget *widget, gpointer whatever);
static void doSaveSettings(GtkWidget *widget, gpointer whatever);
static void doCancelSettings(GtkWidget *widget, gpointer whatever);

static void readOptionsCfg(GtkWidget *w, ConduitCfg *state);
static void setOptionsCfg(GtkWidget *w, ConduitCfg *state);
static void readStateCfg(GtkWidget *w, ConduitCfg *state,gboolean *);
static void setStateCfg(GtkWidget *w, ConduitCfg *state,gboolean);

gint pilotId;
static GnomePilotClient *gpc;

CORBA_Environment ev;

static void 
load_configuration(ConduitCfg **c,guint32 pilotId) 
{
	gchar *prefix;
	gchar *exclude;
	gchar **excs;
	guint i;

	*c = g_new0(ConduitCfg,1);
	(*c)->child = -1;
	
	prefix = g_strdup_printf("/gnome-pilot.d/backup-conduit/Pilot_%u/",pilotId);
	
	gnome_config_push_prefix(prefix);
	(*c)->backup_dir = gnome_config_get_string("backup_dir");
	(*c)->updated_only = gnome_config_get_bool("updated_only=TRUE");
	(*c)->remove_deleted = gnome_config_get_bool("remove_deleted=FALSE");	
	exclude = gnome_config_get_string("exclude_files");
	if(exclude != NULL) {
	  (*c)->exclude_files = NULL;
	  excs = g_strsplit( exclude, ",", 0 );
	  for( i = 0; excs[i] != NULL ; i++ ) {
	    (*c)->exclude_files = g_list_insert_sorted( (*c)->exclude_files , 
							g_strdup(excs[i]),
							(GCompareFunc)g_strcasecmp);
	  }
	  g_strfreev(excs);
	}
	gnome_config_pop_prefix();
	
	(*c)->files_in_backup = NULL;

	if ((*c)->backup_dir == NULL) {
		gchar *tmp;
		gnome_pilot_client_get_pilot_base_dir_by_id(gpc,pilotId,&tmp);
		(*c)->backup_dir = g_strdup_printf("%s/backup/",tmp);
		g_free(tmp);
	}
	
	if(mkdir((*c)->backup_dir,(mode_t)0755) < 0) { /* Wow, I never though I would
							  use octal in C :) */
		if(errno != EEXIST) {
			/* YECH! 
			   CONDUIT_CFG(c.gpilotd_methods)->log_error("Cannot open whatever...");
			*/
		}
	}    
	(*c)->pilotId = pilotId;
	g_free(prefix);
}

static void 
save_configuration(ConduitCfg *c) 
{
	gchar *prefix;
	gchar **exclude;
	gchar *exc=NULL;
	GList *iterator;
	guint i;

	g_return_if_fail(c!=NULL);
       	
	prefix= g_strdup_printf("/gnome-pilot.d/backup-conduit/Pilot_%u/",c->pilotId);
	
	if( c->exclude_files != NULL ) {
	  iterator = c->exclude_files;
	  exclude = g_malloc( sizeof(char *) * (g_list_length(iterator)+1) );
	  for( i=0 ; iterator != NULL ; iterator = iterator->next, i++ ) {
	    exclude[i] = iterator->data;
	  }
	  exclude[i] = NULL;
	  exc = g_strjoinv( ",", exclude);
	  g_free(exclude);
	}
	gnome_config_push_prefix(prefix);
	gnome_config_set_string("backup_dir",c->backup_dir);
	gnome_config_set_bool("updated_only",c->updated_only);
	gnome_config_set_bool("remove_deleted",c->remove_deleted);
	gnome_config_set_string("exclude_files",exc);
	gnome_config_pop_prefix();
	gnome_config_sync();
	gnome_config_drop_all();
	g_free(prefix);
	g_free(exc);
}
static void 
copy_configuration(ConduitCfg *d, ConduitCfg *c)
{
        g_return_if_fail(c!=NULL);
        g_return_if_fail(d!=NULL);
	if(d->backup_dir) g_free(d->backup_dir);
	d->backup_dir = g_strdup(c->backup_dir);
	d->remove_deleted = c->remove_deleted;
	d->updated_only = c->updated_only;
	d->pilotId = c->pilotId;
	d->child = c->child;
	if(d->exclude_files) g_list_free(d->exclude_files);
	d->exclude_files = g_list_copy(c->exclude_files);
	if(d->files_in_backup) g_list_free(d->files_in_backup);
	d->files_in_backup = g_list_copy(c->files_in_backup);
}

static ConduitCfg*
dupe_configuration(ConduitCfg *c) 
{
	ConduitCfg *d;
	g_return_val_if_fail(c!=NULL,NULL);
	d = g_new0(ConduitCfg,1);
	copy_configuration(d,c);
	return d;
}

/** this method frees all data from the conduit config */
static void 
destroy_configuration(ConduitCfg **c) 
{
	g_return_if_fail(c!=NULL);
	g_return_if_fail(*c!=NULL);

	if((*c)->remove_deleted) {
		g_list_free((*c)->files_in_backup);
	}
	
	g_list_free((*c)->exclude_files);
	g_free((*c)->backup_dir);
	g_free(*c);
	*c = NULL;
}

static void 
setSettings(ConduitCfg* conduitCfg, gboolean active)
{
    if(active)
      gnome_pilot_conduit_config_enable(conduit_config,GnomePilotConduitSyncTypeCustom);
    else
      gnome_pilot_conduit_config_disable(conduit_config);

    save_configuration(conduitCfg);
}

static void
doTrySettings(GtkWidget *widget, gpointer whatever)
{
	readStateCfg(cfgStateWindow, curState,&activated);
	readOptionsCfg(cfgOptionsWindow, curState);
	setSettings(curState,activated);
}

static void
doSaveSettings(GtkWidget *widget, gpointer whatever)
{
	doTrySettings(widget, whatever);
}

static void
doCancelSettings(GtkWidget *widget, gpointer whatever)
{
        setSettings(origState,org_activation_state);
}

static void
doRevertSettings(GtkWidget *widget, gpointer whatever)
{
	activated = org_activation_state;
	copy_configuration(curState,origState);
	setOptionsCfg(cfgOptionsWindow, curState);
	setStateCfg(cfgStateWindow, curState,activated);
	setSettings(curState,activated);
}

static void
insert_dir_callback (GtkEditable    *editable, const gchar    *text,
		     gint len, gint *position, void *data)
{
	gint i;
	gchar *curname;

	curname = gtk_entry_get_text(GTK_ENTRY(editable));
	if (*curname == '\0' && len > 0) {
		if (isspace(text[0])) {
			gtk_signal_emit_stop_by_name(GTK_OBJECT(editable), "insert_text");
			return;
		}
	} else {
		for (i=0; i<len; i++) {
			if (isspace(text[i])) {
				gtk_signal_emit_stop_by_name(GTK_OBJECT(editable), 
							     "insert_text");
				return;
			}
		}
	}
}
static void
statechange_cb(GtkEditable    *editable, const gchar    *text,
		     gint            length, gint           *position,
		     void *data)
{
	if (!ignore_changes)
		capplet_widget_state_changed(CAPPLET_WIDGET(capplet), TRUE);
}

static void about_cb (GtkWidget *widget, gpointer data) {
	GtkWidget *about;
	const gchar *authors[] = {_("Eskil Heyn Olsen <deity@eskil.dk>"),NULL};
  
	about = gnome_about_new(_("Gpilotd backup conduit"), VERSION,
				_("(C) 1998 the Free Software Foundation"),
				authors,
				_("Configuration utility for the backup conduit.\n"
				  "The backup conduit is responsible for copying "
				  "files from the pilot to the local host, without "
				  "doing any conversion, whereby ensuring that a "
				  "proper restore can be done"),
				_("gnome-unknown.xpm"));
	gtk_widget_show (about);
  
	return;
}

static void toggled_cb(GtkWidget *widget, gpointer data) 
{
  if(!ignore_changes) {
	gtk_widget_set_sensitive(cfgOptionsWindow,GTK_TOGGLE_BUTTON(widget)->active);
	capplet_widget_state_changed(CAPPLET_WIDGET(capplet), TRUE);
  }
}

static void button_toggled_cb(GtkWidget *widget, gpointer data)
{
   if(!ignore_changes) {
	capplet_widget_state_changed(CAPPLET_WIDGET(capplet), TRUE);
   }
}
static GtkWidget
*createStateCfgWindow(void)
{
	GtkWidget *vbox, *table;
	GtkWidget *label;
	GtkWidget *button;

	vbox = gtk_vbox_new(FALSE, GNOME_PAD);

	table = gtk_table_new(2, 2, FALSE);
	gtk_table_set_row_spacings(GTK_TABLE(table), 4);
	gtk_table_set_col_spacings(GTK_TABLE(table), 10);
	gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, GNOME_PAD);

	label = gtk_label_new(_("Enabled"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1,2);

	button = gtk_check_button_new();
	gtk_object_set_data(GTK_OBJECT(vbox), "conduit_on_off", button);
	gtk_signal_connect(GTK_OBJECT(button), "toggled",
			   GTK_SIGNAL_FUNC(toggled_cb),
			   NULL);
	gtk_table_attach_defaults(GTK_TABLE(table), button, 1, 2, 1,2);

	return vbox;
}

static void
setStateCfg(GtkWidget *cfg, ConduitCfg *state, gboolean active)
{
	GtkWidget *button;

	button = gtk_object_get_data(GTK_OBJECT(cfg), "conduit_on_off");

	g_assert(button!=NULL);

	ignore_changes = TRUE;
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button),active);
	gtk_widget_set_sensitive(cfgOptionsWindow,active);
	ignore_changes = FALSE;
}


static void
readStateCfg(GtkWidget *cfg, ConduitCfg *state, gboolean *active)
{
	GtkWidget *button;

	button  = gtk_object_get_data(GTK_OBJECT(cfg), "conduit_on_off");
  
	g_assert(button!=NULL);

	*active = GTK_TOGGLE_BUTTON(button)->active;
}

/* object data:                        */
/* "userid"   - entry box for pilot user id */
/* "username" - entry box for pilot user name */
static GtkWidget
*createCfgWindow(void)
{
	GtkWidget *vbox, *table;
	GtkWidget *entry, *label;
	GtkWidget *button;

	vbox = gtk_vbox_new(FALSE, GNOME_PAD);

	table = gtk_table_new(2, 3, FALSE);
	gtk_table_set_row_spacings(GTK_TABLE(table), 4);
	gtk_table_set_col_spacings(GTK_TABLE(table), 10);
	gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, GNOME_PAD);

	label = gtk_label_new(_("Backup directory"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 0, 1);
    
	entry = gtk_entry_new_with_max_length(128);
	gtk_object_set_data(GTK_OBJECT(vbox), "dir", entry);
	gtk_table_attach_defaults(GTK_TABLE(table), entry, 1, 2, 0, 1);
	gtk_signal_connect(GTK_OBJECT(entry), "insert_text",
			   GTK_SIGNAL_FUNC(insert_dir_callback),
			   NULL);
	gtk_signal_connect_after(GTK_OBJECT(entry), "insert_text",
				 GTK_SIGNAL_FUNC(statechange_cb),
				 NULL);
	gtk_signal_connect_after(GTK_OBJECT(entry), "delete_text",
				 GTK_SIGNAL_FUNC(statechange_cb),
				 NULL);

	label = gtk_label_new(_("Only backup changed bases"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1,2);

	button = gtk_check_button_new();
	gtk_object_set_data(GTK_OBJECT(vbox), "only_changed", button);
	gtk_table_attach_defaults(GTK_TABLE(table), button, 1, 2, 1,2);
	gtk_signal_connect(GTK_OBJECT(button), "toggled",
			   GTK_SIGNAL_FUNC(button_toggled_cb),
			   NULL);

	label = gtk_label_new(_("Remove local base if deleted on pilot"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 2,3);

	button = gtk_check_button_new();
	gtk_object_set_data(GTK_OBJECT(vbox), "remove_local", button);
	gtk_table_attach_defaults(GTK_TABLE(table), button, 1, 2, 2,3);
	gtk_signal_connect(GTK_OBJECT(button), "toggled",
			   GTK_SIGNAL_FUNC(button_toggled_cb),
			   NULL);


	return vbox;
}

static void
setOptionsCfg(GtkWidget *pilotcfg, ConduitCfg *state)
{
	GtkWidget *dir,*updated_only,*remove_deleted;

	dir  = gtk_object_get_data(GTK_OBJECT(pilotcfg), "dir");
	updated_only = gtk_object_get_data(GTK_OBJECT(pilotcfg), "only_changed");
	remove_deleted = gtk_object_get_data(GTK_OBJECT(pilotcfg), "remove_local");

	g_assert(dir!=NULL);
	g_assert(updated_only!=NULL);
	g_assert(remove_deleted!=NULL);

	ignore_changes = TRUE;

	gtk_entry_set_text(GTK_ENTRY(dir), state->backup_dir);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(updated_only), state->updated_only);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(remove_deleted), state->remove_deleted);

	ignore_changes = FALSE;
}


static void
readOptionsCfg(GtkWidget *pilotcfg, ConduitCfg *state)
{
	GtkWidget *dir,*updated_only,*remove_deleted;

	dir  = gtk_object_get_data(GTK_OBJECT(pilotcfg), "dir");
	updated_only = gtk_object_get_data(GTK_OBJECT(pilotcfg), "only_changed");
	remove_deleted = gtk_object_get_data(GTK_OBJECT(pilotcfg), "remove_local");
  
	state->backup_dir = g_strdup(gtk_entry_get_text(GTK_ENTRY(dir)));
	state->updated_only = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(updated_only));
	state->remove_deleted = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(remove_deleted));
}

static void
pilot_capplet_setup(void)
{
	GtkWidget *frame, *table;

	capplet = capplet_widget_new();

	table = gtk_table_new(1, 2, FALSE);
	gtk_container_border_width(GTK_CONTAINER(table), GNOME_PAD);
	gtk_container_add(GTK_CONTAINER(capplet), table); 

	frame = gtk_frame_new(_("Conduit state"));
	gtk_container_border_width(GTK_CONTAINER(frame), GNOME_PAD_SMALL);
	gtk_table_attach_defaults(GTK_TABLE(table), frame, 0, 1, 0, 1);
	cfgStateWindow = createStateCfgWindow();
	gtk_container_add(GTK_CONTAINER(frame), cfgStateWindow);

	frame = gtk_frame_new(_("Backup options"));
	gtk_container_border_width(GTK_CONTAINER(frame), GNOME_PAD_SMALL);
	gtk_table_attach_defaults(GTK_TABLE(table), frame, 0, 1, 1, 2);
	cfgOptionsWindow = createCfgWindow();
	gtk_container_add(GTK_CONTAINER(frame), cfgOptionsWindow);

	gtk_signal_connect(GTK_OBJECT(capplet), "try",
			   GTK_SIGNAL_FUNC(doTrySettings), NULL);
	gtk_signal_connect(GTK_OBJECT(capplet), "revert",
			   GTK_SIGNAL_FUNC(doRevertSettings), NULL);
	gtk_signal_connect(GTK_OBJECT(capplet), "ok",
			   GTK_SIGNAL_FUNC(doSaveSettings), NULL);
	gtk_signal_connect(GTK_OBJECT(capplet), "cancel",
			   GTK_SIGNAL_FUNC(doCancelSettings), NULL);
	gtk_signal_connect(GTK_OBJECT(capplet), "help",
			   GTK_SIGNAL_FUNC(about_cb), NULL);


	setStateCfg(cfgStateWindow, curState,activated);
	setOptionsCfg(cfgOptionsWindow, curState);

	gtk_widget_show_all(capplet);
}

static void run_error_dialog(gchar *mesg,...) {
	char tmp[80];
	va_list ap;

	va_start(ap,mesg);
	vsnprintf(tmp,79,mesg,ap);
	dialogWindow = gnome_message_box_new(mesg,GNOME_MESSAGE_BOX_ERROR,GNOME_STOCK_BUTTON_OK,NULL);
	gnome_dialog_run_and_close(GNOME_DIALOG(dialogWindow));
	va_end(ap);
}

static gint 
get_pilot_id_from_gpilotd() 
{
	GList *pilots=NULL;
	gint pilot;
	int i,err;
  
	i=0;
	/* we don't worry about leaking here, so pilots isn't freed */
	switch(err = gnome_pilot_client_get_pilots(gpc,&pilots)) {
	case GPILOTD_OK: {
		if(pilots) {
			for(i=0;i<g_list_length(pilots);i++) {
				g_message("pilot %d = \"%s\"",i,(gchar*)g_list_nth(pilots,i)->data); 
			}
			if(i==0) {
				run_error_dialog(_("No pilot configured, please choose the\n'Pilot Link Properties' capplet first."));
				return -1;
			} else {
				gnome_pilot_client_get_pilot_id_by_name(gpc,
									pilots->data,  /* this is the first pilot */
									&pilot);
				if(i>1) {
					g_message("too many pilots...");
					/* need a choose here */
				}
				return pilot;
			}
		} else {
			run_error_dialog(_("No pilot configured, please choose the\n'Pilot Link Properties' capplet first."));
			return -1;
		}    
		break;
	}
	case GPILOTD_ERR_NOT_CONNECTED:
		run_error_dialog(_("Not connected to the gnome-pilot daemon"));
		return -1;
		break;
	default:
		g_warning("gnome_pilot_client_get_pilot_ids(...) = %d",err);
		run_error_dialog(_("An error occured when trying to fetch\npilot list from the gnome-pilot daemon"));
		return -1;
		break;
	}
}

int
main( int argc, char *argv[] )
{
	bindtextdomain (PACKAGE, GNOMELOCALEDIR);
	textdomain (PACKAGE);

	/* we're a capplet */
	gnome_capplet_init ("backup conduit control applet", NULL, argc, argv, 
			    NULL,
			    0, NULL);

	gpc = gnome_pilot_client_new();
	gnome_pilot_client_connect_to_daemon(gpc);
	pilotId = get_pilot_id_from_gpilotd();
	if(pilotId==-1) return -1;

	/* Open the conduit management for "gpbackup1" which is
	   the id for the backup conduit, as found in backup.conduit */
	conduit = gnome_pilot_conduit_management_new("gpbackup1",GNOME_PILOT_CONDUIT_MGMT_ID);
	if (conduit==NULL) {
		g_warning("Could not create object for conduit. The .conduit file may be missing.");
		run_error_dialog(_("Could not create object for conduit.\n"
				   "Perhaps the conduit isn't properly installed ?"));
		return -1;
	}

	conduit_config = gnome_pilot_conduit_config_new(conduit,pilotId);
	load_configuration(&origState,pilotId);
	curState = dupe_configuration(origState);

	org_activation_state = activated = gnome_pilot_conduit_config_is_enabled(conduit_config,NULL);
    
	pilot_capplet_setup();

	/* done setting up, now run main loop */
	capplet_gtk_main();

	gnome_pilot_conduit_management_destroy(conduit);
	destroy_configuration(&origState);
	destroy_configuration(&curState);

	return 0;
}    
