#ifndef __EXPENSE_CONDUIT_H__
#define __EXPENSE_CONDUIT_H__

#include <unistd.h>
#include <pi-expense.h>


enum ExpenseOutputFormat { 
  eSimpleFormat, eComplexFormat
};

typedef struct ConduitCfg {
  gchar *dir;
  gchar *dateFormat;
  mode_t dirMode;
  mode_t fileMode;
  enum ExpenseOutputFormat outputFormat;

  guint32 pilotId;
  pid_t child;
} ConduitCfg;

typedef struct ConduitData {
  struct ExpenseAppInfo ai;
  struct ExpensePref pref;
  GnomePilotDBInfo *dbi;
} ConduitData;

#define GET_CONFIG(c) ((ConduitCfg*)(gtk_object_get_data(GTK_OBJECT(c),"conduit_config")))
#define GET_CONDUIT_DATA(c) ((ConduitData*)gtk_object_get_data(GTK_OBJECT(c),"conduit_data"))

/* #define EC_DEBUG */
#ifdef EC_DEBUG
#define LOG(format,args...) g_log (G_LOG_DOMAIN, \
                                   G_LOG_LEVEL_MESSAGE, \
                                   "expense: "##format, ##args)
#else
#define LOG(format,args...)
#endif

static void 
load_configuration(ConduitCfg **c,guint32 pilotId) 
{
	gchar *prefix;
	gchar *tempbuf;

	g_assert(c!=NULL);
	*c = g_new0(ConduitCfg,1);
	(*c)->child = -1;

	prefix = g_strdup_printf("/gnome-pilot.d/expense-conduit/Pilot_%u/",pilotId);
  
	gnome_config_push_prefix(prefix);
	(*c)->dir = gnome_config_get_string( "dir");
	(*c)->dateFormat = gnome_config_get_string( "date_format=%x");
	(*c)->outputFormat = gnome_config_get_int("output_format=0");
	tempbuf = gnome_config_get_string("dir mode=0700");
	(*c)->dirMode =(mode_t)strtol(tempbuf,NULL,0);
	g_free(tempbuf);
	tempbuf = gnome_config_get_string("file mode=0600");
	(*c)->fileMode =(mode_t)strtol(tempbuf,NULL,0);
	g_free(tempbuf);

	gnome_config_pop_prefix();

	(*c)->pilotId = pilotId;

	/* make a default directory if nothing was defined */
	if((*c)->dir == NULL) {
		tempbuf =gpilotd_get_pilot_basedir_by_id(pilotId);
		(*c)->dir =g_strdup_printf ("%s/expense", tempbuf);
		g_free(tempbuf);
	}
	g_free(prefix);
}


static void 
save_configuration(ConduitCfg *c) 
{
	gchar *prefix;

	g_assert(c!=NULL);

	prefix = g_strdup_printf("/gnome-pilot.d/expense-conduit/Pilot_%u/",c->pilotId);

#ifdef EC_DEBUG
	fprintf(stderr,"expense_conduit: savecfg: prefix = %s\n",prefix);
#endif

	gnome_config_push_prefix(prefix);
	gnome_config_set_string("dir", c->dir);
	gnome_config_set_string("date_format", c->dateFormat);
	gnome_config_set_int("output_format", c->outputFormat);
	g_snprintf(prefix,255,"0%o", c->dirMode);
	gnome_config_set_string("dir mode", prefix);
	g_snprintf(prefix,255,"0%o", c->fileMode);
	gnome_config_set_string("file mode", prefix);

	gnome_config_pop_prefix();

	gnome_config_sync();
	gnome_config_drop_all();
	g_free(prefix);
}

static ConduitCfg*
dupe_configuration(ConduitCfg *c) {
	ConduitCfg *retval;
	g_assert(c!=NULL);
	retval = g_new0(ConduitCfg,1);
	retval->dir = g_strdup(c->dir);
	retval->dateFormat = g_strdup(c->dateFormat);
	retval->outputFormat = c->outputFormat;
	retval->dirMode = c->dirMode;
	retval->fileMode = c->fileMode;

	retval->pilotId = c->pilotId;

	return retval;
}

/** this method frees all data from the conduit config */
static void 
destroy_configuration(ConduitCfg **c) 
{
	g_assert(c!=NULL);
	g_assert(*c!=NULL);
	g_free( (*c)->dir);
	g_free( (*c)->dateFormat);
	g_free(*c);
	*c = NULL;
}

#endif /* __EXPENSE_CONDUIT_H__ */
