/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*-
 *
 * Copyright (C) 2007 William Jon McCann <mccann@jhu.edu>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include "config.h"

#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <errno.h>
#include <pwd.h>

#ifdef ENABLE_RBAC_SHUTDOWN
#include <auth_attr.h>
#include <secdb.h>
#endif

#include <glib.h>
#include <glib/gi18n.h>
#include <glib/gstdio.h>
#include <glib-object.h>

#include <gdk/gdkkeysyms.h>
#include <gdk/gdkx.h>

#include <gtk/gtk.h>

#include <glade/glade-xml.h>

#define DBUS_API_SUBJECT_TO_CHANGE
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>

#ifdef HAVE_POLKIT_GNOME
#include <polkit-gnome/polkit-gnome.h>
#endif

#include "gdm-greeter-login-window.h"
#include "gdm-user-chooser-widget.h"

#if HAVE_PAM
#include <security/pam_appl.h>
#define PW_ENTRY_SIZE PAM_MAX_RESP_SIZE
#else
#define PW_ENTRY_SIZE GDM_MAX_PASS
#endif

#define CK_NAME      "org.freedesktop.ConsoleKit"
#define CK_PATH      "/org/freedesktop/ConsoleKit"
#define CK_INTERFACE "org.freedesktop.ConsoleKit"

#define CK_MANAGER_PATH      "/org/freedesktop/ConsoleKit/Manager"
#define CK_MANAGER_INTERFACE "org.freedesktop.ConsoleKit.Manager"
#define CK_SEAT_INTERFACE    "org.freedesktop.ConsoleKit.Seat"
#define CK_SESSION_INTERFACE "org.freedesktop.ConsoleKit.Session"

#define GLADE_XML_FILE "gdm-greeter-login-window.glade"

#define GDM_GREETER_LOGIN_WINDOW_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GDM_TYPE_GREETER_LOGIN_WINDOW, GdmGreeterLoginWindowPrivate))

enum {
        MODE_SELECTION = 0,
        MODE_AUTHENTICATION
};

struct GdmGreeterLoginWindowPrivate
{
        GladeXML        *xml;
        GtkWidget       *user_chooser;
        gboolean         display_is_local;
        char            *timeformat;

        guint            animation_timeout_id;
};

enum {
        PROP_0,
        PROP_DISPLAY_IS_LOCAL,
};

enum {
        BEGIN_VERIFICATION,
        BEGIN_VERIFICATION_FOR_USER,
        QUERY_ANSWER,
        USER_SELECTED,
        DISCONNECTED,
        CANCELLED,
        LAST_SIGNAL
};

static guint signals [LAST_SIGNAL] = { 0, };

static void     gdm_greeter_login_window_class_init   (GdmGreeterLoginWindowClass *klass);
static void     gdm_greeter_login_window_init         (GdmGreeterLoginWindow      *greeter_login_window);
static void     gdm_greeter_login_window_finalize     (GObject                    *object);

G_DEFINE_TYPE (GdmGreeterLoginWindow, gdm_greeter_login_window, GTK_TYPE_WINDOW)

static void
set_busy (GdmGreeterLoginWindow *login_window)
{
        GdkCursor *cursor;

        cursor = gdk_cursor_new (GDK_WATCH);
        gdk_window_set_cursor (GTK_WIDGET (login_window)->window, cursor);
        gdk_cursor_unref (cursor);
}

static void
set_ready (GdmGreeterLoginWindow *login_window)
{
        GdkCursor *cursor;

        cursor = gdk_cursor_new (GDK_LEFT_PTR);
        gdk_window_set_cursor (GTK_WIDGET (login_window)->window, cursor);
        gdk_cursor_unref (cursor);
}

static void
set_sensitive (GdmGreeterLoginWindow *login_window,
               gboolean               sensitive)
{
        GtkWidget *box;

        box = glade_xml_get_widget (login_window->priv->xml, "auth-input-box");
        gtk_widget_set_sensitive (box, sensitive);

        box = glade_xml_get_widget (login_window->priv->xml, "buttonbox");
        gtk_widget_set_sensitive (box, sensitive);
}

static void
set_focus (GdmGreeterLoginWindow *login_window)
{
        GtkWidget *entry;

        entry = glade_xml_get_widget (GDM_GREETER_LOGIN_WINDOW (login_window)->priv->xml, "auth-prompt-entry");

        gdk_window_focus (GTK_WIDGET (login_window)->window, GDK_CURRENT_TIME);

        if (GTK_WIDGET_REALIZED (entry) && ! GTK_WIDGET_HAS_FOCUS (entry)) {
                gtk_widget_grab_focus (entry);
        }
}


static void
set_message (GdmGreeterLoginWindow *login_window,
             const char            *text)
{
        GtkWidget *label;

        label = glade_xml_get_widget (login_window->priv->xml, "auth-message-label");
        gtk_label_set_text (GTK_LABEL (label), text);
}

static void
show_widget (GdmGreeterLoginWindow *login_window,
             const char            *name,
             gboolean               visible)
{
        GtkWidget *widget;

        widget = glade_xml_get_widget (login_window->priv->xml, name);
        if (widget != NULL) {
                if (visible) {
                        gtk_widget_show (widget);
                } else {
                        gtk_widget_hide (widget);
                }
        }
}

static gboolean
get_show_restart_buttons (GdmGreeterLoginWindow *login_window)
{
        gboolean show;

        show = TRUE;

#ifdef ENABLE_RBAC_SHUTDOWN
        {
                char *username;

                username = g_get_user_name ();
                if (username == NULL || !chkauthattr (RBAC_SHUTDOWN_KEY, username)) {
                        show = FALSE;
                        g_debug ("Not showing stop/restart buttons for user %s due to RBAC key %s",
                                 username, RBAC_SHUTDOWN_KEY);
                } else {
                        g_debug ("Showing stop/restart buttons for user %s due to RBAC key %s",
                                 username, RBAC_SHUTDOWN_KEY);
                }
        }
#endif
        return show;
}

static void
switch_mode (GdmGreeterLoginWindow *login_window,
             int                    number)
{
        const char *default_name;
        GtkWidget  *user_chooser;
        GtkWidget  *box;
        gboolean    show_restart_shutdown = TRUE;

        /* FIXME: do animation */
        default_name = NULL;

        show_restart_shutdown = get_show_restart_buttons (login_window);

        switch (number) {
        case MODE_SELECTION:
                show_widget (login_window, "log-in-button", FALSE);
                show_widget (login_window, "cancel-button", FALSE);
                show_widget (login_window, "shutdown-button",
                             login_window->priv->display_is_local && show_restart_shutdown);
                show_widget (login_window, "restart-button",
                             login_window->priv->display_is_local && show_restart_shutdown);
                show_widget (login_window, "suspend-button", login_window->priv->display_is_local);
                show_widget (login_window, "disconnect-button", ! login_window->priv->display_is_local);
                show_widget (login_window, "auth-input-box", FALSE);
                default_name = NULL;
                break;
        case MODE_AUTHENTICATION:
                show_widget (login_window, "log-in-button", TRUE);
                show_widget (login_window, "cancel-button", TRUE);
                show_widget (login_window, "shutdown-button", FALSE);
                show_widget (login_window, "restart-button", FALSE);
                show_widget (login_window, "suspend-button", FALSE);
                show_widget (login_window, "disconnect-button", FALSE);
                default_name = "log-in-button";
                break;
        default:
                g_assert_not_reached ();
        }

        box = glade_xml_get_widget (login_window->priv->xml, "buttonbox");
        gtk_button_box_set_layout (GTK_BUTTON_BOX (box),
                                   (number == MODE_SELECTION) ? GTK_BUTTONBOX_SPREAD : GTK_BUTTONBOX_END );

        user_chooser = glade_xml_get_widget (login_window->priv->xml, "user-chooser");
        box = gtk_widget_get_parent (user_chooser);
        if (GTK_IS_BOX (box)) {
                guint       padding;
                GtkPackType pack_type;

                gtk_box_query_child_packing (GTK_BOX (box),
                                             user_chooser,
                                             NULL,
                                             NULL,
                                             &padding,
                                             &pack_type);
                gtk_box_set_child_packing (GTK_BOX (box),
                                           user_chooser,
                                           number == MODE_SELECTION,
                                           number == MODE_SELECTION,
                                           padding,
                                           pack_type);
        }

        if (default_name != NULL) {
                GtkWidget *widget;

                widget = glade_xml_get_widget (login_window->priv->xml, default_name);
                gtk_widget_grab_default (widget);
        }
}

static void
do_cancel (GdmGreeterLoginWindow *login_window)
{

        gdm_user_chooser_widget_set_chosen_user_name (GDM_USER_CHOOSER_WIDGET (login_window->priv->user_chooser), NULL);

        switch_mode (login_window, MODE_SELECTION);
        set_busy (login_window);
        set_sensitive (login_window, FALSE);

        g_signal_emit (login_window, signals[CANCELLED], 0);

        set_ready (login_window);
}

static void
reset_dialog (GdmGreeterLoginWindow *login_window)
{
        GtkWidget  *entry;
        GtkWidget  *label;

        g_debug ("GdmGreeterLoginWindow: Resetting dialog");

        entry = glade_xml_get_widget (GDM_GREETER_LOGIN_WINDOW (login_window)->priv->xml, "auth-prompt-entry");
        gtk_entry_set_text (GTK_ENTRY (entry), "");
        gtk_entry_set_visibility (GTK_ENTRY (entry), TRUE);

        label = glade_xml_get_widget (GDM_GREETER_LOGIN_WINDOW (login_window)->priv->xml, "auth-prompt-label");
        gtk_label_set_text (GTK_LABEL (label), "");

        set_message (login_window, "");

        gdm_user_chooser_widget_set_chosen_user_name (GDM_USER_CHOOSER_WIDGET (login_window->priv->user_chooser), NULL);

        switch_mode (login_window, MODE_SELECTION);

        set_sensitive (login_window, TRUE);
        set_ready (login_window);
        set_focus (GDM_GREETER_LOGIN_WINDOW (login_window));
}

gboolean
gdm_greeter_login_window_ready (GdmGreeterLoginWindow *login_window)
{
        g_return_val_if_fail (GDM_IS_GREETER_LOGIN_WINDOW (login_window), FALSE);

        set_sensitive (GDM_GREETER_LOGIN_WINDOW (login_window), TRUE);
        set_ready (GDM_GREETER_LOGIN_WINDOW (login_window));
        set_focus (GDM_GREETER_LOGIN_WINDOW (login_window));

        return TRUE;
}

gboolean
gdm_greeter_login_window_reset (GdmGreeterLoginWindow *login_window)
{
        g_return_val_if_fail (GDM_IS_GREETER_LOGIN_WINDOW (login_window), FALSE);

        reset_dialog (GDM_GREETER_LOGIN_WINDOW (login_window));

        return TRUE;
}

gboolean
gdm_greeter_login_window_info (GdmGreeterLoginWindow *login_window,
                               const char            *text)
{
        g_return_val_if_fail (GDM_IS_GREETER_LOGIN_WINDOW (login_window), FALSE);

        g_debug ("GdmGreeterLoginWindow: info: %s", text);

        set_message (GDM_GREETER_LOGIN_WINDOW (login_window), text);

        return TRUE;
}

gboolean
gdm_greeter_login_window_problem (GdmGreeterLoginWindow *login_window,
                                  const char            *text)
{
        g_return_val_if_fail (GDM_IS_GREETER_LOGIN_WINDOW (login_window), FALSE);

        g_debug ("GdmGreeterLoginWindow: problem: %s", text);

        set_message (GDM_GREETER_LOGIN_WINDOW (login_window), text);

        return TRUE;
}

gboolean
gdm_greeter_login_window_info_query (GdmGreeterLoginWindow *login_window,
                                     const char            *text)
{
        GtkWidget  *entry;
        GtkWidget  *label;

        g_return_val_if_fail (GDM_IS_GREETER_LOGIN_WINDOW (login_window), FALSE);

        g_debug ("GdmGreeterLoginWindow: info query: %s", text);

        entry = glade_xml_get_widget (GDM_GREETER_LOGIN_WINDOW (login_window)->priv->xml, "auth-prompt-entry");
        gtk_entry_set_text (GTK_ENTRY (entry), "");
        gtk_entry_set_visibility (GTK_ENTRY (entry), TRUE);

        label = glade_xml_get_widget (GDM_GREETER_LOGIN_WINDOW (login_window)->priv->xml, "auth-prompt-label");
        gtk_label_set_text (GTK_LABEL (label), text);

        show_widget (login_window, "auth-input-box", TRUE);
        set_sensitive (GDM_GREETER_LOGIN_WINDOW (login_window), TRUE);
        set_ready (GDM_GREETER_LOGIN_WINDOW (login_window));
        set_focus (GDM_GREETER_LOGIN_WINDOW (login_window));

        return TRUE;
}

gboolean
gdm_greeter_login_window_secret_info_query (GdmGreeterLoginWindow *login_window,
                                            const char            *text)
{
        GtkWidget  *entry;
        GtkWidget  *label;

        g_return_val_if_fail (GDM_IS_GREETER_LOGIN_WINDOW (login_window), FALSE);

        entry = glade_xml_get_widget (GDM_GREETER_LOGIN_WINDOW (login_window)->priv->xml, "auth-prompt-entry");
        gtk_entry_set_text (GTK_ENTRY (entry), "");
        gtk_entry_set_visibility (GTK_ENTRY (entry), FALSE);

        label = glade_xml_get_widget (GDM_GREETER_LOGIN_WINDOW (login_window)->priv->xml, "auth-prompt-label");
        gtk_label_set_text (GTK_LABEL (label), text);

        show_widget (login_window, "auth-input-box", TRUE);
        set_sensitive (GDM_GREETER_LOGIN_WINDOW (login_window), TRUE);
        set_ready (GDM_GREETER_LOGIN_WINDOW (login_window));
        set_focus (GDM_GREETER_LOGIN_WINDOW (login_window));

        return TRUE;
}

static void
_gdm_greeter_login_window_set_display_is_local (GdmGreeterLoginWindow *login_window,
                                                gboolean               is)
{
        login_window->priv->display_is_local = is;
}

static void
gdm_greeter_login_window_set_property (GObject      *object,
                                       guint         prop_id,
                                       const GValue *value,
                                       GParamSpec   *pspec)
{
        GdmGreeterLoginWindow *self;

        self = GDM_GREETER_LOGIN_WINDOW (object);

        switch (prop_id) {
        case PROP_DISPLAY_IS_LOCAL:
                _gdm_greeter_login_window_set_display_is_local (self, g_value_get_boolean (value));
                break;
        default:
                G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
                break;
        }
}

static void
gdm_greeter_login_window_get_property (GObject    *object,
                                       guint       prop_id,
                                       GValue     *value,
                                       GParamSpec *pspec)
{
        GdmGreeterLoginWindow *self;

        self = GDM_GREETER_LOGIN_WINDOW (object);

        switch (prop_id) {
        case PROP_DISPLAY_IS_LOCAL:
                g_value_set_boolean (value, self->priv->display_is_local);
                break;
        default:
                G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
                break;
        }
}

static void
log_in_button_clicked (GtkButton             *button,
                       GdmGreeterLoginWindow *login_window)
{
        GtkWidget  *entry;
        const char *text;

        set_busy (login_window);
        set_sensitive (login_window, FALSE);

        entry = glade_xml_get_widget (login_window->priv->xml, "auth-prompt-entry");
        text = gtk_entry_get_text (GTK_ENTRY (entry));

        g_signal_emit (login_window, signals[QUERY_ANSWER], 0, text);
}

static void
cancel_button_clicked (GtkButton             *button,
                       GdmGreeterLoginWindow *login_window)
{
        do_cancel (login_window);
}

static gboolean
try_system_stop (DBusGConnection *connection,
                 GError         **error)
{
        DBusGProxy      *proxy;
        gboolean         res;

        g_debug ("GdmGreeterLoginWindow: trying to stop system");

        proxy = dbus_g_proxy_new_for_name (connection,
                                           CK_NAME,
                                           CK_MANAGER_PATH,
                                           CK_MANAGER_INTERFACE);
        res = dbus_g_proxy_call_with_timeout (proxy,
                                              "Stop",
                                              INT_MAX,
                                              error,
                                              /* parameters: */
                                              G_TYPE_INVALID,
                                              /* return values: */
                                              G_TYPE_INVALID);
        return res;
}

static gboolean
try_system_restart (DBusGConnection *connection,
                    GError         **error)
{
        DBusGProxy      *proxy;
        gboolean         res;

        g_debug ("GdmGreeterLoginWindow: trying to restart system");

        proxy = dbus_g_proxy_new_for_name (connection,
                                           CK_NAME,
                                           CK_MANAGER_PATH,
                                           CK_MANAGER_INTERFACE);
        res = dbus_g_proxy_call_with_timeout (proxy,
                                              "Restart",
                                              INT_MAX,
                                              error,
                                              /* parameters: */
                                              G_TYPE_INVALID,
                                              /* return values: */
                                              G_TYPE_INVALID);
        return res;
}

#ifdef HAVE_POLKIT_GNOME
static void
system_restart_auth_cb (PolKitAction          *action,
                        gboolean               gained_privilege,
                        GError                *error,
                        GdmGreeterLoginWindow *login_window)
{
        GError          *local_error;
        DBusGConnection *connection;
        gboolean         res;

        g_debug ("GdmGreeterLoginWindow: system restart auth callback gained=%s", gained_privilege ? "yes" : "no");

        if (! gained_privilege) {
                if (error != NULL) {
                        g_warning ("GdmGreeterLoginWindow: system restart error: %s", error->message);
                }
                return;
        }

        local_error = NULL;
        connection = dbus_g_bus_get (DBUS_BUS_SYSTEM, &local_error);
        if (connection == NULL) {
                g_warning ("Unable to get system bus connection: %s", local_error->message);
                g_error_free (local_error);
                return;
        }

        res = try_system_restart (connection, &local_error);
        if (! res) {
                g_warning ("Unable to restart system: %s", local_error->message);
                g_error_free (local_error);
                return;
        }
}

static void
system_stop_auth_cb (PolKitAction          *action,
                     gboolean               gained_privilege,
                     GError                *error,
                     GdmGreeterLoginWindow *login_window)
{
        GError          *local_error;
        DBusGConnection *connection;
        gboolean         res;

        g_debug ("GdmGreeterLoginWindow: system stop auth callback gained=%s", gained_privilege ? "yes" : "no");

        if (! gained_privilege) {
                if (error != NULL) {
                        g_warning ("GdmGreeterLoginWindow: system stop error: %s", error->message);
                }
                return;
        }

        local_error = NULL;
        connection = dbus_g_bus_get (DBUS_BUS_SYSTEM, &local_error);
        if (connection == NULL) {
                g_warning ("Unable to get system bus connection: %s", local_error->message);
                g_error_free (local_error);
                return;
        }

        res = try_system_stop (connection, &local_error);
        if (! res) {
                g_warning ("Unable to stop system: %s", local_error->message);
                g_error_free (local_error);
                return;
        }
}

static PolKitAction *
get_action_from_error (GError *error)
{
        PolKitAction *action;
        const char   *paction;

        action = polkit_action_new ();

        paction = NULL;
        if (g_str_has_prefix (error->message, "Not privileged for action: ")) {
                paction = error->message + strlen ("Not privileged for action: ");
        }
        g_debug ("Requesting priv for '%s'", paction);

        polkit_action_set_action_id (action, paction);

        return action;
}
#endif

static void
do_system_restart (GdmGreeterLoginWindow *login_window)
{
        gboolean         res;
        GError          *error;
        DBusGConnection *connection;

        error = NULL;
        connection = dbus_g_bus_get (DBUS_BUS_SYSTEM, &error);
        if (connection == NULL) {
                g_warning ("Unable to get system bus connection: %s", error->message);
                g_error_free (error);
                return;
        }

        res = try_system_restart (connection, &error);
#ifdef HAVE_POLKIT_GNOME
        if (! res) {
                g_debug ("GdmGreeterLoginWindow: unable to restart system: %s: %s",
                         dbus_g_error_get_name (error),
                         error->message);

                if (dbus_g_error_has_name (error, "org.freedesktop.ConsoleKit.Manager.NotPrivileged")) {
                        PolKitAction *action;
                        guint         xid;
                        pid_t         pid;

                        action = get_action_from_error (error);

                        xid = 0;
                        pid = getpid ();

                        g_error_free (error);
                        error = NULL;
                        res = polkit_gnome_auth_obtain (action,
                                                        xid,
                                                        pid,
                                                        (PolKitGnomeAuthCB) system_restart_auth_cb,
                                                        login_window,
                                                        &error);
                        polkit_action_unref (action);

                        if (! res) {
                                g_warning ("Unable to request privilege for action: %s", error->message);
                                g_error_free (error);
                        }

                }
        }
#endif
}

static void
do_system_stop (GdmGreeterLoginWindow *login_window)
{
        gboolean         res;
        GError          *error;
        DBusGConnection *connection;

        error = NULL;
        connection = dbus_g_bus_get (DBUS_BUS_SYSTEM, &error);
        if (connection == NULL) {
                g_warning ("Unable to get system bus connection: %s", error->message);
                g_error_free (error);
                return;
        }

        res = try_system_stop (connection, &error);
#ifdef HAVE_POLKIT_GNOME
        if (! res) {
                g_debug ("GdmGreeterLoginWindow: unable to stop system: %s: %s",
                         dbus_g_error_get_name (error),
                         error->message);

                if (dbus_g_error_has_name (error, "org.freedesktop.ConsoleKit.Manager.NotPrivileged")) {
                        PolKitAction *action;
                        guint         xid;
                        pid_t         pid;

                        xid = 0;
                        pid = getpid ();

                        action = get_action_from_error (error);

                        g_error_free (error);
                        error = NULL;
                        res = polkit_gnome_auth_obtain (action,
                                                        xid,
                                                        pid,
                                                        (PolKitGnomeAuthCB) system_stop_auth_cb,
                                                        login_window,
                                                        &error);
                        polkit_action_unref (action);

                        if (! res) {
                                g_warning ("Unable to request privilege for action: %s", error->message);
                                g_error_free (error);
                        }

                }
        }
#endif
}

static void
restart_button_clicked (GtkButton             *button,
                        GdmGreeterLoginWindow *login_window)
{
        g_debug ("GdmGreeterLoginWindow: restart button clicked");
        do_system_restart (login_window);
}

static void
shutdown_button_clicked (GtkButton             *button,
                         GdmGreeterLoginWindow *login_window)
{
        g_debug ("GdmGreeterLoginWindow: stop button clicked");
        do_system_stop (login_window);
}

static void
on_user_chosen (GdmUserChooserWidget  *user_chooser,
                GdmGreeterLoginWindow *login_window)
{
        char *user_name;

        user_name = gdm_user_chooser_widget_get_chosen_user_name (GDM_USER_CHOOSER_WIDGET (login_window->priv->user_chooser));
        if (user_name == NULL) {
                return;
        }

        g_signal_emit (G_OBJECT (login_window), signals[USER_SELECTED],
                       0, user_name);

        if (strcmp (user_name, GDM_USER_CHOOSER_USER_OTHER) == 0) {
                g_signal_emit (login_window, signals[BEGIN_VERIFICATION], 0);
        } else if (strcmp (user_name, GDM_USER_CHOOSER_USER_GUEST) == 0) {
                /* FIXME: handle guest account stuff */
        } else {
                g_signal_emit (login_window, signals[BEGIN_VERIFICATION_FOR_USER], 0, user_name);
        }

        switch_mode (login_window, MODE_AUTHENTICATION);

        g_free (user_name);
}

static void
on_user_unchosen (GdmUserChooserWidget  *user_chooser,
                  GdmGreeterLoginWindow *login_window)
{
        do_cancel (login_window);
}

static gboolean
on_computer_info_label_button_press (GtkWidget             *widget,
                                     GdkEventButton        *event,
                                     GdmGreeterLoginWindow *login_window)
{
        GtkWidget *notebook;
        int        current_page;
        int        n_pages;

        /* switch page */
        notebook = glade_xml_get_widget (login_window->priv->xml, "computer-info-notebook");
        current_page = gtk_notebook_get_current_page (GTK_NOTEBOOK (notebook));
        n_pages = gtk_notebook_get_n_pages (GTK_NOTEBOOK (notebook));

        if (current_page + 1 < n_pages) {
                gtk_notebook_next_page (GTK_NOTEBOOK (notebook));
        } else {
                gtk_notebook_set_current_page (GTK_NOTEBOOK (notebook), 0);
        }

        return FALSE;
}

static char *
file_read_one_line (const char *filename)
{
        FILE *f;
        char *line;
        char buf[4096];

        line = NULL;

        f = fopen (filename, "r");
        if (f == NULL) {
                g_warning ("Unable to open file %s: %s", filename, g_strerror (errno));
                goto out;
        }

        if (fgets (buf, sizeof (buf), f) == NULL) {
                g_warning ("Unable to read from file %s", filename);
        }

        line = g_strdup (buf);
        g_strchomp (line);

 out:
        fclose (f);

        return line;
}

static const char *known_etc_info_files [] = {
        "redhat-release",
        "SuSE-release",
        "gentoo-release",
        "arch-release",
        "debian_version",
        "mandriva-release",
        "slackware-version",
        NULL
};


static char *
get_system_version (void)
{
        char *version;
        int i;

        version = NULL;

        for (i = 0; known_etc_info_files [i]; i++) {
                char *path1;
                char *path2;

                path1 = g_build_filename (SYSCONFDIR, known_etc_info_files [i], NULL);
                path2 = g_build_filename ("/etc", known_etc_info_files [i], NULL);
                if (g_access (path1, R_OK) == 0) {
                        version = file_read_one_line (path1);
                } else if (g_access (path2, R_OK) == 0) {
                        version = file_read_one_line (path2);
                }
                g_free (path2);
                g_free (path1);
                if (version != NULL) {
                        break;
                }
        }

        if (version == NULL) {
                char *output;
                output = NULL;
                if (g_spawn_command_line_sync ("uname -sr", &output, NULL, NULL, NULL)) {
                        version = g_strchomp (output);
                }
        }

        return version;
}

static void
create_computer_info (GdmGreeterLoginWindow *login_window)
{
        GtkWidget *label;

        label = glade_xml_get_widget (login_window->priv->xml, "computer-info-name-label");
        if (label != NULL) {
                gtk_label_set_text (GTK_LABEL (label), g_get_host_name ());
        }

        label = glade_xml_get_widget (login_window->priv->xml, "computer-info-version-label");
        if (label != NULL) {
                char *version;
                version = get_system_version ();
                gtk_label_set_text (GTK_LABEL (label), version);
                g_free (version);
        }
}

#define INVISIBLE_CHAR_DEFAULT       '*'
#define INVISIBLE_CHAR_BLACK_CIRCLE  0x25cf
#define INVISIBLE_CHAR_WHITE_BULLET  0x25e6
#define INVISIBLE_CHAR_BULLET        0x2022
#define INVISIBLE_CHAR_NONE          0

static GtkWidget *
custom_widget_constructor (GladeXML              *xml,
                           char                  *func_name,
                           char                  *name,
                           char                  *string1,
                           char                  *string2,
                           int                    int1,
                           int                    int2,
                           GdmGreeterLoginWindow *login_window)
{
        GtkWidget *widget;

        g_assert (GLADE_IS_XML (xml));
        g_assert (name != NULL);
        g_assert (GDM_IS_GREETER_LOGIN_WINDOW (login_window));

        widget = NULL;

        if (strcmp (name, "user-chooser") == 0) {
               widget = gdm_user_chooser_widget_new ();
        }

        return widget;
}

static void
load_theme (GdmGreeterLoginWindow *login_window)
{
        GtkWidget *entry;
        GtkWidget *button;
        GtkWidget *box;

        glade_set_custom_handler ((GladeXMLCustomWidgetHandler) custom_widget_constructor,
                                  login_window);
        login_window->priv->xml = glade_xml_new (GLADEDIR "/" GLADE_XML_FILE,
                                                 "window-box",
                                                 PACKAGE);

        g_assert (login_window->priv->xml != NULL);

        box = glade_xml_get_widget (login_window->priv->xml, "window-box");
        gtk_container_add (GTK_CONTAINER (login_window), box);

        login_window->priv->user_chooser =
                glade_xml_get_widget (login_window->priv->xml, "user-chooser");

        if (login_window->priv->user_chooser == NULL) {
                g_critical ("Userlist box not found");
        }

        gdm_user_chooser_widget_set_show_only_chosen (GDM_USER_CHOOSER_WIDGET (login_window->priv->user_chooser), TRUE);

        /* FIXME: set from gconf */
        gdm_user_chooser_widget_set_show_other_user (GDM_USER_CHOOSER_WIDGET (login_window->priv->user_chooser), TRUE);
        gdm_user_chooser_widget_set_show_guest_user (GDM_USER_CHOOSER_WIDGET (login_window->priv->user_chooser), TRUE);

        g_signal_connect (login_window->priv->user_chooser,
                          "activated",
                          G_CALLBACK (on_user_chosen),
                          login_window);
        g_signal_connect (login_window->priv->user_chooser,
                          "deactivated",
                          G_CALLBACK (on_user_unchosen),
                          login_window);

        gtk_widget_show (login_window->priv->user_chooser);

        button = glade_xml_get_widget (login_window->priv->xml, "log-in-button");
        gtk_widget_grab_default (button);
        g_signal_connect (button, "clicked", G_CALLBACK (log_in_button_clicked), login_window);

        button = glade_xml_get_widget (login_window->priv->xml, "cancel-button");
        g_signal_connect (button, "clicked", G_CALLBACK (cancel_button_clicked), login_window);

        button = glade_xml_get_widget (login_window->priv->xml, "restart-button");
        g_signal_connect (button, "clicked", G_CALLBACK (restart_button_clicked), login_window);
        button = glade_xml_get_widget (login_window->priv->xml, "shutdown-button");
        g_signal_connect (button, "clicked", G_CALLBACK (shutdown_button_clicked), login_window);

        entry = glade_xml_get_widget (login_window->priv->xml, "auth-prompt-entry");
        /* Only change the invisible character if it '*' otherwise assume it is OK */
        if ('*' == gtk_entry_get_invisible_char (GTK_ENTRY (entry))) {
                gunichar invisible_char;
                invisible_char = INVISIBLE_CHAR_BLACK_CIRCLE;
                gtk_entry_set_invisible_char (GTK_ENTRY (entry), invisible_char);
        }

        create_computer_info (login_window);

        box = glade_xml_get_widget (login_window->priv->xml, "computer-info-event-box");
        g_signal_connect (box, "button-press-event", G_CALLBACK (on_computer_info_label_button_press), login_window);

        switch_mode (login_window, MODE_SELECTION);
}

static gboolean
fit_window_to_children (GdmGreeterLoginWindow *window)
{
        int x;
        int y;
        int width;
        int height;
        int height_step;
        int width_step;

        /* FIXME: this animation logic is really dumb
         */

        if (!GTK_WIDGET_REALIZED (GTK_WIDGET (window))) {
                return FALSE;
        }

        gdk_window_get_geometry (GTK_WIDGET (window)->window,
                                 &x, &y, &width, &height, NULL);

        if (height == GTK_WIDGET (window)->requisition.height) {
                return FALSE;
        }

        if (width < GTK_WIDGET (window)->requisition.width) {
                width_step = MIN (1, GTK_WIDGET (window)->requisition.width - width);
        } else if (width > GTK_WIDGET (window)->requisition.width) {
                width_step = -1 * MIN (1, width - GTK_WIDGET (window)->requisition.width);
        } else {
                width_step = 0;
        }

        if (height < GTK_WIDGET (window)->requisition.height) {
                height_step = MIN ((int) 25, GTK_WIDGET (window)->requisition.height - height);
        } else if (height > GTK_WIDGET (window)->requisition.height) {
                height_step = -1 * MIN ((int) 25, height - GTK_WIDGET (window)->requisition.height);
        } else {
                height_step = 0;
        }

        gdk_window_resize (GTK_WIDGET (window)->window,
                           width + width_step,
                           height + height_step);

        return TRUE;
}

static gboolean
gdm_greeter_login_window_key_press_event (GtkWidget   *widget,
                                          GdkEventKey *event)
{
        if (event->keyval == GDK_Escape) {
                reset_dialog (GDM_GREETER_LOGIN_WINDOW (widget));
        }

        return GTK_WIDGET_CLASS (gdm_greeter_login_window_parent_class)->key_press_event (widget, event);
}

static void
gdm_greeter_login_window_size_request (GtkWidget      *widget,
                                       GtkRequisition *requisition)
{
        int            screen_w;
        int            screen_h;
        GtkRequisition child_requisition;

        if (GTK_WIDGET_CLASS (gdm_greeter_login_window_parent_class)->size_request) {
                GTK_WIDGET_CLASS (gdm_greeter_login_window_parent_class)->size_request (widget, requisition);
        }

        screen_w = gdk_screen_get_width (gtk_widget_get_screen (widget));
        screen_h = gdk_screen_get_height (gtk_widget_get_screen (widget));

        gtk_widget_size_request (GTK_BIN (widget)->child, &child_requisition);
        *requisition = child_requisition;

        requisition->width += 2 * GTK_CONTAINER (widget)->border_width;
        requisition->height += 2 * GTK_CONTAINER (widget)->border_width;

        requisition->width = MIN (requisition->width, .50 * screen_w);
        requisition->height = MIN (requisition->height, .80 * screen_h);
}

static void
clear_animation_timeout_id (GdmGreeterLoginWindow *window)
{
        window->priv->animation_timeout_id = 0;
}

static void
gdm_greeter_login_window_size_allocate (GtkWidget      *widget,
                                        GtkAllocation  *allocation)
{
        GdmGreeterLoginWindow *window;

        window = GDM_GREETER_LOGIN_WINDOW (widget);

        if (window->priv->animation_timeout_id == 0) {
                window->priv->animation_timeout_id =
                    g_timeout_add_full (G_PRIORITY_DEFAULT_IDLE, 20,
                                        (GSourceFunc) fit_window_to_children,
                                        widget,
                                        (GDestroyNotify) clear_animation_timeout_id);
        }

        GTK_WIDGET_CLASS (gdm_greeter_login_window_parent_class)->size_allocate (widget, allocation);
}

static GObject *
gdm_greeter_login_window_constructor (GType                  type,
                                      guint                  n_construct_properties,
                                      GObjectConstructParam *construct_properties)
{
        GdmGreeterLoginWindow      *login_window;
        GdmGreeterLoginWindowClass *klass;

        klass = GDM_GREETER_LOGIN_WINDOW_CLASS (g_type_class_peek (GDM_TYPE_GREETER_LOGIN_WINDOW));

        login_window = GDM_GREETER_LOGIN_WINDOW (G_OBJECT_CLASS (gdm_greeter_login_window_parent_class)->constructor (type,
                                                                                                                      n_construct_properties,
                                                                                                                      construct_properties));


        load_theme (login_window);

        return G_OBJECT (login_window);
}

static void
gdm_greeter_login_window_class_init (GdmGreeterLoginWindowClass *klass)
{
        GObjectClass   *object_class = G_OBJECT_CLASS (klass);
        GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (klass);

        object_class->get_property = gdm_greeter_login_window_get_property;
        object_class->set_property = gdm_greeter_login_window_set_property;
        object_class->constructor = gdm_greeter_login_window_constructor;
        object_class->finalize = gdm_greeter_login_window_finalize;

        widget_class->key_press_event = gdm_greeter_login_window_key_press_event;
        widget_class->size_request = gdm_greeter_login_window_size_request;
        widget_class->size_allocate = gdm_greeter_login_window_size_allocate;

        signals [BEGIN_VERIFICATION] =
                g_signal_new ("begin-verification",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_LAST,
                              G_STRUCT_OFFSET (GdmGreeterLoginWindowClass, begin_verification),
                              NULL,
                              NULL,
                              g_cclosure_marshal_VOID__VOID,
                              G_TYPE_NONE,
                              0);
        signals [BEGIN_VERIFICATION_FOR_USER] =
                g_signal_new ("begin-verification-for-user",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_LAST,
                              G_STRUCT_OFFSET (GdmGreeterLoginWindowClass, begin_verification_for_user),
                              NULL,
                              NULL,
                              g_cclosure_marshal_VOID__STRING,
                              G_TYPE_NONE,
                              1, G_TYPE_STRING);
        signals [QUERY_ANSWER] =
                g_signal_new ("query-answer",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_LAST,
                              G_STRUCT_OFFSET (GdmGreeterLoginWindowClass, query_answer),
                              NULL,
                              NULL,
                              g_cclosure_marshal_VOID__STRING,
                              G_TYPE_NONE,
                              1, G_TYPE_STRING);
        signals [USER_SELECTED] =
                g_signal_new ("user-selected",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_LAST,
                              G_STRUCT_OFFSET (GdmGreeterLoginWindowClass, user_selected),
                              NULL,
                              NULL,
                              g_cclosure_marshal_VOID__STRING,
                              G_TYPE_NONE,
                              1, G_TYPE_STRING);
        signals [CANCELLED] =
                g_signal_new ("cancelled",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_LAST,
                              G_STRUCT_OFFSET (GdmGreeterLoginWindowClass, cancelled),
                              NULL,
                              NULL,
                              g_cclosure_marshal_VOID__VOID,
                              G_TYPE_NONE,
                              0);
        signals [DISCONNECTED] =
                g_signal_new ("disconnected",
                              G_TYPE_FROM_CLASS (object_class),
                              G_SIGNAL_RUN_LAST,
                              G_STRUCT_OFFSET (GdmGreeterLoginWindowClass, disconnected),
                              NULL,
                              NULL,
                              g_cclosure_marshal_VOID__VOID,
                              G_TYPE_NONE,
                              0);
        g_object_class_install_property (object_class,
                                         PROP_DISPLAY_IS_LOCAL,
                                         g_param_spec_boolean ("display-is-local",
                                                               "display is local",
                                                               "display is local",
                                                               FALSE,
                                                               G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

        g_type_class_add_private (klass, sizeof (GdmGreeterLoginWindowPrivate));
}

static void
gdm_greeter_login_window_init (GdmGreeterLoginWindow *login_window)
{
        login_window->priv = GDM_GREETER_LOGIN_WINDOW_GET_PRIVATE (login_window);

        gtk_window_set_title (GTK_WINDOW (login_window), _("Login Window"));
        gtk_window_set_opacity (GTK_WINDOW (login_window), 0.85);
        gtk_window_set_position (GTK_WINDOW (login_window), GTK_WIN_POS_CENTER_ALWAYS);
        gtk_window_set_deletable (GTK_WINDOW (login_window), FALSE);
        gtk_window_set_decorated (GTK_WINDOW (login_window), FALSE);
        gtk_window_set_skip_taskbar_hint (GTK_WINDOW (login_window), TRUE);
        gtk_window_set_skip_pager_hint (GTK_WINDOW (login_window), TRUE);
        gtk_window_stick (GTK_WINDOW (login_window));
        gtk_container_set_border_width (GTK_CONTAINER (login_window), 25);
}

static void
gdm_greeter_login_window_finalize (GObject *object)
{
        GdmGreeterLoginWindow *login_window;

        g_return_if_fail (object != NULL);
        g_return_if_fail (GDM_IS_GREETER_LOGIN_WINDOW (object));

        login_window = GDM_GREETER_LOGIN_WINDOW (object);

        g_return_if_fail (login_window->priv != NULL);

        G_OBJECT_CLASS (gdm_greeter_login_window_parent_class)->finalize (object);
}

GtkWidget *
gdm_greeter_login_window_new (gboolean is_local)
{
        GObject *object;

        object = g_object_new (GDM_TYPE_GREETER_LOGIN_WINDOW,
                               "display-is-local", is_local,
                               NULL);

        return GTK_WIDGET (object);
}
