/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/***************************************************************************
 *            kolab-mail-side-cache.h
 *
 *  Tue Dec 21 17:58:20 2010
 *  Copyright  2010  Christian Hilberg
 *  <hilberg@kernelconcepts.de>
 ****************************************************************************/

/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */
 
/*----------------------------------------------------------------------------*/

/**
 * SECTION: kolab-mail-side-cache
 * @short_description: read/write offline cache for PIM folders and PIM emails
 * @title: KolabMailSideCache
 * @section_id:
 * @see_also: #KolabMailAccess, #KolabMailSynchronizer, #KolabSettingsHandler
 * @stability: unstable
 * 
 * This class holds functionality to temporarily cache PIM emails (message parts)
 * and not-yet-existing IMAP folders which will be generated by book/cal when
 * new entries are created while in #KolabMailAccess offline mode.
 * It is essential that those mails be cached separate from the
 * #KolabMailImapClient offline cache for sync reasons (they must not be deleted
 * from or directly uploaded to the Kolab server upon IMAP reconnect, since we
 * need to be able to intercept PIM data collisions (sync conflicts)).
 *
 * The initial implementation of this class works without user interaction.
 *
 * This means that for resolving true sync conflicts, a user prefs file must be
 * honoured. This shall be a GLib key file, and the options shall be "newer",
 * "server", "local" (or something similar), to denote that in case of an object
 * being changed on both, local and server side, the newer object, that of the
 * server or the local copy, respectively, should survive the conflict resolution.
 * It should be possible, however, to read this preference from Evolution settings
 * later on, and to even add an "ask" option so the user can resolve the conflict
 * manually. A #KolabSettingsHandler instance is used for that matter.
 */

/*----------------------------------------------------------------------------*/

/* The main functionality of this class will be:
 * - operate transparently in online mode (do not cache then)
 * - in offline mode, cache PIM emails separately from IMAPX and
 *   - for changed PIM objects, hold a reference into the IMAPX
 *     email cache
 *   - alongside each reference, maintain an MD5 checksum of
 *     the referenced original email object (that is, as calculated
 *     right after going offline)
 * - with reference and MD5 sum, the following can be deduced when leaving
 *   offline mode and going online again (for each object in the side cache):
 *   - side cache email does not hold a reference into IMAPX cache:
 *     --> we have a new object and can safely upload it to kolab
 *   - side cache email does hold a reference into IMAPX cache:
 *     - the reference is dangling (original object got deleted from server
 *       while we were offline, so email in IMAPX cache got deleted on
 *       reconnect):
 *       --> we have an update for a kolab object that no longer exists on server
 *           (we should upload, but need to get a new kolab object id so
 *	     there will be no false references in kolab)
 *     - the reference is okay, but side cache MD5 sum differs from the email in
 *       IMAPX cache:
 *       --> true sync conflict (both local and remote pim objects have been
 *	     changed, resolve accouding to user preference (setting in key file)
 *
 */

/*----------------------------------------------------------------------------*/

#ifndef _KOLAB_MAIL_SIDE_CACHE_H_
#define _KOLAB_MAIL_SIDE_CACHE_H_

/*----------------------------------------------------------------------------*/

#include <glib.h>
#include <glib-object.h>

#include "kolab-mail-mime-builder.h"
#include "kolab-settings-handler.h"
#include "kolab-folder-summary.h"
#include "kolab-mail-handle.h"

/*----------------------------------------------------------------------------*/

G_BEGIN_DECLS

#define KOLAB_TYPE_MAIL_SIDE_CACHE             (kolab_mail_side_cache_get_type ())
#define KOLAB_MAIL_SIDE_CACHE(obj)             (G_TYPE_CHECK_INSTANCE_CAST ((obj), KOLAB_TYPE_MAIL_SIDE_CACHE, KolabMailSideCache))
#define KOLAB_MAIL_SIDE_CACHE_CLASS(klass)     (G_TYPE_CHECK_CLASS_CAST ((klass), KOLAB_TYPE_MAIL_SIDE_CACHE, KolabMailSideCacheClass))
#define KOLAB_IS_MAIL_SIDE_CACHE(obj)          (G_TYPE_CHECK_INSTANCE_TYPE ((obj), KOLAB_TYPE_MAIL_SIDE_CACHE))
#define KOLAB_IS_MAIL_SIDE_CACHE_CLASS(klass)  (G_TYPE_CHECK_CLASS_TYPE ((klass), KOLAB_TYPE_MAIL_SIDE_CACHE))
#define KOLAB_MAIL_SIDE_CACHE_GET_CLASS(obj)   (G_TYPE_INSTANCE_GET_CLASS ((obj), KOLAB_TYPE_MAIL_SIDE_CACHE, KolabMailSideCacheClass))

typedef struct _KolabMailSideCacheClass KolabMailSideCacheClass;
typedef struct _KolabMailSideCache KolabMailSideCache;

struct _KolabMailSideCacheClass
{
	GObjectClass parent_class;
};

struct _KolabMailSideCache
{
	GObject parent_instance;
};

GType kolab_mail_side_cache_get_type (void) G_GNUC_CONST;

gboolean kolab_mail_side_cache_configure (KolabMailSideCache *self, KolabSettingsHandler *ksettings, KolabMailMimeBuilder *mimebuilder, GError **err);
gboolean kolab_mail_side_cache_bringup (KolabMailSideCache *self, GError **err);
gboolean kolab_mail_side_cache_shutdown (KolabMailSideCache *self, GError **err);

GList* kolab_mail_side_cache_query_foldernames (KolabMailSideCache *self, GError **err);
gboolean kolab_mail_side_cache_delete_folder (KolabMailSideCache *self, const gchar *foldername, GError **err);

gboolean kolab_mail_side_cache_store (KolabMailSideCache *self, KolabMailHandle *kmailhandle, const gchar *foldername, GError **err);
gboolean kolab_mail_side_cache_retrieve (KolabMailSideCache *self, KolabMailHandle *kmailhandle, GError **err);
gboolean kolab_mail_side_cache_delete_by_uid (KolabMailSideCache *self, const gchar *uid, const gchar *foldername, GError **err);
gboolean kolab_mail_side_cache_delete (KolabMailSideCache *self, KolabMailHandle *kmailhandle, GError **err);

G_END_DECLS

/*----------------------------------------------------------------------------*/

#endif /* _KOLAB_MAIL_SIDE_CACHE_H_ */

/*----------------------------------------------------------------------------*/
