/***************************************************************************
 *            camel-imapx-metadata.h
 *
 *  Mon Oct 11 12:43:03 2010
 *  Copyright  2010  Christian Hilberg
 *  <hilberg@kernelconcepts.de>
 ****************************************************************************/

/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with main.c; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */

/*----------------------------------------------------------------------------*/
/* ANNOTATEMORE / METADATA (RFC 5464) */

#ifndef _CAMEL_IMAPX_METADATA_H_
#define _CAMEL_IMAPX_METADATA_H_

/*----------------------------------------------------------------------------*/

#include <glib.h>

#include <camel/camel.h>

#include "camel-imapx-stream.h"

/*----------------------------------------------------------------------------*/

/* ANNOTATEMORE vs. METADATA:
 *
 * ANNOTATEMORE [05]:
 * The value of a named entry can contain a multitude of attributes, each
 * with a .priv and .shared suffix, all stored in one entry string.
 *
 * METADATA:
 * Each named entry has a '/private' or '/shared' name prefix,
 * and a single value (string or binary) attached to them. This means there
 * are separate entries for shared and private annotations.
 *
 * Result:
 * 
 *
 * Since METADATA allows for binary values, we need to take care that we
 * can handle NUL data bytes, hence the choice of GByteArray for values (which
 * needs to be stored as a blob in the metadata cache db).
 */

/*----------------------------------------------------------------------------*/

typedef enum {
	CAMEL_IMAPX_META_ATTRIB_TYPE_UNSET = 0,
	CAMEL_IMAPX_META_ATTRIB_TYPE_NIL,
	CAMEL_IMAPX_META_ATTRIB_TYPE_UTF8,
	CAMEL_IMAPX_META_ATTRIB_TYPE_BINARY,
	CAMEL_IMAPX_META_ATTRIB_LAST_TYPE
} imapx_meta_attrib_type_t;

typedef enum {
	CAMEL_IMAPX_METADATA_ACCESS_PRIVATE = 0,
	CAMEL_IMAPX_METADATA_ACCESS_SHARED,
	CAMEL_IMAPX_METADATA_LAST_ACCESS 
} imapx_meta_data_access_t;

typedef enum {
	CAMEL_IMAPX_METADATA_PROTO_INVAL = 0,
	CAMEL_IMAPX_METADATA_PROTO_ANNOTATEMORE,
	CAMEL_IMAPX_METADATA_PROTO_METADATA,
	CAMEL_IMAPX_METADATA_LAST_PROTO
} imapx_meta_data_proto_t;


typedef struct _CamelIMAPXMetaDataSpec {
	imapx_meta_data_proto_t proto;
	gchar *mailbox_name;
	gchar *entry_name;
	gchar *attrib_name;
} CamelIMAPXMetaDataSpec;

typedef struct _CamelIMAPXMetaAttrib {
	/* attribute name kept as hash table key in CamelIMAPXMetaEntry */
	GByteArray *data[CAMEL_IMAPX_METADATA_LAST_ACCESS]; /* attribute data (UTF-8/Binary/NIL/unset) */
	imapx_meta_attrib_type_t type[CAMEL_IMAPX_METADATA_LAST_ACCESS]; /* type of data */
} CamelIMAPXMetaAttrib;

typedef struct _CamelIMAPXMetaEntry {
	/* entry name kept as hash table key in CamelIMAPXMetaAnnotation */
	GHashTable *attributes; /* for CamelIMAPXMetaAttribute */
} CamelIMAPXMetaEntry;

typedef struct _CamelIMAPXMetaAnnotation {
	/* annotation name kept as hash table key in CamelIMAPXMetaData */
	GHashTable *entries;
} CamelIMAPXMetaAnnotation;

typedef struct _CamelIMAPXMetaData {
	imapx_meta_data_proto_t proto;
	GHashTable *mboxes;
	/* lock needed? */
	GMutex *md_lock;
} CamelIMAPXMetaData;

/*----------------------------------------------------------------------------*/
/* TODO check which of these can be made static */



CamelIMAPXMetaAttrib *__KOLAB_imapx_meta_attrib_new (void);
gboolean __KOLAB_imapx_meta_attrib_free (CamelIMAPXMetaAttrib*);

CamelIMAPXMetaEntry *__KOLAB_imapx_meta_entry_new (void);
gboolean __KOLAB_imapx_meta_entry_free (CamelIMAPXMetaEntry*);

CamelIMAPXMetaAnnotation *__KOLAB_imapx_meta_annotation_new (void);
gboolean __KOLAB_imapx_meta_annotation_free (CamelIMAPXMetaAnnotation*);

CamelIMAPXMetaData *__KOLAB_imapx_meta_data_new (imapx_meta_data_proto_t, gboolean);
CamelIMAPXMetaData *__KOLAB_imapx_meta_data_resect (CamelIMAPXMetaData**);
gboolean __KOLAB_imapx_meta_data_free (CamelIMAPXMetaData*);
imapx_meta_data_proto_t __KOLAB_imapx_meta_data_get_proto (CamelIMAPXMetaData*);
gboolean __KOLAB_imapx_meta_data_add_from_server_response (CamelIMAPXMetaData*, CamelIMAPXStream*, GError**);
GSList *__KOLAB_imapx_meta_data_new_commandlist (const CamelIMAPXMetaData*);

CamelIMAPXMetaDataSpec *__KOLAB_imapx_meta_data_spec_new (CamelIMAPXMetaData*, const gchar*, const gchar*, const gchar*, GError**);
gboolean __KOLAB_imapx_meta_data_spec_free (CamelIMAPXMetaDataSpec*);

CamelIMAPXMetaAttrib *__KOLAB_imapx_meta_data_get_attrib_from_entry (CamelIMAPXMetaEntry*, CamelIMAPXMetaDataSpec*);
CamelIMAPXMetaAttrib *__KOLAB_imapx_meta_data_get_attrib_from_annotation (CamelIMAPXMetaAnnotation*, CamelIMAPXMetaDataSpec*);
CamelIMAPXMetaAttrib *__KOLAB_imapx_meta_data_get_attrib_from_metadata (CamelIMAPXMetaData*, CamelIMAPXMetaDataSpec*);

/*----------------------------------------------------------------------------*/

#endif /* _CAMEL_IMAPX_METADATA_H_ */

/*----------------------------------------------------------------------------*/
