/***************************************************************************
 *            camel-kolab-session.c
 *
 *  Tue Aug 10 15:04:38 2010
 *  Copyright  2010  Christian Hilberg
 *  <hilberg@kernelconcepts.de>
 ****************************************************************************/

/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with main.c; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */
 
/*----------------------------------------------------------------------------*/

/*
 * The CamelSession class for Kolab access. To be instantiated once for
 * each IMAPX Camel.Provider we have. Within EDS, this is one for address book
 * and one for calendar access. Within Evolution, there should be one more
 * for email.
 */

/*----------------------------------------------------------------------------*/

#include <nss/pk11func.h>
#include <nspr/prtypes.h>

#include <libedataserver/e-account.h>
#include <libedataserverui/e-passwords.h>

#include <camel/camel.h>
#include <camel/camel-session.h>

/* Kolab error reporting */
#include <libekolabutil/kolab-util-error.h>

#include "camel-kolab-session.h"

/*----------------------------------------------------------------------------*/

static CamelSessionClass *parent_class = NULL;
static gchar *nss_tok_pin = NULL; /* FIXME better solution for this */

/*----------------------------------------------------------------------------*/

static gchar* camel_kolab_session_get_password (CamelSession*, CamelService*, const gchar*, const gchar*, const gchar*, guint32, CamelException*);
static void camel_kolab_session_forget_password (CamelSession*, CamelService*, const gchar*, const gchar*, CamelException*);

/*----------------------------------------------------------------------------*/

static void
camel_kolab_session_init (CamelKolabSession *self)
{
	g_assert (CAMEL_KOLAB_IS_SESSION (self));
	
	self->data_dir = NULL;
	self->config_dir = NULL;
	self->nss_tok_pwd = NULL;
	self->passwd = NULL;
	self->is_initialized = FALSE;

}

static void
camel_kolab_session_finalize (CamelKolabSession *self)
{
	g_assert (CAMEL_KOLAB_IS_SESSION (self));

	if (self->data_dir != NULL)
		g_free (self->data_dir);
	if (self->config_dir != NULL)
		g_free (self->config_dir);
	if (self->nss_tok_pwd != NULL)
		g_free (self->nss_tok_pwd);
	if (self->passwd)
		g_free (self->passwd);
}

static void
camel_kolab_session_class_init (CamelKolabSessionClass *klass)
{
	CamelSessionClass *camel_session_class = CAMEL_SESSION_CLASS (klass);
	parent_class = CAMEL_SESSION_CLASS (camel_type_get_global_classfuncs (CAMEL_SESSION_TYPE));

	/* virtual method definition */
	camel_session_class->get_service = parent_class->get_service;
	camel_session_class->get_storage_path = parent_class->get_storage_path;

	camel_session_class->thread_msg_new = parent_class->thread_msg_new;
	camel_session_class->thread_msg_free = parent_class->thread_msg_free;
	camel_session_class->thread_queue = parent_class->thread_queue;
	camel_session_class->thread_wait = parent_class->thread_wait;
	camel_session_class->thread_status = parent_class->thread_status;

	camel_session_class->forward_to = parent_class->forward_to;	

	/* virtual method override */
        camel_session_class->get_password = camel_kolab_session_get_password;
	camel_session_class->forget_password = camel_kolab_session_forget_password;
}

CamelType
camel_kolab_session_get_type (void)
{
	/* TODO might need another mechanism than 'static'
	 *      when dealing with multiple session instances
	 */
        static CamelType camel_kolab_session_type = CAMEL_INVALID_TYPE;

	/* TODO check whether one single KolabSession type
	 *      is okay (it should, as all Camel.Providers
	 *      we use will be IMAPX providers under the hood)
	 */
        if (camel_kolab_session_type == CAMEL_INVALID_TYPE) {
                camel_kolab_session_type = camel_type_register (
                        camel_session_get_type (),
                        "CamelKolabSession",
                        sizeof (CamelKolabSession),
                        sizeof (CamelKolabSessionClass),
                        (CamelObjectClassInitFunc) camel_kolab_session_class_init,
                        NULL,
                        (CamelObjectInitFunc) camel_kolab_session_init,
                        (CamelObjectFinalizeFunc) camel_kolab_session_finalize);
        }

        return camel_kolab_session_type;
}

/*----------------------------------------------------------------------------*/
/* passwords */

static gchar*
camel_kolab_session_get_password (CamelSession *self,
                                  CamelService *service,
                                  const gchar *domain,
                                  const gchar *prompt,
                                  const gchar *item,
                                  guint32 flags,
                                  CamelException *ex)
{
	CamelKolabSession *myself = CAMEL_KOLAB_SESSION (self);

	(void)service;
	(void)domain;
	(void)prompt;
	(void)item;
	(void)flags;
	(void)ex;

	return g_strdup (myself->passwd);
}

static void
camel_kolab_session_forget_password (CamelSession *self,
                                     CamelService *service,
                                     const gchar *domain,
                                     const gchar *item,
                                     CamelException *ex)
{
	CamelKolabSession *myself = CAMEL_KOLAB_SESSION (self);

	(void)service;
	(void)domain;
	(void)item;
	(void)ex;
	
	if (myself->passwd != NULL)
		g_free (myself->passwd);

	myself->passwd = NULL;
}

void
camel_kolab_session_set_password (CamelKolabSession *self,
                                  const gchar *passwd)
{
	g_assert (CAMEL_KOLAB_IS_SESSION (self));

	if (self->passwd != NULL)
		g_free (self->passwd);

	self->passwd = g_strdup (passwd);
}

void
camel_kolab_session_set_token_pin (CamelKolabSession *self,
                                   const gchar *pin)
{
	g_assert (CAMEL_KOLAB_IS_SESSION (self));

	if (nss_tok_pin != NULL)
		g_free (nss_tok_pin);

	nss_tok_pin = g_strdup (pin);
}

/* NSS token pin callback */
static gchar* PR_CALLBACK
pk11_password (PK11SlotInfo *slot,
               PRBool retry,
               gpointer arg)
{
	(void)slot;
	(void)arg;

	/* as recommended by PK11_SetPasswordFunc docs */
	if (retry)
		return NULL;

	return PL_strdup (nss_tok_pin);
}

/*----------------------------------------------------------------------------*/

CamelKolabSession*
camel_kolab_session_new (void)
{
	CamelKolabSession *session = NULL;

	session = CAMEL_KOLAB_SESSION (camel_object_new (CAMEL_KOLAB_SESSION_TYPE));
	
	/* init the EAccount setup */
	/* e_account_writable (NULL, E_ACCOUNT_SOURCE_SAVE_PASSWD); */

	return session;
}

gboolean
camel_kolab_session_bringup (CamelKolabSession *self,
                             GError **err)
{
	g_assert (CAMEL_KOLAB_IS_SESSION (self));
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	if (self->is_initialized)
		return TRUE;

	/* TODO need to supply (parts of) storage paths here so the imap
	 *	objects for the backends will store their files in their
	 *	own respective directories
	 */
	
	/* TODO check whether this is The Right Way to
	 *	initialize the session parent
	 */
	if (self->data_dir == NULL) {
		g_set_error (err,
		             KOLAB_CAMEL_KOLAB_ERROR,
		             KOLAB_CAMEL_KOLAB_ERROR_GENERIC,
		             "%s: data dir not configured",
		             __func__);
		return FALSE;
	}
        camel_session_construct (CAMEL_SESSION (self), self->data_dir);

	/* TODO junk settings */
	
	/* setup further NSS bits here */
	PK11_SetPasswordFunc (pk11_password);

	self->is_initialized = TRUE;
	g_debug ("%s: camel session initialized",
	         __func__);

	return TRUE;
}

gboolean
camel_kolab_session_shutdown (CamelKolabSession *self,
                              GError **err)
{
	g_assert (CAMEL_KOLAB_IS_SESSION (self));
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	if (! self->is_initialized)
		return TRUE;

	g_debug ("%s: camel session shut down",
	         __func__);

	return TRUE;
}

/*----------------------------------------------------------------------------*/

void
camel_kolab_session_set_data_dir (CamelKolabSession *self,
                                  gchar *datadir)
{
	g_assert (CAMEL_KOLAB_IS_SESSION (self));
	g_assert (datadir != NULL);

	if (self->data_dir != NULL)
		g_free (self->data_dir);
	
	self->data_dir = datadir;
}

const gchar*
camel_kolab_session_get_data_dir (CamelKolabSession *self)
{
	g_assert (CAMEL_KOLAB_IS_SESSION (self));
        return self->data_dir;
}

void
camel_kolab_session_set_config_dir (CamelKolabSession *self,
                                    gchar *configdir)
{
	g_assert (CAMEL_KOLAB_IS_SESSION (self));
	g_assert (configdir != NULL);

	if (self->config_dir != NULL)
		g_free (self->config_dir);
	
	self->config_dir = configdir;
}

const gchar*
camel_kolab_session_get_config_dir (CamelKolabSession *self)
{
	g_assert (CAMEL_KOLAB_IS_SESSION (self));
        return self->config_dir;
}

/*----------------------------------------------------------------------------*/
