/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/*
 * evolution-kolab
 * Copyright (C) Silvan Marco Fin 2011 <silvan@kernelconcepts.de>
 * 
 * evolution-kolab is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * evolution-kolab is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <libekolab/kolab-mail-access.h>

#include <libebook/e-contact.h>
#include <libecal/e-cal.h>

#include "kolab-util-contact.h"

gboolean
kolab_util_contact_has_id (EContact *contact) 
{
	gchar *uid = NULL;

	uid = (gchar *) e_contact_get (contact, E_CONTACT_UID);

	if (uid == NULL)
		return FALSE;

	g_free (uid);

	return TRUE;
}

void
kolab_util_contact_gen_uid_if_none (EContact **contact)
{
	gchar *uid = NULL;

	if (! kolab_util_contact_has_id (*contact)) {
		/* no uid yet */
		uid = e_cal_component_gen_uid ();
		e_contact_set (*contact, E_CONTACT_UID, (gconstpointer) uid);
	}

	g_free (uid);

}

KolabMailAccessOpmodeID
kolab_util_contact_deploy_mode_by_koma (KolabMailAccess *koma, KolabMailAccessOpmodeID koma_mode)
{
	KolabMailAccessOpmodeID tmp_mode;
	GError *error = NULL;
	gboolean ok;
g_debug ("%s()[%u] called.", __func__, __LINE__);
	if (koma == NULL) {
		g_debug ("%s()[%u] error: KolabMailAccess object not existent.", __func__, __LINE__);
		return KOLAB_MAIL_ACCESS_OPMODE_INVAL;
	}
	tmp_mode = kolab_mail_access_get_opmode (koma, &error);
	if (error != NULL) {
		g_warning ("%s()[%u] error getting mode: %s",__func__, __LINE__, error->message);
		g_error_free (error);
		return KOLAB_MAIL_ACCESS_OPMODE_INVAL;
	}
	if (tmp_mode < KOLAB_MAIL_ACCESS_OPMODE_OFFLINE) {
		g_debug ("%s()[%u] KolabMailAccess object not ready, deferring.", __func__, __LINE__);
		return tmp_mode;
	}
	ok = kolab_mail_access_set_opmode (koma, koma_mode, &error);
	if (error != NULL) {
		g_warning ("%s()[%u] error setting mode: %s", __func__, __LINE__, error->message);
		g_error_free (error);
		return KOLAB_MAIL_ACCESS_OPMODE_INVAL;
	}
	return koma_mode;
} /* kolab_util_contacts_deploy_by_koma_mode () */

gboolean
kolab_util_contact_deploy_mode_by_backend (KolabMailAccess *koma, 
                                            GNOME_Evolution_Addressbook_BookMode backend_mode)
{
	KolabMailAccessOpmodeID koma_mode;
	KolabMailAccessOpmodeID tmp_mode;

	switch (backend_mode) {
	case GNOME_Evolution_Addressbook_MODE_LOCAL:
		koma_mode = KOLAB_MAIL_ACCESS_OPMODE_OFFLINE;
		break;
	default:
		koma_mode = KOLAB_MAIL_ACCESS_OPMODE_ONLINE;
	}
	tmp_mode = kolab_util_contact_deploy_mode_by_koma (koma, koma_mode);

	return tmp_mode == koma_mode;
}

void
kolab_util_contact_store (EContact *econtact, KolabMailAccess *koma, const gchar *uri, GError **error)
{
	const gchar *sourcename = NULL;
	KolabMailHandle *kmh = NULL;
	GError *tmp_error = NULL;
	KolabSettingsHandler *ksettings = NULL;

	g_return_if_fail (error == NULL || *error == NULL);

	ksettings = kolab_mail_access_get_settings_handler (koma);
	if (ksettings == NULL)
		/* FIXME: obviously, this lacks some conclusive error notification */
		return;

	sourcename = kolab_util_backend_get_relative_path_from_uri (uri);

	kolab_util_backend_modtime_set_on_econtact (econtact);
	kmh = kolab_mail_handle_new_from_econtact (econtact);

	if (! kolab_mail_access_store_handle (koma, kmh, sourcename, &tmp_error))
		g_propagate_error (error, tmp_error);

	g_object_unref (ksettings);
} /* kolab_util_contact_store () */

EBookBackendSyncStatus
kolab_util_contact_map_error (GError *error)
{
	EBookBackendSyncStatus status = GNOME_Evolution_Addressbook_Success;

	if (error == NULL)
		return GNOME_Evolution_Addressbook_Success;
	
	switch (error->code) {
		case KOLAB_BACKEND_ERROR_SYNC_NOTSTORED:
			status = GNOME_Evolution_Addressbook_Success;
			break;
		case KOLAB_BACKEND_ERROR_NOTFOUND:
			status = GNOME_Evolution_Addressbook_ContactNotFound;
			break;
		case KOLAB_BACKEND_ERROR_CONTEXT_MISUSE:
		case KOLAB_BACKEND_ERROR_INFODB_NOFOLDER:
			status = GNOME_Evolution_Addressbook_NoSuchBook;
			break;
		default:
			status = GNOME_Evolution_Addressbook_OtherError;
	}

	g_debug ("%s()[%u] EBookBackendSyncStatus: %i", __func__, __LINE__, status);
	return status;
}
