/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*- */
/*
 * This file is part of capuchin-glib. 
 *
 * Copyright (C) Sebastian Pölsterl 2008 <marduk@k-d-w.org>
 * 
 * capuchin-glib is free software.
 * 
 * You may redistribute it and/or modify it under the terms of the
 * GNU General Public License, as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option)
 * any later version.
 * 
 * capuchin-glib is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with capuchin-glib.
 * If not, see <http://www.gnu.org/licenses/>.
 */

#include <capuchin-glib/capuchin-g-app-object.h>
#include <capuchin-glib/capuchin-g-type-builtins.h>
#include <capuchin-glib/capuchin-marshal.h>

#define CAPUCHIN_G_APP_OBJECT_METHOD_GetApplicationName "GetApplicationName"
#define CAPUCHIN_G_APP_OBJECT_METHOD_Update "Update"
#define CAPUCHIN_G_APP_OBJECT_METHOD_Install "Install"
#define CAPUCHIN_G_APP_OBJECT_METHOD_GetAvailablePlugins "GetAvailablePlugins"
#define CAPUCHIN_G_APP_OBJECT_METHOD_GetAvailableUpdates "GetAvailableUpdates"
#define CAPUCHIN_G_APP_OBJECT_METHOD_GetPluginsWithTag "GetPluginsWithTag"
#define CAPUCHIN_G_APP_OBJECT_METHOD_GetPluginName "GetPluginName"
#define CAPUCHIN_G_APP_OBJECT_METHOD_GetPluginDescription "GetPluginDescription"
#define CAPUCHIN_G_APP_OBJECT_METHOD_GetPluginChanges "GetPluginChanges"
#define CAPUCHIN_G_APP_OBJECT_METHOD_GetPluginTags "GetPluginTags"
#define CAPUCHIN_G_APP_OBJECT_METHOD_GetPluginAuthor "GetPluginAuthor"
#define CAPUCHIN_G_APP_OBJECT_METHOD_GetPluginVersion "GetPluginVersion"
#define CAPUCHIN_G_APP_OBJECT_METHOD_GetTags "GetTags"
#define CAPUCHIN_G_APP_OBJECT_METHOD_Close "Close"

#define CAPUCHIN_G_APP_OBJECT_SIGNAL_InstallFinished "InstallFinished"
#define CAPUCHIN_G_APP_OBJECT_SIGNAL_Status "Status"

#define CAPUCHIN_G_APP_OBJECT_PRIVATE(o)  (G_TYPE_INSTANCE_GET_PRIVATE ((o), CAPUCHIN_TYPE_G_APP_OBJECT, CapuchinGAppObjectPrivate))

struct _CapuchinGAppObjectPrivate
{
	DBusGProxy* proxy;
};

/* Signals */
enum
{
	UPDATE_FINISHED,
	INSTALL_FINISHED,
	STATUS,
	
	LAST_SIGNAL
};

static guint capuchin_g_app_object_signals[LAST_SIGNAL] = {0};

/* private methods */
static void _install_finished_cb (DBusGProxy *proxy,
								  const gchar *plugin_id,
								  CapuchinGAppObject *appobject);

static void _status_cb (DBusGProxy *proxy,
						gint type,
						const gchar *plugin_id,
						gdouble progress,
						gint speed,
						CapuchinGAppObject *appobject);

static void _update_returned_cb (DBusGProxy *proxy,
								 DBusGProxyCall *call,
								 gpointer data);

G_DEFINE_TYPE (CapuchinGAppObject, capuchin_g_app_object, G_TYPE_OBJECT);

static void
capuchin_g_app_object_init (CapuchinGAppObject *object)
{
	object->priv = CAPUCHIN_G_APP_OBJECT_PRIVATE (object);
	object->priv->proxy = NULL;
}

static void
capuchin_g_app_object_finalize (GObject *object)
{
	CapuchinGAppObject *appobject;
	
	appobject = CAPUCHIN_G_APP_OBJECT (object);
	
	dbus_g_proxy_disconnect_signal (appobject->priv->proxy,
									CAPUCHIN_G_APP_OBJECT_SIGNAL_InstallFinished,
									G_CALLBACK (_install_finished_cb),
									appobject);
	
	dbus_g_proxy_disconnect_signal (appobject->priv->proxy,
									CAPUCHIN_G_APP_OBJECT_SIGNAL_Status,
									G_CALLBACK (_status_cb),
									appobject);
	
	g_object_unref (appobject->priv->proxy);

	G_OBJECT_CLASS (capuchin_g_app_object_parent_class)->finalize (object);
}

static void
capuchin_g_app_object_class_init (CapuchinGAppObjectClass *klass)
{
	GObjectClass* object_class = G_OBJECT_CLASS (klass);
	GObjectClass* parent_class = G_OBJECT_CLASS (klass);

	g_type_class_add_private (klass, sizeof (CapuchinGAppObjectPrivate));

	object_class->finalize = capuchin_g_app_object_finalize;
	
	/**
	  * CapuchinGAppObject::update-finished:
	  * @appobject: The object on which the signal is emitted
	  *
	  * Emitted when the repository has been updated
	  * after calling capuchin_g_app_object_update()
	  */
	capuchin_g_app_object_signals[UPDATE_FINISHED] =
		g_signal_new ("update-finished",
					  CAPUCHIN_TYPE_G_APP_OBJECT,
					  G_SIGNAL_RUN_LAST,
					  0, 
					  NULL,
					  NULL,
					  g_cclosure_marshal_VOID__VOID,
					  G_TYPE_NONE,
					  0);
		
	/**
	  * CapuchinGAppObject::install-finished
	  * @appobject: The object on which the signal is emitted
	  * @plugin_id: Plugin's ID
	  *
	  * Emitted when the plugin has been downloaded
	  * and unpacked and is ready to be (re-)loaded 
	  * by the application
	  */
	capuchin_g_app_object_signals[INSTALL_FINISHED] =
		g_signal_new ("install-finished",
					  CAPUCHIN_TYPE_G_APP_OBJECT,
					  G_SIGNAL_RUN_LAST,
					  0, 
					  NULL,
					  NULL,
					  g_cclosure_marshal_VOID__STRING,
					  G_TYPE_NONE,
					  1,
					  G_TYPE_STRING);
	
	/**
	  * CapuchinGAppObject::status
	  * @appobject: The object on which the signal is emitted
	  * @action: Current action
	  * @plugin_id: The ID of the plugin that the status is related to
	  * @progress: The fraction of the progress that's complete (from 0 to 1),
	  * or -1.0 if no information is available on how long it will take
	  * @speed: Download speed
	  *
	  * Emitted when the plugin is getting downloaded and extracted after
	  * calling capuchin_g_app_object_install()
	  */
	capuchin_g_app_object_signals[STATUS] =
		g_signal_new ("status",
					  CAPUCHIN_TYPE_G_APP_OBJECT,
					  G_SIGNAL_RUN_LAST,
					  0, 
					  NULL,
					  NULL,
					  capuchin_marshal_VOID__ENUM_STRING_DOUBLE_INT,
					  G_TYPE_NONE,
					  4,
					  CAPUCHIN_TYPE_G_APP_OBJECT_ACTION,
					  G_TYPE_STRING,
					  G_TYPE_DOUBLE,
					  G_TYPE_INT);
		
}

/**
  * capuchin_g_app_object_new
  * @proxy: A #DBusGProxy object that represents
  * the AppObject for the repository
  * @returns: A new #CapuchinGAppObject object
  *
  * Usually, you don't have to use this method,
  * because capuchin_g_app_object_manager_get_appobject()
  * is more convenient
  */
CapuchinGAppObject*
capuchin_g_app_object_new (DBusGProxy *proxy)
{
	CapuchinGAppObject *appobject;
	
	appobject = CAPUCHIN_G_APP_OBJECT (g_object_new (CAPUCHIN_TYPE_G_APP_OBJECT, NULL));
	appobject->priv->proxy = proxy;
	
	dbus_g_object_register_marshaller (capuchin_marshal_VOID__ENUM_STRING_DOUBLE_INT,
									   G_TYPE_NONE,
									   G_TYPE_INT, G_TYPE_STRING, G_TYPE_DOUBLE, G_TYPE_INT,
									   G_TYPE_INVALID);
	
	// plugin_id
	dbus_g_proxy_add_signal (proxy,
							 CAPUCHIN_G_APP_OBJECT_SIGNAL_InstallFinished,
							 G_TYPE_STRING, G_TYPE_INVALID);
	
	// action, plugin_id, progress, speed
	dbus_g_proxy_add_signal (proxy,
							 CAPUCHIN_G_APP_OBJECT_SIGNAL_Status,
							 G_TYPE_INT, G_TYPE_STRING, G_TYPE_DOUBLE, G_TYPE_INT);
							 
	dbus_g_proxy_connect_signal (proxy,
								 CAPUCHIN_G_APP_OBJECT_SIGNAL_InstallFinished,
								 G_CALLBACK (_install_finished_cb),
								 appobject, NULL);
	
	dbus_g_proxy_connect_signal (proxy,
								 CAPUCHIN_G_APP_OBJECT_SIGNAL_Status,
								 G_CALLBACK (_status_cb),
								 appobject, NULL);
	
	return appobject;
}

/**
 * capuchin_g_app_object_update
 * @appobject: #CapuchinGAppObject of the repository
 * @force_update: Whether to force to download the XML file from the server
 * or use the cached one, if no newer version is available
 * @error: a GError location to store the error occuring, or NULL to ignore.
 *
 * Load the repository. This method returns when the update process
 * completed.
 */
void
capuchin_g_app_object_update (CapuchinGAppObject* appobject, gboolean force_update,
							  GError **error)
{	
	if (!dbus_g_proxy_call (appobject->priv->proxy,
							CAPUCHIN_G_APP_OBJECT_METHOD_Update,
							error,
							G_TYPE_BOOLEAN,
							force_update,
							G_TYPE_INVALID,
							G_TYPE_INVALID))
	{
		return;
	}
	
	g_signal_emit_by_name (appobject, "update-finished");
}

/**
 * capuchin_g_app_object_update_async
 * @appobject: #CapuchinGAppObject of the repository
 * @force_update: Whether to force to download the XML file from the server
 * or use the cached one, if no newer version is available
 *
 * Load the repository.
 * The repository has been initialized when the CapuchinGAppObject::update-finished
 * signal has been emitted.
 */
void
capuchin_g_app_object_update_async (CapuchinGAppObject* appobject, gboolean force_update)
{
	dbus_g_proxy_begin_call (appobject->priv->proxy,
							 CAPUCHIN_G_APP_OBJECT_METHOD_Update,
							 (DBusGProxyCallNotify) _update_returned_cb,
							 (gpointer) appobject,
							 NULL,
							 G_TYPE_BOOLEAN,
							 force_update,
							 G_TYPE_INVALID);
}

/**
 * capuchin_g_app_object_get_application_name
 * @appobject: #CapuchinGAppObject of the repository
 * @error: a GError location to store the error occuring, or NULL to ignore.
 * @returns: The name of the application the repository belongs to
 * or NULL if an error occured
 *
 * Get the name of the application the repository belongs to.
 * You have to free it with g_free().
 */
gchar*
capuchin_g_app_object_get_application_name (CapuchinGAppObject *appobject, GError **error)
{
	gchar *name = NULL;
	
	if (!dbus_g_proxy_call (appobject->priv->proxy,
							CAPUCHIN_G_APP_OBJECT_METHOD_GetApplicationName,
							error,
							G_TYPE_INVALID,
							G_TYPE_STRING, &name,
							G_TYPE_INVALID))
	{
		return NULL;
	}
	
	return name;
}

/**
 * capuchin_g_app_object_get_available_plugins
 * @appobject: #CapuchinGAppObject of the repository
 * @error: a GError location to store the error occuring, or NULL to ignore.
 * @returns: An array of plugin IDs or NULL if an error occured
 *
 * Get all plugins from the repository.
 * You have to free it with g_strfreev().
 */
gchar**
capuchin_g_app_object_get_available_plugins (CapuchinGAppObject *appobject, GError **error)
{
	gchar **plugins;
	
	if (!dbus_g_proxy_call (appobject->priv->proxy,
							CAPUCHIN_G_APP_OBJECT_METHOD_GetAvailablePlugins,
							error,
							G_TYPE_INVALID,
							G_TYPE_STRV, &plugins,
							G_TYPE_INVALID))
	{
		return NULL;
	}
	
	return plugins;
}

/**
 * capuchin_g_app_object_get_available_updates
 * @appobject: #CapuchinGAppObject of the repository
 * @plugins: A null terminated array of #CapuchinGPluginInfo objects
 * @error: a GError location to store the error occuring, or NULL to ignore.
 * representing the plugins that should be checked for updates
 * @returns: An array of plugin IDs or NULL if an error occured
 *
 * Get all available updates.
 * You have to free it with g_strfreev().
 */
gchar**
capuchin_g_app_object_get_available_updates (CapuchinGAppObject *appobject,
											 CapuchinGPluginInfo **plugins,
											 GError **error)
{
	gchar *plugin_id;
	gchar *plugin_version;
	gchar **plugin_ids;
	CapuchinGPluginInfo *plugin;
	GPtrArray *plugin_data;
	gint length = 0;
	gint j;
	
	g_assert (plugins != NULL);
	
	// Get array length
	for (plugin = plugins[0]; plugin != NULL; plugin = plugins[length])
	{
		length++;
	}
	
	plugin_data = g_ptr_array_sized_new (length);
	
	for (j=0; j<length; j++)
	{
		g_assert (plugins[j] != NULL);
		
		g_object_get (G_OBJECT (plugins[j]),
					  "id", &plugin_id,
					  "version", &plugin_version,
					  NULL);

		// Has to be null terminated
		gchar **plug = g_new (gchar*, 3);
		plug[0] = plugin_id;
		plug[1] = plugin_version;
		plug[2] = NULL;
		
		g_ptr_array_add (plugin_data, plug);
	}
	
	if (!dbus_g_proxy_call (appobject->priv->proxy,
							CAPUCHIN_G_APP_OBJECT_METHOD_GetAvailableUpdates,
							error,
							(dbus_g_type_get_collection ("GPtrArray", G_TYPE_STRV)), plugin_data,
							G_TYPE_INVALID,
							G_TYPE_STRV, &plugin_ids,
							G_TYPE_INVALID))
	{
		plugin_ids = NULL;
	}					
	
	// Free the stuff
	for (j=0; j<length; j++)
	{
		g_strfreev ((gchar**) g_ptr_array_index (plugin_data, j));
	}
	g_ptr_array_free (plugin_data, TRUE);
	
	return plugin_ids;
}

/**
 * capuchin_g_app_object_install
 * @appobject: #CapuchinGAppObject of the repository
 * @plugin_id: Plugin's ID
 * @error: a GError location to store the error occuring, or NULL to ignore.
 *
 * Update the plugin given ID.
 * During the installation progess the CapuchinGAppObject::status
 * signal will be emitted. When the process is complete the
 * CapuchinGAppObject::install-finished signal will be emitted.
 */
void
capuchin_g_app_object_install (CapuchinGAppObject *appobject, const gchar *plugin_id,
							   GError **error)
{
	g_assert (plugin_id != NULL);
	
	dbus_g_proxy_call (appobject->priv->proxy,
							CAPUCHIN_G_APP_OBJECT_METHOD_Install,
							error,
							G_TYPE_STRING, plugin_id,
							G_TYPE_INVALID,
							G_TYPE_INVALID);
}

/**
 * capuchin_g_app_object_get_plugins_with_tag
 * @appobject: #CapuchinGAppObject of the repository
 * @tag: A tag
 * @error: a GError location to store the error occuring, or NULL to ignore.
 * @returns: A list of plugin IDs or NULL if an error occured
 *
 * Get all plugins that are tagged with the given tag.
 * You have to free it with g_strfreev().
 */
gchar**
capuchin_g_app_object_get_plugins_with_tag (CapuchinGAppObject *appobject,
											const gchar *tag, GError **error)
{
	gchar **plugins;
	
	g_assert (tag != NULL);
	
	if (!dbus_g_proxy_call (appobject->priv->proxy,
							CAPUCHIN_G_APP_OBJECT_METHOD_GetPluginsWithTag,
							error,
							G_TYPE_STRING, tag,
							G_TYPE_INVALID,
							G_TYPE_STRV, &plugins,
							G_TYPE_INVALID))
	{
		return NULL;
	}

	return plugins;
}

/**
 * capuchin_g_app_object_get_plugin_name
 * @appobject: #CapuchinGAppObject of the repository
 * @plugin_id: Plugin's ID
 * @error: a GError location to store the error occuring, or NULL to ignore.
 * @returns: Plugin's name or NULL if an error occured
 *
 * Get name of plugin with given ID.
 * You have to free it with g_free().
 */
gchar*
capuchin_g_app_object_get_plugin_name (CapuchinGAppObject *appobject,
									   const gchar *plugin_id, GError **error)
{
	gchar *name = NULL;
	
	g_assert (plugin_id != NULL);
	
	if (!dbus_g_proxy_call (appobject->priv->proxy,
							CAPUCHIN_G_APP_OBJECT_METHOD_GetPluginName,
							error,
							G_TYPE_STRING, plugin_id,
							G_TYPE_INVALID,
							G_TYPE_STRING, &name,
							G_TYPE_INVALID))
	{
		return NULL;
	}
	
	return name;
}

/**
 * capuchin_g_app_object_get_plugin_description
 * @appobject: #CapuchinGAppObject of the repository
 * @plugin_id: Plugin's ID
 * @error: a GError location to store the error occuring, or NULL to ignore.
 * @returns: Plugin's descriptio
 *
 * Get description of plugin with given ID.
 * You have to free it with g_free().
 */
gchar*
capuchin_g_app_object_get_plugin_description (CapuchinGAppObject *appobject,
											  const gchar *plugin_id, GError **error)
{
	gchar *desc;
	
	g_assert (plugin_id != NULL);
	
	if (!dbus_g_proxy_call (appobject->priv->proxy,
							CAPUCHIN_G_APP_OBJECT_METHOD_GetPluginDescription,
							error,
							G_TYPE_STRING, plugin_id,
							G_TYPE_INVALID,
							G_TYPE_STRING, &desc,
							G_TYPE_INVALID))
	{
		return NULL;
	}
	
	return desc;
}

/**
 * capuchin_g_app_object_get_plugin_changes
 * @appobject: #CapuchinGAppObject of the repository
 * @plugin_id: Plugin's ID
 * @version: Plugin's version that you want to get the changes from
 * @error: a GError location to store the error occuring, or NULL to ignore.
 * @returns: Changes or NULL if an error occured
 *
 * Get changes for plugin with given ID made in given version.
 * You have to free it with g_free().
 */
gchar* 
capuchin_g_app_object_get_plugin_changes (CapuchinGAppObject *appobject,
										  const gchar *plugin_id, const gchar *version,
										  GError **error)
{
	gchar *changes;
	
	g_assert (plugin_id != NULL && version != NULL);
	
	if (!dbus_g_proxy_call (appobject->priv->proxy,
							CAPUCHIN_G_APP_OBJECT_METHOD_GetPluginChanges,
							error,
							G_TYPE_STRING, plugin_id,
							G_TYPE_STRING, version,
							G_TYPE_INVALID,
							G_TYPE_STRING, &changes,
							G_TYPE_INVALID))
	{
		return NULL;
	}
	
	return changes;
}

/**
 * capuchin_g_app_object_get_plugin_tags
 * @appobject: #CapuchinGAppObject of the repository
 * @plugin_id: Plugin's ID
 * @error: a GError location to store the error occuring, or NULL to ignore.
 * @returns: Array of tags for the plugin or NULL if an error occured
 *
 * Get tags for the plugin given ID.
 * You have to free it with g_strfreev().
 */
gchar**
capuchin_g_app_object_get_plugin_tags (CapuchinGAppObject *appobject,
									   const gchar *plugin_id, GError **error)
{
	gchar **tags;
	
	g_assert (plugin_id != NULL);
	
	if (!dbus_g_proxy_call (appobject->priv->proxy,
							CAPUCHIN_G_APP_OBJECT_METHOD_GetPluginTags,
							error,
							G_TYPE_STRING, plugin_id,
							G_TYPE_INVALID,
							G_TYPE_STRV, &tags,
							G_TYPE_INVALID))
	{
		return NULL;
	}
	
	return tags;
}

/**
 * capuchin_g_app_object_get_plugin_author
 * @appobject: #CapuchinGAppObject of the repository
 * @plugin_id: Plugin's ID
 * @error: a GError location to store the error occuring, or NULL to ignore.
 * @returns: #CapuchinGPluginAuthor or NULL if an error occured
 *
 * Get the author's name and e-mail address for the plugin given ID
 */
CapuchinGPluginAuthor*
capuchin_g_app_object_get_plugin_author (CapuchinGAppObject *appobject,
										 const gchar *plugin_id, GError **error)
{
	gchar **author_arr;
	CapuchinGPluginAuthor *author_obj = NULL;
	
	g_assert (plugin_id != NULL);
	
	if (dbus_g_proxy_call (appobject->priv->proxy,
							CAPUCHIN_G_APP_OBJECT_METHOD_GetPluginAuthor,
							error,
							G_TYPE_STRING, plugin_id,
							G_TYPE_INVALID,
							G_TYPE_STRV, &author_arr,
							G_TYPE_INVALID))
	{
		author_obj = capuchin_g_plugin_author_new (author_arr[0], author_arr[1]);
		g_strfreev (author_arr);
	}
	
	return author_obj;
}

/**
 * capuchin_g_app_object_get_plugin_version
 * @appobject: #CapuchinGAppObject of the repository
 * @plugin_id: Plugin's ID
 * @error: a GError location to store the error occuring, or NULL to ignore.
 * @returns: Plugin's version or NULL if an error occured
 *
 * Get the version of the plugin with given ID.
 * You have to free it with g_free().
 */
gchar*
capuchin_g_app_object_get_plugin_version (CapuchinGAppObject *appobject,
										  const gchar *plugin_id, GError **error)
{
	gchar *version;
	
	g_assert (plugin_id != NULL);
	
	if (!dbus_g_proxy_call (appobject->priv->proxy,
							CAPUCHIN_G_APP_OBJECT_METHOD_GetPluginVersion,
							error,
							G_TYPE_STRING, plugin_id,
							G_TYPE_INVALID,
							G_TYPE_STRING, &version,
							G_TYPE_INVALID))
	{
		return NULL;
	}					
	
	return version;
}

/**
 * capuchin_g_app_object_get_tags
 * @appobject: #CapuchinGAppObject of the repository
 * @error: a GError location to store the error occuring, or NULL to ignore.
 * @returns: A list of tags or NULL when an error occured
 *
 * Get all available tags available in this repository.
 * You have to free it with g_strfreev().
 */
gchar**
capuchin_g_app_object_get_tags (CapuchinGAppObject *appobject, GError **error)
{
	gchar **tags;
	
	if (!dbus_g_proxy_call (appobject->priv->proxy,
							CAPUCHIN_G_APP_OBJECT_METHOD_GetTags,
							error,
							G_TYPE_INVALID,
							G_TYPE_STRV, &tags,
							G_TYPE_INVALID))
	{
		return NULL;
	}
	
	return tags;
}

/**
 * capuchin_g_app_object_close
 * @appobject: #CapuchinGAppObject of the repository
 * @error: a GError location to store the error occuring, or NULL to ignore.
 *
 * Tell the object that it isn't needed anymore
 */
void
capuchin_g_app_object_close (CapuchinGAppObject *appobject, GError **error)
{	
	dbus_g_proxy_call (appobject->priv->proxy,
							CAPUCHIN_G_APP_OBJECT_METHOD_Close,
							error,
							G_TYPE_INVALID,
							G_TYPE_INVALID);
}

static void
_install_finished_cb (DBusGProxy *proxy, const gchar *plugin_id, CapuchinGAppObject *appobject)
{
	g_signal_emit_by_name (appobject, "install-finished",
						   plugin_id);
}

static void
_status_cb (DBusGProxy *proxy, gint type,
			const gchar *plugin_id,	gdouble progress, gint speed, CapuchinGAppObject *appobject)
{	
	g_signal_emit_by_name (appobject, "status",
						   type,
						   plugin_id,
						   progress,
						   speed);
}

static void
_update_returned_cb (DBusGProxy *proxy, DBusGProxyCall *call, gpointer data)
{
	GError *error = NULL;
	CapuchinGAppObject *appobject = CAPUCHIN_G_APP_OBJECT (data);
	
	if (!dbus_g_proxy_end_call (proxy,
								call,
								&error,
								G_TYPE_INVALID))
	{
		g_critical (error->message);
		g_error_free (error);
	}						 
	
	g_signal_emit_by_name (appobject, "update-finished");
}
