/******************************************************************************/
/* blboard.cpp (c)1999 Bozo                                                   */
/******************************************************************************/

#define DEBUG

#include "../core/core.h"

#include "../server/ts.h"

#include "seconst.h"
#include "seprot.h"
#include "seerror.h"

#include "cmdlink.h"
#include "datalink.h"
#include "blboard.h"


// Creation of the instance
C_BlBoard* C_BlBoard::s_pInstance = NULL;


//------------------------------------------------------------------------------
// Instanciation of the class
//------------------------------------------------------------------------------
C_BlBoard* C_BlBoard::Create(handle hLog)
{
	C_BlBoard* pResult = NULL;

	if(s_pInstance == NULL)
	{
		pResult = new C_BlBoard(hLog);
		s_pInstance = pResult;
	}

	return pResult;
}


//------------------------------------------------------------------------------
// Constructor
//------------------------------------------------------------------------------
C_BlBoard::C_BlBoard(handle hLog) : m_cCmdLink(hLog),
																		m_cPmtStructures(17),
																		m_cDataLinksByPgrmNumber(17),
																		m_cDataLinksByPID(17),
																		m_cHandles(17),
																		m_cPats(17),
																		m_cPmts(17)
{
	ASSERT(s_pInstance == NULL);
	s_pInstance = this;

	m_hLog = hLog;

	m_iNumBoards = 0;
	m_bAppRegistered = false;
	m_bAppMastered = false;
	m_bBoardInitialized = false;
	m_bPatValid = false;
	m_strDirectoryString = "DIRECTORY_ANSWER 0";
	iRnd = 12;
}

//------------------------------------------------------------------------------
// Destructor
//------------------------------------------------------------------------------
C_BlBoard::~C_BlBoard()
{
	ASSERT(s_pInstance == this);
}

//------------------------------------------------------------------------------
// Initialization
//------------------------------------------------------------------------------
int C_BlBoard::Init()
{
	ASSERT(s_pInstance == this);

	Log(m_hLog, LOG_NOTE, "Starting initialization of BlBoard");

	SE_STATUS seStatus = ADPTSE_QueryNumBoards (&m_iNumBoards);

	if (seStatus == SE_OK)
	{
//		printf ("NumBoards == %d\n", m_iNumBoards);
		if (m_iNumBoards <= 0) seStatus = SE_ERROR;
	}
	else
	{
		Log(m_hLog, LOG_ERROR,
				"Unable to query the number of boards... Will stop.");
	}

	if (seStatus == SE_OK)
	{
		ADPTSE_APP_CALLBACK sAppCallback;
		sAppCallback.pfADPTSE_MasterStatusGranted =
			&C_BlBoard::CBMasterStatusGranted;
		sAppCallback.pfADPTSE_TableSectionArrived =
			&C_BlBoard::CBTableSectionArrived;
		sAppCallback.pfADPTSE_StreamDataArrived =
			&C_BlBoard::CBStreamDataArrived;
		sAppCallback.pfADPTSE_CAMessageReceived =
			&C_BlBoard::CBCAMessageReceived;
		sAppCallback.pfADPTSE_AddMulticastAddress =
			&C_BlBoard::CBAddMulticastAddress;
		sAppCallback.pfADPTSE_RemoveMulticastAddress =
			&C_BlBoard::CBRemoveMulticastAddress;

		seStatus = ADPTSE_RegisterApp (0, &sAppCallback, 1);

		switch (seStatus)
		{
		case SE_OK :
			// We are master and we can init the card 
			Log(m_hLog, LOG_NOTE, "BlBoard registered. Initialising...");
			m_bAppRegistered = true;
			m_bAppMastered = true;
			seStatus = InitBoard();
			break;
		case SE_OK_MASTER_STATUS_PENDING:
			// We will have to wait for the master to be released: at that time
			// the callback Master status granted will be called
			Log(m_hLog, LOG_NOTE, "BlBoard registered. Waiting to be master to initialise");
			m_bAppRegistered = true;
			// En attendant d'avoir les semaphore, on boucle comme un cochon.
	    	while(!m_bAppMastered)
	           Pause(1);
			seStatus = InitBoard();
			break;
		case SE_ERROR :
			Log(m_hLog, LOG_ERROR,
				"Unable to register BlBoard... Will stop.");
			break;
		case SE_PARAMETER_OUT_OF_RANGE :
			Log(m_hLog, LOG_ERROR,
				"At least one callback function pointer is not valid!!! Will Stop.");
		}
	}

	if (seStatus == SE_OK)
	{
		Log(m_hLog, LOG_NOTE, "BlBoard initialized");
		return NO_ERR;
	}
	else
	{
		Log(m_hLog, LOG_ERROR, "BlBoard not initialized");
		return GEN_ERR;
	}
}

//------------------------------------------------------------------------------
// Execution
//------------------------------------------------------------------------------
int C_BlBoard::Run()
{
	ASSERT(s_pInstance == this);

	// Try to reconnect to the vls while it doesn't have to shutdown
	do
	{
		// Try to contact the vls
		m_cCmdLink.Open();

		// Update the PAT because the Broadlogic board can't do it
		// while capturing a PID (for instance de PID 0)
		UpdatePgrmTable();

		// Capture the PID 0 for further use
		ADPTSE_STREAM_SETUP sStreamSetup;

		sStreamSetup.captureType = TRANSPORT_STREAM;
		sStreamSetup.uiPID = 0;
		sStreamSetup.uiBuffers = 300;
		sStreamSetup.ulBufferSize = 188;
		sStreamSetup.pfCallback = &C_BlBoard::CaptureCallback0;

		SE_HANDLE seHandle;

		SE_STATUS seStatus = ADPTSE_OpenStream(0, &sStreamSetup, &seHandle);
		switch (seStatus)
		{
		case SE_OK:
			Log(m_hLog, LOG_NOTE, "Stream opened for PID 0");
			break;
		default:
			Log(m_hLog, LOG_ERROR, "Unable to capture the PID 0");
			m_cCmdLink.Close();
		}

		if (seStatus == SE_OK) seStatus = ADPTSE_StartCapture(0, seHandle);

/*		// Capture the PCRPID

		sStreamSetup.uiPID = 8190;
		sStreamSetup.pfCallback = &C_BlBoard::CaptureCallback2;
		seStatus = ADPTSE_OpenStream(0, &sStreamSetup, &seHandle);
		switch (seStatus)
		{
		case SE_OK:
			Log(m_hLog, LOG_NOTE, "Stream opened for PID 8190");
			break;
		default:
			Log(m_hLog, LOG_ERROR, "Unable to capture the PID 8190");
			m_cCmdLink.Close();
		}
		if (seStatus == SE_OK) seStatus = ADPTSE_StartCapture(0, seHandle);*/

		// Loop until cmdlink is broken
		while (m_cCmdLink.IsLinkOk())
		{
			C_String strRequest = m_cCmdLink.ReadString();

			if (m_cCmdLink.IsLinkOk())
			{
				// Parse the request
				C_StringTokenizer cTokenizer(strRequest, ' ');
				C_String strRequestString = cTokenizer.NextToken();

				//======================================================================
				//======================================================================
				if (strRequestString == "DIRECTORY_REQUEST")
				{
					// Update program table
					int iRc = UpdatePgrmTable();

					m_cCmdLink.WriteString(m_strDirectoryString);
				}
				//======================================================================
				//======================================================================
				else if (strRequestString == "INIT_STREAMING_REQUEST")
				{
					int iRc;

					if (cTokenizer.CountTokens() == 1)
						iRc = NO_ERR;
					else
					{
						Log(m_hLog, LOG_ERROR, "Bad Init Streaming request : " +
								strRequest);
						iRc = GEN_ERR;
					}

					u16 uiPgrmNumber;
					C_String strPgrmNumber;

					if (!iRc)
					{
						strPgrmNumber = cTokenizer.NextToken();
						uiPgrmNumber = atoi((const char*)strPgrmNumber);
					}

					if (!iRc)
					{
						if (m_cDataLinksByPgrmNumber.Get(uiPgrmNumber))
						{
							Log(m_hLog, LOG_ERROR, "Program already broadcasted");
							iRc = GEN_ERR;
						}
					}

					C_DataLink* pDataLink = NULL;

					if (!iRc)
					{
						pDataLink = new C_DataLink(m_hLog);
						ASSERT(pDataLink);
						iRc = pDataLink->Init();
					}

					if (!iRc)
						m_cCmdLink.WriteString("INIT_STREAMING_ANSWER OK grognon " +
																		pDataLink->GetLocalPort());
					else
						m_cCmdLink.WriteString("INIT_STREAMING_ANSWER ERROR");

					if (!m_cCmdLink.IsLinkOk())
						iRc = GEN_ERR;

					C_String strRequest2;

					if (!iRc)
					{
						strRequest2 = m_cCmdLink.ReadString();
						if (!m_cCmdLink.IsLinkOk())
							iRc = GEN_ERR;
					}

					if (!iRc)
					{
						if (strRequest2 != "NOPE")
						{
							C_StringTokenizer cTokenizer2(strRequest2, ' ');

							if (cTokenizer2.CountTokens() == 3)
							{
								C_String strRequest2String = cTokenizer2.NextToken();
								if (strRequest2String == "START_STREAMING_REQUEST")
								{
									C_String strPeerName = cTokenizer2.NextToken();
									C_String strPeerPort = cTokenizer2.NextToken();

									iRc = pDataLink->Connect(strPeerName, strPeerPort);

									if (!iRc)
									{
										Log(m_hLog, LOG_NOTE, "DataLink connected");
										iRc = StartStreaming(uiPgrmNumber, pDataLink);
									}
									if (!iRc)
										m_cCmdLink.WriteString("START_STREAMING_ANSWER OK");
									else
										m_cCmdLink.WriteString("START_STREAMING_ANSWER ERROR");
								}
								else
								{
									Log(m_hLog, LOG_ERROR, "Bad Start Streaming request string: "
											+ strRequest2String);
									iRc = GEN_ERR;
								}
							}
							else
							{
								Log(m_hLog, LOG_ERROR, "Bad Start Streaming request : " +
										strRequest2);
								iRc = GEN_ERR;
							}
						}
						else
						{
							Log(m_hLog, LOG_ERROR,
									"Vls error => don't take care the Init Streaming request");
							iRc = GEN_ERR;
						}
					}
				}
				//======================================================================
				//======================================================================
				else if (strRequestString == "STOP_STREAMING_REQUEST")
				{
					int iRc;

					if (cTokenizer.CountTokens() == 1)
						iRc = NO_ERR;
					else
					{
						Log(m_hLog, LOG_ERROR, "Bad Stop Streaming request : " +
								strRequest);
						iRc = GEN_ERR;
					}

					u16 uiPgrmNumber;
					C_String strPgrmNumber;

					if (!iRc)
					{
						strPgrmNumber = cTokenizer.NextToken();
						uiPgrmNumber = atoi((const char*)strPgrmNumber);
					}

					if (!iRc && m_cDataLinksByPgrmNumber.Get(uiPgrmNumber))
						iRc = StopStreaming(uiPgrmNumber);

					if (!iRc)
						m_cCmdLink.WriteString("STOP_STREAMING_ANSWER OK");
					else
						m_cCmdLink.WriteString("STOP_STREAMING_ANSWER ERROR");
				}
				//======================================================================
				//======================================================================
				else
				{
					Log(m_hLog, LOG_ERROR, "Unknown request : " + strRequest);
					m_cCmdLink.WriteString("TA MERE !!!!");
				}
			}
		}

		m_cCmdLink.Close();

		seStatus = ADPTSE_StopCapture(0, seHandle);
		seStatus = ADPTSE_CloseStream(0, seHandle);
		// Clean the memory (further writing)
//		>>>>>>>>>>
		Sleep(3000);
	} while (false);

	return NO_ERR;
}

//------------------------------------------------------------------------------
// Destruction
//------------------------------------------------------------------------------
int C_BlBoard::Destroy()
{
	ASSERT(s_pInstance == this);

	Log(m_hLog, LOG_NOTE, "Starting BlBoard destruction");

	SE_STATUS seStatus = SE_OK;

	if (m_bAppRegistered)
	{
		seStatus = ADPTSE_UnRegisterApp (0);
		switch (seStatus)
		{
		case SE_OK :
			Log(m_hLog, LOG_NOTE, "BlBoard unregistered");
			m_bAppMastered = false;
			m_bAppRegistered = false;
			break;
		case SE_ERROR :
			Log(m_hLog, LOG_ERROR, "Unable to unregister!!!");
			break;
		case SE_APP_NOT_REGISTERED :
			Log(m_hLog, LOG_ERROR, "BlBoard has already been unregistered!!!");
			m_bAppMastered = false;
			m_bAppRegistered = false;
		}
	}

	if (seStatus == SE_OK)
	{
		Log(m_hLog, LOG_NOTE, "BlBoard destroyed");
		return NO_ERR;
	}
	else
	{
		Log(m_hLog, LOG_ERROR, "BlBoard not correctly destroyed!!!");
		return GEN_ERR;
	}
}

//------------------------------------------------------------------------------
// Board initialization (after master status has been granted)
//------------------------------------------------------------------------------
SE_STATUS C_BlBoard::InitBoard()
{
	ASSERT(s_pInstance == this);

	Log(m_hLog, LOG_NOTE, "Starting Board initialization");

	// Reset the Board
	SE_STATUS seStatus = ADPTSE_ResetBoard(0);

	// Set the LNB Frequencies
	if (seStatus == SE_OK)
	{
		seStatus = ADPTSE_SetLNBFrequency(0, 10000000, 20000000);
		switch (seStatus)
		{
		case SE_OK :
			Log(m_hLog, LOG_NOTE, "LNB frequencies set");
			break;
		case SE_ERROR :
			Log(m_hLog, LOG_ERROR, "Unable to set the LNB frequency!!!");
			break;
		case SE_APP_NOT_MASTER :
			Log(m_hLog, LOG_ERROR, "BlBoard is not master!!!");
			break;
		case SE_PARAMETER_OUT_OF_RANGE :
			Log(m_hLog, LOG_ERROR, "A parameter is out of range!!!");
		}
	}

	// Set the Frequency of the transponder
	if (seStatus == SE_OK)
	{
		m_sSatTunerSettings.ulTunerFrequency_Khz = 11881000;
		m_sSatTunerSettings.polarization = POLARIZATION_VERTICAL;
		m_sSatTunerSettings.ulSymbolRate_KSym = 27500;
		seStatus = ADPTSE_SetTunerWithLNB(0, &m_sSatTunerSettings);
		switch (seStatus)
		{
		case SE_OK :
			Log(m_hLog, LOG_NOTE, "Tuner settings ok");
			break;
		case SE_ERROR :
			Log(m_hLog, LOG_ERROR, "Unable to set the tuner settings!!!");
			break;
		case SE_APP_NOT_MASTER :
			Log(m_hLog, LOG_ERROR, "BlBoard is not master!!!");
			break;
		case SE_TUNER_SAT_FREQUENCY_OUT_OF_RANGE :
			Log(m_hLog, LOG_ERROR, "Tuner frequency is out of range!!! Fuck!!!");
			break;
		case SE_LNB_NOT_SPECIFIED :
			Log(m_hLog, LOG_ERROR, "LNB values have not been set!!!");
		}
	}

	if (seStatus == SE_OK)
	{
		Log(m_hLog, LOG_NOTE, "Board initialized");
		m_bBoardInitialized = true;
	}
	else
	{
		Log(m_hLog, LOG_ERROR, "Board not initialized!!!");
	}

	return seStatus;
}

//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
int C_BlBoard::UpdatePgrmTable()
{
	ASSERT(s_pInstance == this);

	int iRc;

	PAT_TABLE sPatTable;

	SE_STATUS seStatus = ADPTSE_QueryProgramAssociationTable(0, &sPatTable);

	switch (seStatus)
	{
	case SE_OK:
		Log(m_hLog, LOG_NOTE, "PAT Queried");
		break;
	case SE_ERROR:
		Log(m_hLog, LOG_ERROR, "Unable to Query the PAT");
		break;
	case SE_APP_NOT_REGISTERED:
		Log(m_hLog, LOG_ERROR, "BlBoard is not registered!!!");
		break;
	case SE_PSI_PAT_NOT_PRESENT:
		Log(m_hLog, LOG_WARN, "The PAT is not present!!!");
	}

	if (seStatus == SE_OK)
	{
		if ((!m_bPatValid) || (sPatTable.ucVersion != m_sPatTable.ucVersion))
		{
			// Update the PAT and the programs captured

			// Build the PAT String
			m_strDirectoryString = "DIRECTORY_ANSWER ";
			m_strDirectoryString += sPatTable.usNumberOfElements;
			for (int i = 0; i < sPatTable.usNumberOfElements; i++)
			{
				m_strDirectoryString += " ";
				m_strDirectoryString += sPatTable.usProgramNumber[i];
				m_strDirectoryString += " ";
				m_strDirectoryString += sPatTable.usProgramNumber[i];
			}
			m_sPatTable = sPatTable;
			m_bPatValid = true;
		}

		iRc = NO_ERR;
	}
	else if (seStatus != SE_NO_FILTER_AVAILABLE)
	{
		// No programs are available
		m_bPatValid = false;
		m_strDirectoryString = "DIRECTORY_ANSWER 0";
		iRc = GEN_ERR;
	}
	else
	{
		iRc = GEN_ERR;
	}

	if (m_bPatValid)
	{
		for (int i = 0; i < m_sPatTable.usNumberOfElements; i++)
		{
			Log(m_hLog, LOG_NOTE, C_String("Query the PMT for program number ") +
					C_String(m_sPatTable.usProgramNumber[i]));
			PPMT_TABLE pPmtTable = new PMT_TABLE;
			seStatus = ADPTSE_QueryProgramMapTable(0,
																		m_sPatTable.usProgramNumber[i], pPmtTable);
			if (seStatus == SE_OK)
			{
				printf("PCR_PID : %d\n", pPmtTable->usPcrPid);
				PPMT_TABLE pOld= m_cPmtStructures.Modify(m_sPatTable.usProgramNumber[i],
																							pPmtTable);
				if (pOld)
					delete pOld;
			}
			else
				delete pPmtTable;
		}
	}

	return iRc;
}

//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
int C_BlBoard::StartStreaming(u16 uiPgrmNumber, C_DataLink* pDataLink)
{
	ASSERT(s_pInstance == this);
	ASSERT(!m_cDataLinksByPgrmNumber.Get(uiPgrmNumber));
	ASSERT(pDataLink);

	// Retrieve the PMT_TABLE structure
	PPMT_TABLE pPmtTable = m_cPmtStructures.Get(uiPgrmNumber);
	ASSERT(pPmtTable);

	// Create the PAT
	C_Pat* pPat = new C_Pat;
	ASSERT(pPat);
	pPat->BuildHeader(12, m_sPatTable.ucVersion);
	pPat->AddPgrm(uiPgrmNumber, 1);
	m_cPats.Add(uiPgrmNumber, pPat);

	// Create the PMT
	C_Pmt* pPmt = new C_Pmt;
	ASSERT(pPmt);
	pPmt->BuildHeader(1, uiPgrmNumber, pPmtTable->usPcrPid, pPmtTable->ucVersion);
	for (int i = 0; i < pPmtTable->usNumberOfElements; i++)
	{
		pPmt->AddEs(pPmtTable->paePMTTableElements[i].ucStreamType,
								pPmtTable->paePMTTableElements[i].usElementaryPID);
	}
	m_cPmts.Add(uiPgrmNumber, pPmt);

	// Initialize the Stream Setup structure
	ADPTSE_STREAM_SETUP sStreamSetup;
	sStreamSetup.captureType = TRANSPORT_STREAM;
	sStreamSetup.uiBuffers = 300;
	sStreamSetup.ulBufferSize = 188;
	sStreamSetup.pfCallback = &C_BlBoard::CaptureCallback1;

	int iRc = NO_ERR;

	// Capture the PIDs
	for (int j = 0; (j < pPmtTable->usNumberOfElements) && (!iRc); j++)
	{
		if ((pPmtTable->paePMTTableElements[j].ucStreamType == 0x02) ||
				(pPmtTable->paePMTTableElements[j].ucStreamType == 0x04))
		{
			// Register the Data Link for the given PID
			ASSERT(!m_cDataLinksByPID.Get(
									pPmtTable->paePMTTableElements[j].usElementaryPID));
			Log(m_hLog, LOG_NOTE, C_String("Adding PID ") +
					C_String(pPmtTable->paePMTTableElements[j].usElementaryPID) +
					" , stream type : " +
					C_String(pPmtTable->paePMTTableElements[j].ucStreamType));
			m_cDataLinksByPID.Add(pPmtTable->paePMTTableElements[j].usElementaryPID,
														pDataLink);

			// Start the capture of the PID
			sStreamSetup.uiPID = pPmtTable->paePMTTableElements[j].usElementaryPID;
			PSE_HANDLE pHandle = new SE_HANDLE;
			SE_STATUS seStatus = ADPTSE_OpenStream(0, &sStreamSetup, pHandle);
			if (seStatus == SE_OK)
			{
				m_cHandles.Add(sStreamSetup.uiPID, pHandle);
				seStatus = ADPTSE_StartCapture(0, *pHandle);
			}
			else
			{
				delete pHandle;
				iRc = GEN_ERR;
			}
		}
	}

	if (iRc)
	{
		for (int i = 0; (i < pPmtTable->usNumberOfElements) && (!iRc); i++)
		{
			m_cDataLinksByPID.Remove(pPmtTable->paePMTTableElements[i].usElementaryPID);
			m_cHandles.Delete(pPmtTable->paePMTTableElements[i].usElementaryPID);
		}
	}
	else
	{
		// Register the Data Link for the given program
		m_cDataLinksByPgrmNumber.Add(uiPgrmNumber, pDataLink);
		// Capture the PCR_PID
		m_cDataLinksByPID.Add(pPmtTable->usPcrPid, pDataLink);
		sStreamSetup.uiPID = pPmtTable->usPcrPid;
		PSE_HANDLE pHand = new SE_HANDLE;
		ADPTSE_OpenStream(0, &sStreamSetup, pHand);
		m_cHandles.Add(sStreamSetup.uiPID, pHand);
		ADPTSE_StartCapture(0, *pHand);
	}


	return NO_ERR;
}

//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
int C_BlBoard::StopStreaming(u16 uiPgrmNumber)
{
	ASSERT(s_pInstance == this);

	C_DataLink* pDataLink = m_cDataLinksByPgrmNumber.Get(uiPgrmNumber);
	ASSERT(pDataLink);

	PPMT_TABLE pPmtTable = m_cPmtStructures.Get(uiPgrmNumber);
	ASSERT(pPmtTable);

	// Clean the hashtables
	for (int i = 0; i < pPmtTable->usNumberOfElements; i++)
	{
		// Get the PID
		u16 uiPID = pPmtTable->paePMTTableElements[i].usElementaryPID;

		// Stop capture of the PID
		// Must be called first to prevent segmentation faults
		PSE_HANDLE pHandle = m_cHandles.Get(uiPID);
		if (pHandle)
		{
			ADPTSE_StopCapture(0, *pHandle);
			ADPTSE_CloseStream(0, *pHandle);
			m_cHandles.Delete(uiPID);
		}

		// Remove the link from the ByPID hashtable
		C_DataLink* pLink = m_cDataLinksByPID.Get(uiPID);
		if (pLink)
			m_cDataLinksByPID.Remove(uiPID);
		else
			ASSERT(false);
	}
	// Remove the PCR_PID
	u16 uiPcrPID = pPmtTable->usPcrPid;
	PSE_HANDLE pPcrHandle = m_cHandles.Get(uiPcrPID);
	if (pPcrHandle)
	{
		ADPTSE_StopCapture(0, *pPcrHandle);
		ADPTSE_CloseStream(0, *pPcrHandle);
		m_cHandles.Delete(uiPcrPID);
	}
	C_DataLink* pPcrLink = m_cDataLinksByPID.Get(uiPcrPID);
	if (pPcrLink)
		m_cDataLinksByPID.Remove(uiPcrPID);
	else
		ASSERT(false);

	// Delete the link
	m_cDataLinksByPgrmNumber.Delete(uiPgrmNumber);

	// Delete the PAT
	m_cPats.Delete(uiPgrmNumber);

	// Delete the PMT
	m_cPmts.Delete(uiPgrmNumber);

	return NO_ERR;
}

//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_BlBoard::CaptureCallback0(SE_UINT iBoardNumber, SE_HANDLE seHandle,
																	PSE_VOID pvData, SE_UINT uiDataBlockLength)
{
	byte* pData = (byte*)pvData;
	u16 uiPID = ((*(pData + 1) & 0x1f) << 8) + *(pData + 2);
//	printf("CB0 : hey guy ! I have a packet for PID %d !\n", uiPID);
	C_HashTableIterator<u16, C_DataLink>* pIterator =
											s_pInstance->m_cDataLinksByPgrmNumber.CreateIterator();
	while (pIterator->HasNext())
	{
		C_HashTableNode<u16, C_DataLink>* pNode = pIterator->GetNext();

		// Send the PAT
		C_TsPacket* pPacket = new C_TsPacket;
		ASSERT (pPacket);
		s_pInstance->m_cPats.Get(pNode->GetKey())->Write(pPacket);
		pNode->GetValue()->WriteTsPacket((byte*)(*pPacket));
		delete pPacket;

		// Send the PMT
		pPacket = new C_TsPacket;
		ASSERT (pPacket);
		s_pInstance->m_cPmts.Get(pNode->GetKey())->Write(pPacket);
		pNode->GetValue()->WriteTsPacket((byte*)(*pPacket));
		delete pPacket;
	}
	delete pIterator;

// To parse the PAT in hexadecimal format :)))
//	for (int i = 0; i < 94; i++)
//	{
//		printf("%X %X ", *(pData + (i << 1))/* & 0x1fff*/,
//											*(pData + (i << 1) + 1)/* & 0x1fff*/);
//	}
//	printf("\n");
	ADPTSE_FreeBuffer(iBoardNumber, seHandle, pvData);
}

//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_BlBoard::CaptureCallback1(SE_UINT iBoardNumber, SE_HANDLE seHandle,
																	PSE_VOID pvData, SE_UINT uiDataBlockLength)
{
	byte* pData = (byte*)pvData;
	u16 uiPID = ((*(pData + 1) & 0x1f) << 8) + *(pData + 2);
//	if (uiPID == 267)
//		printf("CB1 : hey guy ! I have a packet for PID %d !\n", uiPID);
	s_pInstance->m_cDataLinksByPID.Get(uiPID)->WriteTsPacket(pData);
	ADPTSE_FreeBuffer(iBoardNumber, seHandle, pvData);
}

//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_BlBoard::CaptureCallback2(SE_UINT iBoardNumber, SE_HANDLE seHandle,
																	PSE_VOID pvData, SE_UINT uiDataBlockLength)
{
	byte* pData = (byte*)pvData;
	u16 uiPID = ((*(pData + 1) & 0x1f) << 8) + *(pData + 2);
//	printf("CB2 : hey guy ! I have a packet for PID %d !\n", uiPID);
	C_HashTableIterator<u16, C_DataLink>* pIterator =
											s_pInstance->m_cDataLinksByPgrmNumber.CreateIterator();
	while (pIterator->HasNext())
	{
		C_HashTableNode<u16, C_DataLink>* pNode = pIterator->GetNext();
		pNode->GetValue()->WriteTsPacket(pData);
	}
	delete pIterator;
	ADPTSE_FreeBuffer(iBoardNumber, seHandle, pvData);
}

//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_BlBoard::CBMasterStatusGranted (SE_UINT iBoardNumber)
{
	// Faudra foutre un semphore ici
	Log(C_BlBoard::s_pInstance->m_hLog, LOG_NOTE, "Master status granted");
	C_BlBoard::s_pInstance->m_bAppMastered = true;
}

//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_BlBoard::CBTableSectionArrived (SE_UINT iBoardNumber,
							SE_HANDLE seHandle, PVOID pvTableSection,
							SE_UINT uiDataBlockLength)
{
	Log(C_BlBoard::s_pInstance->m_hLog, LOG_NOTE, "CB Table section arrived");
}

//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_BlBoard::CBStreamDataArrived (SE_UINT iBoardNumber, SE_HANDLE seHandle,
						  PVOID pvStreamData, SE_UINT uiDataBlockLength)
{
	Log(C_BlBoard::s_pInstance->m_hLog, LOG_NOTE, "CB Stream data arrived");
}

//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_BlBoard::CBCAMessageReceived (SE_UINT iBoardNumber, PVOID pvMessage,
						  SE_INT iMessageLength)
{
	Log(C_BlBoard::s_pInstance->m_hLog, LOG_NOTE, "CB CA message received");
}

//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_BlBoard::CBAddMulticastAddress (SE_UINT iBoardNumber,
							SE_ULONG ulEthernetAddress)
{
	Log(C_BlBoard::s_pInstance->m_hLog, LOG_NOTE, "CB Add multicast address");
}

//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_BlBoard::CBRemoveMulticastAddress (SE_UINT iBoardNumber,
							   SE_ULONG ulEthernetAddress)
{
	Log(C_BlBoard::s_pInstance->m_hLog, LOG_NOTE, "CB Remove multicast address");
}
