/*
 * audiodev.c: installs the "/dev/audio" device. It is an
 * STe DMA sound device. This program is free software; see the
 * file "COPYING" for details.
 *
 * This file must be compiled with 16-bit integers.
 *
 * This program was created starting from Thierry Bousch's lpdev v0.6
 *
 * > Author:  Thierry Bousch (bousch@suntopo.matups.fr)
 * > Version: 0.6 (july 93)
 *
 * Author:  Charles Briscoe-Smith (currently cpbs@ukc.ac.uk)
 * Version: 0.1 (April 1995)
 *
 * audiodev revision history:
 *   Fri Apr 14 23:55:26 UCT 1995  (cpbs)
 *      It seems to work.  Released v0.1 on MiNT mailing list.
 *
 * lpdev revision history:
 *  0.1: First attempt, using SLEEP instead of NAP: it didn't work.
 *  0.2: Added version number, napping in lp_write(), and the TIOCFLUSH
 *        ioctl function. Cleaned up things a bit.
 *  0.3: Introduced spl7() and spl() to fix competition problems. Added
 *        a few tests before installation, to check that MiNT is running
 *        and the device is not already installed.
 *  0.4: Added file locking. This is completely untested, so be
 *        careful. More cleanup and sanity checks during installation.
 *        Modified the sleep conditions in lp_write and lp_select.
 *  0.5: Deleted the now unnecessary stuff about low and high water marks.
 *        More comments added.
 *  0.6: Added support for the O_NDELAY and O_LOCK flags, inlined spl7/spl.
 *        Moved the definitions to lpdev.h.
 */

#include "audiodev.h"
#include "dmasnd.h"
#define  AUDIO_VERSION	"0.1"

/*
 * Global variables
 */

char *volatile buffer1_start, *volatile buffer1_tail;
char *volatile buffer2_start, *volatile buffer2_tail;
volatile int filling_buffer, playing_buffer, next_buffer;
volatile long buffer_contents;
long selector = 0L;
struct kerinfo *kernel;
struct flock our_lock = { F_WRLCK, 0, 0L, 0L, -1 };

/*
 * Forward declarations of the device driver functions
 */

long	audio_open	(FILEPTR *f);
long	audio_write	(FILEPTR *f, char *buf, long bytes);
long	audio_read	(FILEPTR *f, char *buf, long bytes);
long	audio_lseek	(FILEPTR *f, long where, int whence);
long	audio_ioctl	(FILEPTR *f, int mode, void *buf);
long	audio_datime	(FILEPTR *f, int *timeptr, int rwflag);
long	audio_close	(FILEPTR *f, int pid);
long	audio_select	(FILEPTR *f, long proc, int mode);
void	audio_unselect	(FILEPTR *f, long proc, int mode);

DEVDRV audio_device = {
	audio_open, audio_write, audio_read, audio_lseek, audio_ioctl,
	audio_datime, audio_close, audio_select, audio_unselect
};

struct dev_descr devinfo = { &audio_device };

/* Initializes the DMA buffers and DMA hardware */

void reset_buffers (void)
{
	int sr = spl7();
	buffer1_tail = buffer1_start;
	buffer2_tail = buffer2_start;
	filling_buffer = 1;
	playing_buffer = 0;
	next_buffer = 0;
	DMActl = 0;
	DMAsmc = 0x0081;
	buffer_contents = 0L;
	spl(sr);
}

/* Copyright information */

void Version (void)
{
	Cconws("STe DMA sound device driver, by C.P.Briscoe-Smith (version "
	AUDIO_VERSION ").\r\n"
	"Based on buffered Centronics parallel driver v0.6 by T.Bousch\r\n"
	"This program is FREE SOFTWARE, and comes with NO WARRANTY.\r\n"
	"See the file \"COPYING\" for more information.\r\n");
}

/*
 * Installs everything, returns 0 on success. Must be executed in
 * supervisor mode.
 */

long install_things (void)
{
	if (Syield() == EINVFN) {
		Cconws("audiodev: MiNT is not running\r\n");
		return EACCDN;
	}
	if (Fsfirst(DEVNAME, 0) == 0) {
		Cconws("audiodev: device \"" DEVNAME "\" already installed\r\n");
		return EACCDN;
	}
	if (*(char*)0xFFFFFA07 &		/* IERB */
	    *(char*)0xFFFFFA13 & 0x20) {	/* IMRB */
		Cconws("audiodev: Timer A interrupt already in use\r\n");
		return EACCDN;
	}
	buffer1_start = (char *)Malloc(BUFSIZE);
	if (buffer1_start)
		buffer2_start = (char *)Malloc(BUFSIZE);
	if (!buffer1_start || !buffer2_start) {
		Cconws("audiodev: not enough memory\r\n");
		if (buffer1_start)
			free(buffer1_start);
		return ENSMEM;
	}
	reset_buffers();

	kernel = (struct kerinfo *)Dcntl(DEV_INSTALL, DEVNAME, &devinfo);
	if ((long)kernel <= 0L) {
		Cconws("audiodev: unable to install device\r\n");
		return EACCDN;
	}
	/* Finally! */
	Xbtimer(0, 8, 1, new_timera_vector);
/*	Jenabint(13); */

	return 0;
}

/*
 * The main routine is very simple now, it just calls install_things
 * and remains resident if everything went well
 */

int main()
{
	long ret;

	ret = Supexec(install_things);
	if (ret < 0)
		return ret;

	/* Installation is complete */
	Version();
	Ptermres(256L + _base->p_tlen + _base->p_dlen + _base->p_blen, 0);
	return -999;	/* never reached, just to make Gcc happy */
}

/*
 * Will wake any process select'ing the printer;
 * this routine is called by the interrupt handler, but also when the
 * buffer is flushed.
 */

void wake_up (void)
{
	if (selector)
		WAKESELECT(selector);	/* wake selector */
}

/*
 * do_timera is called every time the DMA hardware finishes playing a
 * sample frame.
 */

void
do_timera(void)
{
	int sr = spl7();

	if (next_buffer)
	{
		filling_buffer=playing_buffer;
		playing_buffer=next_buffer;
		next_buffer=0;
		DMActl=1;

		if (filling_buffer==1)
			buffer1_tail=buffer1_start;
		else
			buffer2_tail=buffer2_start;

		buffer_contents=0L;
		wake_up();
	}
	else
		playing_buffer=0;

	spl(sr);
}

#if 0
/*
 * Sends as many bytes as possible (usually one) to the printer until
 * he gets busy. This routine is called by audio_write and by the
 * interrupt handler, so it _must_ be multi-thread. It will not work if
 * you remove the spl7()/spl() pair.
 *
 * On a more general note, it is safest to disable all interrupts before
 * modifying the volatile variables (buffer_contents and buffer_head).
 */

#define PRINTER_BUSY	(*(char*)0xFFFFFA01 & 1)

void print_head (void)
{
	int sr = spl7();
	while (!PRINTER_BUSY && buffer_contents) {
		print_byte( *buffer_head );
		--buffer_contents;
		if (++buffer_head >= buffer_end)
			buffer_head -= BUFSIZE;
	}
	spl(sr);
}
#endif

/*
 * Copies from the user's buffer into the DMA buffer currently being
 * filled. We assume that's there enough room for this operation, ie
 * nbytes + buffer_contents <= BUFSIZE
 *
 * Note: the while() loop will be executed at most twice.
 * Note2: the instruction "buffer_contents += N" looks atomic, but it
 *   isn't (the Gcc outputs several assembly instructions). Therefore it
 *   must be wrapped in spl7()/spl().
 */

/* TODO: reduce time at IPL7 */

void copy_in (char *buf, long nbytes)
{
	int sr = spl7();

	switch (filling_buffer)
	{
	case 1:
		bcopy(buf, buffer1_tail, nbytes);
		buffer_contents += nbytes;
		buffer1_tail+=nbytes;
		if (buffer_contents>=MIN_PLAY && playing_buffer==0)
		{
			/* set buffer1 to play, buffer2 to fill */
			DMASetBase(buffer1_start);
			DMASetEnd(buffer1_tail);
			DMActl=1;
			playing_buffer=1;
			next_buffer=0;

			filling_buffer=2;
			buffer_contents=0L;
			buffer2_tail=buffer2_start;
		}
		else if (playing_buffer)
		{
			/* set buffer1 to play next */
			DMASetBase(buffer1_start);
			DMASetEnd(buffer1_tail);
			DMActl=3;

			next_buffer=1;
		}
		break;
	case 2:
		bcopy(buf, buffer2_tail, nbytes);
		buffer_contents += nbytes;
		buffer2_tail+=nbytes;
		if (buffer_contents>=MIN_PLAY && playing_buffer==0)
		{
			/* set buffer2 to play, buffer1 to fill */
			DMASetBase(buffer2_start);
			DMASetEnd(buffer2_tail);
			DMActl=1;
			playing_buffer=2;
			next_buffer=0;

			filling_buffer=1;
			buffer_contents=0L;
			buffer1_tail=buffer1_start;
		}
		else if (playing_buffer)
		{
			/* set buffer2 to play next */
			DMASetBase(buffer2_start);
			DMASetEnd(buffer2_tail);
			DMActl=3;

			next_buffer=2;
		}
		break;
	}
	spl(sr);
}

/*
 * Here are the actual device driver functions
 */

#define  AUDIO_LOCKED	(our_lock.l_pid >= 0)

long audio_open (FILEPTR *f)
{
	TRACE(("audio: open device"));
	return 0;
}

long audio_close (FILEPTR *f, int pid)
{
	TRACE(("audio: close device"));
	if ((f->flags & O_LOCK) && our_lock.l_pid == pid) {
		TRACE(("audio: releasing lock on close"));
		f->flags &= ~O_LOCK;
		our_lock.l_pid = -1;
		WAKE(IO_Q, (long)&our_lock);
	}
	return 0;
}

long audio_read (FILEPTR *f, char *buf, long bytes)
{
	TRACE(("audio: foolish attempt to read"));
	return 0;
}

long audio_datime (FILEPTR *f, int *timeptr, int rwflag)
{
	if (rwflag) {
		DEBUG(("audio: can't modify date/time"));
		return EACCDN;
	}
	TRACE(("audio: read time and date"));
	*timeptr++ = TGETTIME();
	*timeptr   = TGETDATE();
	return 0;
}

long audio_lseek (FILEPTR *f, long where, int whence)
{
	TRACE(("audio: foolish attempt to seek"));
	if (whence < 0 || whence > 2)
		return EINVFN;
	return where ? ERANGE : 0L;
}

long audio_ioctl (FILEPTR *f, int mode, void *buf)
{
	struct flock *g;

	if (mode == FIONREAD) {
		TRACE(("audio: ioctl(FIONREAD)"));
		*(long *)buf = 0L;
	}
	else if (mode == FIONWRITE) {
		TRACE(("audio: ioctl(FIONWRITE)"));
		*(long *)buf = BUFSIZE - buffer_contents;
	}
	else if (mode == TIOCFLUSH) {
		TRACE(("audio: clear buffer"));
		reset_buffers();
		wake_up();	/* Wake up any select'ing process */
	}
	else if (mode == F_GETLK) {
		g = (struct flock *) buf;

		if (AUDIO_LOCKED) {
			TRACE(("audio: get_lock succeeded"));
			*g = our_lock;
		} else {
			TRACE(("audio: get_lock failed"));
			g->l_type = F_UNLCK;
		}
	}
	else if (mode == F_SETLK || mode == F_SETLKW) {
		g = (struct flock *) buf;

		switch (g->l_type) {
		case F_UNLCK:
		    if (!(f->flags & O_LOCK) || g->l_pid != our_lock.l_pid) {
			DEBUG(("audio: no such lock"));
			return ENSLOCK;
		    } else {
			TRACE(("audio: remove lock"));
			f->flags &= ~O_LOCK;
			our_lock.l_pid = -1;
			WAKE(IO_Q, (long)&our_lock);
		    }
		    return 0;
		case F_RDLCK:
		    TRACE(("audio: read locks are ignored"));
		    return 0;
		case F_WRLCK:
		    while (AUDIO_LOCKED) {
			DEBUG(("audio: conflicting locks"));
			if (mode == F_SETLK) {
				*g = our_lock;
				return ELOCKED;
			}
			SLEEP(IO_Q, (long)&our_lock);
		    }
		    TRACE(("audio: set lock"));
		    f->flags |= O_LOCK;
		    our_lock.l_pid = g->l_pid;
		    return 0;
		default:
		    DEBUG(("audio: invalid lock type"));
		    return EINVFN;
		}
	}
	else {
		DEBUG(("audio: invalid ioctl mode"));
		return EINVFN;
	}
	return 0;
}

long audio_write (FILEPTR *f, char *buf, long bytes)
{
	long _bytes = bytes;
	long N;
	int  ndel = (f->flags & O_NDELAY);	/* don't wait */

	while (bytes) {
		N = BUFSIZE - buffer_contents;
		/*
		 * If the data won't fit into the buffer,
		 * and if the buffer itself is almost full, we won't
		 * be able to copy much. So we better sleep a bit (unless
		 * the O_NDELAY flag is set).
		 */
		if (N < bytes && N < BUFSIZE/4 && !ndel) {
			TRACE(("audio: napping in audio_write"));
			NAP(200);	/* let's wait 200 milliseconds */
			continue;	/* and try again */
		}
		if (bytes < N)
			N = bytes;
		/* Now N contains the number of bytes we want to copy
		 * into the buffer */
		copy_in(buf, N);
		buf += N;
		bytes -= N;
		/*
		 * If the O_NDELAY flag is set, we don't make a second
		 * attempt to write the remaining "bytes" bytes.
		 */
		if (ndel)  break;
	}
	TRACE(("audio: wrote %ld bytes, skipped %ld", _bytes-bytes, bytes));
	return _bytes - bytes;
}

/* Bug: only one process can select the audio device */

long audio_select (FILEPTR *f, long proc, int mode)
{
	if (buffer_contents == BUFSIZE && !selector) {
		TRACE(("audio: select returned 0"));
		selector = proc;
		return 0;
	}
	TRACE(("audio: select returned 1"));
	return 1;
}

void audio_unselect (FILEPTR *f, long proc, int mode)
{
	TRACE(("audio: unselect"));
	selector = 0L;
}
