/*
 * $Header: /u1/src/rfmail/RCS/zm.c,v 0.5.0.1 1992/06/15 06:11:25 pgd Exp pgd $
 *
 * $Log: zm.c,v $
 * Revision 0.5.0.1  1992/06/15  06:11:25  pgd
 * Minor compilation bug fixes.
 * Change of all types with u_ prefix to U prefix
 * Change of name of routine msleep() to mssleep()
 *
 * Revision 0.5  1992/05/18  04:27:24  pgd
 * New distribution
 *
 * Revision 0.4.1.6  1992/03/15  07:58:52  pgd
 * Untested version
 *
 * Revision 0.4  1991/05/08  04:23:43  pgd
 * Initial Beta-release
 *
 *
 */

/* File: zm.c */

/*
 * Zmodem file receive/send misc code.
 *
 * Version for rfmail made by Per Lindqvist, pgd@compuram.bbt.se
 *
 * This version in based on the program zm.c by Chuck Forsberg,
 * and on the zmodem code from Binkleyterm by Robert C. Hartman
 * and Vincent E. Perillo
 *
 *   Z M . C
 *    ZMODEM protocol primitives
 *    8-16-86  Chuck Forsberg Omen Technology Inc
 *
 * Entry point Functions:
 *      zsbhdr(type, hdr) send binary header
 *      zshhdr(type, hdr) send hex header
 *      zgethdr(hdr, rxpos) receive header - binary or hex
 *      stohdr(pos) store position data in Txhdr
 *      long rclhdr(hdr) recover position offset from header
 */


#include "fnet.h"

#include <sys/stat.h>

#include "fcall.h"
#include "zmodem.h"
#include "crc.h"

static char hex[] = "0123456789abcdef";

/* Send a byte as two hex digits */
#define zputhex(c) {register I=(c);xsendline(hex[((I)&0xF0)>>4]);xsendline(hex[(I)&0xF]);}

/*
 * Globals used by ZMODEM functions
 */

static int Rxtype;		/* Type of header received */
int Rxtimeout = 10;		/* seconds to wait for something */
char Rxhdr[ZMAXHLEN];		/* Received header */
char Txhdr[ZMAXHLEN];	/* Transmitted header */
char Attn[ZATTNLEN+1];	/* String rx sends to tx on err */
int Crc32t;		/* Controls 32 bit CRC being sent */
			/* 1 = CRC32, 2 = CRC32 + RLE */
int Crc32r;		/* Indicates/controls 32 bit CRC being received */
			/* 0 = CRC16, 1 = CRC32, 2 = CRC32 + RLE */
extern int Txfcs32;	/* TRUE means send binary frames with 32 bit FCS */
boolean Usevhdrs;	/* Use variable length headers */
int Znulls;		/* Number of nulls to send at beginning of ZDATA hdr */
extern int baudrate;	/* Baudrate for line */
char filename[PATH_LEN]; /* Real file name (path) */
int zsize;
char secbuf[WAZOOMAX];	/* Buffer */
int Rxframeind;		/* ZBIN ZBIN32, or ZHEX type of frame */
int Rxhlen;		/* Length of header received */
static int Not8bit;	/* Seven bits seen on header */

static char *frametypes[] = {
	"No Response to Error Correction Request", /* -4 */
        "CARRIER LOST",			/* -3 */
        "TIMEOUT",			/* -2 */
        "ERROR",			/* -1 */
#define FTOFFSET 4
        "ZRQINIT",
        "ZRINIT",
        "ZSINIT",
        "ZACK",
        "ZFILE",
        "ZSKIP",
        "ZNAK",
        "ZABORT",
        "ZFIN",
        "ZRPOS",
        "ZDATA",
        "ZEOF",
        "ZFERR",
        "ZCRC",
        "ZCHALLENGE",
        "ZCOMPL",
        "ZCAN",
        "ZFREECNT",
        "ZCOMMAND",
        "ZSTDERR",
        "xxxxx"
#define FRTYPES 22      /* Total number of frame types in this array */
                        /*  not including psuedo negative entries */
};


LDECLARE(long, rclhdr, (char *));
LDECLARE(int, noxread7, (void));
LDECLARE(int, zrbhdr, (char *));
LDECLARE(int, zrbhdr32, (char *));
LDECLARE(int, zrhhdr, (char *));
LDECLARE(int, zgethex, (void));


/*
 * Send ZMODEM binary header hdr of type type
 */
void 
zsbhdr(type, hdr)
	int type;
	register char *hdr;
{
	int n;
	int len = 4;

	debug(4, "zsbhdr: %c %ld %s %lx", Usevhdrs?'v':'f', len,
	      frametypes[type+FTOFFSET], rclhdr(hdr));

	if (type == ZDATA)
		for (n = Znulls; --n >= 0; )
			zsendline(0);
	xsendline(ZPAD);
	xsendline(ZDLE);
	if (Crc32t = Txfcs32) {
		Ulong crc = 0xffffffffU;
		
		xsendline(Crc32t == 2
			  ? (Usevhdrs ? ZVBINR32:ZBINR32)
			  : (Usevhdrs ? ZVBIN32:ZBIN32));
		if (Usevhdrs)
			zsendline(len);
		zsendline(type);
		crc = updcrc32(type, crc);
		for (n = len; --n >= 0;) {
			register int c = *hdr++ & 0377;
			zsendline(c);
			crc = updcrc32(c, crc);
		}
		crc = ~crc;
		for (n = len; --n >= 0;) {
			zsendline(crc);
			crc >>= 8;
		}
	} else {
		register Ushort crc;

		xsendline(Usevhdrs ? ZVBIN:ZBIN);
		if (Usevhdrs)
			zsendline(len);
		zsendline(type);
		crc = updcrc(type, 0);
		for (n = len; --n >= 0;) {
			register int c = *hdr++ & 0377;
			zsendline(c);
			crc = updcrc(c, crc);
		}
		zsendline((crc >> 8));
		zsendline(crc);
	}
	if (type != ZDATA)
		drainline(1);
	else
		drainline(0);
}


/*
 * Send ZMODEM HEX header hdr of type type
 */
void 
zshhdr(type, hdr)
	int type;
	register char *hdr;
{
	register int n;
	int len = 4;
	register Ushort crc;

	debug(4, "zshhdr: %c %d %s %lx", Usevhdrs ? 'v':'f', len, 
	      frametypes[type+FTOFFSET], rclhdr(hdr));
	uncorktransmitter();			/* Get our transmitter going */
	xsendline(ZPAD);
	xsendline(ZPAD);
	xsendline(ZDLE);
	xsendline(Usevhdrs ? ZVHEX:ZHEX);
	if (Usevhdrs)
		zputhex(len);
	zputhex(type);
	Crc32t = 0;
	crc = updcrc(type, 0);
	for (n = len; --n >= 0;) {
		register int c = *hdr++ & 0377;
		zputhex(c);
		crc = updcrc(c, crc);
	}
/*	crc = updcrc(0,updcrc(0,crc)); */
	zputhex((crc >> 8));
	zputhex(crc);

	/* Make it printable on remote machine */
	xsendline(015);
	xsendline(012);

	/* Uncork the remote in case a fake XOFF has stopped data flow */
	if (type != ZFIN && type != ZACK)
		xsendline(021);
	drainline(1);
}



/*
 * Read a ZMODEM header to hdr, either binary or hex.
 *   On success, set Zmodem to 1 and return type of header.
 *   Otherwise return negative on error
 */
int 
zgethdr(hdr, frxpos)
	char *hdr;
	long *frxpos;
{
	register int c;
	register int n;
	int cancount;
	long rxpos = 0;

	debug(5, "zgethdr");
	n = baudrate;				/* Max characters before */
						/* start of frame */
	cancount = 5;
again:
	Rxframeind = Rxtype = 0;
	switch (c = noxread7()) {

	case ZPAD | 0200:		/* This is what we want */
		debug(5, "zgethdr got ZPAD|0200");
		Not8bit = c;
	case ZPAD:			/* This is what we want */
		debug(5, "zgethdr got ZPAD");
		break;

	case TIMEOUT:
		debug(5, "zgethdr got TIMEOUT");
		goto done;

	case CAN:
	gotcan:
		debug(5, "zgethdr got CAN");
		if (--cancount <= 0) {
			c = ZCAN;
			goto done;
		}
		switch (c = readline(0)) {
		case TIMEOUT:
			goto again;

		case ZCRCW:
			c = ERROR;
			/* fallthrough... */

		case CAN:
			if (--cancount <= 0) {
				c = ZCAN;
				goto done;
			}
			goto again;
		}
		/* fallthrough... */

	default:
		debug(5, "zgethdr get illegal %c (%#x)",
		      c >= ' ' && c < 127 ? c : ' ', c);
agn2:
		if (--n <= 0) {
			log("zmodem 'zgethdr' - too many errors");
			c = ERROR;
			goto done;
		}

		if (c != CAN)
			cancount = 5;
		goto again;

	}                                          

	cancount = 5;

splat:
	debug(5, "zgethdr splat...");

	switch (c = noxread7()) {
	case ZDLE:
		debug(5, "zgethdr splat got ZDLE");
		/*
		 * This is what we want.
		 */
		break;

	case ZPAD:
		debug(5, "zgethdr splat got ZPAD");
		goto splat;

	case TIMEOUT:
		debug(5, "zgethdr splat got TIMEOUT");
		goto done;

	default:
		debug(5, "zgethdr get illegal %c (%#x)",
		      c >= ' ' && c < 127 ? c : ' ', c);
		goto agn2;
	}                                          

	Rxhlen = 4;			/* Default header length */
	Rxframeind = c = noxread7();
	debug(5, "zgethdr is reading header...");
	switch (c) {

	case ZBIN:
		if (Usevhdrs)
			goto agn2;
		Crc32r = 0;
		c = zrbhdr(hdr);
		break;

	case ZVBIN:
		if ((Rxhlen = c = readline(Rxtimeout*1000)) < 0)
			goto done;
		if (c > ZMAXHLEN)
			goto agn2;
		Crc32r = 0;
		c = zrbhdr(hdr);
		break;

	case ZBIN32:
		if (Usevhdrs)
			goto agn2;
		Crc32r = 1;
		c = zrbhdr32(hdr);
		break;

	case ZVBIN32:
		if ((Rxhlen = c = readline(Rxtimeout*1000)) < 0)
			goto done;
		if (c > ZMAXHLEN)
			goto agn2;
		Crc32r = 1;
		c = zrbhdr32(hdr);
		break;

	case ZHEX:
		if (Usevhdrs)
			goto agn2;
		Crc32r = 0;
		c = zrhhdr(hdr);
		break;

	case ZVHEX:
		if ((Rxhlen = c = zgethex()) < 0)
			goto done;
		if (c > ZMAXHLEN)
			goto agn2;
		Crc32r = 0;
		c = zrhhdr(hdr);
		break;

	case CAN:
		goto gotcan;

	case TIMEOUT:
		goto done;

	default:
		goto agn2;

	}                                          
	rxpos = rclhdr(hdr);
	if (frxpos)
		*frxpos = rxpos;
done:
	if (c >= -FTOFFSET && c <= FRTYPES)
		debug(4, "zmodem zgethdr: %c %ld %s %lx",
		      Rxframeind ? Rxframeind : ' ', Rxhlen,
		      frametypes[c+FTOFFSET], rxpos);
	else
		debug(4, "zmodem zgethdr: %d %lx", c, rxpos);
	if (c >= 0 && c <= FRTYPES && Rxframeind & 040)
		Usevhdrs = TRUE;
	debug(5, "zgethdr returns %d", c);
	return c;
}

/*
 * Receive a binary style header (type and position)
 */
static int 
zrbhdr(hdr)
	register char *hdr;
{
	register int c;
	register Uint crc;
	register int n;
	int len = 4;

	debug(4, "zrbhdr ...");
	if ((c = zdlread()) & ~0377)
		return c;
	Rxtype = c;
	crc = updcrc(c, 0);

	for (n = len; --n >= 0;) {
		if ((c = zdlread()) & ~0377)
			return c;
		crc = updcrc(c, crc);
		*hdr++ = c;
	}
	if ((c = zdlread()) & ~0377)
		return c;

	crc = updcrc(c, crc);
	if ((c = zdlread()) & ~0377)
		return c;

	crc = updcrc(c, crc);
	if (crc & 0xFFFF) {
		log("zmodem CRC error");
		return ERROR;
	}
	return Rxtype;
}

/*
 * Receive a hex style header (type and position)
 */
static int 
zrhhdr(hdr)
	register char *hdr;
{
	register int c;
	register Uint crc;
	register int n;

	debug(4, "zrhhdr ...");
	if ((c = zgethex()) < 0)
		return c;
	Rxtype = c;
	crc = updcrc(c, 0);

	for (n = Rxhlen; --n >= 0;) {
		if ((c = zgethex()) < 0)
			return c;
		crc = updcrc(c, crc);
		*hdr++ = c;
	}

	if ((c = zgethex()) < 0)
		return c;
	crc = updcrc(c, crc);
	if ((c = zgethex()) < 0)
		return c;
	crc = updcrc(c, crc);
	if (crc & 0xFFFF) {
		log("zmodem CRC error");
		return ERROR;
	}
	switch (c = readline(1*1000)) {
	case 0215:
		Not8bit = c;
	case 015:
		if ((c = readline(1*1000)) == 012)
			Not8bit |= c;
	}
	return Rxtype;
}


/*
 * Send character c with ZMODEM escape sequence encoding.
 * Escape XON, XOFF. Escape CR following @ (Telenet net escape)
 */
void 
zsendline(c)
	register int c;
{
	static int lastsent;

	c &= 0377;
	switch (c) {
	case 0215:
	case 015:
		if ((lastsent & 0177) != '@')
			goto sendit;
	case 020:
	case 021:
	case 023:
	case 0220:
	case 0221:
	case 0223:
	case ZDLE:
		/*
		 * Quoted characters
		 */
		xsendline(ZDLE);
		c ^= 0100;

	default:
		/*
		 * Normal character output
		 */
sendit:
		xsendline(lastsent = c);
	}                                          
}


/*
 * Decode two lower case hex digits into an 8 bit byte value
 */
static int
zgethex()
{
	register int c, n;

	debug(4, "zgethex...");
	if ((n = noxread7()) < 0) {
		debug(4, "zgethex %d returning", n);
		return n;
	}
	debug(4, "zgethex got %d", n);
	n -= '0';
	if (n > 9)
		n -= ('a' - ':');
	if (n & ~0xF) {
		debug(4, "zgethex return ERROR");
		return ERROR;
	}

	if ((c = noxread7()) < 0) {
		debug(4, "zgethex %d %d returning", n, c);
		return c;
	}
	debug(4, "zgethex got %d %d", n, c);
	c -= '0';
	if (c > 9)
		c -= ('a' - ':');
	if (c & ~0xF) {
		debug(4, "zgethex return ERROR");
		return ERROR;
	}
	debug(4, "zgethex successful, return %d", n << 4 | c);

	return (n << 4 | c);
}


/*
 * Read a byte, checking for ZMODEM escape encoding
 * including CAN*5 which represents a quick abort
 */
int
zdlread()
{
	register int c;

	if ((c = readline(Rxtimeout*1000)) != ZDLE)
		return c;

	switch (c = readline(Rxtimeout*1000)) {
	case CAN:
		return ((c = readline(Rxtimeout*1000)) < 0) ? c :
		    ((c == CAN) && ((c = readline(Rxtimeout*1000)) < 0)) ? c :
		    ((c == CAN) && ((c = readline(Rxtimeout*1000)) < 0)) ? c : ZCAN;

	case ZCRCE:
	case ZCRCG:
	case ZCRCQ:
	case ZCRCW:
		return (c | GOTOR);

	case ZRUB0:
		return 0177;

	case ZRUB1:
		return 0377;

	default:
		return (c < 0) ? c :
		    ((c & 0x60) == 0x40) ? (c ^ 0x40) : ERROR;

	}
}


/*
 * Read a character from the modem line with timeout.
 * Eat parity, XON and XOFF characters.
 */
static int
noxread7()
{
	register int c;

	for (;;) {
		if ((c = readline(Rxtimeout*1000)) < 0) {
			debug(4, "noxread7 TIMEOUT");
			return c;
		}
		c &= 0177;
		switch (c) {
		case XON:
		case XOFF:
			continue;

		default:
			if (!(c & 0x60))
				continue;

		case '\r':
		case '\n':
		case ZDLE:
			return c;
		}
	}
}


/*
 * Store long integer pos in Txhdr
 */
void 
stohdr(pos)
	long pos;
{
	Txhdr[ZP0] = pos;
	Txhdr[ZP1] = pos >> 8;
	Txhdr[ZP2] = pos >> 16;
	Txhdr[ZP3] = pos >> 24;
}

/*
 * Recover a long integer from a header
 */
static long
rclhdr(hdr)
	register char *hdr;
{
	register long l;

	l = hdr[ZP3] & 0377;
	l = (l << 8) | (hdr[ZP2] & 0377);
	l = (l << 8) | (hdr[ZP1] & 0377);
	l = (l << 8) | (hdr[ZP0] & 0377);
	return l;
}


/*
 * Send a string to the modem, processing for \336 (sleep 1 sec)
 * and \335 (break signal)
 */
void
zmputs(s)
	register char *s;
{
	register int c;

	while (*s) {
	switch (c = *s++) {
		case '\336':
			mssleep(1000);
			continue;
		case '\335':
			sendbrk();
			continue;
		default:
			sendline(c);
		}
	}
	uncorktransmitter();			/* Make sure all is well */
}

/*
 * Wait a reasonable amount of time for transmitter buffer to clear.
 * When it does, or when time runs out, turn XON/XOFF off then on.
 * This should release a transmitter stuck by line errors.
 */
void 
uncorktransmitter()
{
	drainline(1);
	xon_flush();
}



/*
 * Receive a binary style header (type and position)
 */
static int 
zrbhdr32(hdr)
	register char *hdr;
{
	register int c;
	register Ulong crc;
	register int n;

	if ((c = zdlread()) & ~0377)
		return c;
	Rxtype = c;
	crc = 0xFFFFFFFFU;
	crc = updcrc32(c, crc);

	for (n = Rxhlen; --n >= 0;) {
		if ((c = zdlread()) & ~0377)
			return c;
		crc = updcrc32(c, crc);
		*hdr++ = c;
	}

	for (n = 4; --n >= 0;) {
		if ((c = zdlread()) & ~0377)
			return c;

		crc = updcrc32(c, crc);
	}

	if (crc != 0xDEBB20E3U) {
		log("zmodem CRC-32 error");
		return ERROR;
	}

	return Rxtype;
}

/*
 * Print throughput message at end of transfer
 */
void
throughput(opt, bytes)
	int opt;
	Ulong bytes;
{
	static Ulong started = 0L;
	static Ulong elapsed;
	static Ulong cps;

	if (!opt)
		started = time(NULL);
	else if (started) {
		elapsed = time(NULL) - started;
		if (elapsed == 0L)
			return;
		cps = bytes / elapsed;
		started = (cps * 1000L) / ((long) baudrate);
		log("throughput CPS: %lu (%lu bytes)  Efficiency: %lu%%",
		    cps, bytes, started);
	}
}

/* END OF FILE: zm.c */

