/*
 * $Header: /u1/src/rfmail/RCS/xtrec.c,v 0.5.0.1 1992/06/15 06:11:25 pgd Exp pgd $
 *
 * $Log: xtrec.c,v $
 * Revision 0.5.0.1  1992/06/15  06:11:25  pgd
 * Minor compilation bug fixes.
 * Change of all types with u_ prefix to U prefix
 * Change of name of routine msleep() to mssleep()
 *
 * Revision 0.5  1992/05/18  04:27:24  pgd
 * New distribution
 *
 * Revision 0.4.1.6  1992/03/15  07:58:52  pgd
 * Untested version
 *
 * Revision 0.4.1.1  1991/05/21  11:13:48  pgd
 * *** empty log message ***
 *
 * Revision 0.4  1991/05/08  04:23:43  pgd
 * Initial Beta-release
 *
 */

/* Receive file(s) using Xmodem/TeLink/MODEM7/SEALINK Batch protocols.

   @(#)Copyright (c) 1987 by Teemu Torma

   Permission is given to distribute this program and alter this code as
   needed to adapt it to forign systems provided that this header is
   included and that the original author's name is preserved. */

#include "fnet.h"

#ifndef HAVE_BCOPY
#include <memory.h>
#endif

#include "fcall.h"
#include "crc.h"
#include "xmodem.h"
     
extern time_t time();

/* General error states to break the loops. */

#define Error           (-1)
#define Done            (-2)

/* States for XMODEM/TeLink receiver. */

#define RecStart        (0)
#define WaitFirst       (1)
#define WaitBlock       (2)

/* States for BATCH file receiver. */

#define RecvName        (0)
#define CheckFNm        (1)
#define CheckFile       (2)

static char *batchrec_states[] = {"RecvName", "CheckFNm", "CheckFile"};

/* States for MODEM7 filename receiver. */

#define SendNak         (0)
#define WaitAck         (1)
#define WaitChar        (2)
#define WaitOkCk        (3)

static int received_block_number = 0;
static char rxbuf[BlockSize];
static long size = 0L;

Telinkblock telinkblock;
boolean zeroblock = FALSE;
boolean receiving_data = FALSE;
char binaryfirstblock[128];
char binarytelinkblock[128];

static boolean
xtrecblk(blocknum, crcmode)
	int blocknum;
	boolean crcmode;
{
	register int c, cnt, checksum;
	register Ushort crc;

	if (blocknum == -1)
		debug(1, "Receiving block ignoring block number");
	else
		debug(2, "Receiving block %d", blocknum);

	received_block_number = blocknum;
  
	if ((c = readline(10*1000)) < 0 || c != (blocknum & 0377)) {
		if (c < 0)
			debug(1, "Timeout on block number receive");
		else {
			if (blocknum == -1)
				if (c == 0 || c == 1) /* Both telink and xmodem allowed */ {
					blocknum = c;
					received_block_number = blocknum;
					debug(1, "Setting block to %d", blocknum);
					goto blockok;
				}
			debug(1, "Bad block number %d, expected %d",
			      c, blocknum & 0377);
		}
		goto blidge;
	}

 blockok:
	if ((c = readline(10*1000)) < 0 || c != (~blocknum & 0377)) {
		if (c < 0)
			debug(1, "Timeout on complement block number receive");
		else
			debug(1, "Bad ~block number %d, expected %d",
			      c, (~blocknum & 0377));
		goto blidge;
	}

	receiving_data = TRUE;
	for (crc = 0, checksum = 0, cnt = 0; cnt < BlockSize; cnt++)
		if ((c = readline(5*1000)) < 0) {
			debug(1, "Timeout while receiving block");
			receiving_data = FALSE;
			goto blidge;
		} else {
			checksum += c;
			crc = updcrc(c, crc);
			rxbuf[cnt] = c;
		}

	receiving_data = FALSE;
	debug(5, "Received %d bytes", BlockSize);
  
	if (crcmode) {
		if (readline(10*1000) != (int) ((crc >> 8) & 0377)
		    || readline(10*1000) != (int) (crc & 0377)) {
			debug(1, "Crc error, block %d", blocknum);
			goto blidge;
		}
	} else {
		if (readline(10*1000) != (checksum & 0377)) {
			debug(1, "Checksum error");
			goto blidge;
		}
      
#ifdef NEEDED
		/*
		 * Just in case, if checksum doesn't match,
		 * see if we can see a crc there?
		 */
		int c;
      
		if ((c = readline(10*1000)) != (checksum & 0377)) {
			if (c != (int) ((crc >> 8) & 0377)
			    || readline(1*1000) != (int) (crc & 0377)) {
				debug(1, "Checksum error");
				goto blidge;
			} else {
				log("Correct CRC, incorrect checksum in checksum mode?");
				return True;
			}
			debug(1, "Checksum error");
			goto blidge;
		}
#endif
	}
	return True;

 blidge:
	if (xmodem_options.quicknak) {
		debug(1, "Fail block %d", blocknum);
		/* Quick nak mode: waiting for input to clean up is handled by look() */
	} else {
		debug(1, "Fail, Skipping rest of the block");
		while (readline(1*1000) != TIMEOUT)
			/* skip rest of the block */;
	}
	return False;
}

/* Copy input block to output file */

static void
writeblock(block, fp)
	register int block;
	FILE *fp;
{
	register char *p;

	register int cnt, bytes_to_write;

	bytes_to_write = BlockSize;
	if (zeroblock && size) {
		if ( (long) block * BlockSize > size)
			bytes_to_write = size - (size / BlockSize * BlockSize);
	}
	for (cnt = 0, p = rxbuf; cnt < bytes_to_write; cnt++, p++)
		(void) putc(*p, fp);

	debug(2, "Block %d written onto disk", block);
	if (block == 1)
		bcopy(rxbuf, binaryfirstblock, 128);
}

/* Sends nak or ack. */
static void
nack(ack, block, crcmode)
	boolean ack;
	int block;
	boolean crcmode;
{
	if (ack)
		sendline(ACK);
	else {
		/*
		 * Sealink is an exception: It always expects 'C'
		 * instead of NAK. Normal telink/xmodem according to
		 * fsc-0001 sends 'C' only for the first block, then
		 * uses NAK even in crc mode? Or have I missed something?
		 */
		if ((crcmode && block <= 1)
		    || (crcmode && xmodem_options.sealink))
			sendline('C');
		else
			sendline(NAK);
	}
	if (xmodem_options.sealink) {
		sendline(block);
		sendline(~block);
	}
}

#define ack() nack(TRUE, block, crcmode)
#define nak() nack(FALSE, block, crcmode)

/*
 * Reads input until it drains or valid block is seen, for which data
 * is then copied to file and received will be proceeded normally.
 * This allows us to restart quicker in case of error.
 */
#define putbuffer(c) \
{ *ip++ = c; if (ip > end) { wrapped = TRUE; ip = buffer; } }

#define readbuffer() \
( (Uint) ((cp >= end) ? cp = buffer, *end : *cp++) & 0377)

/* How many characters to eat until resending NAK */
/* 128 * 6 = 768, 2400 > 1 = 1200 */
#define RETRY_NAK_LIMIT (xmodem_options.speed >> 1)


static boolean
look(block, crcmode)
	int block;
	boolean crcmode;
{
	register char *ip, *end, *cp;
	static char *buffer = NULL;
	boolean wrapped = FALSE;
	int  c, counter = 0;

	debug(1, "Looking for correct block %d in input", block);
	
	if (!buffer) /* Two extra bytes to be sure */
		buffer = (char *) mymalloc( (Uint) (1 + 1 + 1 + BlockSize + 1 + 1 + 2) );
  
	ip = buffer;
  
	/* Points to last byte! */
	end = buffer + (1 + 1 + 1 + BlockSize + 1 + (int)crcmode) - 1;

	while ((c = readline(2*1000)) != TIMEOUT) {
		putbuffer(c);

		/*
		 * If we get lots of trash, it might be a good idea to
		 * retry sending nak to the other end.
		 */
		if (counter++ > RETRY_NAK_LIMIT) {
			debug(1, "Retrying NAK");
			nak();
			counter = 0;
		}
      
		if (wrapped) {
			/*
			 * buffer being size of block + all the other
			 * related hassle, ip should be pointing to
			 * start of block
			 */
			cp = ip;

			/*
			 * Nonstandard: Accept SYN for start of first
			 * block, some systems send first block with
			 * SYN as startup instead of SOH?
			 */
			if (((c = readbuffer()) == SOH
			     || (block == 1 && c == SYN))
			    && (readbuffer() == (block & 0377))
			    && (readbuffer() == (~block & 0377))) {
				register Uint crc, checksum;
				register int count;
				
				if (crcmode) {
					crc = 0;
					for (count = 0; count < BlockSize; count++) {
						c = readbuffer();
						crc = updcrc(c, crc); /* updcrc is a macro */
					}
					count = (readbuffer() == ((crc >> 8) & 0377) &&
						 readbuffer() == (crc & 0377));
				} else {
					checksum = 0;
					for (count = 0; count < BlockSize; count++)
						checksum += readbuffer();
					count = readbuffer() == (checksum & 0377);
				}
				if (count) {
					if (readbuffer() != SOH) {
						log("Internal error: expected SOH");
						xmodem_options.quicknak = FALSE;
						return FALSE;
					}
					
					/* Block number and its complement */
					(void) readbuffer();
					(void) readbuffer();
					
					for (count = 0; count < BlockSize; count++)
						rxbuf[count] = readbuffer();
					return TRUE;
				}
			}
		} /* checked out for block match */
	}
	return FALSE; /* Didn't see valid block until input drained */
}

static void
got_telinkblock()
{
	zeroblock = TRUE;
	bcopy(rxbuf, binarytelinkblock, 128);
	check_out_telinkblock();
	size = telinkblock.fileLength;
	debug(1, "Receiving %ld bytes, %ld blocks", size, size / 128);
	debug(1, "TeLink block received");
}

/* Wait until we can't see anything immedialetly in input */

void
wait_input_to_drain(timeout)
	int timeout;
{
	time_t s_time;
  
	SetStart();
	while (!Timeout(timeout) && readline(2*1000) != TIMEOUT)
		;
}

boolean
xtrec(filename, transfermode)
	char *filename;
	transfermode_t transfermode;
{
	int state = RecStart;
	time_t s_time, starttime, transfertime;
	int tries = 0, early_eot_ignores = 0;
	FILE *fp;
	int c;
	boolean crcmode;
	int block;

	crcmode = True;
	zeroblock = FALSE;
  
	if ((fp = myfopen(filename, "w")) == NULL) {
		log("$Unable to open %s for writing", filename);
		return False;
	}
	log("Receive file %s", filename);

	SetStart();
	starttime = time(NULL);
	bzero( (char *) binarytelinkblock, 128);
	block = 1;
  
	while (state >= RecStart && state <= WaitBlock)
		switch (state) {

		case RecStart:
			debug(2, "Current xtrec state: RecStart");
			debug(1,"Send %s", crcmode ? "C" : "NAK");
			block = 0;
			nak();
			state = WaitFirst;
			break;
	
		case WaitFirst:
			debug(2, "Current xtrec state: WaitFirst");
			if (tries > 20) {
				log("Too many tries on xmodem receive");
				state = Error;
			} else if (Timeout(60)) {
				log("Timeout on xmodem receive start");
				state = Error;
			} else if ((c = readline(10*1000)) == EOT) {
				mssleep(1000);
				flush();
				sendline(ACK);
				log("No file to receive");
				state = Done;
			} else if (c == SYN)  {
				debug(1, "Expect telink block in %s mode",
				      crcmode ? "crc" : "checksum");

				/*
				 * Block number must be 0 for the
				 * telink block.
				 * Telink block always contains only a
				 * checksum
				 */
				block = 0;
				if (!xtrecblk(block, False /* crcmode */ )) {
					debug(1, "Retry, bad block");
					tries++; /* No valid block received */
					wait_input_to_drain(60);
					state = RecStart;
				} else {
					ack(); /* Tell sender we got it */
					block = 1;
					got_telinkblock();
					state = WaitBlock;
				}
			} else if (c == SOH) {
				debug(1, "Expect first block in %s mode",
				      crcmode ? "crc" : "checksum");

				/*
				 * Block number must be 1 for first
				 * block but as some versions send
				 * telink block started up with a SOH,
				 * we have to check it afterwards.
				 */
				if (!xtrecblk(-1, crcmode)) {
					debug(1, "Retry, bad block");
					tries++; /* No valid block received */
					state = RecStart;
				} else {
					block = received_block_number;
					ack(); /* Tell sender we got it */

					if (block == 0) {
						/*
						 * Someone sent us a
						 * telink block
						 * starting with SOH
						 */
						got_telinkblock();
						state = WaitBlock;
					} else {
						/*
						 * Apparently first
						 * block, no telink
						 * block at all?
						 */
						writeblock(block, fp);
						state = WaitBlock;
						debug(2, "First block written onto disk");
					}

					/*
					 * Was either block 0 or 1,
					 * get next
					 */
					block++; 
				} /* Received block correctly */
			} else if (c < 0) {
				debug(1, "Timout on Xmodem rcv start");
	    
				tries++;
				state = RecStart;
			} else if (transfermode == MAILTRANSFER && c == TSYNCH) {
				debug(1, "Got TSYNCH? retry startup");
				
				tries = 0; /* We start from the beginning, so... */
				state = RecStart;
			} else if (!xmodem_options.forcecrc
				   && (tries > xmodem_options.checksum_fallback
				       || Timeout(30))) {
				log("Try checksum mode");
				crcmode = False;
				state = RecStart;
				/* mssleep(1000); Not in standard? */
			}
			break;

		case WaitBlock:
			debug(2, "Current xtrec state: WaitBlock");
			SetStart();
			early_eot_ignores = 0;
			for (tries = 0; state == WaitBlock; tries++) {
				if (tries > 10) {
					log("Too many retries (%d) on %s", tries, filename);
					state = Error;
				} else if (Timeout(60)) {
					log("Timeout on receive %s, tries %d", filename, tries);
					state = Error;
				} else if ((c = readline(10*1000)) == EOT) {
					/*
					 * To determine file size, use
					 * size, which is set from
					 * telink block. If doing pure
					 * xmodem, we don't have this
					 * information, in which case
					 * block number can be used.
					 */
					transfertime = time(NULL) - starttime;
					log("Received %ld bytes, %d blocks, %ld seconds,", 
					    size ? size : (long) (block-1) * BlockSize,
					    (block-1), transfertime);
					debug(1, "size = %d, block = %d, Blocksize = %d", size, block, BlockSize);
					log("%ld characters per second",
					    transfertime ?
					    (size ? size / transfertime :
					     (long) block * BlockSize / transfertime) :  0L);

					/*
					 * If Actual number of bytes
					 * received is less than the
					 * size in telink block, file
					 * transfer was interrupted
					 * prematurely. This happens
					 * with binkleyterm when not
					 * using sealink and we send
					 * binkleyterm nak. Old
					 * version at least cut the
					 * line, but current version
					 * seems to send EOT instead
					 * like transfer was completed
					 * normally?
					 */
					if (zeroblock && size) {
						if ( (long) (block - 1) * BlockSize < size ) {
							if (early_eot_ignores++ <= 2) {
								log("Early EOT at block %d, bytes %d, need %d %s",
								    block, (block - 1) * BlockSize, size,
								    "bytes, retrying\n");
								wait_input_to_drain(60);
								nak();
								state = WaitBlock;
								break;
							} else {
								state = Error;
								log("File %s received %ld bytes,",
								    filename, (long) block * BlockSize);
								log("should have been %ld bytes", size);
								break;
							}
						}
					}
					
					/*
					 * else cannot check this out
					 * if the other end tells us
					 * ok and no telink block!
					 */
					total_bytes += (size ? size : (long) block * BlockSize);
	      
					log("File %s received ok", filename);
					/* sendline(ACK); /* No need for block numbers here */
					state = Done;
				} else if (c == SOH || (block == 1 && c == SYN)) {
					/*
					 * Above c == SYN is
					 * nonstandard. Some systems
					 * send first xmodem block
					 * this way when transferring
					 * files.
					 */
					if (xtrecblk(block,
						     (block == 1 && c == SYN) ? FALSE : crcmode)) {
						ack();
						writeblock(block, fp);
						block++;
						tries = 0;
						early_eot_ignores = 0;
						SetStart();
					} else {
						debug(1, "Block not received correctly, block %d tries %d",
						      block, tries);

						/*
						 * Quick nak mode: nak
						 * as soon as trouble
						 * seen, then start
						 * looking at the
						 * input until
						 * expected block
						 * seen. This is only
						 * applicable to sealink.
						 */
						if (xmodem_options.quicknak && xmodem_options.sealink) {
							nak();

							/*
							 * Wait until
							 * expected
							 * block seen
							 * in input
							 */   
							if (look(block, crcmode)) {
								writeblock(block, fp);
								ack();
								block++;
								tries = 0;
							} else {
								debug(1, "Input drained, sending nak");
								nak();
							}
						} else {
							wait_input_to_drain(60);
							nak();
						}
					}
				} else if (c < 0) {
					debug(1, "Timeout on block %d, tries %d, sending nak",
					      block, tries);
					
					nak();
					/* No need to drain if nothing received */
				} else
					tries = 0; /* Trash from sealink send-ahead... skip it */
			}
			break; /* No-op, terminates case WaitBlock */
		} /* End case */
	
	debug(2, "Current xtrec state: %s", state == Error ? "Error" : "Done");
	if (state != Error) {
		if (fclose(fp) == -1) {
			log("$Close of file %s failed", filename);
			if (xmodem_options.transfer_ack_numbered)
				nak();
			else
				sendline(NAK);
			
			state = Error;
		} else {
			if (xmodem_options.transfer_ack_numbered)
				ack();
			else
				sendline(ACK);
		}
	}
	/* In case of error situation, drop dead */
	
	return (boolean)(state != Error);
}

static int
recmdmfn(filename)
	char *filename;
{
	int state = SendNak;
	time_t s_time;
	int tries = 0, c, pos;
  
	SetStart();
	while (state >= SendNak && state <= WaitOkCk)
		switch (state) {

		case SendNak:
			if (tries > 20) {
				log("Too many tries to get filename");
				state = Error;
			} else if (Timeout(60)) {
				log("Timeout while getting filename");
				state = Error;
			} else {
				sendline(NAK);
				state = WaitAck;
				tries++;
			}
			break;

		case WaitAck:
			switch (readline(5*1000)) {

			case ACK:
				pos = 0;
				state = WaitChar;
				break;

			case EOT:
				debug(1, "Got EOT in WaitAck");
				pos = 0;
				state = Done;
				break;

			case TIMEOUT:
				debug(1, "Timout while waiting ACK/EOT on MDM7");
				state = SendNak;
				break;

			case ENQ:
				/*
				 * This could really exit if we had
				 * implemented file requests
				 */
				debug(1, "Got ENQ while waiting for ACK/EOT on MDM7, continuing");
				state = SendNak;
				break;

			default:
				state = SendNak;
				debug(1, "Garbage on line while getting filename");
				mssleep(1000);
				flush();
				break;
			}
			break;

		case WaitChar:
			switch (c = readline(2*1000)) {

			case EOT:
				pos = 0;
				debug(1, "Got EOT in middle of filename, no files left");
				state = Done;
				break;
				
			case SUB:
				filename[pos] = 0;
				debug(2, "Got fn %s, sending checksum", filename);
				for (pos = 0, c = SUB; filename[pos]; pos++)
					c += filename[pos];
				sendline(c & 0377);
				state = WaitOkCk;
				break;

			case 'u':
				debug(1, "Got 'u', send NAK again");
				state = SendNak;
				break;
				
			case TIMEOUT:
				debug(1, "Timeout while waiting char of filename");
				state = SendNak;
				break;

			default:
				sendline(ACK); /* Was missing? */
				filename[pos++] = c;
				debug(3, "Got char '%c' of filename", c);
				break;
			}
			break;

		case WaitOkCk:
			if ((c = readline(2*1000)) == ACK) {
				/* Standard specified timeout of 1 */
				debug(1, "Got filename %s ok", filename);
				state = Done;
			} else {
				debug(1, "Got <%c>, Checksum failure in filename %s?",
				      c, filename);
				state = SendNak;
			}
			break;
		}
  
	return state == Error ? -1 : pos ? 1 : 0;
}

/*
 * Copy telink block contents to structure for easier handling.
 */
void
check_out_telinkblock()
{
	char *cp;
  
	if (!zeroblock)
		return;
  
	cp = binarytelinkblock;
  
	/* File size */
	telinkblock.fileLength = (long) (Uchar) (*cp) +
		( ((long) (Uchar) (cp[1])) << 8) +
			( ((long) (Uchar) (cp[2])) << 16) +
				( ((long) (Uchar) (cp[3])) << 24);
	cp += 4;
	
	/* CreationTime (ignore) */
	/* CreationDate (ignore) */
	cp += 4;
	bcopy(cp, telinkblock.filename, 16);
	telinkblock.filename[15] = 0;
	cp += 16;
	telinkblock.header_version = *cp++;
	bcopy(cp, telinkblock.sendingProg, 16);
	telinkblock.sendingProg[14] = 0;
	telinkblock.sendingProg[15] = 0;
	cp += 16;
#ifdef NEEDED
	telinkblock.crcMode = *cp++;
	debug(1, "CrcMode %d", telinkblock.crcMode);
#endif
}

boolean
batchrec()
{
	int state = RecvName;
	char filen[100], *filename;
	boolean ok;
	filename = filen;

	while (state >= RecvName && state <= CheckFile) {
		debug(3, "batchrec state: %s", batchrec_states[state]);
		switch (state) {

		case RecvName:
			ok = recmdmfn(filename);
			state = CheckFNm;
			break;

		case CheckFNm:
			switch (ok) {
			case -1:
				debug(1, "Abort batch receive, or no modem7 filename");
				/* Try xmodem receive. Create a file name in case sender doesn't
				   send it to us. */
				uploadpath(filen, "unknown");
				ok = xtrec(filen, FILETRANSFER);
				if (zeroblock) {
					check_out_telinkblock();
					if (strlen(telinkblock.filename)) {
						uploadpath(filename, clean_filename(telinkblock.filename));
						if (myrename(filen, filename) == -1)
							sendadmin("Rename failed",
								  "Could not rename %s to %s",
								  filen, filename);
					}
				}
				state = ok ? CheckFile : Error;
				break;

			case 0:
				log("All files received successfully");
				state = Done;
				break;

			case 1:
				uploadpath(filen, clean_filename(filename));
				filename = filen;
				ok = xtrec(filename, FILETRANSFER);
				state = CheckFile;
			}
			break;

		case CheckFile:
			if (ok) {
				debug(1, "%s received successfully", filename);
				state = RecvName;
			} else {
				log("Batch receive aborted, %s not received", filename);
				state = Error;
			}
			break;
		}
	}
	
	return (boolean)(state != Error);
}
