/*
 * $Header: /u1/src/rfmail/RCS/sched.c,v 0.5.0.1 1992/06/15 06:11:25 pgd Exp pgd $
 *
 * $Log: sched.c,v $
 * Revision 0.5.0.1  1992/06/15  06:11:25  pgd
 * Minor compilation bug fixes.
 * Change of all types with u_ prefix to U prefix
 * Change of name of routine msleep() to mssleep()
 *
 * Revision 0.5  1992/05/18  04:27:24  pgd
 * New distribution
 *
 * Revision 0.4.1.6  1992/03/15  07:58:52  pgd
 * Untested version
 *
 * Revision 0.4.1.3  1991/06/15  09:33:39  pgd
 * *** empty log message ***
 *
 * Revision 0.4  1991/05/08  04:23:43  pgd
 * Initial Beta-release
 *
 *
 */

/*
 * Call Scheduling
 *
 * Author: Per Lindqvist <pgd@compuram.bbt.se>
 */

#include "fnet.h"

#include <sys/stat.h>
#include <ctype.h>

#include "fcall.h"
#include "nodelist.h"
#include "configs.h"
#include "xmodem.h"
#include "zmodem.h"
#include "directory.h"
#include "packet.h"
#include "routing-hash.h"

#ifdef XENIX
DECLARE(int, stat, (char *, struct stat *));
#endif

LDECLARE(void, find_status, (VOID));
LDECLARE(char *, node_statusfile, (Node));
LDECLARE(boolean, check_node_lock, (Node));
LDECLARE(void, reschedule, (VOID));

/*
 * The status for all nodes are collected in the status[] array
 * by the find_status routine.
 * The contents of this array is used to select the next
 * node to call. The reason we do like this is so that the called 
 * nodes are distributed fairly. If we were just scanning the directory 
 * structure without checking all nodes, the node which happens to
 * come first in the out directory, would be unfairly favoured, if the
 * call to it would fail. We anyway would have to look thorough the
 * whole directory, looking for status and lock files.
 */
static struct status {
	Node	node;		/* Node number */
	int	retrycnt;	/* Retry counter */
	time_t	lasttime;	/* Time of last connect/try */
	boolean	hasreq;		/* Has a file request */
	boolean hasout;		/* Has an outgoing packet */
	boolean onhold;		/* The packet put on hold */
	boolean locked;		/* Node has a lock file */
	boolean crash;		/* Node has crash packet. */
	boolean scheduled;	/* True if schedule allows a call. */
	boolean wasbusy;	/* Number was busy at last try */
	Uchar event;		/* Event valid under */
	struct nodelist nodedata;
} *status;

static int maxstatcount;	/* Number of entries allocated to status[] */
static int statcount;		/* Number of entries in the status array */

extern fidonet_options_t fidonet_options;
extern struct nodelist remote_nodeentry;
extern int sched_tag;

Uint event_options;



/*
 * Find status for all nodes.
 *
 * Go through all entries in the out directory tree,
 * and record status for all nodes.
 * Assembles all the data into the status[] array.
 */
static void
find_status()
{
	char *fname;
	int spl;
	char *cp, *ft;
	Node node;
	struct stat sbuf;
	struct status *sp;
	register int i;

	statcount = 0;
	/*
	 * go through each file in the /usr/spool/fnet/out
	 * directory tree.
	 */
	if (!myopendir(config.outdir, TRUE))
		return;
	spl = strlen(config.outdir);
	while (fname = myreaddir(NULL)) {
		/*
		 * The files we are interested in look like
		 * fidonet node specification once the
		 * spool path prefix is removed.
		 * The three character filetype is terminating the
		 * node spec, and indicates what kind of file we have.
		 * If the filename is not fitting this specification,
		 * the file is simply ignored.
		 * Template: /usr/spool/fnet/out/9:999/999.typ
		 */
		cp = fname + spl;
		if (*cp == '/')
			cp++;
		ft = parsenode(cp, &node, ".", NULL);
		if (ft == NULL || *ft++ != '.' || strlen(ft) != 3)
			continue;
		/*
		 * We have the node number, check if we got it before,
		 * else create a new entry in the status array.
		 */
		sp = NULL;
		if (status != NULL) {
			for (i = 0; i < statcount; i++)
				if (NODEEQU(status[i].node, node)) {
					sp = &status[i];
					break;
				}
		}
		if (sp == NULL) {
			if (statcount >= maxstatcount)
				status = (struct status *)
					myrealloc((char *)status,
						  (maxstatcount+=10)
						  *sizeof(struct status));
			sp = &status[statcount];
			sp->node = node;
			sp->hasreq = sp->hasout = sp->onhold
				= sp->wasbusy = sp->locked
				= sp->crash = FALSE;
			sp->scheduled = TRUE;
			sp->lasttime = 0;
			sp->retrycnt = 0;
			if (search_node(node, &sp->nodedata) == NULL) {
				log("Node %s not found", ascnode(node));
				continue;
			}
			statcount++;
		}
		/*
		 * Now get status for file, and figure out what
		 * kind of file it is.
		 * The different kinds are:
		 *	file.?ut   -   Outgoing packet
		 *	file.?eq   -   Outgoing file request
		 *	file.x99   -   Call failure status
		 *	file.occ   -   Call failure status, occupied (busy)
		 *	file.LCK   -   Node is locked
		 */
		if (stat(fname, &sbuf) == -1)
			continue;
		if (sbuf.st_ctime > sp->lasttime)
			sp->lasttime = sbuf.st_ctime;
		if (is_out(fname)) {
			sp->hasout = TRUE;
			if (ft[0] == 'c')
				sp->crash = TRUE;
			else if (isupper(ft[0]))
				sp->onhold = TRUE;
		} else if (is_req(fname)) {
			sp->hasreq = TRUE;
			if (ft[0] == 'c')
				sp->crash = TRUE;
			else if (isupper(ft[0]))
				sp->onhold = TRUE;
		} else if (i = is_failed(fname)) {
			if (strequ(ft, "occ"))
				sp->wasbusy = TRUE;
			sp->retrycnt = i;
		} else if (strequ(ft, "LCK"))
			sp->locked = TRUE;
	}
	myclosedir();
	
}


/*
 * Return the name of the status file for node,
 * or NULL if none found
 */
static char *
node_statusfile(node)
	Node node;
{
	DIR *dir;
	struct dirent *ent;
	char *cp;
	char nn[15];
	int nnl, fc;
	char path[PATH_LEN];

	sprintf(path, "%s/%d:%d", config.outdir, node.zone, node.net);
	if ((dir = opendir(path)) == NULL)
		return NULL;

	/* Form initial part of filename to match this node */
	if (node.point == 0)
		sprintf(nn, "%d.", node.node);
	else
		sprintf(nn, "%d.%d.", node.node, node.point);
	nnl = strlen(nn);
	while (ent = readdir(dir)) {
		/* Check if file is for this node */
		if (!strnequ(ent->d_name, nn, nnl))
			continue;
		fc = is_failed(ent->d_name);
		if (fc > 0)
			break;
	}
	if (ent) {
		cp = strend(path);
		*cp++ = '/';
		strcpy(cp, ent->d_name);
	}
	closedir(dir);
	return ent ? sstrdup(path) : NULL;
}


/*
 * Setup the call status file in the outdir directory.
 */
void
set_call_status(node, occflag)
	Node node;
	boolean occflag;
{
	char *nsf;
	char scratch[PATH_LEN];
	int oldx;
	FILE *fp;

	nsf = node_statusfile(node);

	/*
	 * If nsf != NULL, it is the name of
	 * the old node status file.
	 * It has either ".occ", or ".x99" status.
	 */
	if (nsf) {
		oldx = is_failed(nsf);
		if (unlink(nsf) == -1)
			return;
	} else
		oldx = 0;
	if (occflag)
		sprintf(scratch, "%s.occ", nodepath(config.outdir, node));
	else
		sprintf(scratch, "%s.x%02d", nodepath(config.outdir, node),
			++oldx >= 100 ? 99 : oldx);
	/*
	 * Create new status file
	 */
	fp = myfopen(scratch, "w");
	if (fp == NULL)
		log("Cannot create node status file %s", spoolname(scratch));
	else
		fclose(fp);
}


/*
 * Clear the call status indicator 
 */
void
clear_call_status(node)
	Node node;
{
	char *nsf;

	nsf = node_statusfile(node);

	if (nsf != NULL)
		check_unlink(nsf);
}


/*
 * Select a node to call
 */
Node *
node_to_call()
{
	register struct status *sp;
	time_t t, now;
	struct status *this;
	static Node node;
	struct nodelist nodedata;

	reschedule();

	/*
	 * Go through the status[] array, looking for the node which
	 * is available for calling and  has been waiting the longest.
	 */
 redo:
	t = 0L; this = NULL;
	now = time(NULL);
	for (sp = &status[0]; sp != &status[statcount]; sp++) {
		if (!sp->scheduled
		    || sp->onhold
		    || (!sp->hasreq && !sp->hasout))
			continue;
		/*
		 * Check the retry timer.
		 */
		if (sp->retrycnt > config.retry_max)
			continue;
		if (now < sp->lasttime + config.retry_interval)
			continue;
		if (sp->hasreq && !sp->hasout && !fidonet_options.file_requests)
			continue;
		if (this != NULL && sp->lasttime >= t)
			continue;
		if (search_node(sp->node, &nodedata) == NULL) {
			log("Unknown node %s", ascnode(sp->node));
			continue;
		}

		/*
		 * If this is a file request, check if the node 
		 * accepts file requests.
		 */
		if (sp->hasreq && !sp->hasout) {
			if ((remote_nodeentry.flags & (XA|XB|XC|XR|XW|XX)) == 0) {
				debug(1, "No file requests allowed for %s",
				      ascnode(sp->node));
				sp->hasreq = FALSE;
				continue;
			}
		}

		/*
		 * Select this node
		 */
		this = sp;
	}

	if (this == NULL)
		return NULL;		/* Nothing to call */

	/*
	 * Check if there is a lock file for this node, and
	 * if so, check if lock is valid. Someone else can have
	 * locked the node.
	 */
	if (this->locked && check_node_lock(this->node)) {
		sp->lasttime = time(0L);
		goto redo;
	}

	node = this->node;
	event_options = config.event[this->event].options;
	return &node;
}

boolean
set_node_lock(node)
	Node node;
{
	if (!createlock(sprintfs("%s/%s.LCK", config.outdir,
				 ascnode(node)), config.lock_timeout))
		return FALSE;
	return TRUE;
}


void
clear_node_lock(node)
	Node node;
{
	removelock(sprintfs("%s/%s.LCK", config.outdir, ascnode(node)));
}

static boolean
check_node_lock(node)
	Node node;
{
	register char *name;

	name = lockname(sprintfs("%s/%s.LCK", config.outdir, ascnode(node)));
	if (name == NULL)
		return FALSE;
	return checklock(name);
}

/*
 * Reschedule all nodes for the current time.
 * We go through all the nodes available for calling.
 * and check then with the current schedule, if they
 * should be called or not.
 */
static void
reschedule()
{
	register int i, e;
	int opt;
	struct tm *tp;
	time_t t;
	Uint now;
	int nevents;
	int events[MAX_EVENTS];
	
	/*
	 * Find status is a costly operating, but wee need to do
	 * it here since we do allow more than one instance of fcall.
	 * fpack might also have been running in-between.
	 */
	find_status();

	t = time(0);
	tp = gmtime(&t);
	now = tp->tm_hour * 60 + tp->tm_min;

	/*
	 * Find which events are active right now.
	 */
	nevents = 0;
	for (e = 0; e < config.events; e++) {
		if ((BIT(tp->tm_wday) & config.event[e].days)
		    && now >= config.event[e].start
		    && now < config.event[e].end) {
			events[nevents++] = config.event[e].tag;
		}
	}

	/*
	 * Now go through the selected node numbers, and
	 * see which are callable under the current event(s)
	 */
	for (i = 0; i < statcount; i++) {
		for (e = 0; e < nevents; e++) {
			opt = config.event[e].options;
			if (status[i].crash && opt & EVENT_CM)
				break;
			if (status[i].hasout && opt & EVENT_NONCM)
				break;
			if (status[i].hasreq && opt & EVENT_OBFREQ)
				break;
		}
		if (nevents == 0 || e != nevents) {
			status[i].event = e;
			status[i].scheduled = TRUE;
		} else
			status[i].scheduled = FALSE;
	}
}


/*
 * Setup event_options for the case where we are called up.
 */
void
setup_event_options(node)
	Node node;
{
	register int e;
	time_t now;
	struct tm *tp;
	
	now = time(0);
	tp = gmtime(&now);
	event_options = 0;
	if (config.events == 0) {
		event_options = 0xffff;
		return;
	}
	for (e = 0; e < config.events; e++) {
		if ((BIT(tp->tm_wday) & config.event[e].days)
		    && now >= config.event[e].start
		    && now < config.event[e].end) {
			event_options |= config.event[e].options;
		}
	}

}
