/*
 * $Header: /u1/src/rfmail/RCS/rfmail.c,v 0.5.0.1 1992/06/15 06:11:25 pgd Exp pgd $
 *
 * $Log: rfmail.c,v $
 * Revision 0.5.0.1  1992/06/15  06:11:25  pgd
 * Minor compilation bug fixes.
 * Change of all types with u_ prefix to U prefix
 * Change of name of routine msleep() to mssleep()
 *
 * Revision 0.5  1992/05/18  04:27:24  pgd
 * New distribution
 *
 * Revision 0.4.1.6  1992/03/15  07:58:52  pgd
 * Untested version
 *
 *
 * Allow tabs, as well as spaces, in rfc header
 *
 * Revision 0.4.1.3  1991/06/15  09:33:39  pgd
 * *** empty log message ***
 *
 * Revision 0.4.1.2  1991/06/05  09:13:58  pgd
 * Various Bugfixes:
 *
 * Patches suggested by Marc Boucher (marc):
 * The batch flag (bflag)'s default value comes from config.dobatch and that
 * the -b flag inverses the setting
 *
 * There was a bug in the rfc822 header name detection routine, causing rfmail
 * to pick up the enveloppe-from (From_) instead of the real From:. (wrong
 * thing used as length in a strncmp)
 * 
 * The check for header continuation lines only looked at tabs '\t'. RFC822
 * says that continuation lines can start with spaces or tabs.
 *
 * Patches suggested by Jari Nopanen (JN):
 * 
 * Revision 0.4.1.1  1991/05/21  11:13:48  pgd
 * *** empty log message ***
 *
 * Revision 0.4  1991/05/08  04:23:43  pgd
 * Initial Beta-release
 *
 *
 */


/*
 * Main rfmail program.
 *
 * Call with:
 *	rfmail - send an unix letter to fidonet
 *	rfnews - send an unix news message to fidonet
 *	funpack - unpack fidonet mail
 *	fpack	- pack fidonet mail
 *
 * This module is a merge of the original individual programs.
 *
 * Created by: Per Lindqvist, pgd@compuram.bbt.se
 */


/*
 * Original author copyright:
 *   
 * @(#)Copyright (c) 1987 by Teemu Torma
 * 
 * Permission is given to distribute this program and alter this code as
 * needed to adapt it to forign systems provided that this header is
 * included and that the original author's name is preserved.
 */

/*
 * Authors:
 *
 * Teemu Torma who wrote the original code (?)
 *
 * Heikki Suonsivu (hsu@hutcs.hut.fi) who made a lot of enhancements
 * 
 * Per Lindqvist (pgd@compuram.bbt.se) who continued to enhance rfmail.
 */

#include "fnet.h"

#include <pwd.h>
#include <grp.h>

#include "nodelist.h"
#include "configs.h"
#include "packet.h"
#include "rfmail.h"
#include "routing-hash.h"

#ifndef MSGID_POLITICS
#include "crc.h"
#endif
     

/*
 * Prototypes for local routines
 */
LDECLARE(void, split_address, (char *, char *, char *));
LDECLARE(void, get_fullname, (char *, char *));
LDECLARE(void, parse_rfc_header, (Message *));
LDECLARE(boolean, build_msg, (Message *, FILE *, boolean));
LDECLARE(void, copy_message, (Message *, FILE *));
LDECLARE(boolean, sendmail, (Message *, char *));
LDECLARE(boolean, create_ifna_header, (Message *, boolean));
LDECLARE(void, sendback, (Message *, char *, ...));
LDECLARE(void, fix_replyaddr, (Message *));



DECLARE(int, rfmail, (int, char **));
DECLARE(int, funpack, (int, char **));
DECLARE(int, fpack, (int, char **));
DECLARE(int, rfnews, (int, char **));
DECLARE(char *, myitoa, (int));
DECLARE(char *, convert_full_name, (char *));
DECLARE(void, printusage, (FILE *));


#define	MAXHEADERLINES	50

/* non-private mail */
boolean public = FALSE;

/* version of rfmail, maintained by RCS */
static char *Revision = "$Revision: 0.5.0.1 $";


Node mynode;			/* Our net/node information */

boolean fflag;			/* Indicate we got the "-f" option */
boolean Pflag;
boolean bflag;
Node fnode;			/* Node from the "-f" option */
Node Pnode;
char hostname[32];


/*
 * Storage for the rfc822 header
 *
 * Used for processing incoming unix mail.
 * All field identifiers have to be lower case.
 */
static char *header_from, *header_reply_to, *header_return_path, *header_date,
	*header_subject, *header__from, *header_area, *header_comment_to,
	*header_message_id, *header_received, *header_to, *header_in_reply_to,
	*header_fidonet_flags, *header_newsgroups, *header_charset;

static struct {
	char *field;
	char **ptr;
} header_fields[] = {
	{"from:",		&header_from},
	{">from:",		&header__from},
	{"to:",			&header_to},
	{"reply-to:",		&header_reply_to},
	{"return-path:",	&header_return_path},
	{"date:",		&header_date},
	{"subject:",		&header_subject},
	{"area:",		&header_area},
	{"x-fidonet-comment-to:",&header_comment_to},
	{"x-comment-to:",	&header_comment_to},
	{"message-id:",		&header_message_id},
	{"received:",		&header_received},
	{"in-reply-to:",	&header_in_reply_to},
	{"x-fidonet-flags:",	&header_fidonet_flags},
	{"x-flags:",		&header_fidonet_flags},
	{"newsgroups:",		&header_newsgroups},
	{"x-fsc-charset:",	&header_charset}
};
	

enum  {RFMAIL, RFNEWS, FUNPACK, FPACK} whoamI;


/*
 * This is the main entry-point.
 * 
 * Dispatch to the right part, depending on program name
 */
int
main(argc, argv)
	int argc;
	char **argv;
{
	register char *func;
	int s, c;
	extern char *optarg;
	char *options;

	func = basename(argv[0]);
	if (strequ(func, "rfmail")) {
		whoamI = RFMAIL;
		options = "V:";
	} else if (strequ(func, "rfnews")) {
		whoamI = RFNEWS;
		options = "pV:";
	} else if (strequ(func, "fpack")) {
		whoamI = FPACK;
		options = "f:p:s:V:P:";
	} else if (strequ(func, "funpack")) {
		whoamI = FUNPACK;
		options = "bf:s:V:";
	} else {
		fprintf(stderr, "Illegal call to rfmail\n");
		fprintf(stderr, "Please use one of: rfmail, rfnews, fpack, funpack\n");
		exit(1);
	}

	fixversion(Revision);

	get_configuration(&argc, argv, options);

	gethostname(hostname, sizeof(hostname)-1);


	bflag = config.dobatch;

	/*
	 * Parse command line options
	 */
	while ((c = getopt(argc, argv, options)) != EOF) {
		switch (c) {
		case 'V':	/* Set version (why is this option here?) */
			strcpy(version, optarg);
			break;

		case 'p':
			/* this is not a private message */
			public = TRUE;
			break;

		case 'b':
			bflag = (boolean)!(int)bflag;
			break;

		case 'f':
			if (!parsenode(optarg, &fnode, NULL, &config.mynode))
				fatal(EX_DATAERR, "Could not parse -f option: %s", optarg);
			fflag = TRUE;
			break;


		case 'P':
			if (!parsenode(optarg, &Pnode, NULL, &config.mynode))
				fatal(EX_DATAERR, "Could not parse -P option: %s", optarg);
			Pflag = TRUE;
			break;

		case 's':
			if (strlen(optarg) != 1)
				goto error;
			schedule = *optarg;
			break;

		default:
		error:
			fprintf(stderr, "Illegal option: %c\n", c);
			printusage(errorfp);
			exit(EX_USAGE);
		}
	}


	/*
	 * Dispatch depending on command which got us here
	 */
	switch (whoamI) {
	case RFMAIL:
		s = rfmail(argc, argv);
		break;

	case FUNPACK:
		s = funpack(argc, argv);
		break;

	case FPACK:
		s = fpack(argc, argv);
		break;

	case RFNEWS:
		s = rfnews(argc, argv);
		break;
	}

	exit(s);
	return s;		/* Just to make gcc happy */
}


void
printusage(fp)
	FILE *fp;
{
	switch (whoamI) {
	case RFMAIL:
		fprintf(fp, "rfmail [options] valid-fidonet-recipient-address\n");
		fprintf(fp, "\tSend mail to fidonet.\n");
		fprintf(fp, "Where options is any of:\n");
		fprintf(fp, "-p  Mark message to be marked public in FidoNet.\n");
		break;

	case FUNPACK:
		fprintf(fp, "funpack [-v nn] [-O debuglog] [-b] [-i] [-h]\n");
		fprintf(fp, "        [-f [[<zone>:]<net>/]<node>[.<point>]]\n");
		if (bflag)
			fprintf(fp, "-b          Disable batching.\n");
		else
			fprintf(fp, "-b          Enable batching.\n");
		break;
		
	case FPACK:
		fprintf(fp, "fpack [options], where options are any of:\n");
		fprintf(fp, "Create fidonet mail packet from preprocessed\n");
		fprintf(fp, "messages.\n");
		fprintf(fp, "-f  <valid-fidonet-address> Optional, single node to process.\n");
		fprintf(fp, "-s <schedule tag> Do routing according to the indicated schedule.\n");
		break;
	}
	configprintusage(fp);
}



/*
 * Entrypoint for "rfmail" command
 *
 * stdin has the unix mail message to be send to fidonet
 */
int
rfmail(argc, argv)
	int argc;
	char **argv;
{
	int cnt;
	Message msg;
	

	get_nodelist();			/* Read nodelist index. */

	if (!build_msg(&msg, stdin, FALSE))
		return 1;

	if (optind == argc) {
		/*
		 * No receiver given on command line, try to pick
		 * it up from the "To:" field
		 */
		if (header_to)
			(void)sendmail(&msg, header_to);
		else
			sendback(&msg, "Message receiver missing");
	} else {
		/*
		 * Send mail to all recipients
		 */
		for (cnt = optind; cnt < argc; cnt++)
			(void)sendmail(&msg, argv[cnt]);
	}
		
	return 0;
}


/* 
 * Extract address from Reply-To or From field. Possible formats are:
 *   - Full Name <address>
 *   - address (Full Name)
 *   - address
 */
void
get_address(field, address)
	char *field, *address;
{
	register char *cp, *np;
  
	/* strip possible newline */
	if (cp = strchr(field, '\n'))
		*cp = 0;
  
	if ((cp = strchr(field, '(')) && strchr(cp + 1, ')')) {
		/* format is 'address (full name)' */
		for (np = field; np < cp - 1; np++)
			*address++ = *np;
		*address = 0;
	} else if ((cp = strchr(field, '<')) && (np = strchr(cp + 1, '>'))) {
		/* format is 'full name <address>' */
		cp++;
		while (cp < np)
			*address++ = *cp++;
		*address = 0;
	} else
		/* line contains only address */
		strcpy(address, field);
}

/* Get return address from mail. Extract it from Reply-To: or From: fields. */

void
get_return_address(address)
	char *address;
{
	char reply_to[128], from[128], return_path[128];
	char *cp;
  
	*address = *reply_to = *from = *return_path = 0;
  
	/*
	 * check is there is Reply-To: field,
	 * From: field or a Return-path: field
	 */
	if (header_reply_to)
		get_address(header_reply_to, reply_to);
	else if (header_from)
		get_address(header_from, from);
	else if (header_return_path)
		get_address(header_return_path, return_path);
	if (*reply_to)
		strcpy(address, reply_to);
	else if (*from)
		strcpy(address, from);
	else if (*return_path) {
		/*
		 * Convert the return path to rfmail address by inserting
		 * appending a fidonet-domain
		 * Maybe there should be another config entry for this?
		 * This code assumes the Return-Path entry is a valid
		 * domain address.
		 */
		if (cp = strrchr(return_path, '@')) {
			strncopy(address, return_path, cp-return_path+1);
			strcat(address, *config.fidonet_domain == '.' ?
			       config.fidonet_domain+1 : config.fidonet_domain);
			strcat(address, ".");
			strcat(address, cp+1);
		}
	}
	  
	/* not found, send it to uucp */
	if (*address == 0)
		strcpy(address, "uucp");
}

/* 
 * Send mail back to sender. If Reply-To: of From:-field is present,
 * we'll use address specified in that, otherwise we will not return
 * mail (it also should be able to parse From_ fields). First argument
 * is file containing this mail, others are for vprintf(3S) to be used
 * as transcript. 
 */
static void
sendback(msg, fmt VA_ALIST)
	Message *msg;
	char *fmt;
	VA_DCL
{
	va_list args;
	FILE *mailer;
	char buffer[BUFSIZ];
	char to[128];

	VA_START(args, fmt);
	
	if (!config.return_failed_mail) {
		vfprintf(errorfp, fmt, va_arg(args, char *));
		return;
	}

	get_return_address(to);
	log("Mail failed, return to %s", to);
		
	/* generate shell command and open it */
	sprintf(buffer, "exec %s %s %s", config.rmail, to, config.admin);
	if (mailer = popen(buffer, "w")) {
		/* print correct header for mailer */
		fprintf(mailer, "From MAILER-DAEMON %s remote from %s\n",
			date("%a %h %d %T 19%y", NULL),
			internode(config.mynode));
		fprintf(mailer, "Received: by %s (rfmail %s/%s)\n",
			internode(config.mynode), version,
			config.system_name);
		fprintf(mailer, "\tid AA%05d; %s\n", getpid(),
			date("%a, %d %h %y %T %o (%z)", NULL));
		fprintf(mailer, "Date: %s\n", date("%a, %d %h %y %T %o",
						   NULL));
		fprintf(mailer, "From: FidoNet Mail <%s@%s>\n",
			"MAILER-DAEMON", internode(config.mynode));
		fprintf(mailer, "Subject: Returned mail: %s\n",
			"Unable to deliver mail to FidoNet");
		fprintf(mailer, "Message-ID: <%s.AA%05d@%s>\n",
			date("%y%m%q%H%M", NULL), getpid(),
			internode(config.mynode));
		fprintf(mailer, "To: %s\n", to);
		fprintf(mailer, "Cc: %s\n", config.admin);
		fprintf(mailer, "\n");
		fprintf(mailer,
			"  ----- Transcript of session follows -----\n");
		vfprintf(mailer, fmt, va_arg(args, char *));
		fprintf(mailer, "\n\n");
		fprintf(mailer, "  ----- Unsent message follows -----\n");
		
		/* now copy the message to mailer */
		copy_message(msg, mailer);
		
		/* mail is now sent, close mailer */
		pclose(mailer);
	} else
		log("$Unable to invoke mailer for returned mail");
  
	va_end(args);

}


/* Compare receiver and user in aliasfile. Each line in aliasfile contains
   alias, optional net/node information separated by commas zero or
   more times, white space(s) and rest of the line literally to whom
   mail should be send. Net and node information is format 'net/node'.
   if net is omitted then every net is counted, if node is omitted,
   then all nodes in that net. If net is omitted, slash may be left off.
   
   E.g following are valid aliases:

   tot,504/ Teemu Torma
   foo Foo Bar
   sysop,504/1,504/9 System Operator */


boolean
aliascmp(alias, to, node)
	char *alias, *to;
	Node node;
{
	char buffer[BUFSIZ];
	char *cp, *aliasline;
	Node anode;

	/* Save this for error messages. */
	aliasline = alias;
  
	/* Collect user id, everything until a comma. */
	while (*alias && *alias != ',' && !isspace(*alias) && *to)
		if (*alias++ != *to++)
			return FALSE;
	
	/* 
	 * If to still has characters, it doesn't match with already
	 * terminated alias. */
	if (*to) {
		debug(20, "alias '%s': To still got '%s'.", aliasline, to);
		return FALSE;
	}
  
	if (isspace(*alias)) /* match */ {
		debug(10, "Alias '%s' matched.", aliasline);
		return TRUE;
	}
  
	if (*alias == ',') {
		/* copy alias to buffer and terminate the it after first space */
		strcpy(buffer, alias + 1);
		for (cp = buffer; *cp; cp++) {
			if (isspace(*cp)) {
				*cp = 0;
				break;
			}
		}
		
		/* get net/node information from buffer one at the time */
		for (cp = strtok(buffer, ","); cp; cp = strtok((char *) 0, ",")) {
			debug(2, "Got node '%s'", cp);
			if (parsenode(cp, &anode, NULL, &config.mynode)) {
				if ((anode.zone == NOZONE
				     || anode.zone == node.zone)
				    && (anode.net == NONET
					|| anode.net == node.net)
				    && (anode.node == NONODE
					|| anode.node == node.node)
				    && (anode.point == NOPOINT
					|| anode.point == node.point)) {
					debug(10, "Alias '%s' matched.", aliasline);
					return TRUE;
				}
			} else {
				debug(20, "Parsenode failed, alias '%s'.",
				      aliasline);
				return FALSE;
			}
		}
	} else
		debug(1, "Invalid alias, %c is not ',' or white space", *alias);
	return FALSE;
}


/* 
 * Return receiver's name. If name is aliased, return it, otherwise
 * return receiver's name.
 */
char *
receiver(to, node)
	char *to;
	Node node;
{
	static char name[36];
	char buffer[BUFSIZ];
	register int cnt;
	register char *cp;
	FILE *fp;
  
	debug(3, "Searching for alias for %s", to);
	if (fp = fopen(config.alias, "r")) {
		while (fgets(buffer, BUFSIZ, fp)) {
			if (*buffer != '#' && aliascmp(buffer, to, node)) {
				/* match, save the alias. */
				for (cp = buffer; *cp && !isspace(*cp); cp++)
					/* skip alias itself */;
				while (isspace(*cp))
					cp++;
				if (*cp) {
					for (cnt = 0; *cp && cnt < 35; cnt++, cp++)
						name[cnt] = *cp;
					name[cnt] = 0;
					debug(2, "Got alias %s", name);
					fclose(fp);
					return name;
				} else
					debug(1, "Missing alias");
			}
		}
		fclose(fp);
	} else
		log("$Unable to open aliasfile %s", config.alias);
  
	/* Alias not found. Return the the original receiver with all
	   _-characters replaced by space and all words calitalized. */
  
	for (cp = NULL, cnt = 0; *to && cnt < 35; cnt++, cp = to++) {
		if (*to == '_')
			name[cnt] = ' ';
		else
			if (!cp || *cp == '_' || (cnt > 0 && *(to-1) == ' ')) {
				if (islower(*to))
					name[cnt] = toupper(*to);
				else
					name[cnt] = *to;
			} else {
				if (isupper(*to))
					name[cnt] = tolower(*to);
				else
					name[cnt] = *to;
			}
	}
	name[cnt] = 0;
	debug(2, "No alias, name %s", name);
	return name;
}


/* 
 * Return receiver's name, took Comment-To: field. If not found,
 * call receiver() and return what it returns
 */
char *
newsreceiver(to, node)
	char *to;
	Node node;
{
	debug(2, "Checking To: field");
	if (header_to) {
		debug(1, "Got recipent %s from To: field", header_to);
		strcpy(to, header_to);
	} else {
		debug(2, "Checking Comment-To: field");
		if (header_comment_to) {
			debug(1, "Got recipent %s from Comment-To field",
			      header_comment_to);
			strcpy(to, header_comment_to);
		}
	}
	return receiver(to, node);
}


/* Send message 'msg' to user 'to'. To is in format of
   net!node!receiver, where net and node are destination and
   receiver is receivers name in which blankos are replaces by
   _-characters or alias. */

static boolean
sendmail(msg, to)
	Message *msg;
	char *to;
{
	char *cp;
	char name[36];
	Message m;
	char fullname[80], domainaddr[80];

	/* JN: Private mail is always nonpublic */
	public = FALSE;

	m = *msg;

	split_address(to, fullname, domainaddr);


	/* get receiver and net/node */
	if (cp = parse_address(domainaddr, name, &m.dest)) {
		sendback(msg, "Parse of address failed: %s", cp);
		return FALSE;
	}
  
	if (*fullname)
		strncopy(m.to, fullname, sizeof(m.to)-1);
	else {
		for (cp = name; cp = strchr(name, '_'); )
			*cp++ = ' ';
		strncopy(m.to, name, sizeof(m.to)-1);
	}

#ifdef REMOVE
	/* JN: Convert full name */
	for (cp=m.to; *cp; cp++)
		if ((cp == m.to || *(cp-1) == ' ' || *(cp-1) == '-') &&
		    islower(*cp))
			*cp = toupper(*cp);
#endif
	convert_full_name(m.to);


	/*
	 * Try to convert to internet return address to a
	 * plain fidonet node address.
	 */
	fix_replyaddr(&m);

	/*
	 * Now place message into the msg spool directory
	 */
	if (!write_msg(&m, m.orig, m.dest)) {
		sendback(msg, "Could not send message");
		return FALSE;
	}
	return TRUE;
}



/*
 * Parse the rfc822 header, and decompose into the parts.
 * We will leave the mail file pointer pointing to after the header.
 */
static void
parse_rfc_header(msg)
	Message *msg;
{
	char *cp;
	int l, i;
	struct line *lp;
	char fieldname[30];


	for (i = 0; i < sizeof(header_fields)/sizeof(header_fields[0]); i++)
		*header_fields[i].ptr = NULL;
	if (lp = msg->rfchdr) {
		do {
			if ((cp = strchr(lp->txt, ' ')) == NULL)
				cp = strchr(lp->txt, '\t');
			if (cp) {
				l = cp - lp->txt;
				if (l >= sizeof(fieldname))
					l = sizeof(fieldname) - 1;
				strncopy(fieldname, lp->txt, l);
				strlwr(fieldname);
				for (i = 0; i < sizeof(header_fields)/sizeof(header_fields[0]); i++) {
					if (strnequ(fieldname, header_fields[i].field, strlen(header_fields[i].field))
					    && *header_fields[i].ptr == NULL) {
						*header_fields[i].ptr = cp + 1;
						break;
					}
				}
			}
		} while ((lp = lp->next) != msg->rfchdr);
		if (header_from) {
			if (msg->internet) free(msg->internet);
			msg->internet = strsave(header_from);
		}
		if (header_message_id) {
			if (msg->ufgid) free(msg->ufgid);
			msg->ufgid = strsave(header_message_id);
		}
		if (header_newsgroups) {
			if (msg->ufgroup) free(msg->ufgroup);
			msg->ufgroup = strsave(header_newsgroups);
		}
	}
}


/*
 * Split address into fullname (if there) and domain address.
 */
static void
split_address(from, fullname, domainaddr)
	char *from, *fullname, *domainaddr;
{
	char *cp, *np, *cp1;
	char buffer[128];

	/* This is the uucp/internet return address */
	strcpy(buffer, from);

	/*
	 * First separate the address into full name
	 * and domain address
	 * Name formats:
	 * 	username@domain <full name>
	 *	full name (username@domain)
	 *	username@domain
	 */
	if ((cp = strchr(buffer, '<')) && (np = strchr(cp, '>'))) {
		/* 
		 * Format: full name <username@domain>
		 */
		strncpy(fullname, buffer, cp-buffer);
		cp1 = &fullname[cp-buffer];
		while (cp1 > fullname && cp1[-1] == ' ')
			cp1--;
		while (*++np == ' ')
			;
		if (*np) {
			if (cp1 > fullname)
				*cp1++ = ' ';
			while (*np)
				*cp1++ = *np++;
			while (cp1 > fullname && cp1[-1] == ' ')
				cp1--;
			*cp1 = 0;
		} else
			*cp1 = 0;
		cp1 = buffer;
		for (cp++; *cp != '>'; )
			*cp1++ = *cp++;
		*cp1++ = 0;
	} else if ((cp = strchr(buffer, '(')) && (np = strchr(cp, ')'))) {
		/* 
		 * Format: username@domain (full name)
		 */
		strncopy(fullname, cp+1, np-cp-1);
		while (cp > buffer && cp[-1] == ' ')
			cp--;
		np++;
		while (*np == ' ')
			np++;
		if (*np) {
			if (cp != buffer)
				*cp++ = ' ';
			strcpy(cp, np);
		} else
			*cp = 0;
	} else {
		/*
		 * No fullname in address. We will try to pick it
		 * up below.
		 */
		*fullname = 0;
	}

	/*
	 * We should now have the plain domain address
	 */
	strcpy(domainaddr, buffer);
}


/*
 * Try very hard to get the "full-name" of the sender.
 * Also return unix mail address of originator
 */
static void
get_fullname(addr, fullname)
	char *addr, *fullname;
{
	register char *cp;
	char *home, *getenv();
	FILE *fp;
	boolean pwdtried;
	struct passwd *pwd;
	int uid;
	char fname[128];
	char *np, *cp1;
	char scratch[128];
	char *atpos, *bangpos;


	/*
	 * Strip away mailname
	 */
	strcpy(scratch, addr);
	atpos = strchr(scratch, '@');
	bangpos = strrchr(scratch, '!');
	if (atpos && !bangpos) { 		/* ....@..... */
		*atpos++ = 0;
		if (!strequ(atpos, config.mailname))
			goto notthishost;
	} else if (bangpos && !atpos) {		/* .... ! ..... */
		*bangpos++ = 0;
		if (!strequ(scratch, config.mailname))
			goto notthishost;
		strcpy(scratch, bangpos);
	} else if (atpos && bangpos)		/* ... ! .... @ .... */
		goto notthishost;

	/*
	 * If we did not get a fullname, check if letter is from
	 * this host, and if so, try to pick up full name from
	 * $NAME or if that fails, .fullname or .realname, or as a final
	 * step from the password file.
	 * A path is considered to be local if there is none
	 * of the %!@ characters in the address.
	 */
	if ((cp = strpbrk(scratch, "%!@")) == NULL) {
		uid = getuid();
		pwd = NULL;
		pwdtried = FALSE;
		if (uid >= config.useruid && (cp = getenv("NAME")))
			strcpy(fullname, cp);
		if (*fullname == 0) {
			/*
			 * Obtain home directory even if called by a daemon
			 */
			if ((int)config.useruid == -1
			    || uid < config.useruid
			    || (home = getenv("HOME")) == NULL) {
				if (*scratch)
					pwd = getpwnam(scratch);
				if (!*scratch || pwd == NULL) {
					uid = getuid();
					if (uid >= config.useruid)
						pwd = getpwuid(uid);
				}
				if (pwd)
					home = pwd->pw_dir;
				pwdtried = TRUE;
			}
			if (home) {
				sprintf(fname, "%s/.realname", home);
				if ((fp = fopen(fname, "r"))) {
					if (fgets(fullname, 80, fp)==NULL)
						*fullname = 0;
					fclose(fp);
				}
				if (*fullname == 0) {
					sprintf(fname, "%s/.fullname", home);
					if ((fp = fopen(fname, "r"))) {
						if (fgets(fullname, 80, fp)==NULL)
							*fullname = 0;
						fclose(fp);
					}
				}
				if (*fullname)
					fullname[strlen(fullname)-1] = 0;
			}
		}
		if (*fullname == 0) {
			/*
			 * Try to pick up fullname from passwd file
			 */
			if (pwd == NULL && !pwdtried) {
				pwd = getpwnam(scratch);
				if (pwd == NULL) {
					uid = getuid();
					if (uid >= config.useruid)
						pwd = getpwuid(uid);
				}
			}
			if (pwd) {
		/* There are two commonly used gecos-formats: So called USG
		   format used by System III and System V machines and BSD
		   format used by BSD machines. In USG format name is
		   sandwitched between '-' and '(' characters and in BSD
		   format name is first this in gecos-field up to first comma
		   in it. In many machines there's only name in gecos. */
				if ((cp = strchr(pwd->pw_gecos, '-'))
				    && (np = strchr(cp+1, '('))) {
					/* USG format 'stuff-name(stuff)' */
					strncopy(fullname, cp+1, np-cp-1);
				} else if (cp = strchr(pwd->pw_gecos, ',')) {
					/* BSD format 'name,stuff...' */
					for (cp1 = fullname, cp++;
					     *cp && *cp != ','; cp++)
						*cp1++ = *cp;
					*cp1++ = 0;
				} else if (*pwd->pw_gecos) {
					/* non-empty gecos, assume that there's only name */
					strcpy(fullname, pwd->pw_gecos);
				} else if (*fullname == 0) {
              /* Lazy administrator or user who want's to be anonymous
                 (or this is some administrative uid with no explanation
                 which)... We'll use only the username. */
					strcpy(fullname, pwd->pw_name);
				}
			}
		}
	}
	if (*fullname)
		return;
 notthishost:
	
	/*
	 * Extract receiver name as fullname,
	 * assume user name is in "scratch";
	 */
	strcpy(scratch, addr);
	if (cp = strchr(scratch, '@')) {
		*cp++ = 0;
		if (cp1 = strrchr(scratch, '!'))
			strcpy(fullname, cp1+1);
		else if (cp1 = strchr(scratch, '%'))
			strncopy(fullname, scratch, cp1-scratch-1);
		else
			strcpy(fullname, scratch);
	} else if (cp = strrchr(scratch, '!'))
		strcpy(scratch, cp+1);
	else if (cp = strchr(scratch, '%'))
		strncopy(fullname, scratch, cp-scratch-1);
	else
		strcpy(fullname, scratch);
	cp = fullname;
	while (cp = strchr(cp, '_'))
		*cp++ = ' ';
}


/*
 * Build an internal message, based on the already parsed rfc822
 * header, and the mail body coming from infp.
 */
static boolean
build_msg(msg, infp, newsflag)
	Message *msg;
	FILE *infp;
	boolean newsflag;
{
	char buffer[BUFSIZ];
	boolean inheader;
	int c;
	char *cp;

	bzero((char *)msg, sizeof(Message));

	/*
	 * Read the mail, and store into message structure
	 */
	inheader = TRUE;
	while (fgets(buffer, BUFSIZ, infp)) {
		cp = strend(buffer);
		while (cp > buffer && cp[-1] == '\n')
			*--cp = 0;
		if (inheader) {
			if (*buffer == 0) {
				inheader = FALSE;
				continue;
			}
			/*
			 * See if we have a header continuation line
			 */
			for (;;) {
				c = getc(infp);
				if (c == -1)
					break;
				ungetc(c, infp);
				if (c != '\t' && c != ' ')
					break;
				if (fgets(cp, &buffer[BUFSIZ]-cp, infp) == NULL)
					break;
				cp = strend(cp);
				if (cp[-1] == '\n')
					*--cp = 0;
			}
			newrfchdrline(msg, buffer, 0);
		} else
			newbodyline(msg, buffer, 0, TRUE);
	}

	/*
	 * Parse rfc822 header
	 */
	parse_rfc_header(msg);

	/*
	 * Process/create message header
	 */
	msg->orig = mynode;
	msg->attr = public ? 0 : ATTR_PRIVATE;
	msg->cost = 0;
	

	/*
	 * Convert RFC822 header to IFNA header
	 */
	if (!create_ifna_header(msg, newsflag))
		return FALSE;

	return TRUE;
}


	
/*
 * Copy a message to output
 */
static void
copy_message(msg, fp)
	Message *msg;
	FILE *fp;
{
	struct line *lp;

	if (lp = msg->rfchdr) {
		do {
			fputs(lp->txt, fp);
			putc('\n', fp);
		} while ((lp = lp->next) != msg->rfchdr);
	}
	putc('\n', fp);
	if (lp = msg->body) {
		do {
			fputs(lp->txt, fp);
			putc('\n', fp);
		} while ((lp = lp->next) != msg->body);
	}
}


/* literal months */
static char *months[] = {
	"Jan", "Feb", "Mar", "Apr", "May", "Jun",
	"Jul", "Aug", "Sep", "Oct", "Nov", "Dec", NULL,
};

/* Literal weekdays */
static char *weekdays[] = {
	"Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat", NULL,
};
  


/*
 * Create IFNA header lines and data, based on the rfc822
 * header from a message packet.
 */
static boolean
create_ifna_header(msg, newsflag)
	Message *msg;
	boolean newsflag;
{
	time_t t;
	struct tm *localtime();
	struct tm *mtime;
	static char timebuf[20];
	char fullname[80], domainaddr[128];
	char *cp, *cp1, *cp2, *cp3;
	char scratch[256];
	int attr;
	Node tmpnode;

	/* To: or Comment-To: fields */
	if (header_to == NULL) {
		if (newsflag)
			if (header_comment_to)
				strncpy(msg->to, header_comment_to, 35);
			else
				strcpy(msg->to, "All");
	} else
		strncpy(msg->to, header_to, 35);

	/* jn: Strip @fxxx...  and _ */
	for (cp = msg->to; *cp; cp++)
		if (*cp == '_')
			*cp = ' ';
		else 
			if (*cp == '@') {
				*cp = '\0';
				break;
			}
	convert_full_name(msg->to);    
 
	/* Subject */
	if (header_subject)
		strncopy(msg->subject, header_subject, sizeof(msg->subject)-1);

	/* Date */
	if (header_date) {
		t = parsedate(header_date, NULL);
		if (t == -1) {
			buglog("Unparsable DATE: %s", header_date);
			t = time(NULL);
		}
	} else
		t = time(NULL);
	mtime = localtime(&t);
	/* Convert date to SEAdog format */
	sprintf(timebuf, "%3s %2d %3s %2d %02d:%02d",
		weekdays[mtime->tm_wday], mtime->tm_mday,
		months[mtime->tm_mon], mtime->tm_year,
		mtime->tm_hour, mtime->tm_min);
	strncopy(msg->ascdate, timebuf, sizeof(msg->ascdate)-1);
	msg->date = t;

	/* From: field */
	if (header_from == NULL) {
		log("Message has no 'From:' header field");
		sendback(msg, "Message has no 'From:' header field");
		return FALSE;
	}
	split_address(header_from, fullname, domainaddr);
	if (*domainaddr) {
		if (*fullname == 0)
			get_fullname(domainaddr, fullname);
		msg->replyaddr = strsave(domainaddr);
	}
	if (*fullname)
		strncopy(msg->from, fullname, sizeof(msg->from)-1);

	/* Message-ID: */
	if (header_message_id && newsflag) { /* No MSGID: for mail */
		if ((cp1 = strchr(header_message_id, '<')) == NULL
		    || (cp2 = strchr(cp1, '>')) == NULL
		    || (cp = strchr(cp1, '@')) == NULL
		    || cp > cp2)
			buglog("Illegal RFC822 Message-ID: %s",
			       header_message_id);
		else {
			/*
			 * Change
			 *	Message-ID: <id@domain>
			 * to:
			 *	MSGID: domain id
			 *
			 * Try to convert domain to fidonet
			 * node number.
			 */
			strncopy(scratch, cp+1, cp2-cp-1);
			if (parseinternode(scratch, &tmpnode))
				strcpy(scratch, ascnode(tmpnode));
			cp3 = strend(scratch);
			*cp3++ = ' ';
			strncopy(cp3, cp1+1, cp-cp1-1);
#ifndef MSGID_POLITICS
			/*
			 * The resulting rfc822-type msgid is not very
			 * liked on fidonet. As an alternative, to not
			 * to break too many existing programs, we do
			 * a crc32() on the id part of msgid, to make
			 * it a hex number, if it is not already one.
			 */
			for (cp = cp3; *cp; cp++)
				if (!isxdigit(*cp))
					break;
			if (*cp) {
				Ulong crc = 0xffffffffU;

				while (*cp) {
					register Uchar c = *cp++;
					crc = updcrc32(c, crc);
				}
				sprintf(cp3, "%08x", crc);
			}
#endif
			msg->msgid = strsave(scratch);
		}
	}
	
	/*
	 * Look for Fidonet flags
	 */
	if (header_fidonet_flags) {
		strcpy(scratch, header_fidonet_flags);
		cp = strtok(scratch, " ");
		while (cp) {
			strlwr(cp);
			attr = binary_attribute(cp);
			if (attr == NULL)
				buglog("Illegal X-Flag: %s", cp);
			else
				msg->attr |= attr;
			cp = strtok(NULL, " ");
		}
	}

	/* In-Reply-To: */
	if (header_in_reply_to) {
		if ((cp1 = strchr(header_in_reply_to, '<')) == NULL
		    || (cp2 = strchr(cp1, '>')) == NULL
		    || (cp = strchr(cp1, '@')) == NULL
		    || cp > cp2)
			buglog("Illegal RFC822 In-Reply-To: %s",
			       header_in_reply_to);
		else {
			/*
			 * Change
			 *	In-Reply-To: <id@domain>
			 * to:
			 *	REPLY: domain id
			 *
			 * Convert internet domain address to
			 * fidonet node number, if applicable.
			 */
			strncopy(scratch, cp+1, cp2-cp-1);
			if (parseinternode(scratch, &tmpnode))
				strcpy(scratch, ascnode(tmpnode));
			cp3 = strend(scratch);
			*cp3++ = ' ';
			strncopy(cp3, cp1+1, cp-cp1-1);
			msg->reply = strsave(scratch);
		}
	}
	return TRUE;
}

/* 
 * Spool Usenet news articles to fidonet. Currently there is no news-system
 * in fido, it's done by software after and before mail-slots. Echomail
 * is most popular one, and this program does about the same thing:
 * it sends news-articles as fido-mail containing suitable information
 * for echomail. Echomail relies on normal fido-mail, we send news thru
 * rfmail, because this mechanism requires nothing different from normal
 * mail
 */

int
ngcmp(ng1, ng2)
	Newsgroup *ng1, *ng2;
{
	return strcmp(ng1->ng, ng2->ng);
}


/* 
 * Do actual processing of article. This will copy article, get newsgoroups
 * and save them and then send all rest of article.
 */
int
rfnews(argc, argv)
	int argc;
	char **argv;
{
	char scratch[256];
	char *cp;
	Message msg;
	Newsgroup *ngp, ng;
	int i;

	/* jn: ECHOMAIL is always public, well it should be anyway */
	public = TRUE;

	/* Sort newsgroups according to newsgroup name for easier searching */
  
#ifdef USG
	qsort( (char *) config.ng, (Uint) config.newsgroups,
	      sizeof(Newsgroup), ngcmp);
#endif
#ifdef BSD
	qsort( (char *) config.ng, config.newsgroups,
	      sizeof(Newsgroup), ngcmp);
#endif
  
	get_nodelist();			/* Read nodelist index */

	build_msg(&msg, stdin, TRUE); /* Build a fidonet message */
  
  
	/*
	 * Now the headers are built (except for AREA:)
	 * parse the "Newsgroups:" rfc header line, and 
	 * convert to fidonet area name, and pass on
	 * to all nodes that want to be fed from that newsgroup.
	 */
	if (header_newsgroups) {
		strcpy(scratch, header_newsgroups);
		cp = strtok(scratch, " ,");
		while (cp) {
			/*
			 * We have a newsgroup name, try
			 * check if this has a fidonet equivalent, and
			 * if so, forward it to fidonet.
			 */
			strncopy(ng.ng, cp, sizeof(ng.ng)-1);
			if (ngp = (Newsgroup *)
			    bsearch( (char *) &ng, (char *) config.ng,
				    (Uint) config.newsgroups,
				    sizeof(Newsgroup), ngcmp))
				{
				/*
				 * Add special headers to message
				 * Note, they are put last in the body of the message,
				 * since the header lines comes before the message.
				 */

				newbodyline(&msg, sprintfs("--- rfmail %s", version), 0, TRUE);
				newbodyline(&msg, sprintfs(" * Origin: %s (%s)",
							     config.origin,
							     ngp->my_address.point
							     	? ascnode(ngp->my_address)
							        : ascnode_nopoint(ngp->my_address)),
					    0, TRUE);
				add_seenby(&msg, ngp->my_address);
				for (i = 0; i < config.seenbys; i++)
					add_seenby(&msg, config.seenby[i]);

				sendnews(&msg, ngp);
				}
			cp = strtok(NULL, ", ");
		}
	}
	return EX_OK;
}


/*
 * Send a news message to an echomail area.
 */
boolean
sendnews(msg, ngp)
	Message *msg;
	Newsgroup *ngp;
{
	Message m;
	Node newnode[MAX_NEWSGROUP_NODES];
	Newsgroup *ng[MAX_NEWSGROUP_NODES];
	int newnodes, i, j;

	/*
	 * Copy message
	 */
	m = *msg;
	m.seenby = (Node *)mymalloc(m.maxseenbys * sizeof(Node));
	bcopy((char *)msg->seenby, (char *)m.seenby, m.maxseenbys * sizeof(Node));

	/*
	 * Make a list of all nodes to send news message to
	 */
	newnodes = 0;
	for (i = 0; i < ngp->nodes; i++) {
		for (j = 0; j < m.seenbys; j++)
			if (cmpnode(&ngp->u.node[i], &m.seenby[j]) == 0)
				break;
		if (j == m.seenbys) {
			ng[newnodes] = ngp;
			newnode[newnodes++] = ngp->u.node[i];
			add_seenby(&m, ngp->u.node[i]);
		}
	}

	/*
	 * Now write message to the msg spool directory.
	 */
	for (i = 0; i < newnodes; i++) {
		m.area = ng[i]->echo;
		m.ngp = ng[i];
		write_msg(&m, ngp->my_address, newnode[i]);
	}
	free((char *)m.seenby);
	return TRUE;
}

/*
 * write a message to the fidonet message spool area
 *
 * This routine require that the "msg" parameter is completely
 * set-up for fidonet mail.
 */
boolean
write_msg(msg, fromnode, tonode)
	Message *msg;
	Node fromnode, tonode;
{
	Message m;
	FILE *fp;
	char *sfile;
	struct line *lp;
	int i, attr;
	char scratch[128];
	char *cp, *cp1;
	Node lastnode, dest;
	boolean havevia, newsmsg, haveorig, bossroute;
	struct dest_info destinfo;


	m = *msg;
	m.orig = fromnode;
	m.dest = tonode;
	dest = tonode;

	/*
	 * If destination is a point, and not one of our points,
	 * route the letter to the boss of the point.
	 */
	bossroute = FALSE;
	if (tonode.point != 0) {
		dest = tonode;
		dest.point = 0;
		if (is_our_node(dest)) {
			bossroute = TRUE;
			dest = tonode;
		}
	}

	sfile = spoolfile("M.");
	if (!valid_netnode(m.to, dest)) {
		log("Message is rejected\n");
		return FALSE;
	}
	haveorig = (boolean)(m.orig.net != 0 && m.orig.node != 0);
	if (msg->area != NULL) {
		newsmsg = TRUE;
		msglog("[%s] to AREA: %s@%s",
		       msg->msgid ? msg->msgid : "",
		       msg->area, ascnode(m.dest));
	} else {
		if (!haveorig && m.replyaddr == NULL)
			return FALSE;
		if (haveorig)
			msglog("Mail from %s@%s to %s@%s",
			       m.from, ascnode(m.orig), m.to, ascnode(m.dest));
		else
			msglog("Mail from %s to %s@%s",
			       msg->replyaddr, m.to, ascnode(m.dest));
		newsmsg = FALSE;
	}
	if (bossroute)
		msglog("Message routed to boss");

	if (!haveorig)
		m.orig = config.mynode;
	log("Message file %s", spoolname(sfile));
	fp = myfopen(sfile, "w");
	lock(fileno(fp));

	fprintf(fp, "N %s\n", ascnode(dest));

	if (haveorig)
		fprintf(fp, "O %s\n", ascnode(m.orig));

	fprintf(fp, "T %s\n", m.to);

	/*
	 * replyaddr as 'from' is removed, since
	 * replyaddr does not contain the fullname,	
	 * which gives big complaints from fidonet.
	 */
/*	if (m.replyaddr)
		fprintf(fp, "F %s\n", m.replyaddr);
	else */
		fprintf(fp, "F %s\n", m.from);
	fprintf(fp, "S %s\n", m.subject);
	fprintf(fp, "D %s\n", m.ascdate);
	if (m.replyaddr)
		fprintf(fp, "R %s\n", m.replyaddr);
	if (m.attr & (ATTR_PRIVATE|ATTR_CRASH|ATTR_FILE|ATTR_RRR|ATTR_IRR|ATTR_AUDREQ)) {
		fprintf(fp, "X");
		for (attr = 1; attr <= 0100000; attr <<= 1) {
			if (m.attr & attr) {
				switch (attr) {
				case ATTR_PRIVATE: fprintf(fp, " P"); break;
				case ATTR_CRASH:   fprintf(fp, " C"); break;
				case ATTR_FILE:    fprintf(fp, " F"); break;
				case ATTR_RRR:	   fprintf(fp, " R"); break;
				case ATTR_IRR:     fprintf(fp, " I"); break;
				case ATTR_AUDREQ:  fprintf(fp, " A"); break;
				}
			}
		}
		putc('\n', fp);
	}

	/*
	 * The divider between commands and the message text
	 */
	putc('\n', fp);

	/*
	 * First emit various headers which were speically extracted
	 */
	if (m.area)
		fprintf(fp, "AREA:%s\n", m.area);
	/*
	 * I am not sure about the utility of this header.
	 */
	fprintf(fp, "\001RFMAIL news/mail %s\n", version);
	if (m.dest.zone != config.mynode.zone)
		fprintf(fp, "\001INTL %s %s\n",
			ascnode_nopoint(m.dest), ascnode_nopoint(m.orig));
	if (m.orig.point)
		fprintf(fp, "\001FMPT %d\n", m.orig.point);
	if (m.dest.point)
		fprintf(fp, "\001TOPT %d\n", m.dest.point);
	if (m.replyaddr)
		fprintf(fp, "\001REPLYADDR %s\n", m.replyaddr);
	if (m.msgid)
		fprintf(fp, "\001MSGID: %s\n", m.msgid);
	if (m.reply)
		fprintf(fp, "\001REPLY: %s\n", m.reply);

	/*
	 * Emit headers which were not specially processed
	 * except for "Via" lines, which comes after the message.
	 */
	havevia = FALSE;
	if (lp = m.headers) {
		do {
			if (!newsmsg && strnequ(lp->txt, "Via ", 4))
				havevia = TRUE;
			else
				fprintf(fp, "\001%s\n", lp->txt);
			lp = lp->next;
		} while (lp != m.headers);
	}

	/*
	 * Emit message body
	 */

	/*
	 * I don't understand the utility of these rfc822-type headers
	 * in a fidonet message. But if you like it, you can always
	 * defined HEADER_KLUDGE, and you will get them.
	 */
#ifdef HEADER_KLUDGE
	if (m.internet)
		fprintf(fp, "From: %s\n", m.internet);
	if (m.ufgroup) {
		fprintf(fp, "Newsgroups: %s\n", m.ufgroup);
		if (m.ufgid)
			fprintf(fp, "Message-ID: %s\n", m.ufgid);
	}
#endif	/* HEADER_KLUDGE */
	if (lp = m.body) {
		do {
			/* another kludge for uu2fido.sh */
			if (lp == m.body && strnequ(lp->txt, "To:", 3)) {
				fputs("\n", fp);
			}
			fputs(lp->txt, fp);
			putc(lp->softeol ? '\r' : '\n', fp);
			lp = lp->next;
		} while (lp != m.body);
	}
	
	if (newsmsg) {
		/*
		 * Create the "SEEN-BY:" kludge lines
		 */
		if (msg->seenbys) {
			cp = scratch;
			for (i = 0; i < msg->seenbys; ) {
				if (cp == scratch
				    || lastnode.zone != msg->seenby[i].zone
				    || lastnode.net != msg->seenby[i].net) {
					if (cp == scratch) {
						sprintf(scratch, "SEEN-BY:");
						cp = scratch + 8;
					}
					sprintf(cp, " %s", netnode(msg->seenby[i]));
				} else
					sprintf(cp, " %d", msg->seenby[i].node);
				lastnode = msg->seenby[i];
				cp1 = strend(cp);
				if (cp1-scratch < config.max_linelen) {
					cp = cp1;
					i++;
					continue;
				}
				*cp = 0;
				cp = scratch;
				fputs(scratch, fp);
				putc('\n', fp);
			}
			if (cp != scratch) {
				fputs(scratch, fp);
				putc('\n', fp);
			}

			/*
			 * Emit the final "PATH:" kludge, appending our node
			 * This can be done smarter.
			 */
			destinfo = get_destinfo(tonode);
			if (m.path)
				fprintf(fp, "\001PATH: %s %s\n",
					m.path, netnode(destinfo.my_address));
			else
				fprintf(fp, "\001PATH: %s\n",
					netnode(destinfo.my_address));
		}
	} else {
		/*
		 * Emit the "^AVia" kludge lines.
		 * Add our own via if this message is routed.
		 */
		if (havevia) {
			lp = m.headers;
			do {
				if (strnequ(lp->txt, "Via ", 4))
					fprintf(fp, "\001%s\n", lp->txt);
				lp = lp->next;
			} while (lp != m.headers);
		}

		destinfo = get_destinfo(tonode);	
		if (m.attr & ATTR_INTRANSIT)
			fprintf(fp, "\001Via %s %s, %s (rfmail %s)\n",
				hostname, ascnode(destinfo.my_address),
				date(NULL, NULL), version);
	}

	fclose(fp);
	return TRUE;
}


/*
 * Fixup "replyaddr" kludge
 */
static void
fix_replyaddr(msg)
	Message *msg;
{
	register char *cp;
	char host[64], user[64];
	int target;
	Node node, mask;
	boolean hasnode;
	
	if (msg->replyaddr == NULL)
		return;

	/*
	 * Check if the return address is on the form:
	 *	username@hostdomain
	 * or
	 *	hostdomain!username
	 */
	if (cp = strchr(msg->replyaddr, '@')) {
		strcpy(host, cp+1);
		strncopy(user, msg->replyaddr, cp-msg->replyaddr);
	} else if (cp = strrchr(msg->replyaddr, '!')) {
		strncopy(host, msg->replyaddr, cp-msg->replyaddr);
		strcpy(user, cp+1);
	} else {
		msg->orig = config.mynode;
		return;
	}
	if (strpbrk(user, "%@!")) {
		msg->orig = config.mynode;
		return;
	}
	/*
	 * Check the domain with the routing table,
	 * to see if we can substitute a fidonet node number
	 */
	hasnode = FALSE;
	for (target = 0; target < config.targets; target++) {
		if (config.routing[target].rtype != ROUTING_UNIXMAIL)
			continue;
		if (strequ(host, config.routing[target].unixaddr)
		    && config.routing[target].masks > 0) {
			mask = config.routing[target].mask[0];
			node.zone = mask.zone == NODE_WILDCARD ?
				config.mynode.zone : mask.zone;
			node.net = mask.net == NODE_WILDCARD ?
				config.mynode.net : mask.net;
			node.node = mask.node == NODE_WILDCARD ?
				config.mynode.node : mask.node;
			node.point = mask.point == NODE_WILDCARD ?
				config.mynode.point : mask.point;
			hasnode = TRUE;
			break;
		}
	}

	/*
	 * If we got a fidonet node number, it can uniquely
	 * identify the reply address, otherwise we have to 
	 * generate the "REPLYADDR" kludge line.
	 */
	if (hasnode)
		msg->orig = node;
#ifdef EXTRA_REPLYADDR
	else
		newheaderline(msg, sprintfs("REPLYADDR %s", msg->replyaddr),0);
#endif

}



char *
myitoa(n)
	int n;
{
	static char buffer[40];

	sprintf(buffer, "%d", n);
	return buffer;
}

char *
convert_full_name(name)
	char *name;
{
	char *cp;

	for (cp=name; *cp; cp++)
		if ((cp == name || *(cp-1) == ' ' || *(cp-1) == '-') &&
		    islower(*cp))
			*cp = toupper(*cp);
	return name;
}
