/*
 * $Header: /u1/src/rfmail/RCS/packet.c,v 0.5.0.1 1992/06/15 06:11:25 pgd Exp pgd $
 *
 * $Log: packet.c,v $
 * Revision 0.5.0.1  1992/06/15  06:11:25  pgd
 * Minor compilation bug fixes.
 * Change of all types with u_ prefix to U prefix
 * Change of name of routine msleep() to mssleep()
 *
 * Revision 0.5  1992/05/18  04:27:24  pgd
 * New distribution
 *
 * Revision 0.4.1.6  1992/03/15  07:58:52  pgd
 * Untested version
 *
 * Revision 0.4.1.3  1991/06/15  09:33:39  pgd
 * *** empty log message ***
 *
 * Revision 0.4.1.1  1991/05/21  11:13:48  pgd
 * *** empty log message ***
 *
 * Revision 0.4  1991/05/08  04:23:43  pgd
 * Initial Beta-release
 *
 *
 */

/*
 * Misc routines handling packets and messages
 *
 * Author:
 *
 * Per Lindqvist (pgd@compuram.bbt.se) 
 */

#include "fnet.h"

#include "nodelist.h"
#include "configs.h"
#include "packet.h"
#include "rfmail.h"
  
LDECLARE(boolean, read_date, (FILE *, time_t *, char *));
LDECLARE(char *, fido_date, (time_t));
LDECLARE(char *, saveline, (char *));
LDECLARE(boolean, write_fts0001_header, (struct packet * ,register FILE *));
LDECLARE(boolean, write_fsc0039_header, (struct packet * ,register FILE *));
LDECLARE(boolean, write_fsc0045_header, (struct packet * ,register FILE *));
LDECLARE(boolean, write_fsc0048_header, (struct packet * ,register FILE *));

/*
 * Table with message attributes mapping
 */
static struct {
	int	attr;
	char	*name;
} message_attributes[] = {
	{ATTR_PRIVATE,		"private"},
	{ATTR_CRASH,		"crash"},
	{ATTR_RECD,		"received"},
	{ATTR_SENT,		"sent"},
	{ATTR_FILE,		"file-attach"},
	{ATTR_INTRANSIT,	"in-transit"},
	{ATTR_ORPHAN,		"orphan"},
	{ATTR_KILLSENT,		"kill-when-sent"},
	{ATTR_LOCAL,		"local"},
	{ATTR_HOLD,		"hold"},
	{ATTR_UNUSED,		"unused"},
	{ATTR_FILEREQ,		"file-request"},
	{ATTR_RRR,		"return-receipt-request"},
	{ATTR_IRR,		"is-return-receipt"},
	{ATTR_AUDREQ,		"audit-request"},
	{ATTR_UPDREQ,		"file-update-request"}
};


static char *
saveline(cp)
	register char *cp;
{
	register char *cp1;

	while (isspace(*cp))
		cp++;
	cp1 = strend(cp);
	while (cp1 > cp && isspace(cp1[-1]))
		*--cp1 = 0;
	return strsave(cp);
}


/*
 * Return the ascii mnemonic for a message attribute
 */
char *
ascii_attribute(attr)
	int attr;
{
	register int i;

	for (i = 0; i < sizeof(message_attributes)/sizeof(*message_attributes); i++)
		if (attr & message_attributes[i].attr)
			return message_attributes[i].name;
	return NULL;
}

/*
 * Convert an ascii mnemonic for a message attribute to binary
 */
int
binary_attribute(attr)
	char *attr;
{
	register int i;

	for (i = 0; i < sizeof(message_attributes)/sizeof(*message_attributes); i++)
		if (strequ(attr, message_attributes[i].name))
			return message_attributes[i].attr;
	return 0;
}


boolean
read_packet_header(fp, pp)
	register FILE *fp;
	register struct packet *pp;
{
	register int c;
	int i;
	Uchar hdr[0x3a];
	Ushort hdrver;

	for (i = 0; i < sizeof(hdr); i++) {
		c = getc(fp);
		if (c == -1)
			return FALSE;
		hdr[i] = c;
	}

	pp->ver = hdr[0x12] + (hdr[0x13] << 8);
	if (pp->ver != 2) {
		log("Cannot handle packet version %02x", hdrver);
		return FALSE;
	}

	pp->orig.point = pp->dest.point = 0;
	pp->orig.zone = pp->dest.zone = 0;
	pp->orig.node = hdr[0x00] | (hdr[0x01] << 8);
	pp->dest.node = hdr[0x02] | (hdr[0x03] << 8);
	pp->year      = hdr[0x04] | (hdr[0x05] << 8);
	pp->month     = hdr[0x06] | (hdr[0x07] << 8);
	pp->day       = hdr[0x08] | (hdr[0x09] << 8);
	pp->hour      = hdr[0x0a] | (hdr[0x0b] << 8);
	pp->minute    = hdr[0x0c] | (hdr[0x0d] << 8);
	pp->second    = hdr[0x0e] | (hdr[0x0f] << 8);
	pp->rate      = hdr[0x10] | (hdr[0x11] << 8);
	pp->orig.net  = hdr[0x14] | (hdr[0x15] << 8);
	pp->dest.net  = hdr[0x16] | (hdr[0x17] << 8);
	for (i = 0; i < 8; i++)
		pp->passwd[i] = hdr[0x1a+i];
	pp->passwd[8] = 0;

	if (hdr[0x2c] == hdr[0x29] && (hdr[0x2d] & 0x7f) == (hdr[0x28] & 0x7f))
		pp->capword = hdr[0x2c] | (hdr[0x2d] << 8);
	else
		pp->capword = 0;

	if (pp->capword & 1) {
		/*
		 * We have a FSC-039 or FSC-048 revision 2+ packet.
		 */
		pp->capword = hdr[0x2c] | (hdr[0x2d] << 8);
		pp->prodcode = hdr[0x18] | (hdr[0x2a] << 8);
		pp->prodmaj = hdr[0x19];
		pp->prodmin = hdr[0x2b];
		pp->proddata = hdr[0x36] | (hdr[0x37] << 8)
			| (hdr[0x38] << 16) | (hdr[0x39] << 24);
		pp->pktterm = hdr[0x3a] | (hdr[0x3b] << 8);
		pp->orig.zone = hdr[0x22] | (hdr[0x23] << 8);
		pp->dest.zone = hdr[0x24] | (hdr[0x25] << 8);
		pp->orig.point = hdr[0x32] | (hdr[0x33] << 8);
		pp->dest.point = hdr[0x34] | (hdr[0x35] << 8);
		pp->serialno = 0;

		if ((pp->orig.point != 0) && (pp->orig.net == 0xffff))
			/* 
			 * Got a FSC-048 packet
			 */
			pp->orig.net = hdr[26] | (hdr[27] << 8);

	} else {
		pp->capword = 0;
		pp->prodcode = hdr[0x18];
		pp->prodmaj = pp->prodmin = 0;
		pp->proddata = 0;
		pp->pktterm = 0;
		pp->orig.point = 0;
		pp->dest.point = 0;
		pp->serialno = hdr[0x19];
	}
		
	if (pp->orig.zone == 0)
		pp->orig.zone = config.mynode.zone;
	if (pp->dest.zone == 0)
		pp->dest.zone = config.mynode.zone;
	return TRUE;
}


boolean
read_type2_header(fp, msg)
	FILE *fp;
	struct message *msg;
{
	long pos;

	msg->orig.zone = msg->dest.zone = 0;
	msg->orig.point = msg->dest.point = 0;

	if (!read_int16(fp, &msg->orig.node)) 	goto error;
	if (!read_int16(fp, &msg->dest.node)) 	goto error;
	if (!read_int16(fp, &msg->orig.net)) 	goto error;
	if (!read_int16(fp, &msg->dest.net))	goto error;
	if (!read_int16(fp, &msg->attr))	goto error;
	if (!read_int16(fp, &msg->cost))	goto error;
	if (!read_date(fp, &msg->date, msg->ascdate)) goto error;
	pos = ftell(fp);
	if (!read_string(fp, msg->to, 36))	goto error;
	if (!read_string(fp, msg->from, 36))	goto error;
	if (!read_string(fp, msg->subject, 72))	goto error;
	if (strnequ("AREA:", msg->subject, 5) || strchr(msg->subject, '\r')) {
		/*
		 * Some message packers give a null-terminated 18-character
		 * date, instead of padding to 20 bytes. Some compensation
		 * for this is done in read_date, but if the "To:" field is
		 * null, we still get out of phase.
		 * If we get the "AREA:" line into the subject, that is
		 * probably the case. Here we try to correct for that
		 * problem.
		 */
		fseek(fp, pos, 0);
		strncpy(msg->to, "All", sizeof(msg->to));
		if (!read_string(fp, msg->from, 36))	goto error;
		if (!read_string(fp, msg->subject, 72))	goto error;
	}
	return TRUE;
 error:
	return FALSE;
}

/*
 * Read a message into memory.
 *
 * The type byte is already read.
 */
boolean
read_type2_message(fp, msg, pkt)
	FILE *fp;
	struct message *msg;
	struct packet *pkt;
{
	register int c;
	int n;
	char buf[256];		/* Collect lines here */
	char *cp, *lastsp = NULL, *cp1;
	boolean kludge, needspace, special;
	Node tmpnode;
	struct line *lp, *last;
	int pos;

	bzero((char *)msg, sizeof(struct message));

	if (!read_type2_header(fp, msg))
		return FALSE;

	if (msg->dest.zone == 0)
		msg->dest.zone = pkt->dest.zone;
	if (msg->orig.zone == 0)
		msg->orig.zone = pkt->orig.zone;

	debug(2, "Message From: %s %s To: %s %s",
	      msg->from, ascnode(msg->orig),
	      msg->to, ascnode(msg->dest));

	/* Replace all spaces with '_'s */
	for (cp = msg->from; cp = strchr(cp, ' '); )
		*cp++ = '_';
	/* Replace all spaces with '_'s */
	for (cp = msg->to; cp = strchr(cp, ' '); )
		*cp++ = '_';
      
	/*
	 * Process message body
	 */
	msg->body = msg->headers = msg->rfchdr = NULL;
	*buf = 0;
	cp = buf;
	lastsp = NULL;
	kludge = needspace = special = FALSE;

	pos = 0;
	while ((c = getc(fp)) && c != EOF) {
		if (!special && ((pos > config.max_linelen && lastsp)
				 || c == '\n')) {
			/*
			 * Check for " * Origin:" or "SEEN-BY:"
			 */
			if (strnequ(buf, "SEEN-BY: ", 9)
			    || strnequ(buf, " * Origin: ", 11))
				special = TRUE;
		}
		if (pos > config.max_linelen && lastsp && !special) {
			/*
			 * We have to break this line here, and wrap the
			 * last partial word to the next line.
			 * But watch out for the "* Origin:" line and the
			 * SEEN-BY: lines, which should not wrap.
			 * This code is removing the space, where the
			 * wrapping is taking place. Maybe it should not?
			 */
			newbodyline(msg, buf, lastsp-buf, FALSE);
			lastsp++;
			cp1 = cp;
			cp = buf;
			pos = 0;
			while (lastsp < cp1) {
				*cp = *lastsp++;
				if (*cp == '\t')
					pos = (pos + 8) & ~7;
				else
					pos++;
				cp++;
			}
			lastsp = NULL;
		}
		if (cp == &buf[sizeof(buf)-1] && c != '\r') {
			/*
			 * Buffer is going to overflow. We
			 * pretend that we got an end-line.
			 */
			*cp++ = c;
			c = '\r';
		}
		if (c == '\r' || (special && c == '\n')) {
			/*
			 * Hard end-line, store this line.
			 * We also process the "AREA:" header line
			 * here, since it is not preceeded by
			 * the CTRL-A kludge.
			 */
			*cp = 0;
			if (kludge) {
				newheaderline(msg, cp==buf?"":buf, cp-buf);
				kludge = FALSE;
			} else if (msg->body == NULL
				   && msg->area == NULL
				   && strncmp(buf, "AREA:", 5) == 0)
				msg->area = strsave(buf+5);
			else
				newbodyline(msg, cp==buf?"":buf, 0, TRUE);
			cp = buf;
			special = FALSE;
			lastsp = NULL;
			pos = 0;
		} else if (cp == buf && c == 1)
			kludge = special = TRUE;
		else if (c == '\n' || c == 0215) {
			needspace = (boolean)(cp > buf && cp[-1] != ' '
					      && cp[-1] != '\t');
		} else if (c == ' ' || needspace) {
			needspace = FALSE;
			lastsp = cp;
			*cp++ = c;
			pos++;
		} else if (c == '\t') {
			*cp++ = c;
			pos = (pos + 8) & ~7;
		} else {
			*cp++ = c;
			pos++;
		}
	}

	/*
	 * Scan kludge lines
	 */
	if (lp = msg->headers) {
		last = lp->prev;
		while (last) {
			if (lp == last)
				last = NULL;
			if (strncmp(lp->txt, "TOPT", 4) == 0) {
				if (check_atoi(lp->txt+4, &n)) {
					msg->dest.point = n;
					lp = delline(lp, &msg->headers);
				} else goto bad;
			} else if (strncmp(lp->txt, "FMPT", 4) == 0) {
				if (check_atoi(lp->txt+4, &n)) {
					msg->orig.point = n;
					lp = delline(lp, &msg->headers);
				} else goto bad;
			} else if (strncmp(lp->txt, "INTL", 4) == 0) {
				cp = lp->txt+4;
				if (*cp == ':') cp++;
				while (isspace(*cp)) cp++;
				cp1 = strchr(cp, ' ');
				if (cp1) {
					do {
						*cp1++ = 0;
					} while (isspace(*cp1));
				}
				if (parsenode(cp, &tmpnode, NULL, NULL)) {
					tmpnode.point = msg->dest.point;
					msg->dest = tmpnode;
				} else
					goto bad;
				if (cp1) {
					if (parsenode(cp1, &tmpnode, NULL, NULL)) {
						tmpnode.point = msg->orig.point;
						msg->orig = tmpnode;
					} else
						goto bad;
				}
				lp = delline(lp, &msg->headers);
			} else if (strncmp(lp->txt, "REPLYADDR", 9) == 0) {
				cp = lp->txt+9;
				if (*cp == ':') cp++;
				msg->replyaddr = saveline(cp);
				lp = delline(lp, &msg->headers);
			} else if (strncmp(lp->txt, "PATH:", 5) == 0) {
				msg->path = saveline(lp->txt+5);
				lp = delline(lp, &msg->headers);
			} else if (strncmp(lp->txt, "MSGID:", 6) == 0) {
				msg->msgid = saveline(lp->txt+6);
				lp = delline(lp, &msg->headers);
			} else if (strncmp(lp->txt, "REPLY:", 6) == 0) {
				msg->reply = saveline(lp->txt+6);
				lp = delline(lp, &msg->headers);
			} else if (strncmp(lp->txt, "TO:", 3) == 0) {
				msg->toto = saveline(lp->txt+3);
				lp = delline(lp, &msg->headers);
			} else if (strncmp(lp->txt, "RFMAIL", 6) == 0) {
				/*
				 * What is this header for?
				 * I think we should just quietly delete it.
				 */
				lp = delline(lp, &msg->headers);
			}
				
			goto good;
		bad:
			log("Illegal message header line: %s", lp->txt);
		good:
			if (lp == NULL)
				break;
			lp = lp->next;
		}
	}
	return TRUE;
}

			
/*
 * Write a message to a type-2 packet
 *
 * This rutine assumes the message was read with "read_message",
 * and thus 'body' contains the whole fidonet type message
 * including the header lines.
 */
boolean
write_type2_hdr(msg, fp)
	register struct message *msg;
	register FILE *fp;
{
	/*
	 * Write header
	 */
	if (!write_int16(fp, MSGTYPE)         ||
	    !write_int16(fp, msg->orig.node)  ||
	    !write_int16(fp, msg->dest.node)  ||
	    !write_int16(fp, msg->orig.net)   ||
	    !write_int16(fp, msg->dest.net)   ||
	    !write_int16(fp, msg->attr)       ||
	    !write_int16(fp, msg->cost)	    ||
	    !write_string(fp, msg->ascdate) ||
	    !write_string(fp, msg->to)      ||
	    !write_string(fp, msg->from)    ||
	    !write_string(fp, msg->subject))
		return FALSE;
	return TRUE;
}

/*
 * Store another line into the text body
 */
void
newbodyline(msg, cp, len, hardeol)
	struct message *msg;
	char *cp;
	int len;
	boolean hardeol;
{
	register struct line *lp;

	if (msg->body == NULL)
		msg->body = (lp = addline(NULL, cp, len));
	else
		lp = addline(msg->body, cp, len);
	lp->softeol = (boolean)!hardeol;
}

void
newheaderline(msg, cp, len)
	struct message *msg;
	char *cp;
	int len;
{
	if (msg->headers == NULL)
		msg->headers = addline(NULL, cp, len);
	else
		(void) addline(msg->headers, cp, len);
}

void
newrfchdrline(msg, cp, len)
	struct message *msg;
	char *cp;
	int len;
{
	if (msg->rfchdr == NULL)
		msg->rfchdr = addline(NULL, cp, len);
	else
		(void) addline(msg->rfchdr, cp, len);
}



/*
 * Insert/append a line to the end of the list
 */
struct line *
addline(here, txp, len)
	register struct line *here;
	char *txp;
	int len;
{
	char *s;
	register struct line *lp;

	if (len) {
		s = strnsave(txp, len);
	} else
		s = strsave(txp);
	lp = (struct line *)mymalloc(sizeof(struct line));
	lp->txt = s;
	lp->softeol = FALSE;
	if (here) {
		lp->next = here;
		lp->prev = here->prev;
		here->prev->next = lp;
		here->prev = lp;
	} else
		lp->next = lp->prev = lp;
	return lp;
}

/*
 * Delete a line.
 */
struct line *
delline(this, head)
	struct line *this, **head;
{
	struct line *lp;

	if (this->txt)
		free(this->txt);
	if (this == this->next) {
		free((char *)this);
		if (head)
			*head = NULL;
		return NULL;
	} else {
		this->next->prev = this->prev;
		this->prev->next = this->next;
		lp = this->prev;
		if (head && *head == this)
			*head = this->next;
		free((char *)this);
		return lp;
	}
}

/* 
 * Parse date of message. Date will be read and converted to one
 * long-integer as normal Unix-time will be.
 * 
 * Both SEAdog and Fido date formats are understood.
 * SEAgod: Mon 1 Jan 86 02:34
 * Fido:   01 Jan 86 02:34:56
 *
 * Return FALSE only on read errors.
 */
static boolean
read_date(fp, tm, buffer)
	FILE *fp;
	time_t *tm;
	char *buffer;
{
	time_t timevar;
	int n, c;
  
	/* read date from packet */
	for (n = 0; n < 20; n++) {
		c = getc(fp);
		if (c == EOF)
			return FALSE;
		buffer[n] = c;
	}

	/* Some message-packers do mistakes! Date should be 20 bytes but
	   they start name directly after null terminating 18-char date
	   used in some systems. Check if following char is null or not,
	   and if not, put it back there as it probably is first char in
	   recipent name. This seems to be problem in OMMM, but I'm not
	   sure yet.

	   Wed Nov 16 21:11:34 1988 Seems that the bug is in fsc001, as
	   I constantly keep receiving messages which 19 byte date?
	 */
  
	if (c) {
		ungetc(c, fp);
		buffer[19] = 0;
	}
  
	/* try to get date */
	timevar = parsedate(buffer, NULL);
	if (timevar == -1) {
		*tm = time(NULL);
		buglog("Unparseable date %s, using today", buffer);
	} else
		*tm = timevar;
	return TRUE;
}



/*
 * free all storage associated with a message
 */
void
freemsg(msg)
	struct message *msg;
{
	register struct line *lp;

	if (msg->area) free(msg->area), msg->area=NULL;
	if (msg->replyaddr) free(msg->replyaddr), msg->replyaddr=NULL;
	if (msg->path) free(msg->path), msg->path=NULL;
	if (msg->msgid) free(msg->msgid), msg->msgid=NULL;
	if (msg->reply) free(msg->reply), msg->reply = NULL;
	if (msg->seenby) free((char *)msg->seenby), msg->seenby = NULL;
	if (msg->internet) free((char *)msg->internet), msg->internet = NULL;
	if (msg->ufgid) free((char *)msg->ufgid), msg->ufgid = NULL;
	if (msg->ufgroup) free((char *)msg->ufgroup), msg->ufgroup = NULL;
	for (lp = msg->body; lp; lp = delline(lp, &msg->body))
		;
	msg->body = NULL;
	for (lp = msg->headers; lp; lp = delline(lp, &msg->headers))
		;
	msg->headers = NULL;
	for (lp = msg->rfchdr; lp; lp = delline(lp, &msg->rfchdr))
		;
	msg->rfchdr = NULL;
}

/*
 * Read a message written by "write_msg"
 *
 * Mailfile contains first headers
 * terminated by empty line, after that comes message text.
 * Header consists of one letter id followed by arguments if any.
 * 
 * Id-letters:
 * N - fidonet address information of message, e.g. N 2:504/8.0.
 * T - to whom message is to, e.g. T Teemu Torma.
 * F - from whom message is from, e.g. F Foo Bar.
 * S - subject of message, e.g S How are you?.
 * R - Indicates the uucp/internet reply address
 * O - Indicates the fidonet origin node
 * P - indicates that message is private. (Here is little conflict:
 * 	rfmail uses option -p to indicate that mail to public and here
 * 	P means that mail is private. Maybe this P should mean that
 *	mail is public and default is private).
 *
 * All letters having arguments have only on space after them, and all
 * rest of the line is considered to be that argument.
 *
 * Note that the read_msg routine does not split up the message
 * in header and body, but reads everything into the "body".
 */

/*
 * Read message header.
 * Leave file point pointing at first line of message body.
 */
boolean
read_msg_hdr(msg, fp)
	struct message *msg;
	FILE *fp;
{
	char buffer[BUFSIZ];
	char *cp;
	struct dest_info di;
	boolean gotN, gotO, gotF, gotT, gotR, gotS, gotD, gotP, gotX;

	gotN=gotO=gotF=gotT=gotR=gotS=gotD=gotP=gotX = FALSE;
	bzero((char *)msg, sizeof(struct message));
	while (fgets(buffer, BUFSIZ, fp)) {
		cp = strend(buffer);
		while (cp > buffer && isspace(cp[-1]))
			*--cp = 0;
		if (*buffer == 0)
			break;
		if (buffer[0] && buffer[1] && buffer[1] != ' ') {
			log("Illegal header line: %s", buffer);
			return FALSE;
		}
		switch (*buffer) {
		case 'N': 	/* dest node */
			if (gotN) goto multi;
			gotN = TRUE;
			if (!parsenode(buffer+2, &msg->dest, NULL, &config.mynode)) {
				log("Invalid destination: %s", buffer);
				return FALSE;
			}
			break;
			
		case 'O':	/* orig node */
			if (gotO) goto multi;
			gotO = TRUE;
			if (!parsenode(buffer+2, &msg->orig, NULL, &config.mynode)) {
				log("Invalid originator: %s", buffer);
				return FALSE;
			}
			break;
			
		case 'F':	/* From */
			if (gotF) goto multi;
			gotF = TRUE;
			strncopy(msg->from, buffer+2, sizeof(msg->from)-1);
			break;
			
		case 'T':	/* To */
			if (gotT) goto multi;
			gotT = TRUE;
			strncopy(msg->to, buffer+2, sizeof(msg->to)-1);
			break;
			
		case 'R':	/* Replyaddr */
			if (gotR) goto multi;
			gotR = TRUE;
			msg->replyaddr = strsave(buffer+2);
			break;
			
		case 'S':	/* Subject */
			if (gotS) goto multi;
			gotS = TRUE;
			strncopy(msg->subject, buffer+2, sizeof(msg->subject)-1);
			break;
			
		case 'D':	/* Date */
			if (gotD) goto multi;
			gotD = TRUE;
			strncopy(msg->ascdate, buffer+2, sizeof(msg->ascdate)-1);
			break;

		case 'P':	/* Obsolete */
			if (gotP) goto multi;
			gotP = TRUE;
			msg->attr |= ATTR_PRIVATE;
			break;
			
		case 'X':
			if (gotX) goto multi;
			gotX = TRUE;
			cp = buffer + 1;
			while (isspace(*cp))
				cp++;
			while (*cp) {
				if (cp[1] && !isspace(cp[1]))
					log("Illegal flag line: %s", buffer);
				switch (*cp) {
				case 'P':
					msg->attr |= ATTR_PRIVATE;
					gotP = TRUE;
					break;
				case 'C': msg->attr |= ATTR_CRASH; break;
				case 'F': msg->attr |= ATTR_FILE; break;
				case 'R': msg->attr |= ATTR_RRR; break;
				case 'I': msg->attr |= ATTR_IRR; break;
				case 'A': msg->attr |= ATTR_AUDREQ; break;
				}
				cp++;
				while (isspace(*cp))
					cp++;
			}
			break;
			
		default:
			log("Illegal message header line: %s", buffer);
			return FALSE;
		multi:
			log("Multiple definition of header: %s", buffer);
			return FALSE;
		}
	}
	if (!gotN) {
		log("Message with no destination");
		return FALSE;
	}
	if (!gotO) {
		di = get_destinfo(msg->dest);
		msg->orig = di.my_address;
	}
	if (!gotT) {
		log("Message with no receiver");
		return FALSE;
	}
	/* if from is missing, we'll improvise...*/
	if (!gotF) {
		log("Missing 'from' in message header");
		strcpy(msg->from, "Usenet");
	}
	/* if subject is missing, let's put something there */
	if (!gotS) {
		log("Missing subject in message header");
		strcpy(msg->subject, "Mail from Usenet");
	}
	/* if date is missing, use current date */
	if (!gotD) {
		log("Missing date in message header");
		strcpy(msg->ascdate, fido_date(time((long *) 0)));
	}
	return TRUE;
}

/* 
 * Check that net/node exists and mail can be send to there (ie. it
 * is not down or in hold). Also be will replace name with sysop's name,
 * if mail is for sysop (note that alias will override this sysop-name).
 */
boolean
valid_netnode(name, node)
	char *name;
	Node node;
{
	struct nodelist nlentry;
  
	if (search_node(node, &nlentry)) {
		if (nlentry.type & DOWN) {
			log("Node %s is down", ascnode(node));
			return FALSE;
		} else if (nlentry.type & HOLD)
			log("Node %s is on hold", ascnode(node));
		/* everything was fine */
		if (!stricmp(name, "sysop"))
			strcpy(name, nlentry.sysop);
		return TRUE;
	}
	
#if 0
	/*
	 * Node not found. Check if it is a point, and
	 * route this message to its boss if so.
	 */
	if (node.point != 0) {
		boss = node;
		boss.point = 0;
		if (search_node(boss, &nlentry)) {
			if (nlentry.type & DOWN) {
				log("Boss node %s of node %s is down",
				    ascnode(boss), ascnode(node));
				return FALSE;
			} else {
				log("Mail routed to boss node %s",
				    ascnode(boss));
				return TRUE;
			}
		}
	}
#endif
	log("Node %s does not exist", ascnode(node));
	return FALSE;
}


/*
 * Create packet header for a node
 */
void
create_packet_header(pkt, node)
	struct packet *pkt;
	Node node;
{
	struct tm *tm_s;
	time_t clocktime;
	struct dest_info dest;

	/* get info about destination, if any */

	dest = get_destinfo(node);	

#ifdef DEBUG
	switch (dest.pkttype)
	{
	case FTS0001: 
		strcpy(p, "stoneage"); 
		break; 
	case FSC0039: 
		strcpy(p, "FSC-0039"); 
		break;
	case FSC0045: 
		strcpy(p, "FSC-0045)"); 
		break;
	case FSC0048: 
		strcpy(p, "FSC-0048"); 
		break;		  
	}

	printf(" Node %d:%d/%d.%d gets %s packets with password %s from %d:%d/%d.%d\n",
		node.zone, node.net, node.node, node.point, p, dest.passwd, 
		dest.my_address.zone, dest.my_address.net, dest.my_address.node, dest.my_address.point);
#endif
	bzero((char *)pkt, sizeof(struct packet));
	clocktime = time(NULL);
	tm_s = localtime(&clocktime);

	pkt->orig = dest.my_address;
	pkt->dest = node;
	pkt->year = tm_s->tm_year + 1900;
	pkt->month = tm_s->tm_mon;
	pkt->day = tm_s->tm_mday;
	pkt->hour = tm_s->tm_hour;
	pkt->minute = tm_s->tm_min;
	pkt->second = tm_s->tm_sec;
	pkt->rate = config.maxbaud;
	pkt->ver = HDRVER;
	pkt->prodcode = PRODUCTCODE;
	pkt->prodmin = version_minor;
	pkt->prodmaj = version_major; 
	pkt->serialno = 0;
	strcpy(pkt->passwd, dest.passwd);
	pkt->pkttype = dest.pkttype;
	pkt->capword = CAP_WORD;
}

boolean
write_packet_header(pkt, packet)
	register struct packet *pkt;
	register FILE *packet;
	{
	switch (pkt->pkttype)
		{
	case FTS0001:
		return write_fts0001_header(pkt, packet);
		break;
	case FSC0039:
		return write_fsc0039_header(pkt, packet);
		break;
	case FSC0045:
		return write_fsc0045_header(pkt, packet);
		break;
	case FSC0048:
		return write_fsc0048_header(pkt, packet);
		break;
	default:
		return FALSE;
		}
	}

boolean
write_fts0001_header(pkt, packet)
	register struct packet *pkt;
	register FILE *packet;
{
	register int i;

	if (!write_int16(packet, pkt->orig.node)) 	goto error;
	if (!write_int16(packet, pkt->dest.node))  	goto error;
	if (!write_int16(packet, pkt->year))		goto error;
	if (!write_int16(packet, pkt->month))		goto error;
	if (!write_int16(packet, pkt->day))		goto error;
	if (!write_int16(packet, pkt->hour))		goto error;
	if (!write_int16(packet, pkt->minute))		goto error;
	if (!write_int16(packet, pkt->second))		goto error;
	if (!write_int16(packet, pkt->rate))		goto error;
	if (!write_int16(packet, pkt->ver))		goto error;
	if (!write_int16(packet, pkt->orig.net))	goto error;
	if (!write_int16(packet, pkt->dest.net))	goto error;
	if (!write_char(packet, pkt->prodcode))		goto error;
	if (!write_char(packet, pkt->serialno)) 	goto error;
	if (!write_char(packet, pkt->passwd[0]))	goto error;
	if (!write_char(packet, pkt->passwd[1]))	goto error;
	if (!write_char(packet, pkt->passwd[2]))	goto error;
	if (!write_char(packet, pkt->passwd[3]))	goto error;
	if (!write_char(packet, pkt->passwd[4]))	goto error;
	if (!write_char(packet, pkt->passwd[5]))	goto error;
	if (!write_char(packet, pkt->passwd[6]))	goto error;
	if (!write_char(packet, pkt->passwd[7]))	goto error;
	if (!write_int16(packet, pkt->orig.zone))  	goto error;
	if (!write_int16(packet, pkt->dest.zone))	goto error;
	for (i = 0; i < 20; i++)
		if (putc(0, packet) == EOF)		goto error;
	return TRUE;

 error:
	return FALSE;
}

boolean
write_fsc0039_header(pkt, packet)
	register struct packet *pkt;
	register FILE *packet;
{
	register int i;

	if (!write_int16(packet, pkt->orig.node)) 	goto error;
	if (!write_int16(packet, pkt->dest.node))  	goto error;
	if (!write_int16(packet, pkt->year))		goto error;
	if (!write_int16(packet, pkt->month))		goto error;
	if (!write_int16(packet, pkt->day))		goto error;
	if (!write_int16(packet, pkt->hour))		goto error;
	if (!write_int16(packet, pkt->minute))		goto error;
	if (!write_int16(packet, pkt->second))		goto error;
	if (!write_int16(packet, pkt->rate))		goto error;
	if (!write_int16(packet, pkt->ver))		goto error;
	if (!write_int16(packet, pkt->orig.net))	goto error;
	if (!write_int16(packet, pkt->dest.net))	goto error;
	if (!write_char(packet, pkt->prodcode))		goto error;
	if (!write_char(packet, pkt->prodmaj))		goto error;
	if (!write_char(packet, pkt->passwd[0]))	goto error;
	if (!write_char(packet, pkt->passwd[1]))	goto error;
	if (!write_char(packet, pkt->passwd[2]))	goto error;
	if (!write_char(packet, pkt->passwd[3]))	goto error;
	if (!write_char(packet, pkt->passwd[4]))	goto error;
	if (!write_char(packet, pkt->passwd[5]))	goto error;
	if (!write_char(packet, pkt->passwd[6]))	goto error;
	if (!write_char(packet, pkt->passwd[7]))	goto error;
	if (!write_int16(packet, pkt->orig.zone))  	goto error;
	if (!write_int16(packet, pkt->dest.zone))	goto error;
	/* 2 byte spare */
	if (!write_int16(packet, 0))			goto error;  
	/* byte swapped  capability word */
	if (!write_char(packet, 
			pkt->capword >> 8) == EOF)	goto error;
	if (putc(pkt->capword & 0xff, packet) == EOF)   goto error;
	if (!write_char(packet, pkt->prodcode >> 8))	goto error;
	if (!write_char(packet, pkt->prodmin))		goto error;
	if (!write_int16(packet, pkt->capword))		goto error;
	if (!write_int16(packet, pkt->orig.zone))	goto error;
	if (!write_int16(packet, pkt->dest.zone))	goto error;
	if (!write_int16(packet, pkt->orig.point))	goto error;
	if (!write_int16(packet, pkt->dest.point))	goto error;

	/* product specific data, not used */

	for (i = 0; i < 4; i++)
		if (putc(0, packet) == EOF)		goto error;
	return TRUE;

 error:

	return FALSE;
}

boolean
write_fsc0045_header(pkt, packet)
	register struct packet *pkt;
	register FILE *packet;
{
	register int i;

	if (!write_int16(packet, pkt->orig.node)) 	goto error;
	if (!write_int16(packet, pkt->dest.node))  	goto error;
	if (!write_int16(packet, pkt->orig.point))	goto error;
	if (!write_int16(packet, pkt->dest.point))	goto error;

	/* 8 bytes 0 */
	if (!write_int16(packet, 0))			goto error;
	if (!write_int16(packet, 0))			goto error;
	if (!write_int16(packet, 0))			goto error;
	if (!write_int16(packet, 0))			goto error;

	/* packet subversion = 2 */
	if (!write_int16(packet, 2))			goto error;
	if (!write_int16(packet, pkt->ver))		goto error;
	if (!write_int16(packet, pkt->orig.net))	goto error;
	if (!write_int16(packet, pkt->dest.net))	goto error;
	if (!write_char(packet, pkt->prodcode))		goto error;
	if (!write_char(packet, pkt->prodmin)) 		goto error;
	if (!write_char(packet, pkt->passwd[0]))	goto error;
	if (!write_char(packet, pkt->passwd[1]))	goto error;
	if (!write_char(packet, pkt->passwd[2]))	goto error;
	if (!write_char(packet, pkt->passwd[3]))	goto error;
	if (!write_char(packet, pkt->passwd[4]))	goto error;
	if (!write_char(packet, pkt->passwd[5]))	goto error;
	if (!write_char(packet, pkt->passwd[6]))	goto error;
	if (!write_char(packet, pkt->passwd[7]))	goto error;
	if (!write_int16(packet, pkt->orig.zone))  	goto error;
	if (!write_int16(packet, pkt->dest.zone))	goto error;
	if (!write_char(packet, pkt->orig_domain[0]))	goto error;
	if (!write_char(packet, pkt->orig_domain[1]))	goto error;
	if (!write_char(packet, pkt->orig_domain[2]))	goto error;
	if (!write_char(packet, pkt->orig_domain[3]))	goto error;
	if (!write_char(packet, pkt->orig_domain[4]))	goto error;
	if (!write_char(packet, pkt->orig_domain[5]))	goto error;
	if (!write_char(packet, pkt->orig_domain[6]))	goto error;
	if (!write_char(packet, pkt->orig_domain[7]))	goto error;
	if (!write_char(packet, pkt->dest_domain[0]))	goto error;
	if (!write_char(packet, pkt->dest_domain[1]))	goto error;
	if (!write_char(packet, pkt->dest_domain[2]))	goto error;
	if (!write_char(packet, pkt->dest_domain[3]))	goto error;
	if (!write_char(packet, pkt->dest_domain[4]))	goto error;
	if (!write_char(packet, pkt->dest_domain[5]))	goto error;
	if (!write_char(packet, pkt->dest_domain[6]))	goto error;
	if (!write_char(packet, pkt->dest_domain[7]))	goto error;

	/* 4 byte product specific data */

	for (i = 0; i < 4; i++)
		if (putc(0, packet) == EOF)		goto error;
	return TRUE;

 error:
	return FALSE;
}

boolean
write_fsc0048_header(pkt, packet)
	register struct packet *pkt;
	register FILE *packet;
{
	register int i;

	if (!write_int16(packet, pkt->orig.node)) 	goto error;
	if (!write_int16(packet, pkt->dest.node))  	goto error;
	if (!write_int16(packet, pkt->year))		goto error;
	if (!write_int16(packet, pkt->month))		goto error;
	if (!write_int16(packet, pkt->day))		goto error;
	if (!write_int16(packet, pkt->hour))		goto error;
	if (!write_int16(packet, pkt->minute))		goto error;
	if (!write_int16(packet, pkt->second))		goto error;
	if (!write_int16(packet, pkt->rate))		goto error;
	if (!write_int16(packet, pkt->ver))		goto error;

	if (pkt->orig.point == 0)
		if (!write_int16(packet, pkt->orig.net))
							goto error;
	else
		if (!write_int16(packet, -1 ))		goto error;
	if (!write_int16(packet, pkt->dest.net))	goto error;
	if (!write_char(packet, pkt->prodcode))		goto error;
	if (!write_char(packet, pkt->prodmaj))		goto error;
	if (!write_char(packet, pkt->passwd[0]))	goto error;
	if (!write_char(packet, pkt->passwd[1]))	goto error;
	if (!write_char(packet, pkt->passwd[2]))	goto error;
	if (!write_char(packet, pkt->passwd[3]))	goto error;
	if (!write_char(packet, pkt->passwd[4]))	goto error;
	if (!write_char(packet, pkt->passwd[5]))	goto error;
	if (!write_char(packet, pkt->passwd[6]))	goto error;
	if (!write_char(packet, pkt->passwd[7]))	goto error;
	if (!write_int16(packet, pkt->orig.zone))  	goto error;
	if (!write_int16(packet, pkt->dest.zone))	goto error;

	if (pkt->orig.point == 0)
		if (!write_int16(packet, -1))		goto error;
	else
		if (!write_int16(packet, pkt->dest.zone))
							goto error;

	/* byte swapped  capability word */
	if (!write_char(packet, 
			pkt->capword >> 8) == EOF)	goto error;
	if (putc(pkt->capword & 0xff, packet) == EOF)   goto error;
	if (!write_char(packet, pkt->prodcode >> 8))	goto error;
	if (!write_char(packet, pkt->prodmin))		goto error;
	if (!write_int16(packet, pkt->capword))		goto error;
	if (!write_int16(packet, pkt->orig.zone))	goto error;
	if (!write_int16(packet, pkt->dest.zone))	goto error;
	if (!write_int16(packet, pkt->orig.point))	goto error;
	if (!write_int16(packet, pkt->dest.point))	goto error;

	/* product specific data, not used */

	for (i = 0; i < 4; i++)
		if (putc(0, packet) == EOF)		goto error;

	return TRUE;

 error:
	return FALSE;
}


/* Return date in ascii string that is in fido-format, ie.
   dd mmm yy hh:mm:ss, like 24 Jan 87 06:24:27 */

static char *
fido_date(clocktime)
     time_t clocktime;
{
  struct tm *tm_s;
  static char ddate[20];
  
  /* Literal months */
  static char *months[] = {
    "Jan", "Feb", "Mar", "Apr", "May", "Jun",
    "Jul", "Aug", "Sep", "Oct", "Nov", "Dec",
  };
  
  tm_s = localtime(&clocktime);
  sprintf(ddate, "%02d %3s %02d %02d:%02d:%02d", tm_s->tm_mday,
                 months[tm_s->tm_mon], tm_s->tm_year, tm_s->tm_hour,
                 tm_s->tm_min, tm_s->tm_sec);
  return ddate;
}


void
add_seenby(msg, node)
	struct message *msg;
	Node node;
{
	if (msg->seenbys >= msg->maxseenbys)
		msg->seenby = (Node *)myrealloc((char *)msg->seenby, (msg->maxseenbys += 10)*sizeof(Node));
	msg->seenby[msg->seenbys++] = node;
}


