/*
 * $Header: /u1/src/rfmail/RCS/nodecomp.c,v 0.5.0.1 1992/06/15 06:11:25 pgd Exp pgd $
 *
 * $Log: nodecomp.c,v $
 * Revision 0.5.0.1  1992/06/15  06:11:25  pgd
 * Minor compilation bug fixes.
 * Change of all types with u_ prefix to U prefix
 * Change of name of routine msleep() to mssleep()
 *
 * Revision 0.5  1992/05/18  04:27:24  pgd
 * New distribution
 *
 *
 */

/*
 * Authors:
 *
 * This code is written by Per Lindqvist (pgd@compuram.bbt.se)
 * based on code written by Teemu Torma
 * and Heikki Suonsivu (hsu@hutcs.hut.fi)
 */

/* LINTLIBRARY */

#include "fnet.h"

#ifdef HAVE_VALUES_H
#include <values.h>
#endif

#include <sys/stat.h>
#ifdef XENIX
DECLARE(int, stat, (char *, struct stat *));
#endif

#include "nodelist.h"
#include "configs.h"
#include "shuffle.h"
#include "ndcomp-hash.h"
     
extern long atol();
Node originnode;
extern node_index_t *nodeindex;
name_index_t *name_index_work;
long nodeindexsize, nodeindexpos, nameindexsize, nameindexpos;

int nodes, names;
int compile_zone, compile_region, compile_net, compile_node, compile_type;

LDECLARE(boolean, parse_entry, (struct nodelist *, char *));
LDECLARE(boolean, getfield, (char *, char *, int));
DECLARE(void, printusage, (FILE *));
LDECLARE(char *, checklist, (char *));
LDECLARE(void, compile, (char *));
LDECLARE(boolean, readnodeindex, ());
LDECLARE(boolean, readnameindex, ());
LDECLARE(boolean, writenodeindex, ());
LDECLARE(boolean, writenameindex, ());

FILE *nlb;			/* Binary nodelist */
FILE *nlidx;			/* Nodelist index file */
FILE *nlnidx;			/* Nodelist system name index file */

char *nodelists[MAX_NODELISTS];	/* Pointer to the nodelists to use */
int nodelistcnt;		/* Number of nodelists */
boolean force = FALSE;		/* Flag set to force nodelist update */


/* Compare nodelist index entry (for qsort) */
static int
cmpnodeindex(node1, node2)
	node_index_t *node1, *node2;
{
	return cmpnode(&node1->node, &node2->node);
}


static int
cmpnameindex(name1, name2)
	name_index_t *name1, *name2;
{
	return stricmp(name1->name, name2->name);
}

int
main(argc, argv)
	int argc;
	char **argv;
{
	int i, j, c;
	struct stat nlsbuf, sbuf;
	char nodelisttmp[128], nodetemp[128], nametemp[128];
	static char *options = "fhv:O:X:";
	char *cp;
	int nlidxcount;

	get_configuration(&argc, argv, options);

	while ((c = getopt(argc, argv, options)) != EOF) {
		switch (c) {
		case 'f':	/* Force a nodelist update */
			force = TRUE;
			break;

		default:
			printusage(errorfp);
			exit(EX_USAGE);
		}
	}

	nodes = 0;
	names = 0;

	/*
	 * Make a table of all nodelists to compile
	 */
	nodelistcnt = 0;
	for (i = 0; i < config.nodelists; i++) {
		if (cp = checklist(config.nodelist[i])) {
			nodelists[nodelistcnt++] = strdup(cp);
			log("Adding nodelist %s", cp);
		}
	}

	/*
	 * Check if we really need to update the nodelist
	 */
	if (stat(config.nodelist_name, &nlsbuf) == -1) {
		if (errno != ENOENT) {
			fatal(EX_OSFILE, "$nodecomp: Cannot stat the nodelist (%s)",
			    config.nodelist_name);
		} else
			nlsbuf.st_mtime = 0;
	}
	for (i = 0; i < nodelistcnt; i++) {
		if (stat(nodelists[i], &sbuf) == -1) {
			log("$nodecomp: Cannot stat nodelist file %s",
			    nodelists[i]);
			continue;
		}
		if (sbuf.st_mtime > nlsbuf.st_mtime)
			goto doit;
	}
	if (force)
		log("nodecomp: Forcing a nodelist compilation.");
	else {
		log("nodecomp: Nodelist compilation not neccessary.");
		exit(0);
	}

 doit:
	/*
	 * Create the new nodelist binary file with a temporary name
	 * It will be renamed to its real name after 
	 * successful completion.
	 * The temporary file is created in the final directory
	 * for the nodelist.
	 */
	sprintf(nodelisttmp, "%s/TM.XXXXXX", basepath(config.nodelist_name));
	mktemp(nodelisttmp);
	nlb = myfopen(nodelisttmp, "w+");
	if (nlb == NULL)
		fatal(EX_OSFILE, "$Cannot create temporary file: %s", nodelisttmp);

	/*
	 * Write a dummy header to the nodelist file
	 */
	write_long(nlb, 0);	/* version */
	write_long(nlb, 0);	/* nodes */
	write_long(nlb, 0);	/* nodeindexpos */
	write_long(nlb, 0);	/* nodeindexsize */
	write_long(nlb, 0);	/* names */
	write_long(nlb, 0);	/* nameindexpos */
	write_long(nlb, 0);	/* nameindexsize */

	/*
	 * Create two temporary files for the node index
	 * and the system name index.
	 */
	strcpy(nodetemp, "/tmp/NOC.NO.XXXXXX");
	mktemp(nodetemp);
	nlidx = myfopen(nodetemp, "w+");
	if (nlidx == NULL) {
		log("$Unable to create temp-file for nodelist-index: %s",
		    nodetemp);
		goto fail;
	}

	strcpy(nametemp, "/tmp/NOC.NA.XXXXXX");
	mktemp(nametemp);
	nlnidx = myfopen(nametemp, "w+");
	if (nlnidx == NULL) {
		log("$Unable to create temp-file for name-index: %s",
		    nametemp);
		goto fail;
	}


	/*
	 * Compile the nodelists
	 */
	for (i = 0; i < nodelistcnt; i++)
		compile(nodelists[i]);

	/* Ok, now get both indices back and qsort them */
	
	if (!readnodeindex()) {
		log("$Cannot read temp. file for nodeindex: %s", nodetemp);
		goto fail;
	}

	if (!readnameindex()) {
		log("$Cannot read name temp. file for name index: %s",
		    nametemp);
		goto fail;
	}

#ifdef USG
	log("Sorting node index");
	qsort( (char *) nodeindex, (Uint) nodes,
	      sizeof(node_index_t), cmpnodeindex);

	log("Sorting name index");
	qsort( (char *) name_index_work, (Uint) names,
	      sizeof(name_index_t), cmpnameindex);
#endif

#ifdef BSD
	log("Sorting node index");
	qsort( (char *) nodeindex, nodes,
	      sizeof(node_index_t), cmpnodeindex);

	log("Sorting name index");
	qsort( (char *) name_index_work, names,
	      sizeof(name_index_t), cmpnameindex);
#endif

	nodeindexsize = ftell(nlidx);

	/*
	 * Check nodelist for duplicates.
	 * If yes, use the later entry.
	 */
	log("Dupecheck nodelist");
	j = 0;
	nlidxcount = nodeindexsize/sizeof(node_index_t);
	for (i = 1; i < nlidxcount; i++) {
		if (samenode(nodeindex[j].node, nodeindex[i].node)) {
			debug(1, "Duplicate node %s",
			      ascnode(nodeindex[i].node));
			if (nodeindex[i].offset > nodeindex[j].offset)
				nodeindex[j] = nodeindex[i];
			nodeindexsize -= sizeof(node_index_t);
		} else
			nodeindex[++j] = nodeindex[i];
	}
		
	log("Write indexes to nodelist");
  
	/*
	 * Append the indexes to the nodelist.
	 */
	if (!writenodeindex())
		goto fail;
	if (!writenameindex())
		goto fail;

	rewind(nlb);
	write_long(nlb, NODELIST_ID);	/* Identification */
	write_long(nlb, (long)nodes);
	write_long(nlb, nodeindexpos);
	write_long(nlb, nodeindexsize);
	write_long(nlb, (long)names);
	write_long(nlb, nameindexpos);
	write_long(nlb, nameindexsize);
  
	debug(1,"nodeindex, pos=%d, size=%d\n", nodeindexpos, nodeindexsize);
	debug(1,"nameindex, pos=%d, size=%d\n", nameindexpos, nameindexsize);

	/*
	 * Done, close files. Delete temporary files,
	 * and rename the nodelist to its real name
	 */

	fclose(nlb);
	fclose(nlidx);
	fclose(nlnidx);
	if (unlink(nodetemp) == -1)
		log("$Cannot unlink file: %s", nodetemp);
	if (unlink(nametemp) == -1)
		log("$Cannot unlink file: %s", nametemp);
	if (myrename(nodelisttmp, config.nodelist_name) == -1) {
		if (errno != EEXIST) {
			log("$Error renaming %s to %s",
			    nodelisttmp, config.nodelist_name);
			goto fail;
		}
		if (errno == EEXIST && unlink(config.nodelist_name) == -1) {
			log("$Cannot delete: %s", config.nodelist_name);
			goto fail;
		}
		if (myrename(nodelisttmp, config.nodelist_name) == -1) {
			log("$Cannot rename %s to %s",
			    nodelisttmp, config.nodelist_name);
			goto fail;
		}
	}
	exit(0);

fail:
	unlink(nodetemp);
	unlink(nametemp);
	unlink(nodelisttmp);
	exit(-1);
}


/*
 * Given a nodelist specification, return the
 * name of the newest nodelist
 * or NULL if none.
 */
static char *
checklist(nlspec)
	char *nlspec;
{
	char *fname;
	struct stat sbuf;
	register char *cp;
	char lastname[128];
	int lastweek = 0;
	time_t lastdate = 0;
	int n, i, wskip;

	/*
	 * Look for a trailing '*' signalling we have
	 * to fill in the week number.
	 */
	cp = strend(nlspec);
	if (cp == nlspec || cp[-1] != '*') {
		/*
		 * A normal non-wild nodelist spec.
		 * Check if file is there.
		 */
		if (access(basepath(nlspec), 4) == 0)
			return nlspec;
		else {
			log("No file matching nodelist spec. %s\n", nlspec);
			return NULL;
		}
	}

	/*
	 * Number of characters in basename up to the '*'
	 */
	cp = strend(nlspec);
	if (cp[-1] != '*') {
		buglog("nodecomp: findlatest called with a non-wild spec '%s'",
		       nlspec);
		return NULL;
	}
	cp--;
	wskip = cp - basename(nlspec);

	/*
	 * Weeknumber has to be appended to nodlist name.
	 *
	 * Lookup all files matching this nodelist specification
	 */
	*lastname = 0;
	if (myopendir(basepath(nlspec), FALSE)) {
		while (fname = myreaddir(nlspec)) {
			/*
			 * Decode week-number
			 */
			cp = basename(fname) + wskip;
			if (!isdigit(*cp))
				continue;
			n = 0;
			for (i = 0; isdigit(cp[i]); i++)
				n = n * 10 + cp[i] - '0';
			if (cp[i]) {
				log("Cannot parse weeknumber from filename: '%s'", basename(fname));
				continue;
			}

			if (stat(fname, &sbuf) == -1) {
				log("$Cannot stat file: '%s'", fname);
				continue;
			}

			/*
			 * We now have week number and creation date
			 * Is this nodelist file newer than the previously
			 * selected one?
			 * We have to watch out for the yearly wrap-around.
			 * This algorithm is not completely fool-proof.
			 */
			if (lastdate == 0
			    || n > lastweek
			    || (n < lastweek
				&& lastweek > 300
				&& n < 60
				&& sbuf.st_mtime > lastdate-60*60*24)) {
				lastweek = n;
				lastdate = sbuf.st_mtime;
				strcpy(lastname, fname);
				continue;
			}
		}
	} else
		return NULL;
	myclosedir();
	return *lastname ? lastname : NULL;
}

static void
compile(nlname)
	char *nlname;
{
	char buffer[BUFSIZ];
	FILE *nl;
	struct nodelist node;
	node_index_t nodei;
	name_index_t namei;
	long offset;
	register int i;
	
	/*
	 * At this point we have selected the (hopefully) latest
	 * nodelist to load.
	 */
	log("Compiling nodelist %s", nlname);
	if ((nl = fopen(nlname, "r")) == NULL) {
		log("$Unable to open nodelist file %s", nlname);
		return;
	}

	/*
	 * Nodelist defaults are set to my zone, region and net.
	 * (Is this really the right thing to do?)
	 */
	compile_zone = config.mynode.zone;
	compile_region = config.myregion;
	compile_net = config.mynode.net;
	compile_node = config.mynode.node;
	compile_type = NODE;
	
  
	/* save host/region offsets */
	while (fgets(buffer, BUFSIZ, nl)) {
		if (*buffer == '\n' || *buffer == ';')
			continue;
		
		if (!parse_entry(&node, buffer)) {
			log("\tEntry: %s", buffer);
			continue;
		}

		if (node.type == 0)
		       continue;
		/*
		 * Determine if we should write this entry
		 * or not.
		 */
		for (i = 0; i < config.exclude_zones; i++)
			if (config.exclude_zone[i] == node.node.zone)
				goto dontwrite;
		/*
		 * We default to writing all zones
		 */
		if (config.include_zones == 0)
			goto goodzone;
		for (i = 0; i < config.include_zones; i++)
			if (config.include_zone[i] == node.node.zone)
				goto goodzone;

	dontwrite:
		/*
		 * If we come here, don't write out this entry.
		 */
		continue;

	goodzone:
		/*
		 * Zone should be written. Check the net.
		 */
		for (i = 0; i < config.exclude_nets; i++)
			if (config.exclude_net[i].zone == node.node.zone
			    && config.exclude_net[i].net == node.node.net)
				goto dontwrite;
		if (config.include_nets == 0)
			goto dowrite;
		for (i = 0; i < config.include_nets; i++)
			if (config.include_net[i].zone == node.node.zone
			    && config.include_net[i].net == node.node.net)
				goto dowrite;
		goto dontwrite;

	dowrite:
		offset = ftell(nlb);
		nodei.node = node.node;
		nodei.offset = offset;
		
		if (!write_char(nlb, node.type)        ||
		    !write_int16(nlb, node.node.zone)  ||
		    !write_int16(nlb, node.node.net)   ||
		    !write_int16(nlb, node.node.node)  ||
		    !write_int16(nlb, node.node.point) ||
		    !write_int16(nlb, node.region)     ||
		    !write_int16(nlb, node.speed)      ||
		    !write_int16(nlb, node.flags)      ||
		    !write_long(nlb, node.modemcap)    ||
		    !write_char(nlb, node.mailhour)    ||
		    !write_string(nlb, node.name)      ||
		    !write_string(nlb, node.city)      ||
		    !write_string(nlb, node.sysop)     ||
		    !write_string(nlb, node.phone)     ||
		    !write_string(nlb, node.password)) {
			log("$Cannot write nodelist: %s",
			    config.nodelist_name);
			return;
		}

		/*
		 * Write the nodelist index temporary file
		 */
		if ((node.type & NLENTRYMASK) != REGION
		    && (node.type & NLENTRYMASK) != KENNEL) {
			if (fwrite((char *)&nodei, 1, sizeof(nodei), nlidx)
			    != sizeof(nodei)) {
				log("$Cannot write nodelist index temporary file");
				return;
			}
		}
		

		/*
		 * Write the nodelist name index temporary file
		 */
		if ((node.type & NLENTRYMASK) != REGION
		    && (node.type & NLENTRYMASK) != HOST
		    && (node.type & NLENTRYMASK) != ZONE
		    && (node.type & NLENTRYMASK) != KENNEL) {
			register char *cp;
			
			cp = node.name;
			for (i = 0; i < sizeof(namei.name)-1; i++)
				namei.name[i] = *cp ? *cp++ : 0;
			namei.name[sizeof(namei.name)-1] = 0;
			namei.offset = offset;
			names++;
			if (fwrite((char *)&namei, 1, sizeof(namei), nlnidx)
			    != sizeof(namei)) {
				log("$Cannot write name index temporary file");
				return;
			}
		}
		nodes++;
	}
	fclose(nl);
}


/*
 * Read the temporary node index file
 */
static boolean
readnodeindex()
{
	long size;

	myfseek(nlidx, 0L, SEEK_END);
	size = ftell(nlidx);
	rewind(nlidx);
	nodeindex = (node_index_t *) mymalloc(size);
	nodes = size / sizeof(node_index_t);
	if (fread((char *)nodeindex, 1, size, nlidx) != size) {
		log("$Read error");
		return FALSE;
	}
	return TRUE;
}

/*
 * Read the temporary name index file
 */
static boolean
readnameindex()
{
	long size;

	myfseek(nlnidx, 0L, SEEK_END);
	size = ftell(nlnidx);
	rewind(nlnidx);
	name_index_work = (name_index_t *) mymalloc(size);
	names = size / sizeof(name_index_t);
	if (fread((char *)name_index_work, 1, size, nlnidx) != size) {
		log("$Read error");
		return FALSE;
	}
	return TRUE;
}

/*
 * Write out the name index in compressed format.
 */
static boolean
writenameindex()
{
	register int i;
	register name_index_t *np;

	nameindexpos = ftell(nlb);
	for (i = 0; i < names; i++) {
		np = &name_index_work[i];
		if (!write_long(nlb, np->offset)
		    || !write_string(nlb, np->name)) {
			log("$Write error writing name index");
			return FALSE;
		}
	}
	nameindexsize = ftell(nlb) - nameindexpos;
	return TRUE;
}

static boolean
writenodeindex()
{
	nodeindexpos = ftell(nlb);
	if (fwrite((char *)nodeindex, 1, nodeindexsize, nlb) != nodeindexsize) {
		log("$Write error writing node index");
		return FALSE;
	}
	return TRUE;
}

/* 
 * Get one comma-terminated field from buffer,
 * and place it into entry.
 * Returns true if field is ok, false otherwise
 */
static boolean
getfield(buffer, entry, size)
	char *buffer, *entry;
	int size;
{
	register char *cp;
	static char *bp;
	int cc;

	if (buffer)
		bp = buffer;

	while (isspace(*bp))
		bp++;
	cp = entry;
	cc = size;
	if (*bp == 0)
		return FALSE;
	while (*bp && *bp != ',') {
		if (cc > 0) {
			*cp++ = *bp;
			cc--;
		}
		bp++;
	}
	while (cp > entry && isspace(cp[-1]))
		--cp;
	*cp = 0;
	if (*bp == ',')
		bp++;
	return TRUE;
}

/*
 * Parse one line of the nodelist.
 * Return TRUE if line parsed all right, FALSE if
 * there was an error, or missing fields, in that line.
 * This routine also parses lines from point file.
 * Returns data into a "nodelist" structure.
 */
static boolean
parse_entry(entry, buffer)
	struct nodelist *entry;
	char *buffer;
{
	register char *cp;
	register int n;
	char flag[80];
	int type, hash;
	char field1[40], field2[40];
	Node node;
  
	/* strip newline if exists */
	cp = strend(buffer);
	while (cp > buffer && isspace(cp[-1]))
		*--cp = 0;

	if (!getfield(buffer, field1, sizeof(field1))
	    || !getfield(NULL, field2, sizeof(field2))) {
		log("Illegal type/node in nodelist");
		return FALSE;
	}
	    
  
	/* get type of entry */
	if (*field1 == 0)	/* First string is a null string */
		type = (compile_type==BOSS?POINT:NODE);
	else if (strequ("Zone", field1)) {
		type = ZONE; compile_type = NODE;
	} else if (strequ("Region", field1)) {
		type = REGION; compile_type = NODE;
	} else if (strequ("Host", field1)) {
		type = HOST; compile_type = NODE;
	} else if (strequ("Hub", field1)) {
		type = HUB; compile_type = NODE;
	} else if (strequ("Pvt", field1))
		type = (compile_type == BOSS?POINT:NODE)|PRIVATE;
	else if (strequ("Hold", field1))
		type = (compile_type == BOSS?POINT:NODE)|HOLD;
	else if (strequ("Down", field1))
		type = (compile_type == BOSS?POINT:NODE)|DOWN;
	else if (strequ("Point", field1)) {
		type = POINT; compile_type = NODE;
	} else if (strequ("Boss", field1)) {
		/*
		 * Boss line. This line has a different syntax, and
		 * only sets the compile_fields
		 * It is used to set the boss node for following
		 * point definitions.
		 */
		cp = parsenode(field2, &node, " ", NULL);
		if (cp && *cp) {
			log("Illegal Boss line");
			return FALSE;
		}
		compile_zone = node.zone == NOZONE
			? config.mynode.zone : node.zone;
		compile_net = node.net == NONET
			? config.mynode.net : node.net;
		compile_node = node.node == NONODE
			? config.mynode.node : node.node;
		compile_region = 0;	/* What to put here? */
		compile_type = BOSS;
		entry->type = 0;
		return TRUE;
	} else {
		log("Unknown nodelist type");
		return FALSE;
	}
  
	if (field2[0] == 0) {
		log("Missing zone/net/node/region");
		return FALSE;
	}
	n = 0;
	for (cp = field2; *cp; cp++) {
		if (!isdigit(*cp))
			break;
		n = n * 10 + *cp - '0';
	}
	if (*cp) {
		cp = parsenode(field2, &node, NULL, NULL);
		if (cp && *cp) {
			log("Illegal node-number in nodelist");
			return FALSE;
		}
		if (node.zone != NOZONE) {
			entry->node.zone = compile_zone = node.zone;
			entry->node.net = compile_net = node.net;
			entry->node.node = compile_node = node.node;
			entry->node.point = node.point;
		} if (node.net != NONET) {
			entry->node.zone = compile_zone;
			entry->node.net = compile_net = node.net;
			entry->node.node = compile_node = node.node;
			entry->node.point = node.point;
		} else if (node.node != NONODE) {
			entry->node.zone = compile_zone;
			entry->node.net = compile_net;
			entry->node.node = compile_node = node.node;
			entry->node.point = node.point;
		} else if (node.point != NOPOINT) {
			entry->node.zone = compile_zone;
			entry->node.net = compile_net;
			entry->node.node = compile_node;
			entry->node.point = node.point;
		}
		if (entry->node.point == NOPOINT)
			entry->node.point = 0;
		if (entry->node.zone == NOZONE
		    || entry->node.net == NONET
		    || entry->node.node == NONODE) {
			log("Illegal node %s", ascnode(entry->node));
			return FALSE;
		}
		/*
		 * Region is not properly set in this case.
		 * But we currently do not use the region information.
		 */
		entry->region = compile_region = 0;
		debug(8, "Node %s", ascnode(entry->node));
	} else {
		if (n == 0) {
			log("Value of zone/net/node/region is 0");
			return FALSE;
		}
		entry->type = type;
		switch (type & NLENTRYMASK) {
		case ZONE:
			entry->node.zone = compile_zone = n;
			entry->region = compile_region = 0;
			entry->node.net = compile_net = 0;
			entry->node.node = compile_node = 0;
			entry->node.point = 0;
			debug(8, "Zone %d", compile_zone);
			break;
			
		case REGION:
			entry->node.zone = compile_zone;
			entry->region = compile_region = n;
			/*
			 * Is this right? What should the net
			 * actually be after a region?
			 */
			entry->node.net = compile_net = n;
			entry->node.node = compile_node = 0;
			entry->node.point = 0;
			debug(8, "Region %d", compile_region);
			break;
			
		case HOST:
			entry->node.zone = compile_zone;
			entry->region = compile_region;
			entry->node.net = compile_net = n;
			entry->node.node = compile_node = 0;
			entry->node.point = 0;
			debug(8, "Net %d:%d", compile_zone, compile_net);
			break;
			
		case POINT:
			entry->node.zone = compile_zone;
			entry->region = compile_region;
			entry->node.net = compile_net;
			entry->node.node = compile_node;
			entry->node.point = n;
			break;
			
		case HUB:
		case NODE:
			entry->node.zone = compile_zone;
			entry->region = compile_region;
			entry->node.net = compile_net;
			if (compile_type != BOSS) {
				entry->node.node = compile_node = n;
				entry->node.point = 0;
			} else {
				entry->node.node = compile_node;
				entry->node.point = n;
			}
			break;
		}
	}

	/* get node/net/region name */
	if (!getfield(NULL, entry->name, sizeof(entry->name))) {
		log("Missing name");
		return FALSE;
	}
  
	/* get city */
	if (!getfield(NULL, entry->city, sizeof(entry->city))) {
		log("Missing city");
		return FALSE;
	}
  
	/* get sysop */
	if (!getfield(NULL, entry->sysop, sizeof(entry->sysop))) {
		log("Missing sysop name");
		return FALSE;
	}
  
	/* get phone number */
	if (!getfield(NULL, entry->phone, sizeof(entry->phone))) {
		log("Missing telephone-number");
		return FALSE;
	}
	if (strequ(entry->phone, "-Unpublished-"))
		*entry->phone = 0;
  
	/* get maximum baud rate */
	if (!getfield(NULL, flag, sizeof(flag))) {
		log("Missing baud-rate");
		return FALSE;
	}
	n = 0;
	for (cp = flag; isdigit(*cp); cp++)
		n = n * 10 + *cp - '0';
		;
	if (*cp) {
		log("Illegal digit in Baud-rate");
		return FALSE;
	}
	entry->speed = n;

	/*
	 * Decode flags
	 */
	entry->flags = 0;
	entry->mailhour = 0;
	entry->modemcap = 0;
	*entry->password = 0;
	while (getfield(NULL, flag, sizeof(flag))) {
		if (*flag == 0)
			continue;
		strlwr(flag);
		if (flag[0] == '#' || flag[0] == '!') {
			cp = flag;
			while (*cp == '#' || *cp == '!') {
				n = 0;
				cp++;
				while (isdigit(*cp))
					n = n * 10 + *cp++ - '0';
				if (*cp && *cp != '#' && *cp != '!')
					log("Illegal capabilities flag: %s",
					    flag);
				goto next;
				switch (n) {
				case 1: entry->mailhour |= MH01; break;
				case 2: entry->mailhour |= MH02; break;
				case 8: entry->mailhour |= MH08; break;
				case 9: entry->mailhour |= MH09; break;
				case 18: entry->mailhour |= MH18; break;
				case 20: entry->mailhour |= MH20; break;
				default:
					log("Unknown mail-hour flag '%s' node %s",
					    flag, ascnode(entry->node));
					goto next;
				}
			}
		} else if (flag[0] == 'u')
			continue; /* User defined */
		else {
			hash = nodecomp_hash_function(flag);
			switch (hash) {
			case NODEFLAG_CM: entry->flags |= CM; break;
			case NODEFLAG_MO: entry->flags |= MO; break;
			case NODEFLAG_LO: entry->flags |= LO; break;
			case NODEFLAG_MN: entry->flags |= MN; break;
			case NODEFLAG_XA: entry->flags |= XA; break;
			case NODEFLAG_XB: entry->flags |= XB; break;
			case NODEFLAG_XC: entry->flags |= XC; break;
			case NODEFLAG_XP: entry->flags |= XP; break;
			case NODEFLAG_XR: entry->flags |= XR; break;
			case NODEFLAG_XW: entry->flags |= XW; break;
			case NODEFLAG_XX: entry->flags |= XX; break;
			case NODEFLAG_GUUCP: entry->flags |= GUUCP; break;
			case NODEFLAG_NEC: entry->flags |= NEC; break;
			case NODEFLAG_REC: entry->flags |= REC; break;
			case NODEFLAG_V21:  entry->modemcap |= V21; break;
			case NODEFLAG_V22:  entry->modemcap |= V22; break;
			case NODEFLAG_V22B: entry->modemcap |= V22b; break;
			case NODEFLAG_V23:  entry->modemcap |= V23; break;
			case NODEFLAG_V29:  entry->modemcap |= V29; break;
			case NODEFLAG_V32:  entry->modemcap |= V32; break;
			case NODEFLAG_V32BIS:
			case NODEFLAG_V32B: entry->modemcap |= V32b; break;
			case NODEFLAG_V33:  entry->modemcap |= V33; break;
			case NODEFLAG_V34:  entry->modemcap |= V34; break;
			case NODEFLAG_V42:  entry->modemcap |= V42; break;
			case NODEFLAG_V42BIS: 
			case NODEFLAG_V42B: entry->modemcap |= V42b; break;
			case NODEFLAG_MNP:
			case NODEFLAG_NMP:
			case NODEFLAG_MNP5: entry->modemcap |= MNP; break;
			case NODEFLAG_H96:  entry->modemcap |= H96; break;
			case NODEFLAG_HST:  entry->modemcap |= HST; break;
			case NODEFLAG_MAX:  entry->modemcap |= MAX; break;
			case NODEFLAG_PEP:  entry->modemcap |= PEP; break;
			case NODEFLAG_CSP:  entry->modemcap |= CSP; break;
			case NODEFLAG_ISDNA: entry->modemcap |= ISDNA; break;
			case NODEFLAG_ISDNB: entry->modemcap |= ISDNB; break;
			case NODEFLAG_ISDNC: entry->modemcap |= ISDNC; break;
				/*
				 * Unknown nodelist flags
				 * these are ingnored.
				 */
			case NODEFLAG_DDS:
			case NODEFLAG_DVN:
			case NODEFLAG_MDN:
			case NODEFLAG_PDN:
			case NODEFLAG_SDN:
			case NODEFLAG_SDS:
			case NODEFLAG_WIN:
			case NODEFLAG_H14:
			case NODEFLAG_H16:
			case NODEFLAG_H55:
			case NODEFLAG_ZYX:
				break;
			case NODEFLAG_DS: /* Dual speed */
				entry->modemcap |= V32|HST; break;

			default:
				log("Unknown capabilities flag '%s' node %s",
				    flag, ascnode(entry->node));
			}
		}
	next: ;
	}

	/*
	 * Setup implicit modemcaps.
	 */
	if (entry->modemcap & V42b)
		entry->modemcap |= V42;
	if (entry->modemcap & HST)
		entry->modemcap |= MNP;
	if (entry->modemcap & V32b)
		entry->modemcap |= V32;
	
	/*
	 * Lower speeds are not in the nodelist.
	 * Guess them.
	 */
	if (entry->speed >= 2400)
		entry->modemcap |= V22b;
	if (entry->speed >= 1200 && (entry->modemcap & V22) == 0)
		if (entry->node.zone == 2)
			entry->modemcap |= V22;
		else
			entry->modemcap |= BELL212;
	if (entry->speed >= 300 && (entry->modemcap & V21) == 0)
		if (entry->node.zone == 2)
			entry->modemcap |= V21;
		else
			entry->modemcap |= BELL300;
	return TRUE;
}


void
printusage(fp)
	FILE *fp;
{
	fprintf(fp, "nodecomp [options]\n");
	fprintf(fp, "\tCompile rfmail nodelist.\n");
	fprintf(fp, "Where options are any of:\n");
	fprintf(fp, "-f          Force a nodelist compilation.\n");
	configprintusage(fp);
}


