/*
 * locks.c: Lock file maintenance
 *
 *
 * $Header: /u1/src/rfmail/RCS/locks.c,v 0.5 1992/05/18 04:27:24 pgd Exp pgd $
 *
 * $Log: locks.c,v $
 * Revision 0.5  1992/05/18  04:27:24  pgd
 * New distribution
 *
 * Revision 0.4.1.6  1992/03/15  07:58:52  pgd
 * Untested version
 *
 * Revision 1.1  1991/07/03  02:17:18  mboucher
 * Initial revision
 *
 * Revision 0.4.1.1  1991/05/21  11:13:48  pgd
 * *** empty log message ***
 *
 * Revision 0.4  1991/05/08  04:23:43  pgd
 * Initial Beta-release
 *
 *
 */

/*
 *
 * Authors:
 *
 * Teemu Torma who wrote the original code (?)
 *
 * Heikki Suonsivu (hsu@hutcs.hut.fi) who made a lot of enhancements
 * 
 * Per Lindqvist (pgd@compuram.bbt.se) who continued to enhance rfmail.
 *
 * locking routines rewritten by Marc Boucher <marc@CAM.ORG> for SVR4
 */

#include "fnet.h"

#include <sys/sysmacros.h>

#ifdef FCNTL
#include <fcntl.h>
#endif

#include <sys/stat.h>

#include "nodelist.h"
#include "configs.h"

#ifdef SVR4
#include <sys/mkdev.h>
#endif

#ifdef XENIX
DECLARE(int, stat, (char *, struct stat *));
DECLARE(int, fstat, (int, struct stat *));
#endif

/*
 *	readlock() - read contents of lockfile
 *
 *	Returns pid read, -1 on error or 0 if unknown
 */
static int
readlock(name, size)
	int size;
	char *name;
{
	int fd, pid;
	char apid[16];
	long pid32;	/* This must be at least 4 bytes */
	short pid16;	/* This must be at least 2 bytes */

	if ((fd = open(name, O_RDONLY)) == -1) {
		log("readlock: can't open %s", name);
		return(-1);
	}

	switch(size) {
	case 2:
		if (sizeof(pid16) != 2) {
			log("Unportable code detected, sizeof(pid16) %d != 2",
			    sizeof(pid16));
			close(fd);
			return(-1);
		}
		if (read(fd, (char *) &pid16, 2) != 2) {
			log("$Reading two-byte lock file failed");
			close(fd);
			return(-1);
		}
		close(fd);
		return((int)pid16);

	case 4:
		if (sizeof(pid32) != 4) {
			log("Unportable code detected, sizeof(pid32) %d != 4", sizeof(pid32));
			close(fd);
			return(-1);
		}
		if (read(fd, (char *) &pid32, 4) != 4) {
			log("$Reading four-byte lock file failed");
			close(fd);
			return(-1);
		}
		close(fd);
		return((int)pid32);

	case 11:
		if(read(fd, apid, sizeof(apid)) != 11) {
			log("$Reading eleven-byte lock file failed");
			close(fd);
			return(-1);
		}
		sscanf(apid, "%d", &pid);
		close(fd);
		return(pid);
		break;
	}
	log("Unknown lock file size");
	close(fd);
	return(0);
}



/*
 * Check for a lock.
 * Parameter is the full path of the lock file.
 * Returns:
 *	TRUE if the lock is in use
 *	FALSE if there is no lock of that name
 */
boolean
checklock(name)
	char *name;
{
	int pid, livetime;
	struct stat st;

	if (stat(name, &st) == -1)
		if (errno!=ENOENT) {
			fatal(0, "checklock: error stat()ing %s, errno=%d",
			      name, errno);
			return TRUE;
		} else
			return FALSE;

	switch (pid = readlock(name, st.st_size)) {
	case -1:
		break;

	case 0:
		if(24*3600 > (livetime = time(NULL) - st.st_mtime)) {
			log("Warning; killing lock file %s, age %ld", name, livetime);
			if (unlink(name) < 0) {
				log("$Cannot remove stale lock file %s", name);
				return TRUE;
			}
			return FALSE;
		}
		break;

	default:
		if ((kill(pid, 0) == -1) && errno == ESRCH) {
			log("Warning; killing lock file %s, nonexistant pid %d", name, pid);
			check_unlink(name);
			return FALSE;
		}
	}
	return TRUE;
}

char *
lockname(thing)
	char *thing;
{
	static char name[128];
	boolean notlocked;

	if (*thing != '/') {
		notlocked = TRUE;
		if (hasconfig && strequ(config.locks, "svr4")) {
			struct stat st;

			sprintf(name, "/dev/%s", thing);
			if(stat(name, &st) != -1)
				notlocked = TRUE;
			else {
				sprintf(name, "%s/LK.%03d.%03d.%03d",
					config.lockdir, major(st.st_dev),
					major(st.st_rdev), minor(st.st_rdev));
			}
			notlocked = FALSE;
		}
		if (notlocked)
			sprintf(name, "%s/LCK..%s", config.lockdir, thing);
	} else
		strcpy(name, thing);
	return(name);
}

/*
 * Create a lock for a thing.
 * Returns TRUE if lock successfully created.
 * returns FALSE if we could not lock.
 */
boolean
createlock(s, timeout)
	char *s;	/* Thing to be locked */
	time_t timeout;
{
	int fd, process_id = getpid();
	char tmplock[128], *name;
	char apid[16];
	boolean locked;

	if ((name = lockname(s)) == NULL)
		return FALSE;		/* fail */

	if (timeout) {
		long s_time;

		SetStart();
		locked = TRUE;
		while ((!Timeout(timeout)) && (locked = checklock(name)))
			sleep(config.lock_sleep);
	} else
		locked = checklock(name);
	if (locked)
		return FALSE;		/* locked by another process */

	/*
	 * Thing is not locked.
	 * Create lock for us
	 *
	 * first make the temp file
	 */
	sprintf(tmplock, "%s/TM.rfm.XXXXXX", *name=='/' ? basepath(name)
						        : config.lockdir);
	mktemp(tmplock);
	if(!tmplock[0]) {
		log("createlock: mktemp() failed");
		return FALSE;		/* fail */
	}

	fd = creat(tmplock, 0444);
	if (fd == -1 && errno == ENOENT && makepath(tmplock))
		fd = creat(tmplock, 0444);
	if (fd == -1) {
		log("$createlock: cannot create %s", tmplock);
		return FALSE;		/* fail */
	}

	/* put my pid in it
	 */
	if (!hasconfig
	    || strequ(config.locks, "hdb") || strequ(config.locks, "svr4")) {
		sprintf(apid, "%d\n", process_id);
		write(fd, "            ", 11 - strlen(apid));
		write(fd, apid, strlen(apid));
	} else if (strequ(config.locks, "bsd"))
		write(fd, (char *) &process_id, sizeof(process_id));
	else if (!strequ(config.locks, "empty")) {
		log("Unknown lock type %s", config.locks);
		check_unlink(tmplock);
		return FALSE;		/* fail */
	}

	close(fd);

	/* link it to the lock file
	 */
	if (rename(tmplock, name) == -1) {
		log("$locks (createlock): rename(%s,%s) failed",
		    tmplock, name);
		return FALSE;
	}
	return TRUE;
}


void
removelock(s)
	char *s;	/* thing to remove the lock from */
{
	char *name;

	if(!(name=lockname(s))) {
		log("removelock: can't find name for %s", s);
		return;
	}
	check_unlink(name);
}
