/*
 * $Header: /u1/src/rfmail/RCS/freq.c,v 0.5 1992/05/18 04:27:24 pgd Exp pgd $
 *
 * $Log: freq.c,v $
 * Revision 0.5  1992/05/18  04:27:24  pgd
 * New distribution
 *
 * Revision 0.4.1.6  1992/03/15  07:58:52  pgd
 * Untested version
 *
 * Revision 0.4.1.3  1991/06/15  09:33:39  pgd
 * *** empty log message ***
 *
 * Revision 0.4.1.2  1991/06/05  09:13:58  pgd
 * Various Bugfixes:
 *
 *
 * If file-request-max-files is set to zero, there should be no limitation
 * on the number of files that can be requested.
 *
 * Revision 0.4  1991/05/08  04:23:43  pgd
 * Initial Beta-release
 *
 *
 */

/*
 * File-request routines
 *
 * Author: Per Lindqvist <pgd@compuram.bbt.se>
 */

#include "fnet.h"

#include <sys/stat.h>
#include <ctype.h>

#include "fcall.h"
#include "nodelist.h"
#include "configs.h"
#include "xmodem.h"
#include "zmodem.h"
#include "directory.h"
#include "packet.h"

#ifdef XENIX
DECLARE(int, stat, (char *, struct stat *));
#endif


extern int remote_capabilities;	/* YOOHOO capabilities bits */
extern Node remote_node;	/* Node communicating with */
extern int opacketseq;		/* Sequence number for output packets */
extern struct nodelist remote_nodeentry;

int file_request_files;		/* Number of files transferred */
int file_request_size;		/* Number of kbytes transferred */

LDECLARE(boolean, parsereq, (FILE *, char *, char *, char *));
LDECLARE(int, sendrequest, (char *, boolean));
LDECLARE(boolean, lookup_alias, (FILE *, char *, char *));
LDECLARE(int, lookup_list, (FILE *, char *, char *));

/* 
 * Check if we have a file request file for this node.
 * Send the file request file if yes.
 */
boolean
sendreq(zedzap)
	boolean zedzap;
{
	char reqpath[PATH_LEN];
	char alias[20];
	int s;

	/*
	 * Check if the remote node accepts file requests.
	 */
	if ((remote_capabilities & WZ_FREQ) == 0)
		return FALSE;

	sprintf(reqpath, "%s.req", nodepath(config.outdir, remote_node));
	if (access(reqpath, 4) == 0) {
		log("Sending file requests for %s", ascnode(remote_node));
		sprintf(alias, "%04x%04x.req", remote_node.net, remote_node.node);
		if (zedzap)
			s = sendzmodem(reqpath, alias, ZCNL, ZMCLOB, 0);
		else
			s = batchsend(reqpath, alias);
		switch (s) {
		case OK:
			savepacket(reqpath);
			return TRUE;
		case REJECTED:
			savebad(reqpath);
		case ERROR:
		case FAILED:
			break;
		}
	} else
		debug(1, "No file requests for %s", ascnode(remote_node));
	return FALSE;
}


/*
 * Process file requests from remote
 */
boolean
filerequests(zedzap)
	boolean zedzap;
{
	char filepath[PATH_LEN];
	FILE *reqfp, *aliasfp, *listfp;
	char *cp;
	boolean reqok;
	boolean validated;
	int s;
	char fn[PATH_LEN], pw[20], up[20];
	char reqfile[PATH_LEN];
	boolean something;

	reqfp = aliasfp = listfp = NULL;
	something = FALSE;

	/*
	 * Validate remote node
	 * Remote nodes are of four kinds.
	 * None, known in nodelist, protected connection, anyone
	 */
	switch (config.file_request_mode) {
	case NONE_REQ:			/* No file requests */
		reqok = False;
		break;
	case PROT_REQ:			/* Only from protected links */
		reqok = False;		/* Until we implement passwords */
		break;
	case KNOWN_REQ:			/* Known in nodelist */
		reqok = (boolean)(search_node(remote_node, NULL) != NULL);
		break;
	case ANYONE_REQ:		/* Anyone can do file requests */
		reqok = True;
		break;
	}

	/*
	 * Scan node directory, looking for the next
	 * file with type ".req"
	 * The opening of myopendir for each iteration in the loop
	 * is because we might need "myreaddir" in the loop to get
	 * the filenames for wildcard file requests.
	 */
	for (;;) {
		myopendir(nodepath(config.indir, remote_node), FALSE);
		while (cp = myreaddir(NULL)) {
			if (!is_req(cp))
				continue;
			if ((reqfp = fopen(cp, "r")) == NULL)
				continue;
			log("Processing file requests in %s from %s",
			    basename(cp), ascnode(remote_node));
			something = TRUE;
			break;
		}
		myclosedir();
		if (cp == NULL)
			break;
		strcpy(reqfile, cp);
		/*
		 * Loop that picks up each line from the file request
		 * file and processes each file request.
		 */
		while (parsereq(reqfp, fn, pw, up)) {
			/*
			 * Handle the special files "about" and "files"
			 * Request of "about" is always allowed,
			 * request of "files" need file request allowance.
			 */
			if (strequ(fn, "about")) {
				strcpy(filepath, config.file_request_about);
				goto hasit;
			}
			if (!reqok) {
				log("File request of %s forbidden for this node",
				    fn);
				break;
			}
			if (strequ(fn, "files")) {
				strcpy(filepath, config.file_request_files);
				goto hasit;
			}
			/*
			 * Look up filename in alias file.
			 */
			if (*config.file_request_alias) {
				if (!aliasfp) {
					aliasfp = fopen(config.file_request_alias, "r");
					if (aliasfp == NULL)
						*config.file_request_alias = 0;
				}
			}
			validated = FALSE;
			if (aliasfp && lookup_alias(aliasfp, fn, filepath)) {
				validated = TRUE;
				strcpy(fn, filepath);
			}
			if (!validated && listfp == NULL && *config.file_request_list) {
				listfp = fopen(config.file_request_list, "r");
				if (listfp == NULL)
					*config.file_request_list = 0;
			}
			/*
			 * Check for wildcards
			 * If no wildcards, directly look up filename
			 * in file request list.
			 */
			if (!strpbrk(fn, "*?[]")) {
				if (validated)
					goto hasit;
				if (listfp) {
					s = lookup_list(listfp, fn, pw);
					if (s == OK) {
						strcpy(filepath, fn);
						goto hasit;
					} else if (s == REJECTED)
						goto reject;
				}
			} else if (listfp == NULL && !validated)
				goto reject;
			/*
			 * It is a wildcard filename.
			 * We have to make a directory lookup to
			 * get the real filenames.
			 */
			if (*fn == '/')
				strcpy(filepath, fn);
			else {
				if (*config.file_request_dir == 0)
					goto reject;
				strcpy(filepath, config.file_request_dir);
				strcat(filepath, fn);
			}
			cp = basename(filepath);
			if (cp > filepath && cp[-1] == '/' && *cp)
				*--cp = 0;
			else
				cp = NULL;
			if (strpbrk(filepath, "*?[]"))
				goto reject;
			myopendir(filepath, FALSE);
			if (cp)
				*cp = '/';
			while (cp = myreaddir(filepath)) {
				if (!validated) {
					s = lookup_list(listfp, cp, pw);
					if (s == REJECTED) {
						log("File request of file %s rejected", cp);
						continue;
					} else if (s == ERROR)
						goto fini;
				}
				s = sendrequest(cp, zedzap);
				if (s == ERROR)
					goto fini;
				if (s == FINISH)
					break;
			}
			continue;
		hasit:
			if (sendrequest(filepath, zedzap) == ERROR)
				goto fini;
			continue;
		reject:
			log("File request of file %s rejected", fn);
		}
		fclose(reqfp);
		reqfp = NULL;
		if (*reqfile)
			check_unlink(reqfile);
	}
 fini:
	if (reqfp)
		fclose(reqfp);
	if (aliasfp)
		fclose(aliasfp);
	if (listfp)
		fclose(listfp);
	return something;
}

/*
 * Remove all old file requests from remote node
 * Also do some other cleanup relating to file requests
 */
void
remove_reqs()
{
	char *fname;

	if (myopendir(nodepath(config.indir, remote_node), FALSE)) {
		while ((fname = myreaddir(NULL)))
			if (struequ(basetype(fname), "req"))
				check_unlink(fname);
		myclosedir();
	}
	file_request_files = 0;
	file_request_size = 0;
}

/*
 * Parse a line from a file request file
 * Split into its three components
 * Return TRUE if the line parsed all right, FALSE if something 
 * is wrong.
 */
static boolean
parsereq(fp, fname, pw, upd)
	FILE *fp;
	char *fname, *pw, *upd;
{
	char lbuf[128];
	char *cp;

	*fname = *pw = *upd = 0;
	while (fgets(lbuf, sizeof lbuf - 1, fp)) {
		cp = strend(lbuf);
		while (cp != lbuf && isspace(cp[-1]))
		       *--cp = 0;
		cp = lbuf;
		while (isspace(*cp))
			cp++;
		/*
		 * Strip blank lines and comments
		 * what is a comment in a file request file anyway?
		 */
		if (*cp == ';' || *cp == '#' || *cp == 0)
			continue;
		/*
		 * First collect filename and process it
		 * to remove bad character. Convert dos path
		 * to unix path.
		 * Case is ignored. (unless caller is rfmail, maybe)
		 */
		cp = strtok(cp, " \t");
		if (cp == NULL)
			continue;
		strcpy(fname, cp);
		strlwr(fname);
		for (cp = fname; cp = strchr(cp, '\\'); cp++)
			*cp = '/';
		/*
		 * Look for password and/or update parameters
		 */
		while (cp = strtok(NULL, " \t")) {
			switch (*cp) {
			case '!':	 /* Password */
				if (*pw)
					goto fail;
				strcpy(pw, cp+1);
				break;
			case '+':
			case '-':
				if (*upd)
					goto fail;
				strcpy(upd, cp);
				break;
			case ';':
				break;
			default:
				goto fail;
			}
		}
		return TRUE;
	}
 fail:
	return FALSE;
}

/*
 * Send a file request
 */
static int
sendrequest(fpath, zedzap)
	char *fpath;
	boolean zedzap;
{
	struct stat sbuf;
	int s;

	if (stat(fpath, &sbuf) == 0) {
		if (config.file_request_max_size &&
		    file_request_size + sbuf.st_size
		    			> config.file_request_max_size*1024) {
			log("Request of %s rejected: size limit", fpath);
			return FINISH;
		}
		if (config.file_request_max_files && file_request_files >= config.file_request_max_files) {
			log("Request of %s rejected: file number limit", fpath);
			return FINISH;
		}
		log("File %s requested by file request", fpath);
		if (zedzap)
			s = sendzmodem(fpath, basename(fpath), 0, 0, 0);
		else
			s = batchsend(fpath, basename(fpath));
		switch (s) {
		case OK:
			log("File transmission of %s done", fpath);
			file_request_size += sbuf.st_size;
			file_request_files++;
			break;
		case FAILED:
			log("File transmission of %s failed", fpath);
			break;
		case REJECTED:
			log("File transmission of %s rejected by remote", fpath);
			break;
		case ERROR:
			log("File transmission error");
			break;
		}
		return s;
	} else
		log("File %s requested, but not found", fpath);
	return FAILED;
}

/*
 * Look up in alias file
 */
static boolean
lookup_alias(fp, fn, fpath)
	FILE *fp;
	char *fn, *fpath;
{
	char lbuf[128];
	register char *cp;

	rewind(fp);
	while (fgets(lbuf, sizeof(lbuf)-1, fp)) {
		/*
		 * Each line in the lbuf file
		 * has two fields, separated
		 * by space.
		 */
		if (*lbuf == ';' || *lbuf == '#')
			continue;
		cp = lbuf;
		while (*cp && !isspace(*cp))
			cp++;
		if (*cp)
			*cp++ = 0;
		while (isspace(*cp))
			cp++;
		strlwr(lbuf);
		if (strequ(lbuf, fn)) {
			strcpy(fpath, cp);
			return True;
		}
	}
	return False;
}

/*
 * Lookup a name in file request list
 * The entries in the file request list are either
 * absolute path, or relative to file-request-dir.
 * If name ends with a slash (/) that entry allows file requests
 * from any file from that directory, or any sub-directory from it.
 * If you want to allow request of any file in the directory
 * but not from any subdirectory of that directory, end
 * the filename with "/*"
 * Wildcards are allowed only in the basename of the entry
 * and not in the directory names.
 *
 * Returns:
 *	OK - File request of file is ok
 *	REJECTED - File request of file is not ok
 *	FAILED - File is not found in file request list
 */
static int
lookup_list(fp, fn, upw)
	FILE *fp;
	char *fn, *upw;
{
	char lbuf[128], filepath[PATH_LEN];
	char *cp, *lpp, *fpw, *bnp;
	int isdir;

	/*
	 * Form full file path from root
	 * into filepath, with lpp pointing at
	 * the path local to file_request_dir
	 * and bnp pointing at the basename
	 */
	if (*fn == '/') {
		strcpy(filepath, fn);
		if (*config.file_request_dir
		    && strnequ(filepath, config.file_request_dir, strlen(config.file_request_dir)))
			lpp = filepath + strlen(config.file_request_dir);
		else
			lpp = NULL;
	} else {
		strcpy(filepath, config.file_request_dir);
		lpp = strend(filepath);
		strcpy(lpp, fn);
	}
	bnp = basename(filepath);
	/*
	 * Fail if wildcard in the directory path.
	 */
	if ((cp = strpbrk(filepath, "*?[]")) && cp < bnp)
		return REJECTED;

	/*
	 * Fail if ".." appears in path
	 */
	if (strstr(filepath, ".."))
		return REJECTED;
	/*
	 * Search in file request list 
	 * for this filename
	 */
	rewind(fp);
	while (fgets(lbuf, sizeof(lbuf)-1, fp)) {
		cp = strend(lbuf);
		while (cp > lbuf && isspace(cp[-1]))
			*--cp = 0;
		if (*lbuf == ';' || *lbuf == '#' || *lbuf == 0)
			continue;
		if (cp = strchr(lbuf, '!')) {
			/*
			 * Password needed for file request
			 * separate password from filename.
			 */
			fpw = cp+1;
			*cp = 0;
			while (cp > lbuf && isspace(cp[-1]))
				*--cp = 0;
		} else
			fpw = NULL;
		cp = strend(lbuf);
		isdir = cp > lbuf && cp[-1] == '/';
		cp = *lbuf == '/' ? filepath : lpp;
		if ((isdir ? strnequ(lbuf, cp, strlen(lbuf))
		          : strequ(lbuf, cp)) || match(cp, lbuf)) {
			if (fpw && !strequ(upw, fpw))
				return REJECTED;
			return OK;
		}
	}
	return FAILED;
}
