/*
 * $Header: /u1/src/rfmail/RCS/dial.c,v 0.5.0.1 1992/06/15 06:11:25 pgd Exp pgd $
 *
 * $Log: dial.c,v $
 * Revision 0.5.0.1  1992/06/15  06:11:25  pgd
 * Minor compilation bug fixes.
 * Change of all types with u_ prefix to U prefix
 * Change of name of routine msleep() to mssleep()
 *
 * Revision 0.5  1992/05/18  04:27:24  pgd
 * New distribution
 *
 * Revision 0.4  1991/05/08  04:23:43  pgd
 * Initial Beta-release
 *
 *
 */

/*
 * The new rfmail dialer.
 */

/*
 * Authors:
 * 
 * Per Lindqvist (pgd@compuram.bbt.se)
 */

#include "fnet.h"

#ifdef FCNTL
#include <fcntl.h>
#endif

#include <utmp.h>
#include <sys/stat.h>


#include "nodelist.h"
#include "configs.h"
#include "fcall.h"

#define	UUCP	4

#define LDEVS	"/usr/lib/uucp/Devices"
#define SYSFILE "/usr/lib/uucp/Systems"
#define DIALFILE "/usr/lib/uucp/Dialcodes"
#define DEVDIR	"/dev/"
#define LOCK	"/usr/spool/uucp/LCK.."

static char devname[20];	/* Name of the device we have locked */

/*
 * The new dialer
 */
extern struct nodelist remote_nodeentry;
static Uint ungetty_mask = 0;
static boolean uucp_initted = FALSE;
static boolean need_ungetty_undial;
static int dial_rate;
static char crlf[] = "\r\n";
extern int baudrate;

LDECLARE(Uint, searchmodems, (int));
LDECLARE(void, modemwrite, (char *));
LDECLARE(int, modemread, (char *, struct modem_struct *, Ushort *, Uint *));
LDECLARE(void, uucpinit, (void));
LDECLARE(boolean, checkline, (char *));
LDECLARE(boolean, ungetty, (char *));


/*
 * Return the highest preferred speed for a mode mask.
 */
int
mode_to_baud(mode)
	int mode;
{
	if (mode & PEP) return 19200;
	else if (mode & V32b) return (mode & (V42|V42b)) ? 38400 : 19200;
	else if (mode & (H96|HST|MAX|CSP)) return 9600;
	else if (mode & V32) return (mode & (V42|V42b)) ? 19200 : 9600;
	else if (mode & V29) return 9600;
	else if (mode & V23) return 4800;
	else if (mode & V22b) return 2400;
	else if (mode & V22) return 1200;
	else if (mode & V21) return 300;
	else return 9600;	/* Impossible */
}
	

/*
 * Dial the indicated telephone number
 *
 * Returns the file descriptor for the connected line,
 * or a negative error code.
 */
int
new_dial(telno, attr, nodedata)
	char *telno;
	TTYSTATE *attr;
	struct nodelist *nodedata;
{
	register int i;
	Uint set;
	char *device;
	struct modem_struct *modem;
	Uint linebit;
	boolean gotit;
	int errcode, retcode;
	int fd;
	TTYSTATE ttystate;
	int speed, mode, j, resp, newspeed;
	Ushort flags;
	char buf[128];
	char devpath[80];
	Uint spd;

	if (telno[0] == 0)
		fatal(0, "new_dial: no telephone number specified");

	uucpinit();

	errcode = NO_BD_K;

	/*
	 * Look for modem according to modem search order.
	 */
	mode = 0;
	set = 0;
	fd = 0;
	for (i = 0; config.modem_search_order[i]; i++) {
		if (nodedata->modemcap & config.modem_search_order[i]) {
			set = searchmodems(config.modem_search_order[i]);
			if (set) {
				mode = config.modem_search_order[i];
				errcode = NO_BD_A;
				break;
			}
		}
	}

	/*
	 * We now have a set of modems to try dialing on.
	 * Try them all in turn until we find a free one.
	 */
	gotit = FALSE;
	while (set) {
		for (j = 0; (set & BIT(j)) == 0; j++)
			;
		device = config.line[j].device;
		modem = &config.modem[config.line[j].modem];
		linebit = BIT(j);
		set &= ~linebit;
		if (checkline(device)) {
			/*
			 * Get rid of getty if we are sharing the line.
			 */
			if (linebit & ungetty_mask) {
				if (!ungetty(device)) {
					if (*devname) {
						removelock(devname);
						*devname = 0;
					}
					continue;
				}
			}
			gotit = TRUE;
			break;
		}
	}
	if (!gotit)
		return errcode;

	/*
	 * Select speed to use for speaking with modem.
	 */
	if (modem->interface_speed)
		speed = modem->interface_speed;
	else {
		speed = mode_to_baud(modem->modemcap & nodedata->modemcap);
		if (speed > config.maxbaud)
			speed = config.maxbaud;
		i = baud_to_bsdbaud(speed)-1;
		if ((i & modem->speeds) == 0) {
			/*
			 * Modem does not support this speed.
			 * Select higher speed if possible,	
			 * up to config.maxbaud.
			 */
			j = baud_to_bsdbaud(config.maxbaud)-1;
			while (j > 0 && (modem->speeds & j) == 0)
				j--;
			if (j == 0) {
				buglog("Modem speed mismatch for device %s",
				       device);
				errcode = L_PROB;
				goto error_cleanup;
			}
			speed = bsdbaud_to_baud(j+1);
		}
	}

	/*
	 * We now have a modem line which is reserved for our use.
	 * We need to open with O_NDELAY since we don't have
	 * any carrier at this point.
	 */
	sprintf(devpath, "/dev/%s", devname);
	fd = open(devpath, O_RDWR | O_NDELAY);
	if (fd < 0) {
		retcode = L_PROB;
		goto error_cleanup;
	}

	/*
	 * If we don't have the ungetty program, and
	 * we are sharing the line with getty (HDB uucp)
	 * a hangup on the line will get rid of getty.
	 */
	if ((linebit & ungetty_mask) && *config.ungetty_path == 0) {
		hangup(fd);
		mssleep(500);
	}

	/*
	 * Set correct line parameters. Also
	 * set line in local mode, and turn off O_NDELAY
	 */
#ifdef TERMIO
	attr->c_cflag = (attr->c_cflag & ~CBAUD) | baud_to_bsdbaud(speed);
#else
	attr->sg_ospeed = attr->sg_ispeed = baud_to_bsdbaud(speed);
#endif
	ttystate = *attr;
#ifdef TERMIO
	ttystate.c_lflag &= ~ECHO;
	ttystate.c_cflag &= ~HUPCL;
	ttystate.c_cflag |= CLOCAL;
#else
	ttystate.sg_flags &= ~ECHO;
#endif
	if (!set_terminal_state(fd, &ttystate)) {
		errcode = L_PROB;
		goto error_cleanup;
	}
	line = open(devpath, O_RDWR);
	if (line == -1) {
		retcode = L_PROB;
		goto error_cleanup;
	}
	close(fd);
	fd = line;

	debug(1, "Dialing out on modem %s", device);

	/*
	 * If not fixed interface speed, make the modem
	 * sync to our speed.
	 */
	if (modem->interface_speed == 0) {
		dial_rate = 250;
		modemwrite("aaaa");
	}

	dial_rate = modem->dial_rate; /* Wait-time between each character */

	/*
	 * Send modem the init string
	 */
	modemwrite(modem->init);
	if (modemread(buf, modem, NULL, NULL) != MM_OK) {
		/*
		 * Try a second time before giving up.
		 */
		modemwrite(modem->init);
		modemwrite(crlf);
		if (modemread(buf, modem, NULL, NULL) != MM_OK) {
			log("Modem does not respond properly");
			errcode = L_PROB;
			goto error_cleanup;
		}
	}
	
	/*
	 * Now send the telephone number
	 */
	sprintf(buf, "%s%s%s", modem->dial_prefix, telno, modem->dial_suffix);
	modemwrite(buf);

	do {
		resp = modemread(buf, modem, &flags, &spd);
		switch (resp) {
		case MM_RRING:
			debug(1, "Remote is ringing");
			break;
		case MM_RING:
			log("Got a call on the modem, while dialing");
			errcode = DIAL_RING;
			goto error_cleanup;
		case MM_OK:	/* We should not get an ok here */
			break;
		case MM_NOCARRIER:
			log("Modem got no carrier");
			errcode = DIAL_NO_CARRIER;
			goto error_cleanup;
		case MM_ERROR:
			log("Dial gave ERROR response");
			errcode = DIAL_MODEM_ERROR;
			goto error_cleanup;
		case MM_NODIALTONE:
			errcode = DIAL_NO_DIALTONE;
			goto error_cleanup;
		case MM_BUSY:
			errcode = NO_OCC;
			goto error_cleanup;
		case MM_NOANSWER:
			errcode = NO_ANS;
			goto error_cleanup;
		case MM_CONNECT:
			break;
		case -1:
			errcode = A_PROB;
			goto error_cleanup;
		}
	} while (resp != MM_CONNECT);
	
	/*
	 * We now are connected to the other modem.
	 * flags have the particular flags for this connection.
	 */
	if (spd)
		newspeed = spd;
	else
		newspeed = speed;
	if ((modem->speeds & BIT(baud_to_bsdbaud(newspeed)-1)) == 0) {
		errcode = DIAL_BAD_SPEED;
		goto error_cleanup;
	}
	if (newspeed != speed) {
		debug(1, "Setting new speed for modem line: %d", newspeed);
#ifdef TERMIO
		ttystate.c_cflag
			= attr->c_cflag
			= (attr->c_cflag & ~CBAUD) | baud_to_bsdbaud(speed);
#else
		ttystate.sg_ospeed = ttystate.sg_ispeed
			= attr->sg_ospeed = attr->sg_ispeed
			= baud_to_bsdbaud(speed);
#endif
	}
	baudrate = newspeed;

	/*
	 * Now set correct terminal state for communication.
	 * The reopen of the line is needed on some systems
	 * to drop CLOCAL
	 */
	set_terminal_state(fd, attr);
	fd = open(devpath, O_RDWR);
	if (fd == -1) {
		fd = line;
		retcode = L_PROB;
		goto error_cleanup;
	}
	close(line);
	return fd;

error_cleanup:
	if (fd) {
		new_undial(fd);
		close(fd);
	}
	line = -1;
	return errcode;
}

/*
 * Return the set of lines which have the indicated flag set.
 */
static Uint
searchmodems(flag)
	int flag;
{
	register int i;
	Uint set;

	set = 0;
	for (i = 0; i < config.lines; i++)
		if (config.modem[config.line[i].modem].modemcap & flag)
			set |= BIT(i);
	return set;
}


void
new_undial(fd)
	int fd;
{
	int pid, status;

	debug(1, "Undial called, fd = %d, device = %s", fd, devname);

	if (fd >= 0)
		close(fd);
	if (*devname == 0)
		return;
	/*
	 * Remove lock on device and give it back to getty
	 */
	removelock(devname);

	if (*config.ungetty_path) {
		if ((pid = fork()) == 0) {
			execl(config.ungetty_path,
			      "ungetty", "-t", NULL);
			exit(-1);
		}
		status = mywait(pid);
	}
}

	
/*
 * Check if a line is free for use.
 * As a side effect, sets "devname" to the path of the locked device.
 */
static boolean
checkline(device)
	char *device;
{
	register int i;
	struct utmp *utp, u, *getutline();

	*devname = 0;
	for (i = 0; i < config.lines; i++)
		if (strequ(config.line[i].device, device))
			break;
	/* This condition cannot happen */
	if (i == config.lines)
		return FALSE;
	/*
	 * Check if line is in use by someone.
	 */
	strncpy(u.ut_line, device, sizeof(u.ut_line));
	setutent();
	if ((utp = getutline(&u)) == NULL)
		goto maybe;
	if (utp->ut_type == USER_PROCESS
	    && strequ(utp->ut_user, "DIALOUT")) {
		log("Got line %s, in dialout state",
		    utp->ut_line);
		goto maybe;
	}
	if (kill(utp->ut_pid, 0) == -1) {
		if (errno == ESRCH)
			goto maybe;
		if (errno != EPERM)
			return FALSE;
	}
	if (utp->ut_type != LOGIN_PROCESS)
		return FALSE;
 maybe:
	if (!createlock(device, 0)) {
		*devname = 0;
		return FALSE;
	}
	strcpy(devname, device);
	return TRUE;
}

/*
 * Initialize bitmap with lines that are shared with uucp.
 */
static void
uucpinit()
{
	FILE *fp;
	register int i;
	char buf[256];
	char *ttyline, *devicetype;

	if (uucp_initted)
		return;

	ungetty_mask = 0;

	if ((fp = fopen(config.devices, "r")) == NULL) {
		buglog("Cannot open file %s", config.devices);
		return;
	}
	while (fgets(buf, sizeof(buf)-1, fp)) {
		if (buf[0] == '#')
			continue;
		if ((devicetype = strtok(buf, " \t")) == NULL)
			continue;
		if ((ttyline = strtok(NULL, " \t")) == NULL)
			continue;
		for (i = 0; i < config.lines; i++)
			if (strequ(config.line[i].device, ttyline))
				ungetty_mask |= BIT(i);
	}
	fclose(fp);
	uucp_initted = TRUE;
}


/*
 * Tell getty to release the device.
 */
static boolean
ungetty(device)
	char *device;
{
	int pid;
	int status;

	need_ungetty_undial = FALSE;
	if (config.ungetty_path[0]) {
		if ((pid = fork()) == 0) {
			execl(config.ungetty_path, "ungetty", devname, NULL);
			exit(-1);
		}

		status = mywait(pid);
		switch (status) {
		case 0:		/* Line not enabled by getty */
			return TRUE;

		case 1:		/* Need ungetty -r when ready */
			need_ungetty_undial = TRUE;
			return TRUE;

		case 2:		/* Could not get line */
			return FALSE;

		default:	/* Unknown error code or error. */
			return TRUE;
		}
	}
	return TRUE;
}

static int
dialalarm()
{
}

/*
 * Write a string to the modem.
 */
static void
modemwrite(s)
	char *s;
{
	register int c;

	debug(1, "write to modem: '%s'", s);

	while (c = *s++) {
		switch (c) {
		case '|':
			sendline('\r');
			break;
		case '-':	/* ignore */
			break;
		case '~':	/* A one second delay */
			mssleep(1000);
			break;
		case '`':	/* A .1 second sleep */
			mssleep(100);
			break;
		default:
			sendline(c);
		}
		if (dial_rate)
			mssleep(dial_rate);
	}
}

/*
 * Read a string from the modem
 * Return the message response number,
 * 0 for not found, and -1 for error.
 */
static int
modemread(buf, modem, flags, spd)
	char *buf;
	struct modem_struct *modem;
	Ushort *flags;
	Uint *spd;
{
	int c, l, i;
	long t, ttmo;
	
	debug(10, "Waiting for response from modem");

	/*
	 * Make a long wait before the first character,
	 * and a short one once they start coming.
	 */
	ttmo = time(NULL) + config.dial_timeout;
 redo:
	c = readline(config.dial_timeout*1000);
	if (c < 0) {
		debug(1, "Modem returned EOF");
		t = time(NULL);
		if (t < ttmo) {
			mssleep(100);
			goto redo;
		}
		debug(1, "Modem timed out");
		return -1;
	}
	l = 0;
	for (;;) {
		if (l >= 128) {
			debug(1, "buffer overflow in modemread");
			return -1;
		}
		if (c == '\n' && l != 0)
			break;
		else if (c != '\r' && c != '\n')
			buf[l++] = c;
		c = readline(5000);
		if (c < 0) {
			buf[l] = 0;
			debug(1, "Timeout in modemread. String so far: '%s'",
			      buf);
			return -1;
		}
	}
	buf[l] = 0;
	for (i = 0; i < modem->messages; i++)
		if (strequ(buf, modem->message[i].msg))
			break;
	if (i < modem->messages) {
		debug(1, "Modem response: %s", buf);
		if (flags)
			*flags = modem->message[i].flags;
		if (spd && modem->message[i].type == MM_CONNECT)
			*spd = bsdbaud_to_baud(modem->message[i].misc);
		return modem->message[i].type;
	}
	debug(1, "Unrecognized modem response: %s", buf);
	return 0;
}






