/*
 * Linux tty interface file
 * Pink House Systems, October 1993
 */

#include <stdio.h>
#include <sys/ioctl.h>
#include <termio.h>
#include <unistd.h>
#include <curses.h>
#include <string.h>
#include "tty.h"


static int rows, columns;
static int input  = fileno(stdin);
static int output = fileno(stdout);
static int fg_color, bg_color, br_status, rv_status, bl_status;
static int tty_perm;

static char colors[10][10] = 
{
    "BLACK",
    "RED",
    "GREEN",
    "YELLOW",
    "BLUE",
    "MAGENTA",
    "CYAN",
    "WHITE",
    "OFF",
    "ON"
};



void tty_init(void)
{
    tty_defaults();
    noecho();
    nonl();
    noraw();	// just to turn on signals
    cbreak();
 }


void tty_end(void)
{
    nl();
    echo();
    nocbreak();
    noraw();	// just to turn on signals
    tty_defaults();
}


void tty_clrscr(void)
{
    write(output, "\x1b[H\x1b[J", 6);
}


void tty_cursormove(int y, int x)
{
    char msg[10];
    
    sprintf(msg, "\x1b[%d;%dH", y + 1, x + 1);
    write(output, msg, strlen(msg));
}


void tty_foreground(int color)
{
    char str[] = "\x1b\x5b\x33\x30\x6d";
    str[3] += (fg_color = color);
    write(output, str, 5);
}


void tty_background(int color)
{
    char str[] = "\x1b\x5b\x34\x30\x6d";
    str[3] += (bg_color = color);
    write(output, str, 5);
}


void tty_bright(int status)
{
    br_status = status;
    if (status == ON)
        write(output, "\x1b\x5b\x31\x6d", 4);
    else
        write(output, "\x1b\x5b\x32\x32\x6d", 5);
}


void tty_reverse(int status)
{
    rv_status = status;
    if (status == ON)
        write(output, "\x1b\x5b\x37\x6d", 4);
    else
        write(output, "\x1b\x5b\x32\x37\x6d", 5);
}


void tty_blink(int status)
{
    bl_status = status;
    if (status == ON)
        write(output, "\x1b\x5b\x35\x6d", 4);
    else
        write(output, "\x1b\x5b\x32\x35\x6d", 5);
}


void tty_beep(void)
{
    char c = 7;
    write(output, &c, 1);
}


void tty_defaults(void)
{
    write(output, "\x1b\x5b\x30\x6d", 4);
}


void tty_getcolors(tty_colors *col)
{
    col->fg_color  = fg_color;
    col->bg_color  = bg_color;
    col->br_status = br_status;
    col->rv_status = rv_status;
    col->bl_status = bl_status;
}


void tty_setcolors(tty_colors *col)
{
    tty_foreground(col->fg_color);
    tty_background(col->bg_color);
    tty_bright(col->br_status);
    tty_reverse(col->rv_status);
    tty_blink(col->bl_status);
}


int tty_write(char *buf, int length)
{
    return write(output, buf, length);
}


int tty_read(char *buf, int length)
{
    return read(input, buf, length);
}


int tty_getch(void)
{
    char c;
    read(input, &c, 1);
    return c;
}


void tty_putch(int c)
{
    write(output, &c, 1);
}


int tty_getkey(void)
{
    char c;

    c = tty_getch();
    if (c == 012) return KEY_ENTER;
    if (c != KEY_ESC) return c;
    
    if ((c = tty_getch()) == KEY_ESC) return KEY_ESC;
    else
	if (c >= '0' && c <= '9') return KEY_ALT0 + c - '0';
	else
	    if (c >= 'A' && c <= 'Z') return KEY_ALTA + c - 'A';
	    else
                if (c >= 'a' && c <= 'z') return KEY_ALTa + c - 'a';
                else
                    if (c == KEY_ENTER) return KEY_ALTENTER;

    switch (tty_getch())
    {
        case 0x5b:  	c = tty_getch();
          		return KEY_F1 + c - 0x41;
          		break;
            		
        case 0x31:  	c = tty_getch();
		    	if (c == 0x7e)
		    	    return KEY_HOME;
		    	else
          		{
            		    tty_getch();
            		    return KEY_F6 + c - 0x37;
             		}
            		
        case 0x32:  	c = tty_getch();
                    	if (c == 0x7e) return KEY_INS;
          	    	tty_getch();
           	    	switch (c)
            	    	{
            		    case 0x30: return KEY_F9;
            		    case 0x31: return KEY_F10;
            		    case 0x33: return KEY_F11;
            		    case 0x34: return KEY_F12;
            		    case 0x35: return KEY_SHIFTF3;
            		    case 0x36: return KEY_SHIFTF4;
            		    case 0x38: return KEY_SHIFTF5;
            		    case 0x39: return KEY_SHIFTF6;
                        }
            		
        case 0x33:  	if ((c = tty_getch()) == 0x7e)
          		    return KEY_DEL;
          		else
          		{
          		    tty_getch();
          		    return KEY_SHIFTF7 + c - 0x31;
          		}

        case 0x34:	tty_getch();
          		return KEY_END;

        case 0x35:  	tty_getch();
          		return KEY_PPAGE;

        case 0x36:	tty_getch();
          		return KEY_NPAGE;

        case 0x41:	return KEY_UP;
            
        case 0x42:	return KEY_DOWN;

        case 0x43:	return KEY_RIGHT;

        case 0x44:  	return KEY_LEFT; 
            
        default:	break;
    }	
    return KEY_NOKEY;
}


void tty_getsize(int *_columns, int *_rows)
{
    winsize winsz;

    ioctl(output, TIOCGWINSZ, &winsz);
    if (!winsz.ws_col || !winsz.ws_row)
    {
        columns = 80;
        rows    = 25;
    }
    else
    {
        columns = winsz.ws_col;
        rows    = winsz.ws_row;
    }
    *_columns = columns;
    *_rows    = rows;
}


void tty_getscreen(char *buf)
{
    char *tty_name = ttyname(1);
    buf[0] = 0;
    buf[1] = tty_name[strlen(tty_name) - 1] - '0';
    if (ioctl(output, TIOCLINUX, buf) == -1)
    {
        buf[0] = buf[1] = 0;				// Linux bug
        if (ioctl(output, TIOCLINUX, buf) == -1)	// (bad ioctl on console 8)
        {
	    tty_perm = 0;
	    return;
	}
    }
    tty_perm = 1;
    int index = 2 + rows * columns;
    for (int y = rows - 1; y >= 0; y--)
        for (int x = columns - 1; x >= 0; x--)
            if (buf[--index] != ' ')
                goto found;

   found:

    buf[0] = y + 1;
    buf[1] = 0; 
    tty_cursormove(y + 1, 0);
}


void tty_putscreen(char *buf)
{    
    tty_defaults();
    tty_cursormove(0, 0);
    if (tty_perm)
    {
        write(output, buf + 2, rows * columns);
        tty_cursormove(buf[0], buf[1]);
    }
    else
        tty_clrscr();
}


int tty_getcolorindex(char *colorname)
{
    for (int i = 0; i < 10; i++)
        if (strcmp(colors[i], colorname) == 0)
            return (i < 8) ? i : (i - 8);
    return -1;
}
