/*
 * Main file
 * Pink House Systems, October 1993
 */

#include <curses.h>
#include <unistd.h>
#include <termio.h>
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <signal.h>
#include <time.h>
#include <sys/wait.h>
#include <pwd.h>
#include <grp.h>
#include "tty.h"
#include "window.h"
#include "edit.h"
#include "status.h"
#include "panel.h"
#include "config.h"


#define MAX_STATIC_SIZE 60

static WINDOW *win;

edit *edt;
int  SCREEN_X, SCREEN_Y;
int background = OFF;
char title[] = "Pink House Systems shell 3.2b";
char login[] = "User:";
char tty[]   = "tty:";
char exit_msg[] = "*** (EXIT) Exit Pink House Systems shell ? (ENTER to confirm, TAB to cancel) ***";
char PS1[4]   = " $ ";
char *screen  = NULL;
char configfile[MAXPATHLEN] = "";
char homedirectory[MAXPATHLEN];
char ErrorLogDirectory[MAXPATHLEN];
char NormalModeHelp[256];
char CommandLineModeHelp[256];

static int UserHeartAttack;

#define BUILTIN_OPERATIONS		19

#define BUILTIN_Copy			-1
#define BUILTIN_RenameMove		-2
#define BUILTIN_MkDir			-3
#define BUILTIN_Delete			-4
#define BUILTIN_Exit			-5
#define BUILTIN_HistoryBack		-6
#define BUILTIN_ShowTty			-7
#define BUILTIN_Refresh			-8
#define BUILTIN_GotoRoot		-9
#define BUILTIN_SwitchPanels		-10
#define BUILTIN_HistoryNext		-11
#define BUILTIN_FileDisplayMode		-12
#define BUILTIN_DeleteCmdLn		-13
#define BUILTIN_FileSelect		-14
#define BUILTIN_CopyFileToCmdLn		-15
#define BUILTIN_Home			-16
#define BUILTIN_End			-17
#define BUILTIN_PageUp			-18
#define BUILTIN_PageDown		-19


#define MAX_BUILTIN_NAME		25


char builtin[BUILTIN_OPERATIONS][MAX_BUILTIN_NAME] =
{
    "<Copy>",
    "<RenameMove>",
    "<MkDir>",
    "<Delete>",
    "<Exit>",
    "<HistoryBack>",
    "<ShowTty>",
    "<Refresh>",
    "<GotoRoot>",
    "<SwitchPanels>",
    "<HistoryNext>",
    "<FileDisplayMode>",
    "<DeleteCmdLn>",
    "<FileSelect>",
    "<CopyFileToCmdLn>",
    "<Home>",
    "<End>",
    "<PageUp>",
    "<PageDown>"
};


struct ConfigKey
{
    char  KeyName[10];
    char *OpName;
    char *OpCommand;
    char *OpNewDir;
    char  OpSaveScreen;
    char  OpPause;
    char  OpHide;
    char  OpBuiltin;
};

#define KEYS_FIELDS	119
#define KEYSDATA_FIELDS	8

static ConfigKey ConfigKeysFields[KEYS_FIELDS] = 
{
    { "F1",	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "F2",	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "F3",	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "F4",	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "F5",	  builtin[0],  NULL, NULL, 1, 0, 0, 1 },
    { "F6",	  builtin[1],  NULL, NULL, 1, 0, 0, 1 },
    { "F7",	  builtin[2],  NULL, NULL, 1, 0, 0, 1 },
    { "F8",	  builtin[3],  NULL, NULL, 1, 0, 0, 1 },
    { "F9", 	  NULL,	       NULL, NULL, 1, 0, 0, 0 },
    { "F10",	  builtin[4],  NULL, NULL, 1, 0, 0, 1 },
    { "F11", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "F12", 	  NULL,	       NULL, NULL, 1, 0, 0, 0 },
    { "ShiftF1",  NULL,        NULL, NULL, 1, 0, 0, 0 },	// ignored
    { "ShiftF2",  NULL,        NULL, NULL, 1, 0, 0, 0 },	// ignored
    { "ShiftF3",  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "ShiftF4",  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "ShiftF5",  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "ShiftF6",  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "ShiftF7",  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "ShiftF8",  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "ShiftF9",  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "ShiftF10", NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "ShiftF11", NULL,        NULL, NULL, 1, 0, 0, 0 },	// ignored
    { "ShiftF12", NULL,        NULL, NULL, 1, 0, 0, 0 },	// ignored
    { "CtrlA", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "CtrlB",    NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "CtrlC",    NULL,        NULL, NULL, 1, 0, 0, 0 },	// ignored
    { "CtrlD",    NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "CtrlE",    builtin[5],  NULL, NULL, 1, 0, 0, 1 },
    { "CtrlF",    NULL,	       NULL, NULL, 1, 0, 0, 0 },
    { "CtrlG",    NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "CtrlH",    NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "CtrlI",    NULL,        NULL, NULL, 1, 0, 0, 0 },	// ignored
    { "CtrlJ",    NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "CtrlK",    NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "CtrlL",    NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "CtrlM",    NULL,        NULL, NULL, 1, 0, 0, 0 },	// ignored
    { "CtrlN",    NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "CtrlO",    builtin[6],  NULL, NULL, 1, 0, 0, 1 },
    { "CtrlP",    NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "CtrlQ",    NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "CtrlR",    builtin[7],  NULL, NULL, 1, 0, 0, 1 },
    { "CtrlS",    NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "CtrlT",    builtin[8],  NULL, NULL, 1, 0, 0, 1 },
    { "CtrlU",    builtin[9],  NULL, NULL, 1, 0, 0, 1 },
    { "CtrlV",    NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "CtrlW",    NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "CtrlX",    builtin[10], NULL, NULL, 1, 0, 0, 1 },
    { "CtrlY",    NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "CtrlZ",    NULL,        NULL, NULL, 1, 0, 0, 0 },	// ignored
    { "Alt0",     NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "Alt1", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "Alt2", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "Alt3",     NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "Alt4",     NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "Alt5",     NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "Alt6",     NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "Alt7",     NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "Alt8",     NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "Alt9",     NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "AltA",     NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "AltB",     NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "AltC",     NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "AltD",     builtin[11], NULL, NULL, 1, 0, 0, 1 },
    { "AltE",     NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "AltF",     NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "AltG",     NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "AltH",     NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "AltI", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "AltJ", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "AltK", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "AltL", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "AltM", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "AltN", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "AltO", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "AltP", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "AltQ", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "AltR", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "AltS", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "AltT", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "AltU", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "AltV", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "AltW", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "AltX", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "AltY", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "AltZ", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "Alta", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "Altb", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "Altc", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "Altd", 	  builtin[11], NULL, NULL, 1, 0, 0, 1 },
    { "Alte", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "Altf", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "Altg", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "Alth", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "Alti", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "Altj", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "Altk", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "Altl", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "Altm", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "Altn", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "Alto", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "Altp", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "Altq", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "Altr", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "Alts", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "Altt", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "Altu", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "Altv", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "Altw", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "Altx", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "Alty", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "Altz", 	  NULL,        NULL, NULL, 1, 0, 0, 0 },
    { "Del", 	  builtin[12], NULL, NULL, 1, 0, 0, 1 },
    { "Ins", 	  builtin[13], NULL, NULL, 1, 0, 0, 1 },
    { "AltEnter", builtin[14], NULL, NULL, 1, 0, 0, 1 },
    { "Home", 	  builtin[15], NULL, NULL, 1, 0, 0, 1 },
    { "End", 	  builtin[16], NULL, NULL, 1, 0, 0, 1 },
    { "PageUp",	  builtin[17], NULL, NULL, 1, 0, 0, 1 },
    { "PageDown", builtin[18], NULL, NULL, 1, 0, 0, 1 }
};


#define TITLE_FIELDS	8

static char TitleFields[TITLE_FIELDS][40] = 
{
    "TitleForeground",
    "TitleBackground",
    "TitleBrightness",
    "UserName",
    "TtyName",
    "ClockForeground",
    "ClockBackground",
    "ClockBrightness"
};

static int TitleColors[TITLE_FIELDS] = 
{
    MAGENTA, BLUE, ON, YELLOW, YELLOW, BLACK, CYAN, OFF
};

static int& TitleForeground	= TitleColors[0];
static int& TitleBackground	= TitleColors[1];
static int& TitleBrightness	= TitleColors[2];
static int& UserName 		= TitleColors[3];
static int& TtyName 		= TitleColors[4];
static int& ClockForeground 	= TitleColors[5];
static int& ClockBackground 	= TitleColors[6];
static int& ClockBrightness 	= TitleColors[7];


void settitle(void)
{
    char buf[256], *login_name = getlogin(), *tty_name = ttyname(1);
    window title_win(0, 0, 1, SCREEN_X);

    tty_bright(TitleBrightness);
    tty_foreground(TitleForeground);
    tty_background(TitleBackground);
    title_win.cursormove(0, 0);
    memset(buf, ' ', SCREEN_X);
    title_win.write(buf, SCREEN_X);

    sprintf(buf, "%s %s  %s %s        ", login, login_name, tty, tty_name);
    title_win.cursormove(0, 1);
    title_win.write(title, strlen(title));
    title_win.cursormove(0, SCREEN_X - strlen(buf)  - 1);
    title_win.write(login, strlen(login));
    title_win.putch(' ');
    tty_foreground(UserName);
    title_win.write(login_name, strlen(login_name));
    title_win.putch(' ');
    title_win.putch(' ');
    tty_foreground(TitleForeground);
    title_win.write(tty, strlen(tty));
    title_win.putch(' ');
    tty_foreground(TtyName);
    title_win.write(tty_name, strlen(tty_name));
}


void clock_on(int dummy)
{
    tm *_tm;
    int hour;
    time_t _time;
    char buf[256];
    tty_colors colors;
    
    signal(SIGALRM, clock_on);
    _time = time(NULL);
    _tm = localtime(&_time);
    alarm(60 - _tm->tm_sec);
    tty_getcolors(&colors);
    if ((hour = _tm->tm_hour % 12) == 0) hour = 12;
    sprintf(buf, "%2d:%02d%c", hour, _tm->tm_min, (_tm->tm_hour < 12) ? 'a' : 'p');
    tty_cursormove(0, SCREEN_X - 7);
    tty_foreground(ClockForeground);
    tty_background(ClockBackground);
    tty_bright(ClockBrightness);
    tty_write(buf, strlen(buf));
    tty_setcolors(&colors);
}


void clock_off(void)
{
    signal(SIGALRM, SIG_IGN);
}


void TERMuser_panic(int dummy)
{
    signal(SIGTERM, TERMuser_panic);
    UserHeartAttack = 1; 
}


void INTuser_panic(int dummy)
{
    signal(SIGINT, INTuser_panic);
    UserHeartAttack = 1; 
}


void QUITuser_panic(int dummy)
{
    signal(SIGQUIT, QUITuser_panic);
    UserHeartAttack = 1; 
}


void Background(int dummy)
{
    background = ON;
    clock_off();
    tty_end();
    tty_putscreen(screen);
    signal(SIGTSTP, SIG_DFL);
    kill(getpid(), SIGTSTP);
}


int start(char *cmd, int hide = 0)
{
    int child_exit_code;
    char fname[MAXPATHLEN];
    
    if (hide)
    {
        char *tty_name = ttyname(1);
        clock_off();
        close(1);
        close(2);
        sprintf(fname, "%s/%s.%d", ErrorLogDirectory, "stdout", getpid());
        FILE *stdout_log = fopen(fname, "w");
        sprintf(fname, "%s/%s.%d", ErrorLogDirectory, "stderr", getpid());
	FILE *stderr_log = fopen(fname, "w");
        child_exit_code = system(cmd);
	fclose(stderr_log);
        fclose(stdout_log);
        open(tty_name, O_RDWR);
        open(tty_name, O_RDWR);
    }
    else
    {
        clock_off();
        tty_end();
        tty_putscreen(screen);
        child_exit_code = system(cmd);
	write(1, "\n\n", 2);
        tty_init();
    }
    return child_exit_code;
}


void removelog(void)
{
    char name[MAXPATHLEN];
    
    sprintf(name, "%s/%s.%d", ErrorLogDirectory, "stdout", getpid());
    remove(name);
    sprintf(name, "%s/%s.%d", ErrorLogDirectory, "stderr", getpid());
    remove(name);
}


void panic(char *postmsg)
{
    clock_off();
    tty_clrscr();
    tty_end();
    printf("Fatal error : %s !\n", postmsg);
    delwin(win);
    endwin();
    removelog();
    exit(1);    
}


int getOpCommand(char *CommandName, char *src, char *dest, panel *p, panel *l, edit *e)
{
    char *ptr;
    group *grp;
    passwd *pwd;
    static int busy = 0;
    int  retval, uid, gid;
    char question[MAXLINE];
    char formatted[MAXLINE];
    
    
    while (*src)
    {
        if (*src != '%') *dest++ = *src++;
        else
        {
            switch(*++src)
            {
                case 's': if (busy) { busy = 0; return 0; }
                	  if (*++src != '{') return 0;
                	  if ((ptr = strchr(++src, ',')) == NULL) return 0;
                	  *ptr = 0;
                	  retval = getOpCommand(NULL, src, formatted, p, l, NULL);
                	  if (retval != 1) return retval;
                          sprintf(question, "(%s) %s", CommandName, formatted);
                          *ptr++ = ',';
                	  if ((src = strchr(ptr, '}')) == NULL) return 0;
                	  *src = 0;
                	  question[MAX_STATIC_SIZE] = 0;
                	  retval = getOpCommand(NULL, ptr, formatted, p, l, NULL);
                	  if (retval != 1)
                	  {
                  	      *src = '}';
                	      return retval;
                	  }
                	  if (!e->gets(question, dest, formatted))
                	  {
                  	      *src = '}';
                	      return -1;
                	  }
                	  *src = '}';
                	  break;
                	  
                case 'f': if (p->getcurrentfiletype() != FILE_ENTRY) return -1;
                	  p->getcurrentfilename(dest); 			
                	  break;

                case 'd': if (p->getcurrentfiletype() != DIR_ENTRY) return -1;
                	  p->getcurrentfilename(dest); 			
                	  break;

                case 'l': if (p->getcurrentfiletype() != SYMLINK_ENTRY) return -1;
                	  p->getcurrentfilename(dest); 			
                	  break;
                	  
                case 'a': p->getcurrentfilename(dest); 			
                	  break;
                	  
                case 'm': sprintf(dest, "%o", p->getcurrentfilemode() & 0777); 
                	  break;
                
                case 'o': uid = p->getcurrentfileuid();
                	  pwd = getpwuid(uid);
                	  if (pwd)
                	      sprintf(dest, "%s", pwd->pw_name);
                	  else
                	      sprintf(dest, "%o", uid);
                	  break;

                case 'g': gid = p->getcurrentfilegid();
                	  grp = getgrgid(gid);
                	  if (grp)
                	      sprintf(dest, "%s", grp->gr_name);
                	  else
                	      sprintf(dest, "%o", gid);
                	  break;

                case 'p': p->getpath(dest);
			  break;

                case 'F': if (l->getcurrentfiletype() != FILE_ENTRY) return -1;
                	  l->getcurrentfilename(dest); 			
                	  break;

                case 'D': if (l->getcurrentfiletype() != DIR_ENTRY) return -1;
                	  l->getcurrentfilename(dest); 			
                	  break;

                case 'L': if (l->getcurrentfiletype() != SYMLINK_ENTRY) return -1;
                	  l->getcurrentfilename(dest); 			
                	  break;
                	  
                case 'A': l->getcurrentfilename(dest); 			
                	  break;
                	  
                case 'M': sprintf(dest, "%o", l->getcurrentfilemode() & 0777); 
                	  break;
                
                case 'O': uid = l->getcurrentfileuid();
                	  pwd = getpwuid(uid);
                	  if (pwd)
                	      sprintf(dest, "%s", pwd->pw_name);
                	  else
                	      sprintf(dest, "%o", uid);
                	  break;

                case 'G': gid = l->getcurrentfilegid();
                	  grp = getgrgid(gid);
                	  if (grp)
                	      sprintf(dest, "%s", grp->gr_name);
                	  else
                	      sprintf(dest, "%o", gid);
                	  break;

                case 'P': l->getpath(dest);
			  break;

                default : return 0;
            }
            src++;
            dest += strlen(dest);
        }
    }
    *dest = 0;
    return 1;
}


int getConfigKey(int key)
{
    if (key == KEY_ENTER   || key == KEY_TAB)	   return -1;
    if (key >= KEY_F1      && key <= KEY_F12)      return  0 + key - KEY_F1;
    if (key >= KEY_SHIFTF3 && key <= KEY_SHIFTF10) return 12 + key - KEY_SHIFTF3 + 2;
    if (key >= KEY_CTRLA   && key <= KEY_CTRLZ)    return 24 + key - KEY_CTRLA;
    if (key >= KEY_ALT0    && key <= KEY_ALT9)     return 50 + key - KEY_ALT0;
    if (key >= KEY_ALTA    && key <= KEY_ALTZ)     return 60 + key - KEY_ALTA;
    if (key >= KEY_ALTa    && key <= KEY_ALTz)     return 86 + key - KEY_ALTa;
    if (key == KEY_DEL) 			   return 112;
    if (key == KEY_INS) 			   return 113;
    if (key == KEY_ALTENTER)			   return 114;
    if (key == KEY_HOME) 			   return 115;
    if (key == KEY_END) 			   return 116;
    if (key == KEY_PPAGE)			   return 117;
    if (key == KEY_NPAGE)			   return 118;
    return -1;
}


int main(void)
{
    char cmdln[MAXPATHLEN];
    char *home, *data = NULL;
    char cmdname[MAXPATHLEN];
    char *homedirectory = cmdln;
    char temppath[2][MAXPATHLEN];
    panel *current_panel, *other_panel, *temp_panel;
    int panel_no = 0, action_status, i, j, retval, child_exit_code;
    int sectionptr = 0, index, key, configkey, app_end = 0, len, wait = 0;
            		
    signal(SIGTERM, TERMuser_panic); 
    signal(SIGINT , INTuser_panic);
    signal(SIGQUIT, QUITuser_panic);

    puts("Pink House Systems shell version 3.2b  Copyright (c) October 1993\n");

    strcpy(homedirectory, "/");
    strcpy(temppath[0], ".");
    strcpy(temppath[1], ".");
    
    if (home = getenv("HOME"))
    {
        strcpy(homedirectory, home);
        strcpy(configfile, home);
        if (configfile[strlen(configfile) - 1] != '/') strcat(configfile, "/");
    }
    strcat(configfile, ".pshrc");

    configuration config(configfile, panic);
    if (config.getstatus() != STATUS_OK)
        printf("Cannot open configuration file %s.\nUsing defaults.\n", configfile);

    if (config.getstatus() == STATUS_OK)
    {
        if ((sectionptr = config.getsectionptr("[Colors]")) == -1)
	    puts("[Colors] section missing.");
	else
            for (i = 0; i < TITLE_FIELDS; i++)
            {
                config.getfielddata(sectionptr, TitleFields[i], &data, 1);
                if (!data || (index  = tty_getcolorindex(data))== -1)
                    printf("Invalid %s (%s).\n", TitleFields[i], data);
	        else
                    TitleColors[i] = index;
            }
        if ((sectionptr = config.getsectionptr("[Setup]")) == -1)
	    puts("[Setup] section missing.");
	else
	{
	    config.getfielddata(sectionptr, "NormalModeHelp", &data, 1);
	    if (data) strncpy(NormalModeHelp, data, 256);
	    config.getfielddata(sectionptr, "CommandLineModeHelp", &data, 1);
	    if (data) strncpy(CommandLineModeHelp, data, 256);
	    config.getfielddata(sectionptr, "StartupLeftPanelPath", &data, 1);
	    strncpy(temppath[0], data ? data : ".", MAXPATHLEN);
	    config.getfielddata(sectionptr, "StartupRightPanelPath", &data, 1);
	    strncpy(temppath[1], data ? data : ".", MAXPATHLEN);
	    config.getfielddata(sectionptr, "ErrorLogDirectory", &data, 1);
	    strncpy(ErrorLogDirectory, data ? data : homedirectory, MAXPATHLEN);
	}
        if ((sectionptr = config.getsectionptr("[Keys]")) == -1)
	    puts("[Keys] section missing.");
	else
	{
	    char *contents[KEYSDATA_FIELDS - 2];
	    
	    for (i = 0; i < KEYS_FIELDS; i++)
	    {
	        config.getfielddata(sectionptr, ConfigKeysFields[i].KeyName, contents, KEYSDATA_FIELDS - 2, NO_SEEK);

		if (contents[0])
		{
	            ConfigKeysFields[i].OpName = new char[strlen(contents[0]) + 1];
	            strcpy(ConfigKeysFields[i].OpName, contents[0]);
	        }
	        else
	        {
	            ConfigKeysFields[i].OpName = NULL;
	            continue;
	        }

		if (contents[1])
		{
	            ConfigKeysFields[i].OpCommand = new char[strlen(contents[1]) + 1];
	            strcpy(ConfigKeysFields[i].OpCommand, contents[1]);
	        }
	        else
	        {
	            ConfigKeysFields[i].OpCommand = NULL;
	            continue;
	        }
	        
	        if (contents[2])
	        {
	            ConfigKeysFields[i].OpNewDir = new char[strlen(contents[2]) + 1];
	            strcpy(ConfigKeysFields[i].OpNewDir, contents[2]);
	        }
	        else ConfigKeysFields[i].OpNewDir = NULL;
	        
		if (contents[3]) ConfigKeysFields[i].OpSaveScreen = atoi(contents[3]);
	        else ConfigKeysFields[i].OpSaveScreen = 1;

		if (contents[4]) ConfigKeysFields[i].OpPause = atoi(contents[4]);
	        else ConfigKeysFields[i].OpPause = 0;

		if (contents[5]) ConfigKeysFields[i].OpHide = atoi(contents[5]);
	        else ConfigKeysFields[i].OpHide = 0;
	    }
	    for (i = 0; i < KEYS_FIELDS; i++)
	    {
                ConfigKeysFields[i].OpBuiltin = 0;
	        if (ConfigKeysFields[i].OpName == NULL) continue;
	        for (j = 0; j < BUILTIN_OPERATIONS; j++)
	            if (strcmp(ConfigKeysFields[i].OpName, builtin[j]) == 0)
	            {
	                delete ConfigKeysFields[i].OpName;
	                ConfigKeysFields[i].OpName = builtin[j];
	                ConfigKeysFields[i].OpBuiltin = 1;
	                break;
	            }
	    }
	}
    }

    tty_getsize(&SCREEN_X, &SCREEN_Y);

    screen = new char[2 + SCREEN_X * SCREEN_Y];
    if (screen == NULL)
        panic("not enough memory");
        
    status_init(SCREEN_X, SCREEN_Y - 1, NormalModeHelp, &config);

    if (getuid() == 0) PS1[1] = '#';
    edit edt(SCREEN_X, SCREEN_Y - 2, &config);

    panel  left_panel(SCREEN_Y - 3, SCREEN_X >> 1, 0            , 1, temppath[0],
    		      &edt, panic, &UserHeartAttack, &config);
    panel right_panel(SCREEN_Y - 3, SCREEN_X >> 1, SCREEN_X >> 1, 1, temppath[1], 
    		      &edt, panic, &UserHeartAttack, &config);

    tty_getscreen(screen);
    win = initscr();
    tty_init();

restart:

    signal(SIGTSTP, Background);
    signal(SIGTERM, TERMuser_panic); 
    signal(SIGINT , INTuser_panic);
    signal(SIGQUIT, QUITuser_panic);
    
    if (wait)
        while (!tty_getkey());
    wait = 0;
    
    settitle();
    clock_on(0);


    current_panel = panel_no ? &right_panel : &left_panel;
    other_panel   = panel_no ? &left_panel  : &right_panel;

    current_panel->action(act_REFRESH, other_panel, (void *)-1);
    other_panel->action(act_REFRESH, current_panel, (void *)-1);
    
    current_panel->setfocus(ON, other_panel);
    
    status(NULL, 0, 0, 1, MSG_OK);
    
    edt.gettext(cmdln);
    edt.reset();
    edt.puts(strcat(current_panel->getpath(temppath[0], MAX_STATIC_SIZE), PS1));
    edt.eos();
    edt.puts(cmdln);
    edt.update();
    edt.setcursor();

    while(!app_end)
    {
        UserHeartAttack = 0;
        configkey = getConfigKey(key = tty_getkey());
	if (background)
	{
            tty_init();
	    background = OFF;
	    current_panel->nooptimizations();
	    other_panel->nooptimizations();
            goto restart;
	}
        if (configkey != -1)
            if (ConfigKeysFields[configkey].OpBuiltin)
    	        key = - 1 - (ConfigKeysFields[configkey].OpName - builtin[0]) / MAX_BUILTIN_NAME;
    	    else
    	    {
		if (ConfigKeysFields[configkey].OpName)
		{
		    current_panel->nooptimizations();
		    other_panel->nooptimizations();
		    if (ConfigKeysFields[configkey].OpCommand)
		    {
           	        if (retval = getOpCommand(ConfigKeysFields[configkey].OpName,
           	        			  ConfigKeysFields[configkey].OpCommand,
           	        		          cmdln, current_panel, other_panel, &edt))
           	        {
           	            if (retval == 1)
           	            {
           	                sprintf(cmdname, "Executing %s: %s", ConfigKeysFields[configkey].OpName, cmdln);
				status(cmdname, 0, 0, 0, MSG_WARNING);
           	                child_exit_code = start(cmdln, ConfigKeysFields[configkey].OpHide);
           	                if (ConfigKeysFields[configkey].OpHide)
				{
				    sprintf(cmdln, "%s/%s.%d", ErrorLogDirectory, "stderr", getpid());
				    FILE *stderr_log = fopen(cmdln, "r");
				    if (child_exit_code) tty_beep();
				    if (stderr_log == NULL)
				        status("*** Can't open stderr log file ***", 1, 1, 1, MSG_ERROR);
				    while (fgets(cmdln, SCREEN_X, stderr_log))
				        if (status(cmdln, 1, 0, 0, MSG_ERROR) != KEY_ENTER) break;
				}
				else
           	                {
            	                    if (ConfigKeysFields[configkey].OpSaveScreen)
            	                        tty_getscreen(screen);
            	                    if (ConfigKeysFields[configkey].OpPause)
				        wait = 1;
				}
           	                if (child_exit_code == 0 && ConfigKeysFields[configkey].OpNewDir)
           	                    current_panel->action(act_CHDIR, other_panel, ConfigKeysFields[configkey].OpNewDir);
			        goto restart;
			    }
			}
			else
			{
			    sprintf(cmdln, "Bad configured command for key %s!", ConfigKeysFields[configkey].KeyName);
			    status(cmdln, 1, 1, 1, MSG_ERROR);
			}
		    }
		}
            }
                    
        switch (key)
        {
            case KEY_TAB:    current_panel->setfocus(OFF);
            		     temp_panel = current_panel;
            		     current_panel = other_panel;
            		     other_panel = temp_panel;
            		     panel_no = !panel_no;
            		     current_panel->setfocus(ON, other_panel);
			     edt.gettext(cmdln);
			     edt.reset();
			     edt.puts(strcat(current_panel->getpath(temppath[0], MAX_STATIC_SIZE), PS1));
			     edt.eos();
			     edt.puts(cmdln);
			     edt.update();
            		     break;

            case KEY_UP:     current_panel->action(act_UP    , other_panel, NULL); break;
            case KEY_DOWN:   current_panel->action(act_DOWN  , other_panel, NULL); break;

            case KEY_ENTER:  action_status = 0;
            		     if (*edt.gettext(cmdln))
           		         switch (*cmdln)
           		         {
           		             case '+': action_status = current_panel->action(act_SELECTALL  , other_panel, NULL); break;
           		             case '-': action_status = current_panel->action(act_UNSELECTALL, other_panel, NULL); break;
           		             case '*': action_status = current_panel->action(act_TOGGLE, other_panel, NULL); break;
           		             default : current_panel->nooptimizations();
			                       other_panel->nooptimizations();
           		             	       start(cmdln);
           		             	       action_status = -1; 
           		             	       break;
              			}
              		     else
              		     {
            		         clock_off();
            		         action_status = current_panel->action(act_ENTER , other_panel, screen);
			     }
			     
            		     if (action_status == 1)
            		     {
            		         edt.reset();
            		         edt.puts(strcat(current_panel->getpath(temppath[0], MAX_STATIC_SIZE), PS1));
            		         edt.eos();
            		         edt.update();
            		     }
            		     else
            		         if (action_status == -1) 
            		         {   
            		             tty_getscreen(screen);
            		             edt.history(EDIT_RECORD);
            		             edt.reset();
 				     goto restart;
            		         }
            		     break;

            case BUILTIN_FileSelect:
            		     current_panel->action(act_SELECT  , other_panel, NULL); break;

            case BUILTIN_PageUp:
            		     current_panel->action(act_PGUP    , other_panel, NULL); break;
            case BUILTIN_PageDown:  
            		     current_panel->action(act_PGDOWN  , other_panel, NULL); break;
            case KEY_LEFT:
            case BUILTIN_Home:
            		     current_panel->action(act_HOME    , other_panel, NULL); break;
            case KEY_RIGHT:
            case BUILTIN_End:
            		     current_panel->action(act_END     , other_panel, NULL); break;
            case BUILTIN_GotoRoot:  
            		     current_panel->action(act_CHDIR, other_panel, "/"); break;
            case BUILTIN_Refresh:
            		     current_panel->getpath(temppath[0]);
            		     other_panel->getpath(temppath[1]);
            		     if (strcmp(temppath[0], temppath[1]) == 0)
            		     {
            		         other_panel->action(act_REFRESH, NULL, NULL);
            		         current_panel->action(act_REFRESH, NULL, NULL);
            		     }
            		     else
            		         current_panel->action(act_REFRESH, other_panel, (void *)1);
            		     break;
				
            case BUILTIN_ShowTty:
            		     clock_off();
            		     tty_putscreen(screen);
            		     status(CommandLineModeHelp, 0, 0, 0, MSG_OK);
			     edt.update();
			     edt.setcursor();

        		     while (1)
        		     {
        		         configkey = getConfigKey(key = tty_getkey());
	                         if (background)
	                         {
                                     tty_init();
				     signal(SIGTSTP, Background);
                                     edt.update();
                                     edt.setcursor();
 	                             status(CommandLineModeHelp, 0, 0, 0, MSG_OK);
	                             background = OFF;
	                         }
		                 if (configkey != -1 && ConfigKeysFields[configkey].OpBuiltin)
    	    			     key = - 1 - (ConfigKeysFields[configkey].OpName - builtin[0]) / MAX_BUILTIN_NAME;
            		         if (key == BUILTIN_ShowTty) break;
            		         switch (key)
            		         {
            		             case KEY_ENTER:
	            		          if (*edt.gettext(cmdln))
	            		          {
            		                      tty_putscreen(screen);
	            		              start(cmdln);
            		                      tty_getscreen(screen);
            		                      edt.history(EDIT_RECORD);
            		                      edt.reset();
                                              edt.puts(strcat(current_panel->getpath(temppath[0], MAX_STATIC_SIZE), PS1));
                                              edt.eos();
                                              edt.update();
            		                      status(CommandLineModeHelp, 0, 0, 0, MSG_OK);
	            		          }
            		                  break;

            		             case BUILTIN_HistoryBack:
            		             case KEY_UP:
	    				  edt.history(EDIT_PREVIOUS);
            		             	  break;

            		             case BUILTIN_HistoryNext:
            		             case KEY_DOWN:
					  edt.history(EDIT_NEXT);
					  break;

	                             case BUILTIN_DeleteCmdLn:
	                                  edt.del();
	    		                  break;
	    		     
            		             case BUILTIN_Exit:
            		             	  if (status(exit_msg, 1, 1, 1, MSG_ERROR) == KEY_ENTER)
            		             	  {
	                                      app_end = 1;
	                                      goto end_ctrlo;
	                                  }
		                          status(CommandLineModeHelp, 0, 0, 0, MSG_OK);
	                                  break;

            		             default:
            		                  if (key)
            		                  {
            		                      edt.putch(key);
            		                      edt.update();
            		                  }   
            		             	  break;
            		         }
		                 edt.setcursor();
            		     }
			end_ctrlo:
			     current_panel->nooptimizations();
			     other_panel->nooptimizations();
			     status(NULL, 0, 0, 1, MSG_OK);
            		     if (app_end)
            		         break;
            		     else
            		         goto restart;

	    case BUILTIN_DeleteCmdLn:
	    		     edt.del();
	    		     break;
	    		     
            case BUILTIN_Copy:
            		     current_panel->action(act_COPY, other_panel, NULL);
            		     break;

            case BUILTIN_RenameMove:
            		     current_panel->action(act_MOVE, other_panel, NULL);
            		     break;
            		     
            case BUILTIN_MkDir:
            		     current_panel->action(act_MKDIR, other_panel, NULL);
            		     break;

            case BUILTIN_Delete:
            		     current_panel->action(act_DELETE, other_panel, NULL);
            		     break;

	    case BUILTIN_FileDisplayMode:
	    		     current_panel->action(act_MODE, NULL, NULL);
	    		     break;

            case BUILTIN_Exit:
            		     if (status(exit_msg, 1, 1, 1, MSG_ERROR) == KEY_ENTER)
                                 app_end = 1;
                             break;

            case BUILTIN_CopyFileToCmdLn:
                             edt.gettext(cmdln);
                             len = strlen(cmdln);
                             if (len && isalnum(cmdln[len - 1]))
                                 break;
                             current_panel->getcurrentfilename(&cmdln[len]);
                             while (cmdln[len])
                                 edt.putch(cmdln[len++]);
                             edt.putch(' ');
                             edt.update();
                             break;
            
	    case BUILTIN_HistoryBack:
	    		     edt.history(EDIT_PREVIOUS);
	    		     break;
	    		     
	    case BUILTIN_HistoryNext:
	    		     edt.history(EDIT_NEXT);
	    		     break; 	     

	    case BUILTIN_SwitchPanels:
	    		     current_panel->nooptimizations();
	    		     other_panel->nooptimizations();
	    		     current_panel->action(act_SWITCH, other_panel, NULL);
                             other_panel->action(act_REFRESH, current_panel, (void *)-1);
                             current_panel->action(act_REFRESH, other_panel, (void *)-1);
	    		     break;

            default:	     if (key)
            		     {
            		         edt.putch(key);
            		         edt.update();
            		     }
            		     break;
        }
        edt.setcursor();
    }

    removelog();

    clock_off();
    status_end();
    tty_end();
    tty_clrscr();
    delete screen;
    delwin(win);
    endwin();
}
