/*
 * Panel file
 * Pink House Systems, October 1993
 */

#include <stdio.h>
#include <stdlib.h>
#include <curses.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <fcntl.h>
#include <pwd.h>
#include <grp.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/vfs.h>
#include "tty.h"
#include "window.h"
#include "status.h"
#include "panel.h"


char temp[MAXPATHLEN];
char no_perm[]  = "***** Permission denied ! *****";
char bad_name[] = "***** Invalid name ! *****";
char rights[16] = "rwxrwxrwx";


#define FILE_DISPLAY_MODES	3

char FileDisplayMode[FILE_DISPLAY_MODES][15] = 
{
    "OwnerGroup",
    "DateTime",
    "Size"
};


#define PANEL_FIELDS	17

static char PanelFields[PANEL_FIELDS][40] = 
{
    "PanelFrame",
    "PanelBackground",
    "PanelSelectedFile",
    "PanelSelectedFileBrightness",
    "PanelNotSelectedFile",
    "PanelNotSelectedFileBrightness",
    "PanelCurrentSelectedFile",
    "PanelCurrentNotSelectedFile",
    "PanelCurrentFile",
    "PanelPath",
    "PanelPathBrightness",
    "PanelDeviceFreeSpace",
    "PanelDeviceFreeSpaceBrightness",
    "PanelFileInfo",
    "PanelFileInfoBrightness",
    "PanelFilesInfo",
    "PanelFilesInfoBrightness",
};

static int PanelColors[PANEL_FIELDS] = 
{
    WHITE, BLUE, YELLOW, ON, WHITE, ON, YELLOW, WHITE, 
    CYAN, RED, OFF, RED, OFF, RED, OFF, BLACK, OFF
};

static int& PanelFrame 				= PanelColors[0];
static int& PanelBackground 			= PanelColors[1];
static int& PanelSelectedFile 			= PanelColors[2];
static int& PanelSelectedFileBrightness		= PanelColors[3];
static int& PanelNotSelectedFile		= PanelColors[4];
static int& PanelNotSelectedFileBrightness	= PanelColors[5];
static int& PanelCurrentSelectedFile 		= PanelColors[6];
static int& PanelCurrentNotSelectedFile 	= PanelColors[7];
static int& PanelCurrentFile 			= PanelColors[8];
static int& PanelPath 				= PanelColors[9];
static int& PanelPathBrightness 		= PanelColors[10];
static int& PanelDeviceFreeSpace        	= PanelColors[11];
static int& PanelDeviceFreeSpaceBrightness	= PanelColors[12];
static int& PanelFileInfo 			= PanelColors[13];
static int& PanelFileInfoBrightness 		= PanelColors[14];
static int& PanelFilesInfo			= PanelColors[15];
static int& PanelFilesInfoBrightness		= PanelColors[16];

static int *UserHeartAttack;
static int StartupFileDisplayMode;

extern "C"
{ 
    int tell(int);
}


#define max(a, b) ((a) >= (b) ? (a) : (b))
#define min(a, b) ((a) <= (b) ? (a) : (b))

inline static void xchg(int& a, int& b)
{
    int tmp = a;
    a = b;
    b = tmp;
}


panel::panel(int _lines, int _columns, int _begin_x, int _begin_y, char *_path,
             edit *_edt, void (*_panic)(char *),
             int *_UserHeartAttack, configuration *config)
{
    char *data;
    int sectionptr, index;
    static int configured;
    
    lines   = _lines;
    columns = _columns;
    begin_x = _begin_x;
    begin_y = _begin_y;
    focus   = OFF;
    edt     = _edt;
    panic   = _panic;
    entries = selected_files = 0;
    last_index = -1;
    mode = found_selection = end = 0;
    current_entry = 0;
    first_on_screen = max(0, current_entry - (lines - 2) + 1);
    on_screen = MAX_ENTRIES;
    UserHeartAttack = _UserHeartAttack;
    memset(dir_entry, 0, sizeof(dir_entry));
    strcpy(path, _path);

    win = new window(begin_x, begin_y, lines, columns);

    if (configured)
    {
        mode = StartupFileDisplayMode;
        return;
    }
    if (config->getstatus() == STATUS_OK)
    {
        if ((sectionptr = config->getsectionptr("[Setup]")) != -1)
        {
	    config->getfielddata(sectionptr, "StartupFileDisplayMode", &data, 1);
	    if (data)
	    {
                for (int i = 0; i < FILE_DISPLAY_MODES; i++)
                    if (strcmp(data, FileDisplayMode[i]) == 0)
                        break;
                if (i == FILE_DISPLAY_MODES)
                    printf("Invald StartupFileDisplayMode (%s).\n", data);
                else
                    mode = StartupFileDisplayMode = i;
            }
	}
	
        if ((sectionptr = config->getsectionptr("[Colors]")) != -1)
            for (int i = 0; i < PANEL_FIELDS; i++)
            {
                config->getfielddata(sectionptr, PanelFields[i], &data, 1);
                if (!data || (index = tty_getcolorindex(data)) == -1)
                    printf("Invalid %s (%s).\n", PanelFields[i], data);
	        else
                    PanelColors[i] = index;
            }
    }
    configured = 1;
}


panel::~panel(void)
{
    if (dir)
        closedir(dir);
    for (int i = 0; i < entries; i++)
        if (dir_entry[i].name)
            delete dir_entry[i].name;
    delete win;
}


static int sortfn(const void *_first, const void *_second)
{
    int retval;
    _dir_entry *first  = (_dir_entry *)_first;
    _dir_entry *second = (_dir_entry *)_second;
    int firstisdir  = first->type  == DIR_ENTRY;
    int secondisdir = second->type == DIR_ENTRY;    
    char *pfirst  = strrchr(first->name , '.');
    char *psecond = strrchr(second->name, '.');

    
    if (firstisdir == secondisdir)
        if (*pfirst == *psecond)
            if (*pfirst == '.')
                return (retval = strcmp(++pfirst, ++psecond)) ?
                       retval : strcmp(first->name, second->name);
            else return strcmp(first->name, second->name);
        else return (*pfirst == '.') ? -1 : 1;
    else return firstisdir ? -1 : 1;
}


void panel::recover(void)
{
    char msg[MAXPATHLEN];
    
    first_on_screen = current_entry = 0;
    sprintf(msg, "Can't get permission in directory %s ! (press any key).", path);
    status(msg, 1, 1, 1, MSG_ERROR);
    if (strcmp(path, "/") == 0)
        panic("can't get permission in root directory");
    strcpy(path, "/");
    chdir(path);
    this->action(act_REFRESH, NULL, NULL);
}


int panel::getdirinfo(char *directory, int verify)
{
    DIR *tmpdir;
    dirent *d;
    stat s;
    statfs fstat;
    passwd *pwd;
    group *grp;
    tm *time;
    int old_uid = -1, old_gid = -1, msdosfs = 0;
    int old_entries = 0, backdir_index = -1, sz, hour;
    _dir_entry *old_dir_entry = NULL, tmp;
    char old_path[MAXPATHLEN], temp[MAXPATHLEN];
        
    
    if (!(tmpdir = opendir(directory)))
        return 0;

    if (chdir(directory) == -1)
    {
        closedir(tmpdir);
        return 0;
    }

    closedir(dir);
    dir = tmpdir;
    strcpy(old_path, path);
    getcwd(path, MAXPATHLEN);
    statfs(".", &fstat);
    msdosfs = fstat.f_type == 0x4d44;	// can't get this number without
    					// including linux/msdos_fs.h
    
    if (verify = (verify && selected_files && strcmp(old_path, path) == 0))
    {
        if (!(old_dir_entry = new _dir_entry[MAX_ENTRIES]))
            panic("not enough memory");
        memcpy(old_dir_entry, dir_entry, (old_entries = entries) * sizeof(_dir_entry));
        memset(dir_entry, 0, sizeof(dir_entry));
    }

    selected_files = 0;
    maxname = fstat.f_namelen;
    for (entries = 0; (d = readdir(dir)) && entries < MAX_ENTRIES; entries++)
    {
        if (d->d_name[0] == '.' && !d->d_name[1])
            { entries--; continue; }		// ignore "."
        if (d->d_name[0] == '.' && d->d_name[1] == '.' && !d->d_name[2])
            if (path[1])
                backdir_index = entries;
            else { entries--; continue; }	// ignore ".." if root directory
        s.st_ino = 0;
        stat(d->d_name, &s);
        s.st_mode &= ~07000;
        dir_entry[entries].mode = s.st_mode;
        dir_entry[entries].uid  = s.st_uid;
        dir_entry[entries].gid  = s.st_gid;

        if (verify)
        {
            for (int j = 0; j < old_entries; j++)
                if (strcmp(d->d_name, old_dir_entry[j].name) == 0)
                {
                    selected_files += (dir_entry[entries].selected = old_dir_entry[j].selected);
                    break;
                }
        }
        else
            dir_entry[entries].selected = 0;

        if (s.st_ino)
        {
            if ((s.st_mode & ~0777) == S_IFDIR)
                dir_entry[entries].type = DIR_ENTRY;
            else
                if ((s.st_mode & ~0777) == S_IFREG)
                {
                    dir_entry[entries].type = FILE_ENTRY;
                    dir_entry[entries].executable = ((s.st_mode & 0111) && !msdosfs) ? 1 : 0;
		    // with the new kernel (0.99.12) all MSDOS files
		    // are executable so, when working with msdos
		    // file systems, we had to ignore those bits.
                }
                else
                {
                    dir_entry[entries].type = FILE_ENTRY;
                    dir_entry[entries].executable = OFF;
                }
            dir_entry[entries].size = s.st_size;
        }
        else
        {
            dir_entry[entries].type = SYMLINK_ENTRY;
            sz = readlink(d->d_name, temp, MAXPATHLEN);
            dir_entry[entries].size = (sz == -1) ? 0 : sz;
        }
        
        if (s.st_uid == old_uid)
            memcpy(dir_entry[entries].owner, dir_entry[entries - 1].owner, 7);
        else
        {
            pwd = getpwuid(old_uid = s.st_uid);
            if (pwd)
                sprintf(dir_entry[entries].owner, "%-7s", pwd->pw_name);
	    else
                sprintf(dir_entry[entries].owner, "%-7d", s.st_uid);
	}
	
        if (s.st_gid == old_gid)
            memcpy(dir_entry[entries].group, dir_entry[entries - 1].group, 7);
        else
        {
            grp = getgrgid(old_gid = s.st_gid);
            if (grp)
                sprintf(dir_entry[entries].group, "%-7s", grp->gr_name);
	    else
                sprintf(dir_entry[entries].group, "%-7d", s.st_gid);
	}
	
	time = localtime(&s.st_mtime);
	if ((hour = time->tm_hour % 12) == 0) hour = 12;
	sprintf(dir_entry[entries].date, "%2d-%02d-%02d %2d:%02d%c", 
	        time->tm_mon + 1, time->tm_mday, time->tm_year,
		hour, time->tm_min, (time->tm_hour < 12) ? 'a' : 'p');

        if (dir_entry[entries].name)
            delete dir_entry[entries].name;
        if (!(dir_entry[entries].name = new char[strlen(d->d_name) + 1]))
            panic("not enough memory");
        strcpy(dir_entry[entries].name, d->d_name);
    }

    if (entries == MAX_ENTRIES && readdir(dir))
    {
        sprintf(temp, "Too many directory entries! Only %d will be displayed. (press any key)", MAX_ENTRIES);
        status(temp, 1, 1, 1, MSG_ERROR);
    }
    
    if (verify)    
    {
        for (int i=0; i < old_entries; i++)
            if (old_dir_entry[i].name)
                delete old_dir_entry[i].name;
        delete old_dir_entry;
    }
    
    if (backdir_index != -1)
    {
        tmp = dir_entry[0];
        dir_entry[0] = dir_entry[backdir_index];
        dir_entry[backdir_index] = tmp;
        qsort(dir_entry + 1, entries - 1, sizeof(_dir_entry), sortfn);
    }
    else
        qsort(dir_entry, entries, sizeof(_dir_entry), sortfn);

    return 1;
}


int panel::getnext(void)
{
    if (end) { end = 0; return -1; }
    
    for (int i = last_index + 1; i < entries; i++)
        if (dir_entry[i].selected)
        {
            found_selection = 1;
            return last_index = i;
        }     
    end = !found_selection;
    int retval = found_selection ? -1 : current_entry;
    last_index = -1;
    found_selection = 0;
    return (retval == current_entry && 
            dir_entry[current_entry].type != FILE_ENTRY &&
            dir_entry[current_entry].type != SYMLINK_ENTRY) ? -1 : retval;
}


void panel::update(void)
{
    int i;
    tty_colors c;

    tty_getcolors(&c);

    for (i=first_on_screen; i < entries && (i - first_on_screen < lines - 2); i++)
        update_entry(i);

    tty_background(PanelBackground);
    memset(temp, ' ', columns);
    int limit = min(lines - 2, on_screen);
    for (; i < limit; i++)
    {
        win->cursormove(i - first_on_screen + 1, 1);
        win->write(temp, columns - 2);
    }
    on_screen = entries;
    tty_setcolors(&c);
}


void panel::trunc_fname(char *fname, char *dest, int len)
{
    int flen;
         
    if ((flen = strlen(fname)) > len)
    {
        dest[0] = dest[1] = dest[2] = '.';
        memcpy(dest + 3, fname + flen - len + 3, len - 3);
    }
    else
        memcpy(dest, fname, flen);
}


void panel::update_path(void)
{
    memset(temp, ' ', columns);
    trunc_fname(path, temp, columns - 4 - 11);

    tty_bright(PanelPathBrightness);
    tty_foreground(PanelPath);
    tty_background(PanelFrame);
    win->cursormove(0, 2);
    win->write(temp, columns - 4 - 11);
}


void panel::update_size(void)
{
    statfs fsbuf;
    char sz[16];

    statfs(path, &fsbuf);
    sprintf(sz, "%10d", fsbuf.f_bfree * fsbuf.f_bsize);

    tty_bright(PanelDeviceFreeSpaceBrightness);
    tty_foreground(PanelDeviceFreeSpace);
    tty_background(PanelFrame);
    win->cursormove(0, columns - 2 - 10);
    win->write(sz, strlen(sz));
}


void panel::update_info(void)
{
    char str[256], temp_rights[16], temp_name[16];
    int total_size = 0, total_files = 0, index, mode, len;
    
    if (selected_files)
    {
        while((index = getnext()) != -1)
        {
            total_files ++;
            total_size += dir_entry[index].size;
        }
        sprintf(str, "%d bytes in %d file(s)", total_size, total_files);
        tty_bright(PanelFilesInfoBrightness);
        tty_foreground(PanelFilesInfo);
    }
    else
    {
        strcpy(temp_rights, rights);
        mode = dir_entry[current_entry].mode;
        for (int i = 0; i < 9; mode >>= 1, i++)
            if (!(mode & 1))
                temp_rights[8 - i] = '_';

	strncpy(temp_name, dir_entry[current_entry].name, 14);
        temp_name[14] = 0;
        switch (dir_entry[current_entry].type)
        {
            case FILE_ENTRY:
            case SYMLINK_ENTRY:
                 sprintf(str, "%-14s %9d  %9s", temp_name, dir_entry[current_entry].size, temp_rights);
                 break;

            case DIR_ENTRY:     
                 sprintf(str, "%-14s %9s  %9s", temp_name, 
                         (strcmp(dir_entry[current_entry].name, "..") == 0) ? "UP--DIR" : "SUB-DIR", temp_rights);
		 break;
        }
        tty_bright(PanelFileInfoBrightness);
        tty_foreground(PanelFileInfo);
    }    
    memcpy(temp, str, len = strlen(str));
    memset(temp + len, ' ', columns - 2 - len);
    tty_background(PanelFrame);
    win->cursormove(lines - 1, 2);
    win->write(temp, columns - 4);
}


void panel::update_entry(int entry)
{
    int len;

    memset(temp, ' ', columns);
    len = min(strlen(dir_entry[entry].name), (unsigned)columns - 20);
    memcpy(temp + 1, dir_entry[entry].name, len);
    
    if (entry || !path[1])
        switch (dir_entry[entry].type)
        {
            case DIR_ENTRY:	temp[len + 1] = '/';
            			break;
            			
            case FILE_ENTRY:	if (dir_entry[entry].executable)
            			    temp[len + 1] = '*';
            			break;
            case SYMLINK_ENTRY: temp[len + 1] = '@';
                                break;
        }

    switch (mode)
    {
        case 0:
	     memcpy(temp + columns - 2 - 16, dir_entry[entry].owner, 7);
	     memcpy(temp + columns - 2 -  8, dir_entry[entry].group, 7);
	     break;
	 
	case 1:
	     memcpy(temp + columns - 2 - 16, dir_entry[entry].date, 15);
	     break;
	     
	case 2:
	     char buf[256];
	     sprintf(buf, "%9d", dir_entry[entry].size);
	     memcpy(temp + columns - 2 - 10, buf, 9);
	     break;
		
    	default:
    	     panic("invalid mode");
    }
    
    if (entry == current_entry && focus == ON)
    {
	tty_foreground(dir_entry[entry].selected ? PanelCurrentSelectedFile : PanelCurrentNotSelectedFile);
        tty_background(PanelCurrentFile);
    }
    else
    {
	tty_foreground(dir_entry[entry].selected ? PanelSelectedFile : PanelNotSelectedFile);
        tty_background(PanelBackground);
    }
    
    tty_bright(dir_entry[entry].selected ? PanelSelectedFileBrightness : PanelNotSelectedFileBrightness);
    win->cursormove(entry - first_on_screen + 1, 1);
    win->write(temp, columns - 2);
}


void panel::update_frame(void)
{
    int i;
    tty_colors c;
    char buf[256];
    

    tty_getcolors(&c);
   
    tty_foreground(PanelFrame);
    tty_reverse(ON);
    
    for (i=0; i < lines; i++)
    {
        win->cursormove(i, 0);
        tty_putch(' ');
    }

    for (i=0; i < lines; i++)
    {
        win->cursormove(i, columns - 1);
        tty_putch(' ');
    }
    
    memset(buf, ' ', 256);
    win->cursormove(0, 0);
    tty_write(buf, columns);

    win->cursormove(lines - 1, 0);
    tty_write(buf, columns);

    tty_setcolors(&c);
}


void panel::setfocus(int status, panel *link)
{
    focus = status;
    update_entry(current_entry);
    if (focus)
        if (chdir(path) == -1)
            recover();
}


char *panel::getpath(char *temppath, unsigned len)
{
    trunc_fname(path, temppath, len);
    temppath[min(len, strlen(path))] = 0;
    return temppath;
}


int filelength(int handle)
{
    int temp, length;
    
    temp = tell(handle);
    lseek(handle, 0, SEEK_END);
    length = tell(handle);
    lseek(handle, temp, SEEK_SET);
    return length;
}


enum 
{ 
    SD_OK, 
    SD_CANCEL,
    S_OPENERR, 
    S_READERR, 
    D_CREATERR, 
    D_WRITEERR, 
    SD_NOSPACE,
    SD_NOMEM
};


char copyerr[8][50] =
{
    " ",
    " ",
    "can't open source file !",
    "can't read from source file !",
    "can't create destination file !",
    "can't write into destination file !",
    "not enough free disk space !",
    "not enough memory !"
};


int panel::_copy(char *src, char *dest, int mode)
{
    char *buf;
    char msg[MAXPATHLEN];
    int shandle, dhandle, len, err, key;
    
    if (chkdest && !access(dest, 0))
    {
        for (buf = dest + strlen(dest); *buf != '/'; buf--);

        if (selected_files)
            sprintf(msg, "(COPY) Destination file %s exists. (Overwrite/Skip/All/Cancel) ?", ++buf);
        else
            sprintf(msg, "(COPY) Destination file %s exists. (Overwrite/Skip/Cancel) ?", ++buf);

        switch (key = status(msg, 1, 1, 1, MSG_ERROR))
        {
            case KEY_ENTER:
            case 'O':
            case 'o':
            	 break;
            case 'a':
            case 'A':
		 chkdest = OFF;
		 break;
            case 's':
            case 'S':
                 return SD_OK;
            default:
            	 return SD_CANCEL;
	}
    }
    if ((shandle = open(src, O_RDONLY)) == -1)
        return S_OPENERR;
    if ((dhandle = creat(dest, mode)) == -1)
    {
        close(shandle);
        return D_CREATERR;
    }
    if (!(buf = new char[len = filelength(shandle)]))
    {
        close(shandle);
        close(dhandle);
        remove(dest);
        return SD_NOMEM;
    }
    if (read(shandle, buf, len) != len)
    {
        close(shandle);
        close(dhandle);
        delete buf;
        remove(dest);
        return S_READERR;
    }
    if ((err = write(dhandle, buf, len)) != len)
    {
        close(shandle);
        close(dhandle);
        delete buf;
        remove(dest);
        return SD_NOSPACE;
    }
    else
        if (err == -1)
	{
            close(shandle);
            close(dhandle);
            delete buf;
            remove(dest);
            return D_WRITEERR;
        }

    close(shandle);
    close(dhandle);
    delete buf;
    return SD_OK;
}


enum 
{ 
    FT_OK, 
    FT_CANCEL,
    T_CREATERR, 
    F_DELETERR,
};


char moveerr[5][50] =
{
    " ",
    " ",
    "can't create destination file !",
    "can't delete source file !"
};    


int panel::_move(char *from, char *to)
{
    int key;
    char *buf;
    char msg[MAXPATHLEN];
        
    if (chkdest && !access(to, 0))
    {
        for (buf = to + strlen(to); *buf != '/'; buf--);

        if (selected_files)
            sprintf(msg, "(MOVE) Destination file %s exists. (Overwrite/Skip/All/Cancel) ?", ++buf);
        else
            sprintf(msg, "(MOVE) Destination file %s exists. (Overwrite/Skip/Cancel) ?", ++buf);

        switch (key = status(msg, 1, 1, 1, MSG_ERROR))
        {
            case KEY_ENTER:
            case 'O':
            case 'o':
            	 break;
            case 'a':
            case 'A':
		 chkdest = OFF;
		 break;
            case 's':
            case 'S':
                 return FT_OK;
            default:
            	 return FT_CANCEL;
	}
    }
    
    unlink(to);
    if (link(from, to) == -1)
        return T_CREATERR;
    if (unlink(from) == -1)
        return F_DELETERR;
    return FT_OK;
}


int panel::verify_name(char *file_name)
{
    for (char *ptr = file_name; *ptr; ptr++)
        if (*ptr == '/')
        {
       	    status(bad_name, 1, 1, 1, MSG_ERROR);
            return 0;
        }
    return 1;
}            		            


int panel::getindex(char *str)
{
    str[maxname] = 0;
    for (int i=0; strcmp(str, dir_entry[i].name) && i < entries; i++);
    if (i == entries)
    {
        for (i=0; strcasecmp(str, dir_entry[i].name) && i < entries; i++);
        if (i == entries)
            return 0;
    }
    return i;
}


int panel::action(int action, panel *link, void *aux_info)
{
    int done = 0, old_entry, i, err = 0, entry, len;
    char msg[MAXPATHLEN];
    char text[MAXPATHLEN];
    
    switch (action)
    {
        case act_ENTER:		switch (dir_entry[current_entry].type)
        			{
        			    case DIR_ENTRY:
                                         if ((strcmp(dir_entry[current_entry].name, "..") == 0) && (strcmp(path, "/") == 0)) 
                                             break;
         			         int back = strcmp(dir_entry[current_entry].name, "..") ? FALSE : TRUE;
					 char oldpath[MAXPATHLEN];  
         			         strcpy(oldpath, path);
         			         if (!getdirinfo(dir_entry[current_entry].name))
         			         {
         			             if (back)
         			                 recover();
         			             else
         			                 status(no_perm, 1, 1, 1, MSG_ERROR);
       	    			             break; 
       	    			         }
				         if (back)
				         {
					     for (char *ptr = oldpath + strlen(oldpath) - 1; *ptr != '/'; ptr--);
					     ptr++;
					     for (int i = 0; strcmp(dir_entry[i].name, ptr) && i < entries;  i++);
					     current_entry = i;
					     first_on_screen = max(0, current_entry - (lines - 2) + 1);
				         }
				         else
					     current_entry = first_on_screen = 0;
					
         			         update_path();
         			         update();
         			         update_size();
					 if (strcmp(path, link->path) == 0)
					     link->action(act_REFRESH, this, (void *)-1);
         			         link->update_size();
         			         done = 1;
         			         break;
         			         
         			    case FILE_ENTRY:
         			         if (dir_entry[current_entry].executable)
         			         {
					     tty_end();
					     tty_putscreen((char *)aux_info);
					     system(dir_entry[current_entry].name);
					     write(1, "\n\n", 2);
					     tty_init();
					     nooptimizations();
					     link->nooptimizations();
					     edt->puts(dir_entry[current_entry].name);
					     done = -1;
					  }
         			       	 break;
         			}
				break;

        case act_COPY:		chkdest = ON;
        			if (!selected_files)
        			{
        			    if (dir_entry[current_entry].type == DIR_ENTRY) break;
        			    sprintf(msg, "(COPY) Copying file %s to : ", dir_entry[current_entry].name);
				    if (!edt->gets(msg, text)) break;
            		            if (!text[0])
            		                strcpy(text, dir_entry[current_entry].name);
            		            if (strcmp(path, link->path) == 0)
            		                if (strcmp(dir_entry[current_entry].name, text) == 0)
            		                {
            		                    sprintf(msg, "(COPY) Can't copy file %s to itself! (press any key)", dir_entry[current_entry].name);
            		                    status(msg, 1, 1, 1, MSG_ERROR);
            		                    break;
            		                }
            		            if (!verify_name(text)) break;
            		            strcpy(temp, link->path);
            		            strcat(temp, "/");
            		            sprintf(msg, "(COPY) Copying file %s to %s", dir_entry[current_entry].name, text);
            		            status(msg, 0, 0, 0, MSG_WARNING);
            		            if ((err = _copy(dir_entry[current_entry].name, strcat(temp, text), dir_entry[current_entry].mode & 0777)) != SD_OK)
                       		    {
                       		        if (err == SD_CANCEL) break;
                       		        sprintf(msg, "(COPY) Error copying file %s : %s", dir_entry[current_entry].name, copyerr[err]);
                       		        status(msg, 1, 1, 1, MSG_ERROR);
                       		        break;
                       		    }
                       		    status(NULL, 0, 0, 1, MSG_OK);
                		    update_size();
                		    link->update_size();
        		        }
        		        else
        		        {    
        			    if (strcmp(path, link->path) == 0)
        			    {
        			        status("(COPY) Can't do that! (press any key)", 1, 1, 1, MSG_ERROR);
                       		        break;
                       		    }
        			    if (status("(COPY) Copying file(s) ...  (ENTER to continue, TAB to cancel)", 1, 0, 1, MSG_WARNING) != KEY_ENTER)
                       		        break;
                       		    
                       		    strcpy(temp, link->path);
                       		    strcat(temp, "/");
                       		    len = strlen(temp);

                       		    while ((entry = getnext()) != -1)
                       		    {
                       		        sprintf(msg, "(COPY) Copying %-14s", dir_entry[entry].name);
                       		        status(msg, 0, 0, 0, MSG_WARNING);
                       		        strcpy(temp + len, dir_entry[entry].name);
                       		        if (*UserHeartAttack)
                       		        {
                       		            *UserHeartAttack = 0;
                       		            status("Operation aborted. (press any key)", 1, 1, 1, MSG_ERROR);
                       		            break;
                       		        }
                		        if ((err = _copy(dir_entry[entry].name, temp,  dir_entry[entry].mode)) != SD_OK)
                		        {
                       		            if (err == SD_CANCEL) break;
                		            sprintf(msg, "(COPY) Error copying file %s : %s", dir_entry[entry].name, copyerr[err]);
                		            if (status(msg, 1, 1, 0, MSG_ERROR) != KEY_ENTER)
                		                break;
                		        }
                		        else
                		            dir_entry[entry].selected = 0;
                		        update_size();
                		        link->update_size();
                       		    }
                       		    status(NULL, 0, 0, 1, MSG_OK);
				}
     				if (!link->getdirinfo(link->path))
     				    link->recover();
     				else
     				{
        			    link->update();
        			    link->update_info();
        			}
        			if (!getdirinfo(path))
        			    recover();
        			else
        			{
        			    update();
        			    update_info();
        			}
                       		break;

        case act_DELETE:        if (!selected_files && dir_entry[current_entry].type == DIR_ENTRY)
        			{
        			    if (strcmp(dir_entry[current_entry].name, "..") == 0) break;
        			    sprintf(msg, "(RMDIR) Deleting directory %s ? (ENTER to continue, TAB to cancel)", dir_entry[current_entry].name);
        		            if (status(msg, 1, 0, 1, MSG_ERROR) != KEY_ENTER)
        		                break;
        		            strcpy(temp, path);
        		            strcat(temp, "/");
        		            strcat(temp, dir_entry[current_entry].name);
                       		    if (rmdir(dir_entry[current_entry].name) == -1 && unlink(dir_entry[current_entry].name) == -1)
                       		    {
                       		        sprintf(msg, "(RMDIR) Can't remove directory %s !", dir_entry[current_entry].name);
                       		        status(msg, 1, 1, 0, MSG_ERROR);
                       		        status(NULL, 0, 0, 1, MSG_OK);
                       		        break;
                       		    }
                       		    else
                       		        if (strcmp(temp, link->path) == 0)
                       		        {
                       		            strcpy(link->path, path);
                       		            link->action(act_REFRESH, this, NULL);
                       		        }
                       		    update_size();
                		    link->update_size();
        		        }
        		        else
        		        {    
        			    if (status("(DEL) Deleting file(s) ...  (ENTER to continue, TAB to cancel)", 1, 0, 1, MSG_ERROR) != KEY_ENTER)
                       		        break;
                       		    for (i = 0; i < entries; i++)
                       		        if (dir_entry[i].selected) break;
                       		    
                       		    while ((entry = getnext()) != -1)
                       		    {
                       		        sprintf(msg, "(DEL) Deleting %-14s", dir_entry[entry].name);
                       		        status(msg, 0, 0, 0, MSG_ERROR); 
                       		        if (*UserHeartAttack)
                       		        {
                       		            *UserHeartAttack = 0;
                       		            status("Operation aborted. Not too late I hope ...  (press any key)", 1, 1, 1, MSG_ERROR);
                       		            break;
                       		        }
                		        if (unlink(dir_entry[entry].name) == -1)
                		        {
                		            sprintf(msg, "(DEL) Can't remove file %s !  (press any key)", dir_entry[entry].name);
                		            if (status(msg, 1, 1, 1, MSG_ERROR) != KEY_ENTER)
                		                break;
                		        }
                		        else
                		            dir_entry[entry].selected = 0;
                       		    }
                       		    if (i != entries)
                       		        current_entry = i;
                		    update_size();
                		    link->update_size();
                       		    status(NULL, 0, 0, 1, MSG_OK);
				}
				
        			if (!getdirinfo(path))
        			    recover();
        			else
        			{
        			    current_entry = min(current_entry, entries - 1);
      				    first_on_screen = max(0, current_entry - (lines - 2) + 1);
        			    update();
        			    update_info();
        			}
        			if (strcmp(path, link->getpath(temp)) == 0)
        			{
        			    if (!link->getdirinfo(link->path))
        			        link->recover();
        			    else
        			    {
        			        link->current_entry = min(link->current_entry, link->entries - 1);
      				        link->first_on_screen = max(0, link->current_entry - (link->lines - 2) + 1);
        			        link->update();
        			        link->update_info();
        			    }
        			}
                       		break;

        case act_SELECT:        if (dir_entry[current_entry].type != FILE_ENTRY &&
        			    dir_entry[current_entry].type != SYMLINK_ENTRY)
          			    goto noselect;
        			dir_entry[current_entry].selected++;
        			selected_files += dir_entry[current_entry].selected ? 1 : -1;
        			update_entry(current_entry);
        		      noselect:
        			this->action(act_DOWN, link, NULL);
                       		break;

        case act_SELECTALL:	for (i = 0; i < entries; i++)
        			    if (dir_entry[i].type == FILE_ENTRY ||
        			        dir_entry[i].type == SYMLINK_ENTRY)
        			    {
         			        dir_entry[i].selected = ON;
        			        selected_files++;
                                    }
                                update();
            		        done = 1;
                       		break;

        case act_UNSELECTALL:	for (i = 0; i < entries; i++)
        			    if (dir_entry[i].type == FILE_ENTRY ||
        			        dir_entry[i].type == SYMLINK_ENTRY)
         			            dir_entry[i].selected = OFF;
         			selected_files = 0;
                                update();
            		        done = 1;
                       		break;
                       		
        case act_TOGGLE:	for (i = 0; i < entries; i++)
        			    if (dir_entry[i].type == FILE_ENTRY ||
        			        dir_entry[i].type == SYMLINK_ENTRY)
         			        dir_entry[i].selected = !dir_entry[i].selected;
				selected_files = entries - selected_files;
                                update();
            		        done = 1;
				break;
				
        case act_MKDIR:	        if (!edt->gets("(MKDIR) Enter directory name : ", temp))
          			    break;
            		        if (!temp[0] || !verify_name(temp)) break;
			        if (mkdir(temp, S_IFDIR | S_IRWXU | S_IRWXG) == -1)
          			{
          			    status(no_perm, 1, 1, 1, MSG_ERROR); 
          			    break; 
          			}
        			if (!getdirinfo(path))
        			    recover();
        			else
        			{
        			    current_entry = getindex(temp);
      				    first_on_screen = max(0, current_entry - (lines - 2) + 1);
        			    update();
        			    update_info();
                      		    update_size();
        			}
        			if (strcmp(path, link->getpath(temp)) == 0)
        			{
        			    if (!link->getdirinfo(link->path))
        			        link->recover();
        			    else
        			    {
        			        link->update();
        			        link->update_info();
        			    }
        			}
                		link->update_size();
                       		break;

        case act_MOVE:		int rename_dir = 0;
        			chkdest = ON;
                                if (!selected_files && (dir_entry[current_entry].type == FILE_ENTRY ||
                                                        dir_entry[current_entry].type == SYMLINK_ENTRY))
        			{
        			    sprintf(msg, "(MOVE) Moving file %s to : ", dir_entry[current_entry].name);
				    if (!edt->gets(msg, text)) break;
            		            if (!text[0])
            		                strcpy(text, dir_entry[current_entry].name);
            		            if (strcmp(path, link->path) == 0)
            		                if (strcmp(dir_entry[current_entry].name, text) == 0)
            		                {
            		                    sprintf(msg, "(MOVE) Can't move file %s to itself! (press any key)", dir_entry[current_entry].name);
            		                    status(msg, 1, 1, 1, MSG_ERROR);
            		                    break;
            		                }
            		            
            		            if (!verify_name(text)) break;
            		            strcpy(temp, link->path);
            		            strcat(temp, "/");
            		            sprintf(msg, "(MOVE) Moving file %s to %s", dir_entry[current_entry].name, text);
            		            status(msg, 0, 0, 0, MSG_WARNING);
            		            if ((err = _move(dir_entry[current_entry].name, strcat(temp, text))) != FT_OK)
                       		    {
                       		        if (err == FT_CANCEL) break;
             			        sprintf(msg, "(MOVE) Error moving file %s : %s", dir_entry[current_entry].name, moveerr[err]);
                       		        status(msg, 1, 1, 1, MSG_ERROR);
                       		    }
                       		    status(NULL, 0, 0, 1, MSG_OK);
                      		    update_size();
                		    link->update_size();
        		        }
        		        else
        		        {    
        		            if (!selected_files && dir_entry[current_entry].type == DIR_ENTRY)
        		            {
        		                if (strcmp(dir_entry[current_entry].name, "..") == 0) break;
				        if (!edt->gets("(RENAME) New directory name : ", text) || !text[0])
				            break;
            		                if (rename(dir_entry[current_entry].name, text) == -1)
                       		        {
             			            sprintf(msg, "(RENAME) Error renaming directory : %s !  (press any key)", dir_entry[current_entry].name);
                       		            status(msg, 1, 1, 1, MSG_ERROR);
                       		            break;
                       		        }
	        			rename_dir = 1;
                         	        status(NULL, 0, 0, 1, MSG_OK);
        		            }
        		            else
        		            {
					if (strcmp(path, link->path) == 0)
					{
					    status("(MOVE) Can't do that! (press any key)", 1, 1, 1, MSG_ERROR);
					    break;
					}
        			        if (status("(MOVE) Moving file(s) ...  (ENTER to continue, TAB to cancel)", 1, 0, 1, MSG_WARNING) != KEY_ENTER)
                       		            break;
                       		        strcpy(temp, link->path);
                       		        strcat(temp, "/");
                       		        len = strlen(temp);

                       		        while ((entry = getnext()) != -1)
                       		        {
                       		            sprintf(msg, "(MOVE) Moving %-14s", dir_entry[entry].name);
                       		            status(msg, 0, 0, 0, MSG_WARNING);
                       		            strcpy(temp + len, dir_entry[entry].name);
                       		            if (*UserHeartAttack)
                       		            {
                       		                *UserHeartAttack = 0;
                       		                status("Operation aborted. (press any key)", 1, 1, 1, MSG_ERROR);
                       		                break;
                       		            }
                		            if ((err = _move(dir_entry[entry].name, temp)) != FT_OK)
                		            {
	                       		        if (err == FT_CANCEL) break;
             			                sprintf(msg, "(MOVE) Error moving file %s : %s", dir_entry[entry].name, moveerr[err]);
                		                if (status(msg, 1, 1, 0, MSG_ERROR) != KEY_ENTER)
                		                    break;
                		            }
                		            else
                		                dir_entry[entry].selected = 0;
                       		        }
                       		        status(NULL, 0, 0, 1, MSG_OK);
                       		    }
				}
        			if (!link->getdirinfo(link->path))
        			    link->recover();
        			else
        			{
        			    link->current_entry = min(link->current_entry, link->entries - 1);
      				    link->first_on_screen = max(0, link->current_entry - (link->lines - 2) + 1);
        			    link->update();
        			    link->update_info();
                  		    link->update_size();
        			}
        			if (!getdirinfo(path))
        			    recover();
        			else
        			{
        			    current_entry = rename_dir ? getindex(text) : min(current_entry, entries - 1);
      				    first_on_screen = max(0, current_entry - (lines - 2) + 1);
        			    update();
        			    update_info();
                      		    update_size();
                      		}
                       		break;

        case act_UP:		if (current_entry == 0) break;
        			if (current_entry == first_on_screen)
        			{
        			    current_entry--;
        			    first_on_screen--;
        			    update();
        			}
				else
        			    if (current_entry) 
        			    {
        			        old_entry = current_entry--;
        			        update_entry(old_entry);
        			        update_entry(current_entry);
        			    }
	                       	break;

        case act_DOWN:		if (current_entry < entries - 1)
         			    current_entry++;
         			else break;
        			if (current_entry - first_on_screen >= lines - 2) 
				{
				    first_on_screen++;
				    update();         
				    break;
				}
                                update_entry(current_entry - 1);
                                update_entry(current_entry);
                       		break;

        case act_PGUP:		if (current_entry == 0) break;
        			if (current_entry < lines - 2)
        			    current_entry = first_on_screen = 0;
        			else
        			{
        			    current_entry -= lines - 2;
        			    first_on_screen = max(0, first_on_screen - (lines - 2));
        			}
        			update(); 
        			break;

        case act_PGDOWN:	if (current_entry == entries - 1) break;	
        			if (entries - 1 - first_on_screen < lines - 2)
        			    current_entry = entries - 1;
        			else   
        			    if (entries - 1 - current_entry < lines - 2)
        			    {
        			        current_entry = entries - 1;
        			        first_on_screen = entries - 1 - (lines - 2) + 1;
        			    }
        			    else
        			    {
        			        current_entry += lines - 2;
        			        first_on_screen = min(first_on_screen + lines - 2, (entries - 1) - (lines - 2) + 1);
        			    }
        			update();
        			break;
        			
        case act_HOME:		if (current_entry == 0) break;
        			current_entry = first_on_screen = 0;
        			update();
				break;
        			
	case act_END:		if (current_entry == entries - 1) break;
				current_entry = entries - 1;
				first_on_screen = max(0, (entries - 1) - (lines - 2) + 1);
				update();        
				break;
	
	case act_CHDIR:		first_on_screen = current_entry = 0;
    				strcpy(path, (char *)aux_info);
				this->action(act_REFRESH, NULL, NULL);
				break;

	case act_MODE:		mode = (mode + 1) % 3;
				update();
				break;

	case act_SWITCH:	xchg(lines,   link->lines);
                                xchg(columns, link->columns);
                                xchg(begin_x, link->begin_x);
                                xchg(begin_y, link->begin_y);
                                delete win;
                                win = new window(begin_x, begin_y, lines, columns);
                                delete link->win;
				link->win = new window(link->begin_x, link->begin_y, link->lines, link->columns);
				break;
				
	case act_REFRESH:	int verify = aux_info == (void *)-1;
				char oldentry[256];
				strcpy(oldentry, dir_entry[current_entry].name);
				if (!getdirinfo(path, verify))
				    recover();
				else
                                    if (aux_info == (void *)-1)
                                    {
				        current_entry = min(getindex(oldentry), entries - 1);
				        first_on_screen = max(0, current_entry - (lines - 2) + 1);
				    }
				    else
				        current_entry = first_on_screen = 0;
				update_frame();
				update_path();
	 			update_info();
                      		update_size();
	 			update();
	 			if (aux_info == (void *)1)
 			                link->update_size();
				break;

        default: 		panic("no action");
        			break;
    }
    if (done != -1) update_info();
    return done;
}

