/*
 * Edit line file
 * Pink House Systems, October 1993
 */

#include <curses.h>
#include <string.h>
#include "tty.h"
#include "window.h"
#include "edit.h"


static char etemp[256];

#define EDIT_FIELDS	3

static char EditFields[EDIT_FIELDS][40] = 
{
    "CommandLineForeground",
    "CommandLineBackground",
    "CommandLineBrightness"
};

static int EditColors[EDIT_FIELDS] = 
{
    WHITE, BLACK, ON
};

static int& CommandLineForeground	= EditColors[0];
static int& CommandLineBackground	= EditColors[1];
static int& CommandLineBrightness	= EditColors[2];


edit::edit(int _columns, int _begin_y, configuration *config)
{
    char *data;
    int sectionptr, index;

    columns = _columns;
    records = current_record = 0;
    buf         = new char[EDIT_BUFLEN];
    history_buf = new char[EDIT_HISTBUFLEN];
    memset(history_buf, 0, EDIT_HISTBUFLEN);
    reset();
    win = new window(0, _begin_y, 1, _columns);

    if (config->getstatus() == STATUS_OK)
    {
        if ((sectionptr = config->getsectionptr("[Colors]")) != -1)
            for (int i = 0; i < EDIT_FIELDS; i++)
            {
                config->getfielddata(sectionptr, EditFields[i], &data, 1);
                if (!data || (index = tty_getcolorindex(data)) == -1)
                    printf("Invalid %s (%s).\n", EditFields[i], data);
	        else
                    EditColors[i] = index;
            }
    }
} 


void edit::putch(int key)
{
    if (key == KEY_BACKSPACE && index == static_size) return;
    if (key != KEY_BACKSPACE && index == EDIT_BUFLEN - 1)
        { tty_beep(); return; }
        
    if (key < ' ' || key > 0x7F) return;
         
    if (key == KEY_BACKSPACE)
        index--;
    else
        buf[index++] = key;

    buf[index] = 0;
}


void edit::update(void)
{
    tty_colors c;
    
    tty_getcolors(&c);
    tty_bright(CommandLineBrightness);
    tty_foreground(CommandLineForeground);
    tty_background(CommandLineBackground);
    
    memset(etemp, ' ', columns);
    unsigned len = (index >= columns) ? index - columns + 1 : 0;
    memcpy(etemp, buf + static_size + len, index - static_size - len);
    win->cursormove(0, 0);
    win->write(buf, static_size);
    win->write(etemp, columns - static_size);
    tty_setcolors(&c);
}


void edit::setcursor(void)
{
    tty_bright(CommandLineBrightness);
    tty_foreground(CommandLineForeground);
    tty_background(CommandLineBackground);
    win->cursormove(0, strlen(buf));
}


void edit::del(void)
{
    buf[index = static_size] = 0;
    update();
}


void edit::puts(char *str)
{
    strncpy(buf + index, str, EDIT_BUFLEN - static_size - 1);
    index = strlen(buf);
}


char *edit::gets(char *static_text, char *dest, char *default_str)
{
    int key;
    int oldindex = index, oldstatic_size = static_size;
    char oldbuf[EDIT_BUFLEN];
     
    strcpy(oldbuf, buf);
    reset();
    puts(static_text);
    eos();
    if (default_str) puts(default_str);
    update();
    setcursor();
            		     
    while ((key = tty_getkey()) != KEY_ENTER && key != KEY_TAB)
    {
        putch(key);
        update();
	setcursor();
    }
    gettext(dest);

    strcpy(buf, oldbuf);
    index = oldindex;
    static_size = oldstatic_size;
    update();
    setcursor();
    return (key != KEY_TAB) ? dest : NULL;
}


void edit::history(int dir)
{
    switch (dir)
    {
        case EDIT_PREVIOUS:
	     if (current_record) current_record--;
	     strncpy(buf + static_size, history_buf + current_record * EDIT_RECLEN, EDIT_BUFLEN - static_size - 1);
	     index = strlen(buf);
	     update();
             break;
             
        case EDIT_NEXT:
             if (current_record < records)
             {
                 strncpy(buf + static_size, history_buf + ++current_record * EDIT_RECLEN, EDIT_BUFLEN - static_size - 1);
                 index = strlen(buf);
                 update();
             }
             else
                 del();
             break;
             
        case EDIT_RECORD:
             if ((records && strcmp(&history_buf[(records - 1) * EDIT_RECLEN], buf + static_size) == 0) ||
                 buf[static_size] == 0)
             {
                 current_record = records;
                 break;
             }
             if (records == EDIT_HISTRECS)
             {
                 memcpy(history_buf, history_buf + EDIT_RECLEN, EDIT_HISTBUFLEN - EDIT_RECLEN);
                 records--;
             }
             strcpy(history_buf + records * EDIT_RECLEN, buf + static_size);
             current_record = ++records;
             break;
             
        default:
             break;
    }
}
