/* $Id: pop.c,v 1.9 1996/06/26 01:27:17 oh Exp $
 *
 *  Copyright (C) 1996	Seung-Hong Oh
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <pwd.h>

#include "fetchpop.h"
#include "socket.h"
#include "config.h"
#include "pop.h"
#include "retrieve.h"

int pop(struct optrep *options)
{
  int rpy,newmsg;
  int sockfd;
  int howmany, last, index;
  time_t now;
  FILE *logfd = NULL;
  char buf[POPBUFLEN];
  
  if ( options->logging ) 
    if ( (logfd = fopen(options->logfile,"a")) == NULL ) {
      perror("opening logfile");
      fprintf(stderr,"continuing without logging.\n");
      options->logging = 0;
    }

  now = time(NULL);
  if ( (sockfd=get_connect(options->host, POP_PORT)) < 0 )  {
    if ( sockfd == SOCK_NOEXIST ) 
      sprintf(buf,"[-Error-] %s does not exist.\n",options->host);
    else
      sprintf(buf,"[-Error-] %s is not responding at %s",options->host,ctime(&now));
    printmsg(buf,options->logging, options->daemon, logfd, 1, options->whichdeliver==4);
    return DOWN;
  }
  if ( (rpy = get_reply(sockfd, buf) ) ) {
    sprintf(buf,"[-Error-] Problem with host %s. Server closed connection.\n",options->host);
    printmsg(buf, options->logging, options->daemon, logfd, 1, options->whichdeliver == 4);
    close(sockfd);	
    return rpy;
  }

#ifndef LogOnlyNew
  sprintf(buf,"Connected to pop server %s at %s",options->host,ctime(&now));
  printmsg(buf, options->logging, options->daemon, logfd, 0, options->whichdeliver == 4);
#endif

  if ((rpy = pop_auth(sockfd, options->userid, options->password, options->logging, options->daemon, logfd)))  {
    printmsg("[-Error-] Authentication failed. Disconnecting from pop server.\n",options->logging, options->daemon, logfd, 1, options->whichdeliver==4);
    pop_quit(sockfd);
    close(sockfd);
    return rpy;
  }

  if ( (rpy = pop_stat (sockfd, &howmany)) )  {
    printmsg("[-Error-] Getting POP status. Disconnecting from pop server.\n",options->logging, options->daemon, logfd,1, options->whichdeliver==4);
    pop_quit(sockfd);
    close(sockfd);
    return rpy;
  }

  if ( (rpy=pop_last(sockfd,&last)) != 0 ) {
    printmsg("[-Error-] Getting status of LAST read mail. Disconnecting from the pop server.\n",options->logging, options->daemon, logfd,1, options->whichdeliver==4);
    pop_quit(sockfd);
    close(sockfd);
    return rpy;
  }

  newmsg = howmany-last;

#ifndef LogOnlyNew
  sprintf(buf,"There are %d new messages and total of %d messages.\n",newmsg,howmany);
  printmsg(buf,options->logging, options->daemon, logfd, 0, options->whichdeliver == 4);
  rpy = 0;
#else
  if (!options->daemon)
    printf("There are %d new messages and total of %d messages.\n",newmsg,howmany);
#endif
  if ((options->remove == 2) && (last > 0)) {
    sprintf(buf,"[Remove] Removing old messages from  #1 to #%d in the remote server.\n",last);
    printmsg(buf,options->logging,options->daemon,logfd,0, options->whichdeliver == 4);
    for ( index = 1 ; index <= last ; index++ ) {
      if ( ( rpy = pop_dele(sockfd, index)) != 0) {
	printmsg("[-Error-] deleting old messages. Reseting and Disconnecting.\n",options->logging,options->daemon,logfd,1,options->whichdeliver == 4);
	pop_rset(sockfd);
	pop_quit(sockfd);
	close(sockfd);
	return rpy;
      }
      sprintf(buf,"[Remove #%2d] old message #%2d was removed in the remote server.\n",index,index);
      printmsg(buf,options->logging,options->daemon,logfd,0, options->whichdeliver == 4);
    }
  } else if ( (options->remove == 2 ) && ( last <= 0 ) ) 
    printmsg("[-Warning-] You do not have old messages. I won't delete new messages.\n",options->logging,options->daemon,logfd,0, options->whichdeliver == 4);
  else if ( (options->check == 0) && ( newmsg > 0) )
    rpy = fetchmail( options, sockfd, howmany, last, logfd);
  else if ( ( options->check == 2 )&&  ( howmany>0 ) )
    rpy = fetchmail( options, sockfd, howmany, last, logfd);
  else if ( ( options->check == 3 ) && ( howmany>=options->msgnum) )
    rpy = fetchmail( options, sockfd, options->msgnum, options->msgnum-1, logfd);
  else
    ;
  if ( rpy ) 
    return 1;
  if ((options->remove == 3) && (last > 0)) {
    sprintf(buf,"[Remove #%2d] Removing the specific message #%2d in the remote server.\n",options->msgnum2, options->msgnum2);
    printmsg(buf,options->logging,options->daemon,logfd,0, options->whichdeliver == 4);
    if ((rpy = pop_dele(sockfd, options->msgnum2)) != 0) {
      printmsg("[-Error-] deleting old messages. Reseting and Disconnecting.\n",options->logging,options->daemon,logfd,1,options->whichdeliver == 4);
      pop_rset(sockfd);
      pop_quit(sockfd);
      close(sockfd);
      return rpy;
    }
  }

  if ((rpy=pop_quit(sockfd)))
    printmsg("[-Error-] Disconnecting.\n",options->logging, options->daemon, logfd, 1, options->whichdeliver==4);
  
  if (options->logging)
    fclose(logfd);
  close(sockfd);
  return 0;
}


int get_reply(int sockfd, char *argbuf)
{ 
  int rpy;
  
  if (!readsock(sockfd, argbuf, POPBUFLEN)) {
#ifdef DEBUG
    printf("argbuf is %s\n", argbuf);
#endif
    if ( argbuf[0] != '+' && argbuf[0] !='-' )
      return 1;
    
    if (!strncmp(argbuf,"+OK",3))
      rpy = 0;
    else if (!strncmp(argbuf,"-ERR",4))
      rpy= POP_ERROR;
    else
      rpy = WRONG_PROTOCOL;
  } else
    rpy = DOWN;
  return rpy;
}

int pop_auth(int sockfd, char *userid, char *passwd, short int logging, short int daemon, FILE *fd)
{
  int rpy;
  char buf[POPBUFLEN];
  
    sockprintf(sockfd, "USER %s\r\n",userid);
    if ((rpy= get_reply(sockfd, buf)) == 0 ) {
      sockprintf( sockfd, "PASS %s\r\n",passwd);
      rpy= get_reply(sockfd, buf);
    } else 
	;
    if ( rpy ) 
      printmsg(buf,logging,daemon,fd, 0, 0);
    return rpy;
}

int pop_quit (int sockfd)
{
  /*  fd_set rfds;
    struct timeval tv; 

     FD_ZERO(&rfds);
     FD_SET(sockfd, &rfds);
     tv.tv_sec =0;
     tv.tv_usec=0;
     retval = select(10,NULL,&rfds,&rfds,&tv);
     */
    
  if (writeln(sockfd,"QUIT\r\n")) 
    return 1;
  else 
    return 0;
}

int pop_rset (int sockfd)
{
  if (writeln(sockfd,"RSET\r\n")) 
    return 1;
  else 
    return 0;
}

int pop_stat ( int sockfd, int *msgcount)
{
  int rpy;
  char buf[POPBUFLEN];
  
  writeln(sockfd, "STAT\r\n");
  rpy = get_reply(sockfd, buf);
  if (! rpy )
    sscanf(buf,"%*s %d %*d", msgcount);
  
  return rpy;
}

int pop_retr ( int sockfd, int msgnum)
{
  char buf[POPBUFLEN];
  
  sockprintf(sockfd, "RETR %d\r\n", msgnum);
  return(get_reply(sockfd, buf));
}

int pop_dele ( int sockfd, int msgnum)
{
  char buf[POPBUFLEN];
  
  sockprintf(sockfd, "DELE %d\r\n", msgnum);
  return(get_reply(sockfd, buf));
}

int openmbox (signed char deliver, char *output)
{
  int fd;
  struct flock lock;
  
  if (deliver == 4 )
    fd = 1;
  else 
    if ((fd=open(output, O_CREAT | O_WRONLY | O_APPEND, 0600) ) >= 0 ) {
      lock.l_type = F_WRLCK;
      lock.l_start = 0;
      lock.l_whence = SEEK_SET;
      lock.l_len = 0;
      if  (fcntl(fd,F_SETLKW,&lock) < 0 ) { /* No deadlock */
	close(fd);
	fd = -1;
      }
    }
  return fd;
}

int closembox ( int fd )
{
  int err=0;
  
  if ( fd != 1 )
    err = close(fd);
  
  if ( err )
    perror ("closembox");
  return err;
}

int pop_top(int sockfd, int num, int lines)
{
  char buf[POPBUFLEN];
  
  sockprintf(sockfd,"TOP %d %d\r\n", num, lines);
  return get_reply(sockfd,buf);
}

int pop_last(int sockfd, int *lastread)
{
  int rpy;
  char buf[POPBUFLEN];
  
  writeln(sockfd,"LAST\r\n");
  rpy=get_reply(sockfd,buf);
  if (!rpy)
    sscanf(buf,"%*s %d",lastread);
  return rpy;
}

void printmsg(char *msg, signed char logging, signed char daemon, FILE *logfd, unsigned short int err, short int tostdout)
{
  if ( logging == 1) {
    fprintf(logfd,"%s",msg);
    if ( err )
      fclose(logfd);
  }
  else if (tostdout != 0 ) ; 
  /* if mails are directed to standard output, fetchpop should not print anything to stdout besides the fetched mails */
  else if ( !daemon ) {
    if ( err )
      fprintf(stderr,"%s",msg);
    else 
      printf("%s",msg);
  } else
    ;
}
