/*
 * $Id: fetchpop.c,v 1.9 1996/06/26 01:27:09 oh Exp $
 *
 *    Copyright (C) 1996	Seung-Hong Oh
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "config.h"
#include "fetchpop.h"
#include "pop.h"

int readcfg(struct optrep *options, char rcfile[]);
int daemon_init(char *lockfile);
int do_kill(char *lockfile);
int cmdline(int argc, char **argv, struct optrep *options,char *lockfile);
int makecfg(struct optrep options);
void encryptit(char *to, char *from);
int decryptit(char *to, char *from);

#define VERSION "fetchpop 1.9 release"
#define DATE "Jun 1996"

int main(int argc, char** argv)
{
  char localhost[HOSTLEN];
  struct optrep options;
  char *lockfile;
  int prevmask, rpy;
  
  prevmask = umask(0077);

  gethostname(localhost,HOSTLEN);
  if ((lockfile = (char *) malloc( strlen(getenv("HOME")) + strlen("/.lockfetch-") + strlen(localhost)+ 1)) == NULL) {
    fprintf(stderr,"[-Error-] Cannot allocate memory for .lockfetch... Exiting.\n");
    exit(1);
  }
  strcpy(lockfile, getenv("HOME"));
  strcat(lockfile,"/.lockfetch-");
  strcat(lockfile,localhost);
  
  if ( (rpy = cmdline(argc, argv, &options,lockfile)) != 0 ) {
    free(lockfile);
    if (rpy == KILLED )
      exit(0);
    else
      exit(1);
  }
  
  if (makecfg(options) != 0) { 
    fprintf(stderr,"[-Error-] creating config file for you.\nPerhaps you don't have write permission?\n");
    free(lockfile);
    exit(1);
  }
  
  if (readcfg(&options, options.rcfile) != 0) {
    fprintf(stderr,"Run fetchpop again to create the new correct host info file.\n");
    if (options.daemon)
      remove(lockfile); 
    free(lockfile);
    exit(2);
  }
  
  if (options.whichdeliver != 4) {
    fprintf(stderr,"\r%s, by Seung-Hong Oh.\n", VERSION);
    if ((options.whichdeliver==0) || (options.whichdeliver==1))
      fprintf(stderr,"using mailbox folder %s\n",options.output);
#ifdef HAVEMDA
    else if ( options.whichdeliver == 2) 
      fprintf(stderr,"using %s for mail delivery.\n",DefaultMDA);
#endif
#ifdef HAVEPROCMAIL
    else if ( options.whichdeliver == 3)
      fprintf(stderr,"using procmail for mail  delivery.\n");
#endif
    else
      ;
    if ( options.check == 2)
      fprintf(stderr,"Fetching all mail from the remote server.\n");
    else if ( options.check == 1)
      fprintf(stderr,"Checking mail, not fetching.\n");
    else if ( options.check == 3)
      fprintf(stderr,"Fetching message #%d only.\n",options.msgnum);
    else
      ;
    if (options.toplines)
      fprintf(stderr,"Retrieving at most %d lines of each mail message.\n",options.toplines);
    
    if (options.daemon)     
      fprintf(stderr,"Running fetchpop in the background every %d seconds.\r\n", options.sleep_time);
  }
  if (options.daemon) 
    if (daemon_init(lockfile) != 0) {
      free(lockfile);
      exit(1);
    }
  while (1) {
    rpy = pop(&options);
    if ( options.daemon )
      sleep(options.sleep_time);
    else 
      break; 
  }
  free(lockfile);
  umask(prevmask);
  exit(rpy);
}

int cmdline(int argc, char **argv, struct optrep *options, char *lockfile)
{
  int c;
  signed char iserror = 0 ;
  signed char flag1 = 0;
  signed char flag2 = 0;
  int version = 0;
  extern int optind, opterr;
  extern char *optarg;
  FILE *tmpfd;

  bzero(options,sizeof(struct optrep));

  while( !(iserror) && (c=getopt(argc,argv,"kqrdcampvhxbe:t:f:o:g:l:")) != EOF) {
    switch(c) {
    case 'k': case 'q':
      do_kill(lockfile);
      return(KILLED);
      break;
    case 'r':
      options->remove = 1;
      break;
    case 'x':
      if (!flag1 && !flag2) {
	options->remove = 2;
	if (!flag1)
	  ++flag1;
	else
	  ++flag2;
      } else
	iserror++;
      break;
    case 'd':
      options->daemon = 1;
      break;
    case 'c':
      if ( !flag2 ) {
	flag2++;
	options->check = 1;
      } else
	iserror++;
      break;
    case 'a':
      if ( !flag2 ) {
	flag2++;
	options->check = 2;
      } else
	iserror++;
      break;
    case 'b':
      options->hack_addresses = 1;
      break;
#ifdef HAVEMDA
    case 'm':
      if ( !flag1 ) {
	flag1++;
	options->whichdeliver = 2;
      } else
	iserror++;
      break;
#endif
#ifdef HAVEPROCMAIL
    case 'p':
      if (!flag1) {
	flag1++;
	options->whichdeliver = 3;
      } else
	iserror++;
      break;
#endif
    case 'v':
      version = 1;
      break;
    case 'h':
      iserror++;
      break;
    case 'e':
      options->remove = 3;
      options->msgnum2 = atoi(optarg);
      break;
    case 't':
      options->toplines = atoi(optarg);
      if ( options->toplines <= 0 ) {
	fprintf(stderr,"The number must be positive.\n");
	iserror++; 
      }
      break;
    case 'f':
      options->altrc = 1;
      strcpy(options->rcfile,optarg);
      break;
    case 'o':
      if  (!flag1) { 
	strcpy(options->output,optarg);
	if ( (strcmp(options->output,"stdout") == 0 ) ||( (strcmp(options->output,"-") == 0 )))
	  options->whichdeliver = 4;
	else
	  options->whichdeliver = 1;
	flag1++;
      } else {
	fprintf(stderr,"-o can not be used with -m, -p and -c flag.\n");
	iserror++;
      }
      break;
    case 'g':
      if (!flag2) {
	flag2++;
	options->check = 3;
	options->msgnum = atoi(optarg);
	if ((options->msgnum ) < 1 )
	  iserror++;
      } else
	iserror++;
      break;
    case 'l':
      options->logging = 1;
      strcpy(options->logfile,optarg);
      /* fopen with "a" does not create the file on my system. Strange.
         so I create here if the log file does not exist. */
      if ( access(options->logfile,F_OK) != 0 ) {
	if ( (tmpfd=fopen(options->logfile,"w")) != NULL )
	  fclose(tmpfd);
      }
      break;
    default:
      iserror++;
      break;
    }
  }

  if (options->whichdeliver == 0) {
    if (getenv("MAIL") != NULL)
      strcpy(options->output,getenv("MAIL"));
    else {
      strcpy(options->output,MAILSPOOLDIR);
      strcat(options->output,"/");
      strcat(options->output,whoami);
    }
  }
  
  if ( options->altrc == 0 ) {
    strcpy(options->rcfile,getenv("HOME"));
    strcat(options->rcfile,"/.fetchhost");
  }
  
  if ( version ) {
    fprintf(stderr,"FETCHPOP Version and Compile-time Configuration Information:\n\n");
    fprintf(stderr,"\t%s, %s\n",VERSION,DATE);
    fprintf(stderr,"\tCopyrigt (c) 1996 Seung-Hong Oh \n");
    fprintf(stderr,"\t------------------------------------------\n");
    fprintf(stderr,"\t Default deliver: ");
    fprintf(stderr,"Direct modifying mailbox\n");
    fprintf(stderr,"\t MDA (-m flag) support: ");
#ifdef HAVEMDA
    fprintf(stderr,"Use %s\n",DefaultMDA);
#else
    fprintf(stderr,"DISABLED\n");
#endif
    fprintf(stderr,"\t Procmail(-p flag) support: ");
#ifdef HAVEPROCMAIL
    fprintf(stderr,"Use %s\n",FORMAIL);
#else
    fprintf(stderr,"DISABLED\n");
#endif
    fprintf(stderr,"\t Mail spool directory: %s\n",MAILSPOOLDIR);
    fprintf(stderr,"\t Log file: ");
#ifdef LogShowfrom
    fprintf(stderr,"Show From and Subject lines.\n");
#else
    fprintf(stderr,"Don't show from and subject lines.\n");
#endif
#ifdef LogOnlyNew
    fprintf(stderr,"\t           Logging only when there is new mail.\n");
#else
    fprintf(stderr,"\t           Logging all the time.\n");
#endif    
    fprintf(stderr,"\t Minimum sleep second: %s\n",MINSLEEPSECOND);
    return(1);
  }

  if (optind < argc )
    iserror++;
  else
    ;
  
  if (iserror) {
    fprintf(stderr,"usage: fetchpop [-v] [-k | -q] [-d] [-r] [-x | -e msgID] [-c | -a | -g msg ID] [-h] [-t limit] [-f alternative rcfile] [-l log file] [");
#ifdef HAVEMDA
    fprintf(stderr," -m | ");
#endif
#ifdef HAVEPROCMAIL
    fprintf(stderr,"-p | ");
#endif
    fprintf(stderr,"-o user mailfolder]\n");
    fprintf(stderr,"  -v   : show version and compile-time configuration information.\n");
    fprintf(stderr,"  -k,-q: kill the fetchpop daemon process.\n");
    fprintf(stderr,"  -d   : run fetchpop as a daemon process.\n");
    fprintf(stderr,"  -r   : removes the mail messages from remote server after retrieving them.\n");
    fprintf(stderr,"  -x   : delete all the old messages. No retrieval is done.\n");
    fprintf(stderr,"  -e   : delete a specific message.\n");
    fprintf(stderr,"  -c   : check how many messages are in the pop server. No retrueval is done.\n");
    fprintf(stderr,"  -a   : retrieve all the messages, including old mails.\n");
    fprintf(stderr,"  -g   : get a specific message only. Number must be positive.\n");
    fprintf(stderr,"  -t   : retrieve only the given number of lines for each message.\n");
    fprintf(stderr,"  -b   : hack local addresses in From and Cc lines.\n"); 
    fprintf(stderr,"  -f   : specify different config file other than $HOME/.fetchhost.\n");
#ifdef HAVEMDA
    fprintf(stderr,"  -m   : use %s for mail delivery.\n",DefaultMDA);
#endif
#ifdef HAVEPROCMAIL
    fprintf(stderr,"  -p   : use procmail for mail delivery.\n");
#endif
    fprintf(stderr,"  -o   : write the fetched message to user defined mail folder.\n");
    fprintf(stderr,"  -o - : (-o stdout) write the fetched message to standard output device.\n");
    fprintf(stderr,"  -l   : log fetchpop session to log file.\n");
    fprintf(stderr,"  -h   : show usage help.\n");
  }
  return iserror;
}

int readcfg(struct optrep *options, char rcfile[])
{
  FILE *cfg;
  char buffer[MSGBUFLEN];
  struct stat statbuf;
  char host[HOSTLEN];
  char user[USERIDLEN];
  char password[PASSWORDLEN];
  char decrypted[ENCRYPTLEN];

  
  if (lstat(rcfile, &statbuf) < 0 ) {
    fprintf(stderr,"The file %s not found.\n", rcfile);
    return(-1);
  } else
    if (statbuf.st_mode & ~(S_IFREG | S_IRUSR | S_IWUSR )) {
      fprintf(stderr,"%s: file permission is dangerous. Remove %s first.\n",rcfile,rcfile);
      return -1 ;
    }
  
  if ((cfg = fopen(rcfile,"r")) == NULL) {
    fprintf(stderr,"can not open file %s\n", rcfile);
    return -1;
  }
  
  if ((fgets(buffer, MSGBUFLEN, cfg)) == NULL) {
    fprintf(stderr, "\rCannot get content of %s. Remove %s first.\n", rcfile, rcfile);
    fclose(cfg);
    return -1;
  }
  
  if (sscanf(buffer,"%s %s %s %d", host, user, password, &(options->sleep_time)) != 4 ) {
    fclose(cfg);
    fprintf(stderr,"\r%s is in the wrong format. Remove %s first.\n",rcfile,rcfile);
    return -1;
  }

  strcpy(options->host,host);

  decryptit(decrypted,user);
  strcpy(options->userid, decrypted);

  decryptit(decrypted,password);
  strcpy(options->password, decrypted);

  if (options->sleep_time < atoi(MINSLEEPSECOND)) {
    fprintf(stderr,"\r\nsleepsecond %d seconds is BELOW MINSLEEPSECOND.\nContact Adminstrator for this issue.\nAssuming sleepsecond to be %s\n", options->sleep_time,MINSLEEPSECOND);
    options->sleep_time = atoi(MINSLEEPSECOND);
  }

  fclose(cfg);
  return 0;
}

int makecfg(struct optrep options)
{
  FILE *fd;
  char rcfile[PATHLEN];
  char host[HOSTLEN];
  char user[USERIDLEN];
  char password[PASSWORDLEN];
  char sleeptime[32];
  char encrypted[ENCRYPTLEN];
  
  if ( options.altrc ) 
    strcpy(rcfile,options.rcfile);
  else {
    strcpy(rcfile,getenv("HOME"));
    strcat(rcfile,"/.fetchhost");
  }
  
  if (access (rcfile, F_OK)) {
    fprintf(stderr,"The %s file does not exit. I will make it for you.\n",rcfile);
    printf("Enter the pop server address.\nPop3 Host: ");
    fgets(host,HOSTLEN,stdin);
    host[strlen(host)-1] = '\0';
    printf("Enter login name on %s\nUser: ",host);
    fgets(user,USERIDLEN,stdin);
    user[strlen(user)-1] = '\0';
    printf("Enter the password for user %s\nPassword: ",user);
    fgets(password,PASSWORDLEN,stdin);
    password[strlen(password)-1] = '\0';
    printf("Enter time for fetchpop to sleep(>=%s) when running as daemon (-d flag).\nSleep time(in seconds): ",MINSLEEPSECOND);
    fgets(sleeptime,32,stdin);
    
    if (atoi(sleeptime) < atoi(MINSLEEPSECOND)) {
      fprintf(stderr,"\r\nThe sleep time you entered is below MINSLEEPSECOND %s.\nSaving it as %s\n",MINSLEEPSECOND,MINSLEEPSECOND);
      strcpy(sleeptime, MINSLEEPSECOND);
    }

    if (( fd = fopen(rcfile,"w") ) == NULL ) {
      return 2;
    }
    fputs(host,fd); fputc(' ',fd);
    encryptit(encrypted,user); fputs(encrypted,fd); fputc(' ',fd);
    encryptit(encrypted,password); fputs(encrypted,fd); fputc(' ',fd);
    fputs(sleeptime,fd);
    
    fclose(fd);
  }
  return 0;
}

int do_kill(char *lockfile)
{
  pid_t pid;
  FILE* fd;
  
  if ( (fd = fopen(lockfile, "r")) == NULL ) {
    fprintf(stderr,"No daemon is running\n");
    return(KILLED);
  }
  
  fscanf(fd,"%d",&pid);
  fprintf(stderr,"Killing fetchpop at PID %d\n",pid);
  if ( kill(pid,SIGKILL) < 0 )
    fprintf(stderr,"Error killing the process %d\n.",pid);
  else
    fprintf(stderr,"Done. See you later...\n");
  
  fclose(fd);
  remove(lockfile);
  return KILLED;
}

int daemon_init(char *lockfile)
{
  pid_t pid;
  FILE *tmpfd;
  
  if (( pid = fork () ) < 0 )
    return (1);
  else if ( pid >0 )
    exit (0); 
  else
    ;
  
  if ( (pid = setsid()) == -1 ) {
    fprintf(stderr,"\r\nError calling setsid\n");
    return(1);
  }
  
  if (!access(lockfile,F_OK)) {
    tmpfd = fopen(lockfile, "r");
    fscanf(tmpfd,"%d",&pid);
    fprintf(stderr,"\r\nI found that another fetchpop daemon is running at PID %d.\nIf you are sure that this is incorrect, remove the file %s.\n",(int) pid, lockfile);
    return 1;
  }
  if ((tmpfd = fopen(lockfile,"w")) != NULL) {
    fprintf(tmpfd,"%d",pid);
    fclose(tmpfd);
  }
  return 0;
}

/* 
   There are public libraries for encrytion and decrytion. Those are
   too big and too complicated for simple encrytions, though.
   I wrote encryptit and decryptit functions by hacking
   uuencode and uudecode, which are written by Ian Lance Taylor.
   So I believe the credit for these 2 functions should go to him.
   I will implement these functions with my own algorithm sometime later.
   Actually, it is not necessary to encrypt $HOME/.fetchhost file 
   because it already has permission 600, though. 
*/

#define ENC(c) ((c) ? ((c) & 077) + ' ' +2 : '`' + 2)
#define DEC(c) ((((c) - ' ') & 077) -2)

void encryptit(char to[], char from[])
{
  
  int ch,n;
  char *p;
  int index = 0;
  
  bzero(to,ENCRYPTLEN);
  n = strlen(from);
  to[index++] = ENC(n);
  for (p = from; n > 0; n -= 3, p += 3) {
    ch = *p >> 2;
    ch = ENC (ch);
    to[index++] = ch;
    ch = ((*p << 4) & 060) | ((p[1] >> 4) & 017);
    ch = ENC (ch);
    to[index++] = ch;
    ch = ((p[1] << 2) & 074) | ((p[2] >> 6) & 03);
    ch = ENC (ch);
    to[index++] = ch;
    ch = p[2] & 077;
    ch = ENC (ch);    to[index++] = ch;
  }
}

int decryptit(char to[], char from[])
{
  int ch,n;
  char *p;
  int index = 0;
   
  bzero(to,ENCRYPTLEN);
  p = from;
  n = DEC (*p);
  if (n <= 0)
    return 1;
  for (++p; n > 0; p += 4, n -= 3) {
    if (n >= 3) {
      ch = DEC (p[0]) << 2 | DEC (p[1]) >> 4;
      to[index++] = ch;
      ch = DEC (p[1]) << 4 | DEC (p[2]) >> 2;
      to[index++] = ch;
      ch = DEC (p[2]) << 6 | DEC (p[3]);
      to[index++] = ch;
    } else {
      if (n >= 1) {
	ch = DEC (p[0]) << 2 | DEC (p[1]) >> 4;
	to[index++] = ch;
      }
      if (n >= 2) {
	ch = DEC (p[1]) << 4 | DEC (p[2]) >> 2;
	to[index++] = ch;
      }
    }
  }
  return 0;
}

