/*
 * fdfcomm.c
 *
 * common functions for find duplicates program
 *
 * Roy Bixler
 * March 23, 1991
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 1, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * fdfcomm.c
 *
 * common functions for find duplicates program
 */



#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <time.h>
#ifdef unix
#include <unistd.h>
#endif

#include "fdfcomm.h"
#include "elib.h"



FILE_LIST *I_del_menu[N_INTERACTIVE];
CMP_FLIST Sort_criteria[MAX_SORT_CRITERIA] = { NULL, NULL, NULL, NULL };
int Sort_order[MAX_SORT_CRITERIA] = { -1, -1, -1, -1 };
char Match_criteria;
char i_flag = 0;	/* interactive delete mode */
char l_flag = 0;	/* long listing? */
char v_flag = 0;	/* verbose? */
static char Sorted;



/*
 * ins_sort_criteria
 *
 * Insert a sorting function into the global Sort_criteria array.
 */
void ins_sort_criteria(CMP_FLIST cmp_func)

{
    int i;

    for (i=0; i<MAX_SORT_CRITERIA; i++)
	if (Sort_criteria[i] == NULL)  {
	    Sort_criteria[i] = cmp_func;
	    return;
	}
}



/*
 * ins_sort_order
 *
 * Insert a sorting order into the global Sort_order array.
 */
void ins_sort_order(int sort_order)

{
    int i;

    for (i=0; i<MAX_SORT_CRITERIA; i++)
	if (Sort_order[i] < 0)  {
	    Sort_order[i] = sort_order;
	    return;
	}
}



/*
 * has_sort_crit
 *
 * Returns non-zero if one of the sort criteria is the given pointer to
 * a file comparison function.
 */
int has_sort_crit(CMP_FLIST cmp_flist_func)

{
    int i;

    for (i=0; ((i < MAX_SORT_CRITERIA) && (Sort_criteria[i] != NULL)); i++)
	if (cmp_flist_func == Sort_criteria[i])
	    return 1;

    return 0;
}



/*
 * print_fdate
 *
 * given a dta structure, print out the date in it
 */
void print_fdate(struct ffblk *dta)

{
#ifdef MSDOS
    printf("%02d/%02d/%02d",
	   (dta->ff_fdate >> 9) + 80,
	   (dta->ff_fdate >> 5) & 0xf,
	   (dta->ff_fdate) & 0x1f);
#else /* unix */
    struct tm *tmbuf = localtime(&dta->st_buf.st_mtime);

    printf("%02d/%02d/%02d", tmbuf->tm_year, tmbuf->tm_mon+1, tmbuf->tm_mday);
#endif
}



/*
 * print_ftime
 *
 * given a dta structure, print out the time in it
 */
void print_ftime(struct ffblk *dta)

{
#ifdef MSDOS
    printf("%02d:%02d:%02d",
	   (dta->ff_ftime >> 11),
	   (dta->ff_ftime >> 5) & 0x3f,
	   (dta->ff_ftime & 0x1f));
#else /* unix */
    struct tm *tmbuf = localtime(&dta->st_buf.st_mtime);

    printf("%02d:%02d:%02d", tmbuf->tm_hour, tmbuf->tm_min, tmbuf->tm_sec);
#endif
}



/*
 * print_fdatetime
 *
 * print the given dta's date, a space and the dta's time
 */
void print_fdatetime(struct ffblk *dta)

{
    print_fdate(dta);
    printf(" ");
    print_ftime(dta);
}



/*
 * print_fpath
 *
 * given a path and a dta, print the path with the dta's name tacked on the
 * end
 */
void print_fpath(char *path, char *f_name)

{
    static char *dir_sep = PATH_SEPARATOR;
    int path_len;

    if (path != NULL) {
	path_len = strlen(path);
	printf("%s%s%s", path,
	       ((path_len > 0) && (path[path_len-1] == dir_sep[0]))
	         ? ""
	         : PATH_SEPARATOR,
	       f_name);
    }
}



/*
 * long_listing
 *
 * print out the complete information on the given file
 */
void long_listing(char *path, struct ffblk *dta)

{
    if (path != NULL) {
#ifdef MSDOS
	printf("%c ", (dta->ff_attrib & FA_DIREC) ? 'd' : '-');
	print_fdatetime(dta);
	printf(" %8ld ", dta->ff_fsize);
	print_fpath(path, dta->ff_name);
#else /* unix */
	printf("%c ", S_ISDIR(dta->st_buf.st_mode) ? 'd' : '-');
	print_fdatetime(dta);
	printf(" %8ld ", dta->st_buf.st_size);
	print_fpath(path, dta->name);
#endif
	printf("\n");
    }
}



/*
 * print_match_header
 *
 * prints the first match of files
 */
void print_match_header(FILE_LIST *start)

{
	if (l_flag)
		return;	/* no header */
	else if (Match_criteria & NAMES_MATCH)
#ifdef MSDOS
		printf("file %s found in directories:\n", start->dta.ff_name);
#else
		printf("file %s found in directories:\n", start->dta.name);
#endif
	else if (Match_criteria & CONTENTS_MATCH)
#ifdef MSDOS
		printf("files with matching contents (size = %ld) found:\n",
			   start->dta.ff_fsize);
#else
		printf("files with matching contents (size = %ld) found:\n",
			   start->dta.st_buf.st_size);
#endif
	else if (Match_criteria & SIZES_MATCH)
#ifdef MSDOS
		printf("files of length %ld found:\n", start->dta.ff_fsize);
#else
		printf("files of length %ld found:\n",
		       start->dta.st_buf.st_size);
#endif
	else if (Match_criteria & TIMES_MATCH) {
		printf("files with timestamp ");
		print_fdatetime(&start->dta);
		printf(" found:\n");
	}
}



/*
 * print_next_match
 *
 * prints out any matches after the first one
 */
void print_next_match(FILE_LIST *next, int menu_num)

{
    if (menu_num >= 0)
	printf("%d)%s", menu_num, (l_flag) ? " " : "");
    if (l_flag)
	long_listing(next->path, &next->dta);
    else if (Match_criteria & NAMES_MATCH)
	printf("\t%s\n", next->path);
    else
#ifdef MSDOS
	printf("\t%s%s%s\n", next->path, PATH_SEPARATOR, next->dta.ff_name);
#else
	printf("\t%s%s%s\n", next->path, PATH_SEPARATOR, next->dta.name);
#endif

    next->printed = 1;
}

/*
 * print_id_menu
 *
 * given a starting point in the file list, print out an interactive delete
 * menu.
 */
void print_id_menu(FILE_LIST **menu, int num_items)

{
	int i;

	print_match_header(menu[0]);
	for (i = 0; i < num_items; i++)
		print_next_match(menu[i], i+1);
}

int cmp_flist_date(FILE_LIST *fd, char *path, struct ffblk *dta,
		   int sort_order)

{
#ifdef MSDOS
    return (sort_order == DESCENDING)
	     ? -cmpdate(&fd->dta, dta)
	     : cmpdate(&fd->dta, dta);
#else /* unix */
    return (sort_order == DESCENDING)
             ? -cmpdate(fd->dta.st_buf.st_mtime, dta->st_buf.st_mtime)
             : cmpdate(fd->dta.st_buf.st_mtime, dta->st_buf.st_mtime);
#endif
}

int cmp_flist_name(FILE_LIST *fd, char *path, struct ffblk *dta,
		   int sort_order)

{
    int tmp;

#ifdef MSDOS
    if ((tmp = strcmp(fd->dta.ff_name, dta->ff_name)) == 0)
	tmp = strcmp(fd->path, path);
#else /* unix */
    if ((tmp = strcmp(fd->dta.name, dta->name)) == 0)
	tmp = strcmp(fd->path, path);
#endif

   return (sort_order == DESCENDING) ? -tmp : tmp;
}

int cmp_flist_size(FILE_LIST *fd, char *path, struct ffblk *dta,
		   int sort_order)

{
    int ret_val;

#ifdef MSDOS
    ret_val = (fd->dta.ff_fsize == dta->ff_fsize)
	        ? 0
		: ((fd->dta.ff_fsize > dta->ff_fsize) ? 1 : -1);
#else /* unix */
    ret_val = (fd->dta.st_buf.st_size == dta->st_buf.st_size)
               ? 0
               : ((fd->dta.st_buf.st_size > dta->st_buf.st_size) ? 1 : -1);
#endif

   return (sort_order == DESCENDING) ? -ret_val : ret_val;
}

int cmp_flist_time(FILE_LIST *fd, char *path, struct ffblk *dta,
		   int sort_order)

{
#ifdef MSDOS
    return (sort_order == DESCENDING)
	     ? -cmptime(&fd->dta, dta)
	     : cmptime(&fd->dta, dta);
#else /* unix */
    return (sort_order == DESCENDING)
             ? cmptime(fd->dta.st_buf.st_mtime, dta->st_buf.st_mtime)
             : -cmptime(fd->dta.st_buf.st_mtime, dta->st_buf.st_mtime);
#endif
}



/*
 * cmpflist
 *
 * returns non-zero if the FILE_LIST parameter is less/greater than the given
 * dta/path combination according to the sort criteria.  If sort
 * order is ASCENDING,  non-zero returned if FILE_LIST less than dta/path
 * combo, vice-versa for DESCENDING order.
 */
int cmpflist(FILE_LIST *fd, char *path, struct ffblk *dta)

{
    int i, ret_val = 0;

    for (i=0; ((i < MAX_SORT_CRITERIA) && (Sort_criteria[i] != NULL));
	 i++)
	if  ((ret_val = (*Sort_criteria[i])(fd, path, dta, Sort_order[i])) != 0)
	    break;

    return ret_val;
}



/*
 * cmpflist_eq
 *
 * returns non-zero if the two FILE_LIST parameters are equal according to the
 * sort criteria.
 */

int cmpflist_eq(FILE_LIST *fd1, FILE_LIST *fd2)

{
    if (fd1 == NULL)
	return (fd2 == NULL);
    else if (fd2 == NULL)
	return 0;
    else
	return cmpflist(fd1, fd2->path, &fd2->dta);
}



/*
 * files_match
 *
 * return non-zero if the given two files 'match' according to names and the
 * criteria set by the flags (contents, date/time, size or almost any
 * combination of these).
 */
#ifdef FDF
int files_match(FILE_LIST *file1, FILE_LIST *file2)

{
#ifdef MSDOS
    return (((!(Match_criteria & NAMES_MATCH)) ||
	     (!strcmp(file1->dta.ff_name, file2->dta.ff_name))) &&
	    ((!(Match_criteria & SIZES_MATCH)) ||
	     (file1->dta.ff_fsize == file2->dta.ff_fsize)) &&
	    ((!(Match_criteria & TIMES_MATCH)) ||
	     (!cmptime(&file1->dta, &file2->dta))) &&
	    ((!(Match_criteria & CONTENTS_MATCH)) ||
	     (compare_path_name_files(file1->path, file1->dta.ff_name,
				      file2->path, file2->dta.ff_name))));
#else
    return (((!(Match_criteria & NAMES_MATCH)) ||
	     (!strcmp(file1->dta.name, file2->dta.name))) &&
	    ((!(Match_criteria & SIZES_MATCH)) ||
	     (file1->dta.st_buf.st_size == file2->dta.st_buf.st_size)) &&
	    ((!(Match_criteria & TIMES_MATCH)) ||
	     (!cmptime(file1->dta.st_buf.st_mtime,
		       file2->dta.st_buf.st_mtime))) &&
	    ((!(Match_criteria & CONTENTS_MATCH)) ||
	     (compare_path_name_files(file1->path, file1->dta.name,
				      file2->path, file2->dta.name))));
#endif
}
#endif



/*
 * gen_list
 *
 * given a path, generate a list of all files in the path and its
 * subdirectories.
 */
#ifdef MSDOS
void gen_list(FILE_LIST **f_list, char *path, char dirs)

{
    struct ffblk dta;
    char *path_list, *save_path;

    if (path == NULL)
	return;
    path_list = append_dir_to_path(path, ADD_TO_PATH);
    if ((findfirst(path_list, &dta, FA_DIREC) >= 0) &&
	((save_path = malloc((size_t) (strlen(path)+1))) != NULL)) {
	strcpy(save_path, path);
	do {
	    if (dta.ff_attrib & FA_DIREC)
		if ((dirs) && (!is_special(dta.ff_name))) {
		    add_file_to_list(f_list, save_path, &dta);
		    gen_list_of_dir(f_list, path, dta.ff_name, dirs);
		}
		else /* do nothing */;
	    else
		add_file_to_list(f_list, save_path, &dta);
	} while (!findnext(&dta));
    }

    free(path_list);
}
#else /* unix */
void gen_list(FILE_LIST **f_list, char *path, char dirs)

{
    DIR *dirp;
    struct dirent *dentp;
    struct ffblk dta;
    char *save_path;

    if (path == NULL)
	return;
    if ((dirp = opendir(path)) != NULL) {
	chdir(path);
	if ((save_path = strdup(path)) != NULL)
	    while ((dentp = readdir(dirp)) != NULL) {
		dta.name = strdup(dentp->d_name);
		if (stat(dta.name, &dta.st_buf) == 0) {
		    if (S_ISDIR(dta.st_buf.st_mode))
			if ((dirs) && (!is_special(dta.name))) {
			    add_file_to_list(f_list, save_path, &dta);
			    gen_list_of_dir(f_list, path, dta.name, dirs);
			    chdir(path);
			}
			else if (dta.name != NULL)
			    free(dta.name);
			else /* do nothing */;
		    else
			add_file_to_list(f_list, save_path, &dta);
		}
		else {
		    fprintf(stderr, "%s%s%s: ", path, PATH_SEPARATOR, dta.name);
		    perror("cannot 'stat'");
		}
	    }
	closedir(dirp);
    }
}
#endif





/*
 * gen_list_of_dir
 *
 * given a path and a sub-directory, call 'gen_list()' to generate a list
 * of the sub-directory on the path.
 */
void gen_list_of_dir(FILE_LIST **f_list, char *path, char *subdir, char dirs)

{
    char *new_path;

    gen_list(f_list, new_path = append_dir_to_path(path, subdir), dirs);
    if (v_flag > 1) {
	printf("Searching    ");
	print_fpath(path, subdir);
	printf("\n");
    }
    if (new_path != NULL)
	free(new_path);
}



/*
 * add_file_to_list
 *
 * given a path and a file on the path, add it to the list according to the
 * sort criteria.
 */
void add_file_to_list(FILE_LIST **f_list, char *path, struct ffblk *dta)

{
    FILE_LIST *prev = NULL, *new, *cur = *f_list;

    if (Sorted)
	while ((cur != NULL) && (cmpflist(cur, path, dta) < 0)) {
	    prev = cur;
	    cur = cur->next;
	}

    if ((new = malloc((size_t) sizeof(FILE_LIST))) == NULL) {
	printf("add_file_to_list: memory allocation failure\n");
	exit(-1);
    }
    else {
	memcpy(&new->dta, dta, sizeof(struct ffblk));
	new->path = path;
	new->added = 0;
#ifdef MSDOS
	new->printed = 0;
#else
	new->printed = dta->st_buf.st_mode & S_IFDIR;
#endif
	new->next = cur;
	if (prev == NULL)
	    *f_list = new;
	else
	    prev->next = new;
    }
}



/*
 * list_files
 *
 * given a path, generate a linked list of files on the path or (recursively)
 * its subdirectories.  If 'sorted' is non-zero, make sure the list is
 * sorted.
 */
void list_files(FILE_LIST **f_list, char *prog_name, char *path, char dirs,
                char sorted)

{
    struct stat statbuf;
#ifdef MSDOS
    static char target_dir[MAX_STR];
#else
    char *cur_dir;
    long path_max;
#endif

#ifdef MSDOS
    if (just_disk(path)){
	get_cur_path(path, target_dir);
	path = target_dir;
    }
#endif

    if (access(path, 0) == 0) {
	if ((dirs) && (!stat(path, &statbuf)) &&
	    (!(statbuf.st_mode & S_IFDIR))) {
	    printf("%s: %s is not a directory\n", prog_name, path);
	    print_help();
	    exit(-1);
	}
	Sorted = sorted;
#ifdef unix	
	path_max = pathconf(".", _PC_PATH_MAX);
	if ((cur_dir = malloc(path_max+2)) == NULL) {
	    printf("list_files: memory allocation failure\n");
	    exit(-1);
	}
	getcwd(cur_dir, path_max+2);
#endif
	gen_list(f_list, path, dirs);
#ifdef unix
	chdir(cur_dir);
	free(cur_dir);
#endif
	if (v_flag > 1)
	    printf("\n");
    }
    else {
	printf("%s: %s does not exist\n", prog_name, path);
	print_help();
	exit(-1);
    }
}

void free_file_list(FILE_LIST *f_list)

{
    FILE_LIST *next;

    for (; f_list != NULL; f_list = next)  {
	next = f_list->next;
#ifdef unix
	if (f_list->dta.name != NULL)
	    free(f_list->dta.name);
#endif
	if (f_list->path != NULL)
	    free(f_list->path);
	free(f_list);
    }
}


/*
 * print_flist
 *
 * given a pointer to the first entry of a file list, dump the list to standard
 * output
 */
void print_flist(FILE_LIST *flist)

{
    for (;flist != NULL; flist = flist->next)
#ifdef MSDOS
	printf("path = %s\tname = %s\n", flist->path, flist->dta.ff_name);
#else
	printf("path = %s\tname = %s\n", flist->path, flist->dta.name);
#endif
}


#ifdef FDF
/*
 * compare_path_name_files
 *
 * given a pair of files specified by path and name, return non-zero if their
 * contents match, zero if contents don't match.
 */
int compare_path_name_files(char *path1, char *name1, char *path2, char *name2)

{
    int ret_val;
    char *file1;
    char *file2;

    file1 = append_dir_to_path(path1, name1);
    file2 = append_dir_to_path(path2, name2);
    if ((file1 == NULL) || (file2 == NULL)) {
	printf("compare_path_name_files: memory allocation failed!\n");
	exit(-1);
    }

    ret_val = compare_files(file1, file2);

    free(file1);
    free(file2);

    return ret_val;
}
#endif



/*
 * delete_path_name_file
 *
 * given a file specified by path and name, return zero if the file was
 * successfully deleted, non-zero if not.  'force' parameter, if non-zero, will
 * try to change the mode of a file from read-only to delete it.
 */
int delete_path_name_file(char *path, char *f_name, char force)

{
    char *file = append_dir_to_path(path, f_name);
    int ret_val;

    if (file == NULL) {
	printf("delete_path_name_file: memory allocation failed\n");
	exit(-1);
    }

    ret_val = delete_file(file, force);

    free(file);

    return ret_val;
}



#ifdef MSDOS
void get_cur_path(char *new, char *target_dir)
{
    char cur_dir[MAX_STR];

    getcwd(cur_dir, MAX_STR);
    change_disk(new, cur_dir);
    getcwd(target_dir, MAX_STR);
    setdisk(*cur_dir - 'A');
    chdir(cur_dir);
}


void change_disk(char *dir, char *cur_dir)
{
    int maxdrives;

    if (isupper(*dir))
	tolower(*dir);
    maxdrives = setdisk(*dir - 'a');
    if (maxdrives < (*dir - 'a') + 1){
	setdisk(*cur_dir - 'A');
	printf("drive '%c' is not accessable\n", *dir);
	print_help();
	exit(-1);
    }

}

/*
 *	just_disk()
 *
 *	Input:
 *		dir - the directory to be checked
 *	Output:
 *		returns : 
 *			1 - if 'dir' is a current directory (ie 'X:')
 *			0 - otherwise
 */

int just_disk(char *dir)
{
    if ((strlen(dir) == 2) && (*(dir + 1) == ':'))
	return(1);
    else
	return(0);
}
#endif
