/*---------------------------------------------------------------------------
  Name:		dates.c
  Version:	0.7
  Internal:	27
  Date:		27 may 97
  Author:	Marco Rivellino

  Description:	program main file.

		See the README file for a quick overview of the program
		installation and usage.

---------------------------------------------------------------------------*/
/*
    Copyright (C) 1997  Marco Rivellino & Fabio Menegoni

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/* program global constants + definitions */
#include "config.h"

/* date_entry data type */
#include "datentry.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#ifdef __TURBOC__
	/* neded for searchpath() */
#include <dir.h>
#else
	/* needed to get the home dir name & "lock file" handling */
#include <unistd.h>
#include <pwd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <utime.h>
#endif


#ifndef __NOEDIT
#ifndef __TURBOC__
extern	void edit_data_file(char *filename, int edit_mode);
#endif
#endif




/*---------------------------------------------------------------------------*/




/*
 *	Open file
 *
 */
FILE *open_file(char *filename, char *openmode)
{
	FILE *fileptr;

	if ((fileptr = fopen(filename, openmode)) == NULL) {
		fprintf(stderr, "Cannot open file: %s\n", filename);
		perror("fopen");
	}
	return fileptr;
}




/*
 *	If pointed char is a comment one return true
 *	comment chars are:   ;  *  #
 *
 */
int is_comment_char(char *string)
{
	return (*string == ';' || *string == '*' || *string == '#');
}




/*
 *	Skip all space or tab chars of src
 *
 */
char *skip_spaces(char *src)
{
	while (*src == ' ' || *src == '\t')
		src++;
	return src;
}




#ifdef __DEBUG
/*
 *	Skip all non space or tab chars of src
 *
 */
char *skip_non_spaces(char *src)
{
	while (*src != ' ' && *src != '\t')
		src++;
	return src;
}
#endif




/*
 *	strip ending LF from src and return ptr to beginning of src
 *
 */
char *strip_lf(char *src)
{
	char *tmp = src;

	while (*tmp && *tmp != '\n')
		tmp++;
	*tmp = '\0';
	return src;
}




/*
 *	store today's date into 'de'
 *
 */
void store_current_date(struct date_entry *de)
{
	time_t curtime = time(NULL);
	struct tm *loctime = localtime(&curtime);

	de->day = loctime->tm_mday;
	de->month = (loctime->tm_mon) + 1;
	de->year = (loctime->tm_year) + 1900;
}




/*
 *	print comment msg of date "stored" if it falls into the stored->look_distance period.
 *	if stored->look_distance == -1 || force_cli_look_distance	=>	use cli_look_distance given
 *		as an agrument in the command line interface (-d option).
 *	if the comment is print =>	return TRUE
 *				else	return FALSE
 *
 */
int print_message(struct date_entry *current, struct date_entry *stored, int distance, int cli_look_distance, int force_cli_look_distance, int compact_output, int look_mode, int look_select)
{
	char *month_name[] = {"Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"};

	int look_distance = (stored->look_distance != -1 && !force_cli_look_distance ? stored->look_distance : cli_look_distance);

	int print_rest_of_message = FALSE;

	/* stored date will recur in at maximum 'look_distance' days */
	if (
	    (look_select == LOOK_FORWARD || look_select == LOOK_ROUND) && distance >= 0 &&
	    ( (look_mode == LOOK_DIST && distance <= look_distance) ||
	      (look_mode == LOOK_MONTH && stored->month == current->month)
	    )
	   ) {
		if (distance == 0)
			printf("today ");
		if (distance == 1)
			printf("tomorrow ");
		if (compact_output) {
			if (distance >= 2)
				printf("%4d ", distance);
			printf("-> ");
		} else {
			if (distance >= 2)
				printf("in %d days time ", distance);
			if (stored->type == DE_RECU)
				printf("it recurs ");
			else
				printf("it is ");
		}
		print_rest_of_message = TRUE;
	}
	/* stored date has recurred since maximum 'look_distance' days */
	if (
	    (look_select == LOOK_BACKWARD || look_select == LOOK_ROUND) && distance < 0 &&
	    ( (look_mode == LOOK_DIST && distance >= -look_distance) ||
	      (look_mode == LOOK_MONTH && stored->month == current->month)
	    )
	   ) {
		if (distance == -1)
			printf("yesterday ");
		if (compact_output) {
			if (distance <= -2)
				printf("%4d ", distance);
			printf("-> ");
		} else {
			if (distance <= -2)
				printf("%d days ago ", -distance);
			if (stored->type == DE_RECU)
				printf("it recurred ");
			else
				printf("it was ");
		}
		print_rest_of_message = TRUE;
	}

	if (print_rest_of_message) {
		if (stored->year) {				/* absolute recurrence (e.g. birthday, anniversary..) */
			printf("%2d %s %d: %s",
				stored->day,
				month_name[stored->month - 1],
				stored->year,
				stored->comment);
			if (stored->type == DE_RECU)
				printf(" (%d years)\n", current->year - stored->year);
			else
				printf("\n");
		} else {					/* annual recurrence (e.g. Christmas, Workers' day...) */
			printf("%2d %s: %s\n",
				stored->day,
				month_name[stored->month - 1],
				stored->comment);
		}
		return TRUE;
	}
	return FALSE;
}




/*
 *	print the format error message associated to "format_error"
 *
 */
void print_format_error_msg(char *filename, char *format_error_line, int format_error)
{
	fprintf(stderr, "Format error in data file: %s\n"
			"line: %s"
			, filename, format_error_line);
	switch (format_error) {
		case DA_E_DF_BADEOF:	fprintf(stderr, "unespected end of file.\n");
					break;

		case DA_E_DF_NODAY:	fprintf(stderr, "day number is missing.\n");
					break;

		case DA_E_DF_NOTYPE:	fprintf(stderr, "type word is missing.\n");
					break;

		case DA_E_DF_NOLOOKDIST:
					fprintf(stderr, "look distance number is missing.\n");
					break;
	}
}




void print_help_text(void)
{
	printf( "dates v" DA_MAJVERSION "." DA_MINVERSION " Copyright (C)1997  Marco Rivellino & Fabio Menegoni.\n"
#ifdef __DEBUG
		"DEBUG VERSION: "__DATE__"   "__TIME__"\n\n"
#endif
		"This program is distributed under the terms of the GNU GPL 2.0.\n"
		"See the accluded COPYING file for the licence terms and conditions.\n"
		"\n"
		"This program reminds you important dates in the future and the past.\n"
		"\n"
		"Usage:    dates [options] [data_file]\n"
		"\n"
		"options:  -b    backward look mode (show past dates only)\n"
		"          -c    compact output format\n"
		"          -dN   set look distance to N days (default = %d, N = 0-%d)\n"
#ifndef __TURBOC__
#ifndef __NOEDIT
		"          -e    edit data file (all date entries)\n"
		"          -ed   edit data file (dates only)\n"
		"          -ei   edit data file (invalid entries only)\n"
		"          -er   edit data file (recurrencies only)\n"
#endif
#endif
		"          -f    forward look mode (show future dates only)\n"
		"          -h    show this help\n"
#ifndef __TURBOC__
		"          -l    daily lock (show output once a day)\n"
#endif
		"          -m    show this month's dates\n"
		"          -pN   set page length to N lines (default = %d, N >= 2)\n"
		"\n"
		"data_file is the file with dates to remember "
#ifdef __TURBOC__
		"(default = DATES.LST in DOS path).\n"
#else
		"(default = ~/.dates.lst).\n"
#endif
		, DEFAULT_LOOK_DISTANCE, MAX_LOOK_DISTANCE, DEFAULT_PAGE_LENGTH);
}




/*---------------------------------------------------------------------------*/
void main(int argc, char *argv[])
{
	int 	look_select = LOOK_ROUND, 		/* LOOK_FORWARD = print future dates only
							   LOOK_BACKWARD = print past dates only 
							   LOOK_ROUND = print both future+past dates
							*/
		look_mode = LOOK_DIST,			/* LOOK_DIST = print dates within look_distance
							   LOOK_MONTH = print this month's dates
							*/
		compact_output = FALSE,			/* use compact output format */
		filename_arg,
		force_cli_look_distance = FALSE,
		edit_mode = EDIT_NONE,			/* edit mode: EDIT_NONE=normal mode, all other EDIT_*=run edit interface (-e* option) */
		daily_lock = FALSE;			/* daily lock: TRUE=show output once a day */

	int	cli_look_distance = DEFAULT_LOOK_DISTANCE,	/* day distance to look through (backward and forward) */
		page_length = DEFAULT_PAGE_LENGTH;	/* lines to print before pausing */

	FILE	*infile;				/* input file handle */
	char	*filename = NULL;			/* input file name */
	char	*lock_filename = NULL;			/* lock file name */
	char	*home_dir = NULL;			/* user's home dir */

	int	format_error;				/* data file format error flag */
	char	*format_error_line;			/* data file format error line */

	int	curr_arg = 1,				/* argv counter (shell options) */
		distance,				/* distance between stored and current dates */
		printedlines;				/* printed line cnt */

	struct date_entry	stored,			/* date stored in current line of dates.lst */
				current,		/* current date */
				tmpdate;


	if (argc > 1) {
		if (!STRCASECMP(argv[1], "-h")) {
			print_help_text();
			exit(EXIT_SUCCESS);
		}
		filename_arg = FALSE;
		while (curr_arg < argc && !filename_arg) {
			if (argv[curr_arg][0] == '-') {
				switch(argv[curr_arg][1] & 0xDF) {
					case 'B':
						look_select = LOOK_BACKWARD;
						break;

					case 'C':
						compact_output = TRUE;
						break;

					case 'D':
						if (argv[curr_arg][2])
							cli_look_distance = atoi(&argv[curr_arg][2]);
						else
							cli_look_distance = 0;
						if (cli_look_distance < 0)
							cli_look_distance = 0;
						if (cli_look_distance > MAX_LOOK_DISTANCE)
							cli_look_distance = MAX_LOOK_DISTANCE;
						force_cli_look_distance = TRUE;
						break;
#ifndef __TURBOC__
					case 'E':
						switch(argv[curr_arg][2] & 0xDF) {
							case 'D':	edit_mode = EDIT_DATE;
									break;
							case 'I':	edit_mode = EDIT_INVALID;
									break;
							case 'R':	edit_mode = EDIT_RECU;
									break;
							default:	edit_mode = EDIT_ALL;
									break;
						}
						break;
#endif
					case 'F':
						look_select = LOOK_FORWARD;
						break;
#ifndef __TURBOC__
					case 'L':
						daily_lock = TRUE;
						break;
#endif
					case 'M':
						look_mode = LOOK_MONTH;
						break;

					case 'P':
						if (argv[curr_arg][2])
							page_length = atoi(&argv[curr_arg][2]);
						else
							page_length = 2;
						if (page_length < 2)
							page_length = 2;
						break;

					default:
						filename_arg = TRUE;
						curr_arg--;
						break;			/* if no option matches, this is a "-*" filename */
				}
				curr_arg++;
			} else
				filename_arg = TRUE;
		}
	}  /* end of argv[] parsing */


	store_current_date(&current);

#ifndef __TURBOC__
	{						/* Linux: get user's home dir name */
		struct passwd *my_passwd;
		my_passwd = getpwuid(getuid());
		if (!my_passwd) {
			fprintf(stderr, "Couldn't get passwd infos.\n");
			exit(EXIT_FAILURE);
		}
		home_dir = (char *) malloc(strlen(my_passwd->pw_dir) + 1);
		strcpy(home_dir, my_passwd->pw_dir);
	}

	if (daily_lock) {
		struct stat stat_buf;
		struct tm *mod_time;

		lock_filename = (char *) malloc(strlen(home_dir) + strlen("/.dates.lck") + 1);
		strcpy(lock_filename, home_dir);
		strcat(lock_filename, "/.dates.lck");			/* set lock file name to ~/.dates.lck */

		if (stat(lock_filename, &stat_buf) == -1 && errno == ENOENT)
			fclose(fopen(lock_filename, "w"));

		mod_time = localtime(&stat_buf.st_mtime);

		if (current.day == mod_time->tm_mday  &&  current.month == (mod_time->tm_mon) + 1  &&  current.year == (mod_time->tm_year) + 1900)
			exit(EXIT_SUCCESS);
		else
			utime(lock_filename, NULL);		/* update lock-file's mtime with current time */
	}
#endif


	if (curr_arg <= argc - 1) {
		filename = (char *) malloc(strlen(argv[curr_arg]) + 1);
		strcpy(filename, argv[curr_arg]);
	}
	else
#ifdef __TURBOC__
	{
		char *tmpstr = searchpath("dates.lst");
		filename = (char *) malloc(strlen(tmpstr) + 1);
		strcpy(filename, tmpstr);
							/* DOS: search default data file name
							   in DOS path */
	}
#else
	{						/* Linux: set default name of data file to ~/.dates.lst */
		filename = (char *) malloc(strlen(home_dir) + strlen("/.dates.lst") + 1);
		strcpy(filename, home_dir);
		strcat(filename, "/.dates.lst");
	}
#endif

#ifndef __TURBOC__
	if (edit_mode != EDIT_NONE) {
#ifndef __NOEDIT
		edit_data_file(filename, edit_mode);
#endif
		free(filename);
		exit(EXIT_SUCCESS);
	}
#endif

	if ((infile = open_file(filename, "r")) == NULL)
		exit(EXIT_FAILURE);

	/* get stored date from file & compute distance with today's date */
	printedlines = 1;
	while (read_date_entry(&stored, infile, &format_error_line, &format_error) && stored.state != DA_E_DE_INVALID) {
		copy_date_entry(&tmpdate, &stored);
		if (stored.month == 0) {
			stored.month = current.month;
			tmpdate.month = current.month;
			tmpdate.year = current.year;
			distance = day_distance(&current, &tmpdate);
			if (print_message(&current, &stored, distance, cli_look_distance, force_cli_look_distance, compact_output, look_mode, look_select))
				printedlines++;
			else {
				stored.month = current.month + 1;
				tmpdate.month = current.month + 1;
				distance = day_distance(&current, &tmpdate);
				if (print_message(&current, &stored, distance, cli_look_distance, force_cli_look_distance, compact_output, look_mode, look_select))
					printedlines++;
			}
		} else {
			tmpdate.year = current.year;
			distance = day_distance(&current, &tmpdate);
			if (print_message(&current, &stored, distance, cli_look_distance, force_cli_look_distance, compact_output, look_mode, look_select))
				printedlines++;
			else {
				tmpdate.year = current.year + 1;
				distance = day_distance(&current, &tmpdate);
				if (print_message(&current, &stored, distance, cli_look_distance, force_cli_look_distance, compact_output, look_mode, look_select))
					printedlines++;
			}
		}
		if (printedlines % page_length == 0) {
			printf("press ENTER...\n");
			fflush(stdin);
			getchar();
			printedlines = 1;
		}
	} /* while */

	if (format_error != DA_E_DF_OK) {
		print_format_error_msg(filename, format_error_line, format_error);
		exit(EXIT_FAILURE);
	}
	if (stored.state == DA_E_DE_INVALID) {
		fprintf(stderr, "\n**Invalid date entry:\n\n"
				"  %d/%d/%d\n"
				"  %s\n"
				"  %d\n"
				"  %s\n\n"
				"**found in data file: %s\n"
				"**Run \"dates -ei %s\" to correct it.\n\n",
			stored.day, stored.month, stored.year,
			(stored.type == DE_DATE ? "DATE" : "RECU"),
			stored.look_distance,
			stored.comment,
			filename,
			filename);
		exit(EXIT_FAILURE);
	}

	fclose(infile);

	free(filename);
	free(lock_filename);
	free(home_dir);
	exit(EXIT_SUCCESS);
}
