/*---------------------------------------------------------------------*/
/*    Copyright (c) 1993 by Manuel Serrano. All rights reserved.       */
/*                                                                     */
/*                                     ,--^,                           */
/*                               _ ___/ /|/                            */
/*                           ,;'( )__, ) '                             */
/*                          ;;  //   L__.                              */
/*                          '   \    /  '                              */
/*                               ^   ^                                 */
/*                                                                     */
/*                                                                     */
/*    This program is distributed in the hope that it will be useful.  */
/*    Use and copying of this software and preparation of derivative   */
/*    works based upon this software are permitted, so long as the     */
/*    following conditions are met:                                    */
/*           o credit to the authors is acknowledged following         */
/*             current academic behaviour                              */
/*           o no fees or compensation are charged for use, copies,    */
/*             or access to this software                              */
/*           o this copyright notice is included intact.               */
/*      This software is made available AS IS, and no warranty is made */
/*      about the software or its performance.                         */
/*                                                                     */
/*      Bug descriptions, use reports, comments or suggestions are     */
/*      welcome Send them to                                           */
/*        <Manuel.Serrano@inria.fr>                                    */
/*        Manuel Serrano                                               */
/*        INRIA -- Rocquencourt                                        */
/*        Domaine de Voluceau, BP 105                                  */
/*        78153 Le Chesnay Cedex                                       */
/*        France                                                       */
/*---------------------------------------------------------------------*/


/*---------------------------------------------------------------------*/
/*    serrano/prgm/project/bigloo/runtime1.3/Clib/ports.c ...          */
/*                                                                     */
/*    Author      :  Manuel Serrano                                    */
/*    Creation    :  Thu Jul 23 15:34:53 1992                          */
/*    Last change :  Sat Jun 12 12:20:02 1993  (serrano)               */
/*                                                                     */
/*    La manipulation des ports d'entree et de sortie                  */
/*---------------------------------------------------------------------*/
#include <stdio.h>
#include <errno.h>
#include <termio.h>
#if( !defined( sony_news ) )
#   include <unistd.h>
#endif
#include <sys/file.h>
#include <bigloo.h>

/*---------------------------------------------------------------------*/
/*    Des variables globales                                           */
/*---------------------------------------------------------------------*/
obj_t current_output_port, current_input_port, current_error_port;
obj_t default_io_bufsiz;

/*---------------------------------------------------------------------*/
/*     make_output_port ...                                            */
/*---------------------------------------------------------------------*/
obj_t
make_output_port( name, file )
char *name;
FILE *file;
{
   obj_t new_output_port;

   new_output_port = MAKE_ATOMIC_OBJECT( OUTPUT_PORT_SIZE,
                                         HEADER_OUTPUT_PORT );
   
   new_output_port->output_port_t.file = file;
   new_output_port->output_port_t.name = name;
   
   return BREF( new_output_port );
}

/*---------------------------------------------------------------------*/
/*    open_output_file ...                                             */
/*---------------------------------------------------------------------*/
obj_t
open_output_file( name )
obj_t name;
{
   FILE *file;
   
   if( !(file = fopen( STRING( name ).string, "w" )) )
      return BFALSE;

   return make_output_port( STRING( name ).string, file );
}

/*---------------------------------------------------------------------*/
/*    append_output_file ...                                           */
/*---------------------------------------------------------------------*/
obj_t
append_output_file( name )
obj_t name;
{
   FILE *file;
   
   if( !(file = fopen( STRING( name ).string, "w+" )) )
      return BFALSE;

   return make_output_port( STRING( name ).string, file );
}

/*---------------------------------------------------------------------*/
/*    close_output_port ...                                            */
/*---------------------------------------------------------------------*/
obj_t
close_output_port( port )
obj_t port;
{
   fclose( OUTPUT_PORT( port ).file );

   return port;
}

/*---------------------------------------------------------------------*/
/*    open_input_file ...                                              */
/*    -------------------------------------------------------------    */
/*    !!! WARNING !!! WARNING !!! WARNING !!! WARNING !!! WARNING !!!  */
/*    Quand on ouvre un input_port on est oblige, pour des histoires   */
/*    de coherence de remplir le buffer.                               */
/*---------------------------------------------------------------------*/
obj_t
open_input_file( name, bufsiz )
obj_t name, bufsiz;
{
   FILE *file;
   obj_t new_input_port;
   int   buflen;
   int   c_bufsiz = CINT( bufsiz );
   char *buffer;
   
   if( !(file = fopen( STRING( name ).string, "r" )) )
      return BFALSE;
      
   new_input_port = MAKE_ATOMIC_OBJECT( INPUT_PORT_SIZE + c_bufsiz + 1,
                                        HEADER_INPUT_PORT );

   new_input_port->input_port_t.class    = CLASS_FILE;
   new_input_port->input_port_t.file     = file;
   new_input_port->input_port_t.eof      = 0L;
   new_input_port->input_port_t.bufsiz   = ADD_I( bufsiz, BINT( 1 ) );
   new_input_port->input_port_t.backward = BINT( 0 );
   new_input_port->input_port_t.forward  = BINT( 0 );
   new_input_port->input_port_t.remember = BINT( 0 );
   new_input_port->input_port_t.name     = STRING( name ).string;
   
/*--- On remplis le buffer --------------------------------------------*/
   buflen = fread( (char *)&(new_input_port->input_port_t.buffer),
                   1,
                   c_bufsiz,
                   file );

/*--- A-t-on lu la fin du fichier ? -----------------------------------*/
   if( buflen < c_bufsiz )
      new_input_port->input_port_t.eof = BINT( 1 );

/*--- On n'oublie pas de placer le `\0' de fin de buffer  -------------*/
   ((char *)&(new_input_port->input_port_t.buffer))[ buflen ] = '\0';

   return BREF( new_input_port );
}

/*---------------------------------------------------------------------*/
/*    open_input_console ...                                           */
/*---------------------------------------------------------------------*/
obj_t
open_input_console()
{
   obj_t new_input_port;
   int   buflen;
   int   c_bufsiz = CINT( default_io_bufsiz );
   char *buffer;
   
   new_input_port = MAKE_OBJECT( INPUT_PORT_SIZE + c_bufsiz + 1,
                                 HEADER_INPUT_PORT );

   new_input_port->input_port_t.class    = CLASS_CONSOLE;
   new_input_port->input_port_t.file     = stdin;
   new_input_port->input_port_t.eof      = 0L;
   new_input_port->input_port_t.bufsiz   = ADD_I( default_io_bufsiz, BINT( 1 ) );
   new_input_port->input_port_t.backward = default_io_bufsiz;
   new_input_port->input_port_t.forward  = default_io_bufsiz;
   new_input_port->input_port_t.remember = default_io_bufsiz;
   new_input_port->input_port_t.name     = "[stdin]";
   
/*--- On ne remplis pas le buffer, on le vide -------------------------*/
   bzero( (char *)&(new_input_port->input_port_t.buffer), c_bufsiz );

/*--- On vide le buffer de stdin --------------------------------------*/
   fflush( stdin );
   
   return BREF( new_input_port );
}

/*---------------------------------------------------------------------*/
/*    open_input_string ...                                            */
/*---------------------------------------------------------------------*/
obj_t
open_input_string( string )
obj_t string;
{
   obj_t new_input_port;
   int   c_bufsiz = CINT( STRING_LENGTH( string ) );
   char *buffer;

   new_input_port = MAKE_OBJECT( INPUT_PORT_SIZE + c_bufsiz + 1,
                                 HEADER_INPUT_PORT );

   new_input_port->input_port_t.class    = CLASS_STRING;
   new_input_port->input_port_t.file     = 0L;
   new_input_port->input_port_t.eof      = 0L;
   new_input_port->input_port_t.bufsiz   = ADD_I( BINT( c_bufsiz ),
                                                  BINT( 1 ) );
   new_input_port->input_port_t.backward = BINT( 0 );
   new_input_port->input_port_t.forward  = BINT( 0 );
   new_input_port->input_port_t.remember = BINT( 0 );
   new_input_port->input_port_t.name     = "[string]";
    
/*--- On remplis le buffer --------------------------------------------*/
   strcpy( (char *)&(new_input_port->input_port_t.buffer),
           STRING( string ).string );

/*--- On a lu la fin du fichier (bien sur) ! --------------------------*/
   new_input_port->input_port_t.eof = BINT( 1 );

   return BREF( new_input_port );   
}

/*---------------------------------------------------------------------*/
/*    input_port_debug ...                                             */
/*---------------------------------------------------------------------*/
obj_t
input_port_debug( s, f )
obj_t s , f;
{
   FILE *file = OUTPUT_PORT( f ).file;
   
   fprintf( file, "====== input_port_debug ===============================\n" );
   fprintf( file, "\n" );
   fprintf( file, "   backward   : %d '%c'\n",
            CINT( INPUT_PORT( s ).backward ),
            BUFFER( s )[ CINT( INPUT_PORT( s ).backward ) ]);
   fprintf( file, "   forward    : %d '%c'\n",
            CINT( INPUT_PORT( s ).forward ),
            BUFFER( s )[ CINT( INPUT_PORT( s ).forward ) ]);
   fprintf( file, "   remember   : %d '%c'\n",
            CINT( INPUT_PORT( s ).remember ),
            BUFFER( s )[ CINT( INPUT_PORT( s ).remember ) ]);
   fprintf( file, "   mark       : %d '%c'\n",
            CINT( INPUT_PORT( s ).mark ),
            BUFFER( s )[ CINT( INPUT_PORT( s ).mark ) ]);
   fprintf( file, "   eof        : %d\n", INPUT_PORT( s ).eof );
   fprintf( file, "   buffer     : [%s]\n", BUFFER( s ) );
   fprintf( file, "===================================================\n" );

   return s;
}

/*---------------------------------------------------------------------*/
/*    close_input_port ...                                             */
/*---------------------------------------------------------------------*/
obj_t
close_input_port( port )
obj_t port;
{
   if( (INPUT_PORTP( port )) && (INPUT_PORT( port ).class == CLASS_FILE) )
   {
      fclose( INPUT_PORT( port ).file );

      strcpy( BUFFER( port ), "input port closed" );
   }

   return port;
}

/*---------------------------------------------------------------------*/
/*    input_port_fill_buffer ...                                       */
/*    -------------------------------------------------------------    */
/*    Si on a lu un end-of-file, on ne peut rien lire de plus.         */
/*    Idem si le buffer est deja plein : runner == 0                   */
/*    -------------------------------------------------------------    */
/*    S'il est encore possible de lire des choses on commence par      */
/*    reajuster le buffer. C'est a dire le recaler a gauche.           */
/*    -------------------------------------------------------------    */
/*    Quand on tompe sur un end-of-file, on n'a pas besoin de          */
/*    mettre un '\0' dans le buffer car C s'en charge.                 */
/*    -------------------------------------------------------------    */
/*    Voici l'etat du buffer avant de lire quoi que ce soit:           */
/*       +-------------------+-+----------------+-+------------+-+     */
/*       |                   | |                | |            |0|     */
/*       +-------------------+-+----------------+-+------------+-+     */
/*       0               backward              forward      bufsiz     */
/*                                                                     */
/*    Apres le rewind l'etat devient:                                  */
/*       +-+-----------------+-+---------------------------------+     */
/*       | |                 | |           |0|?|?|?|?|?|?|?|?|?|?|     */
/*       +-+-----------------+-+---------------------------------+     */
/*     0 = backward        forward                          bufsiz     */
/*                                                                     */
/*    Et pour finir on a:                                              */
/*       +-+-----------------+-+---------------------------------+     */
/*       | |                 | |                               |0|     */
/*       +-+-----------------+-+---------------------------------+     */
/*     0 = backward        forward                          bufsiz     */
/*---------------------------------------------------------------------*/
unsigned char
input_port_fill_buffer( port )
obj_t port;
{
   int backward = CINT( INPUT_PORT( port ).backward );

/*--- du debug --------------------------------------------------------*/
#ifdef DEBUG_PORT
   puts( "Avant le fill_buffer: " );
   input_port_debug( port, current_output_port );
#endif      
/*---------------------------------------------------------------------*/
/*    On a un traitement specifique a faire suivant qu'on est en       */
/*    train de lire sur la console ou dans un fichier                  */
/*---------------------------------------------------------------------*/
   if( INPUT_PORT( port ).class == CLASS_FILE )
   {
      if( INPUT_PORT( port ).eof || !backward )
         return 0;
      else
      {
         char *buffer = BUFFER( port );
         int   bufsiz = CINT( INPUT_PORT( port ).bufsiz );
         int   buflen;

/*--- On recale le buffer a gauche ------------------------------------*/
         strcpy( buffer, buffer + backward );

/*--- Ou commence et ou termine le buffer actuel ?  -------------------*/
         buflen = bufsiz - backward - 1;

/*--- On lit pour remplir le buffer -----------------------------------*/
         buflen += fread( buffer + buflen, 1, bufsiz - buflen - 1,
                          INPUT_PORT( port ).file );

/*--- A-t-on lu la fin du fichier ? -----------------------------------*/
         if( buflen < (bufsiz - 1) )
            INPUT_PORT( port ).eof = BINT( 1 );

/*--- On n'oublie pas de placer le `\0' de fin de buffer  -------------*/
         buffer[ buflen ] = '\0';
      }
   }
   else
      if( INPUT_PORT( port ).class == CLASS_CONSOLE )
      {
         char *buffer = BUFFER( port );
         int   bufsiz = CINT( INPUT_PORT( port ).bufsiz );
         int   buflen;

         if( INPUT_PORT( port ).eof ||
             (CINT( INPUT_PORT( port ).forward ) == (bufsiz - 1)) )
            return 0;
         else
         {
/*--- On recale le buffer a gauche ------------------------------------*/
            strcpy( buffer, buffer + backward );
            
            buflen = CINT( INPUT_PORT( port ).forward )-backward - 1;
            
            fgets( buffer + buflen,
                   bufsiz - buflen - 1,
                   INPUT_PORT( current_input_port ).file );
            
            if( feof( INPUT_PORT( current_input_port ).file ) )
            INPUT_PORT( port ).eof = BINT( 1 );
         }
      }
      else
         return 0;
         
/*--- Et ziou, c'est fini... ------------------------------------------*/
   INPUT_PORT( port ).forward  = SUB_I_PTAG( SUB_I( INPUT_PORT( port ).forward,
                                             INPUT_PORT( port ).backward ),
                                             PSUB_TAG( 1 ) );
   INPUT_PORT( port ).remember = SUB_I( INPUT_PORT( port ).remember,
                                      INPUT_PORT( port ).backward );
      
   INPUT_PORT( port ).backward = BINT( 0 );
      
/*--- du debug --------------------------------------------------------*/
#ifdef DEBUG_PORT
   puts( "Apres le fill_buffer: " );
   input_port_debug( port, current_output_port );
#endif      

   return 1;
}

/*---------------------------------------------------------------------*/
/*    input_port_get_string ...                                        */
/*    -------------------------------------------------------------    */
/*    Cette fonction construit une chaine `Bigloo' qui est extraite    */
/*    du buffer. La chaine commence en `mark' et fini en `backward'.   */
/*    Pour pouvoir faire un strcpy, on efface momentanement le char    */
/*    suivant `backward', on est donc obliger de le backuper.          */
/*---------------------------------------------------------------------*/
obj_t
input_port_get_string( s )
obj_t s;
{
   char  bck;
   obj_t res;

   bck = BUFFER( s )[ (int)CINT( INPUT_PORT( s ).backward ) ];
   
   BUFFER( s )[ (int)CINT( INPUT_PORT( s ).backward ) ] = '\0';
   res = c_string_to_string( &BUFFER( s )[ CINT( INPUT_PORT( s ).mark ) ] );
   BUFFER( s )[ (int)CINT( INPUT_PORT( s ).backward ) ] = bck;

   return res;
}

/*---------------------------------------------------------------------*/
/*    input_port_get_small_string ...                                  */
/*    -------------------------------------------------------------    */
/*    Cette fonction n'existe que pour le lecteur `Bigloo'. Quand      */
/*    le lecteur lit une chaine "...", il veut pouvoir acces tres      */
/*    facilement au coprs de la chaine, sans les guillemets. Cette     */
/*    fonction se charge de ce travail.                                */
/*    -------------------------------------------------------------    */
/*    Cette fonction construit une chaine `Bigloo' qui est extraite    */
/*    du buffer. La chaine commence en `mark' + 1 et fini en           */
/*    `backward' - 1.                                                  */
/*    Pour pouvoir faire un strcpy, on efface momentanement le char    */
/*    suivant `backward', on est donc obliger de le backuper.          */
/*---------------------------------------------------------------------*/
obj_t
input_port_get_small_string( s )
obj_t s;
{
   char  bck;
   obj_t res;
   
   bck = BUFFER( s )[ (int)CINT( INPUT_PORT( s ).backward ) - 1 ];
   
   BUFFER( s )[ (int)CINT( INPUT_PORT( s ).backward ) - 1 ] = '\0';
   res = c_string_to_string( &BUFFER( s )[ CINT( INPUT_PORT( s ).mark ) + 1 ] );
   BUFFER( s )[ (int)CINT( INPUT_PORT( s ).backward ) - 1 ] = bck;

   return res;
}

/*---------------------------------------------------------------------*/
/*    input_port_get_symbol ...                                        */
/*---------------------------------------------------------------------*/
obj_t
input_port_get_symbol( s )
obj_t s;
{
   char  bck;
   obj_t res;
   
   bck = BUFFER( s )[ (int)CINT( INPUT_PORT( s ).backward ) ];
   
   BUFFER( s )[ (int)CINT( INPUT_PORT( s ).backward ) ] = '\0';
   res = c_string_to_symbol( &BUFFER( s )[ CINT( INPUT_PORT( s ).mark ) ] );
   BUFFER( s )[ (int)CINT( INPUT_PORT( s ).backward ) ] = bck;

   return res;
}

/*---------------------------------------------------------------------*/
/*    intput_port_read_string ...                                      */
/*    -------------------------------------------------------------    */
/*    Cette fonction fonctionne aussi bien pour les `consoles' que pour*/
/*    les fichiers.                                                    */
/*---------------------------------------------------------------------*/
obj_t
intput_port_read_string( port, number )
obj_t port, number;
{
   obj_t res;
   int   len;
   char  aux;
   char *buffer = BUFFER( port );

   /* On verifie que la taille de ce qu'on veut lire n'est pas plus grande */
   /* que la taille des buffers du port.                                   */
   len = GT_I( INPUT_PORT( port ).bufsiz, number ) ?
      CINT( number ) : CINT( INPUT_PORT( port ).bufsiz );

   /* On regarde si on n'a pas deja lu eof sur le port */
   if( INPUT_PORT( port ).eof )
   {
      int buf_len;
      int offset;

      /* Puisqu'on a lu eof, il faut regarder quelle est la taille de la */
      /* chaine contenue dans le buffer.                                 */
      buf_len = strlen( &BUFFER( port )[ CINT( INPUT_PORT( port ).backward ) ] );

      len = buf_len > len ? len : buf_len;

      /* On peut maintenant construire la nouvelle chaine. */
      offset = CINT( ADD_I( INPUT_PORT( port ).backward, BINT( len ) ) );
      
      aux = buffer[ offset ];
      buffer[ offset ] = '\0';
      res = c_string_to_string( &BUFFER( port )[ CINT( INPUT_PORT( port ).backward ) ] );
      buffer[ offset ] = aux;
      
      /* on ajuste les curseurs */
      INPUT_PORT( port ).forward = ADD_I( INPUT_PORT( port ).backward,
                                          BINT( len ) );
   }
   else
      /* on regarde si on a assez de chose dans le buffer */
      if( GT_I( SUB_I( INPUT_PORT( port ).bufsiz,
                       INPUT_PORT( port ).backward ),
                       number ) )
      {
         int offset;

         /* on calcule la chaine resultat */
         offset = CINT( ADD_I( INPUT_PORT( port ).backward, number ) );
         
         aux = buffer[ offset ];
         buffer[ offset ] = '\0';
         res = c_string_to_string( &BUFFER( port )[ CINT( INPUT_PORT( port ).backward ) ] );
         buffer[ offset ] = aux;
      
         /* on ajuste les curseurs */
         INPUT_PORT( port ).forward = ADD_I( INPUT_PORT( port ).backward,
                                             number );
      }
      else
      {
         int len_aux;
         
         /* on commence par remplir le buffer */
         INPUT_PORT( port ).forward = ADD_I_PTAG( INPUT_PORT( port ).forward,
                                                  PSUB_TAG( 1 ) );
         input_port_fill_buffer( port );

         /* On regarde la taille de ce qu'on vient de lire */
         if( (len_aux = strlen( buffer )) > len )
         {
            /* on construit la nouvelle chaine */
            aux = buffer[ len ];
            buffer[ len ] = '\0';
            res = c_string_to_string( BUFFER( port ) );
            buffer[ len ] = aux;
      
            /* on ajuste les curseurs */
            INPUT_PORT( port ).forward = ADD_I( INPUT_PORT( port ).backward,
                                                BINT( len ) );
         }
         else
         {
            res = c_string_to_string( BUFFER( port ) );
      
            /* on ajuste les curseurs */
            INPUT_PORT( port ).forward = ADD_I( INPUT_PORT( port ).backward,
                                                BINT( len_aux ) );
         }   
      }

   INPUT_PORT_REMEMBER_REF( port );
   INPUT_PORT_AJUST_CURSOR( port );

   return res;
}

/*---------------------------------------------------------------------*/
/*    input_port_display_error ...                                     */
/*---------------------------------------------------------------------*/
obj_t
input_port_display_error( ip, op )
obj_t ip, op;
{
   FILE *output        = OUTPUT_PORT( op ).file;
   int  forward        = CINT( INPUT_PORT( ip ).forward ) - 1;
   int  end_of_print   = forward;
   int  start_of_print = (int)CINT( INPUT_PORT( ip ).backward ) - 40;
   int  count;
   char bck;
   
   /* on calcule l'endroit ou on va commencer a afficher */
   if( (int)start_of_print < 0 )
      start_of_print = 0;

   count = end_of_print - start_of_print;

   /* on calcule la chaine et la longueur de ce qu'on va afficher */
   while( (BUFFER( ip )[ end_of_print ] != '\0') &&
          (BUFFER( ip )[ end_of_print ] != '\n') &&
          (count < 80) )
      end_of_print++, count++;

   /* on trippote le buffer */
   bck = BUFFER( ip )[ end_of_print ];
   BUFFER( ip )[ end_of_print ] = '\0';

   /* on affiche la chaine */
   fprintf( output, "%s\n",
            &BUFFER( ip )[ start_of_print ] );
   
   /* on restore le buffer */
   BUFFER( ip )[ end_of_print ] = bck;

   return ip;
}

/*---------------------------------------------------------------------*/
/*     init_io ...                                                     */
/*---------------------------------------------------------------------*/
init_io()
{
#if( !defined( _SBFSIZ ) )
#   define _SBFSIZ 8
#endif

   default_io_bufsiz = BINT( BUFSIZ * _SBFSIZ );
   
   current_output_port = make_output_port( "stdout", stdout );
   current_error_port  = make_output_port( "stderr", stderr );
   current_input_port  = open_input_console();
}

/*---------------------------------------------------------------------*/
/*    fexists ...                                                      */
/*---------------------------------------------------------------------*/
unsigned char
fexists( name )
char *name;
{
   return !access( name, R_OK );
}

