;*---------------------------------------------------------------------*/
;*    Copyright (c) 1993 by Manuel Serrano. All rights reserved.       */
;*                                                                     */
;*                                     ,--^,                           */
;*                               _ ___/ /|/                            */
;*                           ,;'( )__, ) '                             */
;*                          ;;  //   L__.                              */
;*                          '   \    /  '                              */
;*                               ^   ^                                 */
;*                                                                     */
;*                                                                     */
;*    This program is distributed in the hope that it will be useful.  */
;*    Use and copying of this software and preparation of derivative   */
;*    works based upon this software are permitted, so long as the     */
;*    following conditions are met:                                    */
;*           o credit to the authors is acknowledged following         */
;*             current academic behaviour                              */
;*           o no fees or compensation are charged for use, copies,    */
;*             or access to this software                              */
;*           o this copyright notice is included intact.               */
;*      This software is made available AS IS, and no warranty is made */
;*      about the software or its performance.                         */
;*                                                                     */
;*      Bug descriptions, use reports, comments or suggestions are     */
;*      welcome Send them to                                           */
;*        <Manuel.Serrano@inria.fr>                                    */
;*        Manuel Serrano                                               */
;*        INRIA -- Rocquencourt                                        */
;*        Domaine de Voluceau, BP 105                                  */
;*        78153 Le Chesnay Cedex                                       */
;*        France                                                       */
;*---------------------------------------------------------------------*/


;*=====================================================================*/
;*    serrano/prgm/project/bigloo/comptime1.3/0cfa/0cfa.scm ...        */
;*                                                                     */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Wed May  5 15:43:49 1993                          */
;*    Last change :  Wed Jun 23 15:05:53 1993  (serrano)               */
;*                                                                     */
;*    Ce module fait l'analyse de control a proprement parle.          */
;*    On annote l'arbre de syntaxe abstraite.                          */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    Le module                                                        */
;*---------------------------------------------------------------------*/
(module 0cfa_0cfa
   (include "Var/variable.sch"
	    "Tools/trace.sch"
	    "0cfa/0cfa.sch"
	    "0cfa/app.sch")
   (import  tools_speek
	    tools_error
	    tools_shape
	    type_misc)
   (export  (0cfa! tree)
	    (cleanup-variables!)
	    (0cfa-exp! exp)))

;*---------------------------------------------------------------------*/
;*    0cfa! ...                                                        */
;*---------------------------------------------------------------------*/
(define (0cfa! tree)
   ;; on lie toutes les parametres formels de toutes les fonctions
   (for-each (lambda (var)
		(for-each create-variable!
			  (function-args (global-value var))))
	     tree)
   ;; on met a bottom les arguments de toutes les procedures exportees
   (for-each (lambda (var)
		(if (eq? (global-import var) 'export)
		    (for-each (lambda (v)
				 (local-A*-set! v 'bottom))
			      (function-args (global-value var)))))
	     tree)
   ;; on commence l'iteration de point fixe
   (let loop ((continue? *continue-0cfa?*))
      (if continue?
	  (begin
	     (trace 0cfa
		    "*****************************************************"
		    #\newline
		    "Iteration No: " *time-stamp* #\Newline)
	     (set! *continue-0cfa?* #f) 
	     (for-each (lambda (var)
			  (trace (loop 0cfa)
				 ":::::::::::::::::::::::::::::::::::::::::::"
				 #\Newline
				 (shape var) #\Newline)
			  (enter-function (shape var))
			  (0cfa-exp! (function-body (global-value var)))
			  (leave-function))
		       tree)
	     (set! *time-stamp* (+fx 1 *time-stamp*))
	     (loop *continue-0cfa?*))
	  (begin
	     (trace-0cfa)
	     tree))))

;*---------------------------------------------------------------------*/
;*    0cfa-exp! ...                                                    */
;*    -------------------------------------------------------------    */
;*    Cette fonction fait des annotations et retourne une liste de     */
;*    lambda qui constitue l'approximation.                            */
;*---------------------------------------------------------------------*/
(define (0cfa-exp! exp)
   (trace (0cfa loop) "exp: " (shape exp) #\Newline)
   (match-case exp
;*--- atom ------------------------------------------------------------*/
      ((atom ?exp)
       (cond
	  ((local? exp)
	   ;; si le champ `info' d'une variable local n'est pas une
	   ;; instance de _variable c'est que la procedure qui contient
	   ;; cette variable n'a jamais ete evalue donc, il faut retourner
	   ;; bottom
	   (if (not (_variable? (local-info exp)))
	       'bottom
	       (local-A* exp)))
	  ((global? exp)
	   ;; pour le moment toutes les variables globales qui ne sont
	   ;; pas liees a des lambda sont bottom (pbm du (define foo foo))
	   (get-global-A*! exp))
	  (else
	   'bottom)))
;*--- quote -----------------------------------------------------------*/
      ((quote ?exp)
       (cond
	  ((symbol? exp)
	   '(bsymbol))
	  ((pair? exp)
	   '(bpair))
	  ((vector? exp)
	   '(bvector))
	  (else
	   'bottom)))
;*--- function --------------------------------------------------------*/
      ((function ?fun)
       (if (global? fun)
	   (get-global-lambda! fun))
       (list fun))
;*--- failure ---------------------------------------------------------*/
      ((failure . ?-)
       (0cfa-exp*! (cdr exp))
       ;; une erreur ne permet aucune approximation. Surtout pas `bobj'
       '())
;*--- cif -------------------------------------------------------------*/
      ((cif ?si ?alors ?sinon)
       (0cfa-exp! si)
       (union (0cfa-exp! alors) (0cfa-exp! sinon)))
;*--- typed-case ------------------------------------------------------*/
      ((typed-case ?type ?test . ?clauses)
       (0cfa-exp! test)
       (union* (map (lambda (clause) (0cfa-exp! (cadr clause)))
		    clauses)))
;*--- begin -----------------------------------------------------------*/
      ((begin . ?-)
       (0cfa-exp*! (cdr exp)))
;*--- set! ------------------------------------------------------------*/
      ((set! ?var ?val)
       (cond
	  ((local? var)
	   (local-A*-set! var (union (local-A* var) (0cfa-exp! val)))
	   'bottom)
	  ((global? var)
	   (let ((A* (get-global-A*! var)))
	      (global-A*-set! var (union A* (0cfa-exp! val)))
	      'bottom))))
;*--- let -------------------------------------------------------------*/
      ((let ?bindings ?body)
       ;; on cree les liaisons si nessecaires
       (if (not (_variable? (local-info (car (car bindings)))))
	   (for-each (lambda (b) (create-variable! (car b)))
		     bindings))
       ;; on scan toutes les expressions locales
       (map (lambda (b)
	       (mark-local-approx! (car b)
				   (0cfa-exp! (cadr b))))
	    bindings)
       ;; on scan le corps du let
       (0cfa-exp! body))
;*--- labels ----------------------------------------------------------*/
      ((labels ?bindings ?body)
       ;; on cree les liaisons si nessecaires
       (if (not (_lambda? (local-info (car (car bindings)))))
	   (for-each (lambda (b)
			(create-lambda! (car b))
			(for-each create-variable!
				  (function-args (local-value (car b)))))
		     bindings))
       ;; on ne scan pas les corps locaux. Il n'est utile de le faire
       ;; que quand la fonction est appellee.
       ;; on scan le corps du labels
       (0cfa-exp! body))
;*--- block -----------------------------------------------------------*/
      ((block ?var ?body)
       (if (not (_lambda? (local-info var)))
	   (create-lambda! var))
       (0cfa-exp! body))
;*--- return-from -----------------------------------------------------*/
      ((return-from ?app-info ?escape ?value)
       (0cfa-approx-application! app-info 'bottom (list (0cfa-exp! value))))
;*--- apply -----------------------------------------------------------*/
      ((apply ?app-info ?fun . ?args)
       ;; comme on ne rentre pas dans les structures `apply' introduit
       ;; des bottoms
       (mark-bottom*! (list (0cfa-exp! fun)))
       (let loop ((args args)
		  (res '()))
	  (if (null? (cdr args))
	      (let ((app-res (0cfa-unknown-application! app-info 'bottom res)))
		 (app-res*-set! app-info app-res)
		 app-res)
	      (loop (cdr args)
		    (cons (0cfa-exp! (car args)) res)))))
;*--- funcall ---------------------------------------------------------*/
      ((funcall ?app-info ?fun . ?args)
       ;; il ne faut pas oublier de retirer le 0 de marquage de fin
       ;; d'expression
       (let loop ((args args)
		  (res '()))
	  (if (null? (cdr args))
	      (let ((app-res (0cfa-unknown-application! app-info
							fun
							(reverse! res))))
		 (trace 0cfa "*** funcall: " (shape exp) " --> "
			(shape app-res)	#\Newline)
		 (app-res*-set! app-info app-res)
		 app-res)
	      (loop (cdr args)
		    (cons (0cfa-exp! (car args)) res)))))
;*--- application -----------------------------------------------------*/
      ((?fun ?app-info . ?args)
       (let ((v (gensym)))
	  (trace 0cfa "*** app." v ": " (shape exp) #\Newline)
	  (let ((args-approx (map 0cfa-exp! args)))
	     (if (and (not (null? (app-fun* app-info)))
		      (equal? args-approx (app-args* app-info)))
		 (app-res* app-info)
		 (let ((app-res (0cfa-application! fun args-approx)))
		    (trace 0cfa " --> " v ": " (shape app-res) #\newline)
		    (app-res*-set!  app-info app-res)
		    (app-args*-set! app-info args-approx)
		    app-res)))))))

;*---------------------------------------------------------------------*/
;*    0cfa-exp*! ...                                                   */
;*    -------------------------------------------------------------    */
;*    On evalue toutes les expressions et on retourne le resultat de   */
;*    la derniere                                                      */
;*---------------------------------------------------------------------*/
(define (0cfa-exp*! exp)
   (let loop ((exp   exp)
	      (last 'bottom))
      (if (null? exp)
	  last
	  (loop (cdr exp) (0cfa-exp! (car exp))))))

;*---------------------------------------------------------------------*/
;*    0cfa-unknown-application! ...                                    */
;*---------------------------------------------------------------------*/
(define (0cfa-unknown-application! app-info fun args-approx)
   (trace (loop 0cfa) ">-0cfa-unknown-application!: " (shape fun)
	  " " (shape args-approx) #\Newline)
   (let ((fun-approx (cond
			((local? fun)
			 (local-A* fun))
			((global? fun)
			 (get-global-A*! fun)
			 (global-A* fun))
			(else
			 (0cfa-exp! fun))))
	 (args-equal? (equal? args-approx (app-args* app-info))))
      (trace (loop 0cfa) "0cfa( " (shape fun) " ): " (shape fun-approx)
	     #\Newline)
      (cond
	 ((eq? fun-approx 'bottom)
	  (if (eq? (app-fun* app-info) 'bottom)
	      (app-res* app-info)
	      (let ((app-res (0cfa-approx-application! app-info
						       'bottom
						       args-approx)))
		 (app-fun*-set! app-info 'bottom)
		 app-res)))
	 (else
	  (let loop ((fun-approx fun-approx)
		     (app-res    (app-res* app-info)))
	     (cond
		((null? fun-approx)
		 (app-args*-set! app-info args-approx)
		 app-res)
		((and args-equal?
		      (memq (car fun-approx) (app-fun* app-info)))
		 (loop (cdr fun-approx) app-res))
		(else
		 (app-fun*-set! app-info (cons (car fun-approx)
					       (app-fun* app-info)))
		 (loop (cdr fun-approx)
		       (union (0cfa-approx-application! app-info
							(car fun-approx)
							args-approx)
			      app-res)))))))))

;*---------------------------------------------------------------------*/
;*    0cfa-approx-application! ...                                     */
;*---------------------------------------------------------------------*/
(define (0cfa-approx-application! app-info fun-approx args-approx)
   (trace (loop 0cfa) "0cfa-approx-application!: " (shape fun-approx) " "
	  (shape args-approx) #\Newline)
   (cond
      ((eq? fun-approx 'bottom)
       (mark-bottom*! args-approx)
       'bottom)
      ((local? fun-approx)
       (if (eq? (local-class fun-approx) 'function)
	   (0cfa-application! fun-approx args-approx)
	   (0cfa-unknown-application! app-info fun-Approx args-approx)))
      ((global? fun-approx)
       (if (eq? (global-class fun-approx) 'function)
	   (0cfa-application! fun-Approx args-approx)
	   (0cfa-unknown-application! app-info fun-Approx args-approx)))
      (else
       (trace 0cfa
	      "!!! WARNING !!! WARNING !!! WARNING !!! WARNING !!!"
	      #\Newline
	      "Probably illegal application of variable `" fun-Approx "'"
	      #\Newline)
       (warning (current-function)
		"Probably illegal application of `" fun-Approx "'" #\Newline)
       (mark-bottom*! args-approx)
       'bottom)))
       
;*---------------------------------------------------------------------*/
;*    0cfa-application! ...                                            */
;*---------------------------------------------------------------------*/
(define (0cfa-application! fun args-approx)
   (trace (loop 0cfa) "0cfa-application!: " (shape fun) " "
	  (shape args-approx) #\Newline)
   (cond
      ((external-function? fun)
       (mark-bottom*! args-approx)
       'bottom)  
      ((foreign-function? fun)
       (mark-bottom*! args-approx)
       (let ((type-res (cadr (foreign-type (global-value fun)))))
	  (if (or (foreign-type? type-res)
		  (eq? type-res 'bobj))
	      'bottom
	      (list type-res))))
      (else
       (let (formals arity)
	  (cond
	     ((local? fun)
	      (if (function? (local-value fun))
		  (begin
		     (set! arity (function-arity (local-value fun)))
		     (set! formals (function-args (local-value fun))))
		  (begin
		     (set! arity 1)
		     (set! formals (return-args (local-value fun))))))
	     (else
	      (set! arity (function-arity (global-value fun)))
	      (set! formals (function-args (global-value fun)))))
	  (if (not (=fx (length formals)
			(length args-approx)))
	      (cond
		 ((>=fx arity 0)
		  (wrong-number! fun formals args-approx))
		 ((>fx (negfx (+fx arity 1)) (length formals))
		  (wrong-number! fun formals args-approx))
		 (else
		  (let loop ((approx  '())
			     (oapprox args-approx)
			     (arity   arity))
		     (if (=fx arity -1)
			 (begin
			    (approx-formals! fun
					     formals
					     (reverse! (cons '(bpair) approx)))
			    (0cfa-lambda-body! fun))
			 (loop (cons (car oapprox) approx)
			       (cdr oapprox)
			       (+fx arity 1))))))
	      (begin
		 (approx-formals! fun formals args-approx)
		 (0cfa-lambda-body! fun)))))))

;*---------------------------------------------------------------------*/
;*    mark-bottom*! ...                                                */
;*---------------------------------------------------------------------*/
(define (mark-bottom*! approxs)
   (trace (loop 0cfa) "*** mark-bottom*!: " (shape approxs) #\Newline)
   (for-each (lambda (as)
		(if (eq? as 'bottom)
		    'nothing
		    (for-each
		     (lambda (v)
			(trace 0cfa "mark-bottom: " (shape v) #\Newline
			            "       site: "
				    (shape (cond
					      ((local? v)
					       (local-site* v))
					      ((global? v)
					       (global-site* v))
					      (else
					       '())))
				    #\Newline)
			(cond
			   ((local? v)
			    (if (not (eq? (local-site* v) 'bottom))
				(begin
				   (local-site*-set! v 'bottom)
				   (if (eq? (local-class v) 'function)
				       (begin
					  (for-each (lambda (a)
						       (mark-local-approx!
							a 'bottom))
						    (function-args
						     (local-value v)))
					  (0cfa-lambda-body! v)))
				   (continue-0cfa!))))
			   ((global? v)
			    (if (not (eq? (global-site* v) 'bottom))
				(begin
				   (global-site*-set! v 'bottom)
				   (if (eq? (global-class v) 'function)
				       (begin
					  (for-each (lambda (a)
						       (mark-local-approx!
							a 'bottom))
						    (function-args
						     (global-value v)))
					  (0cfa-lambda-body! v)))
				   (continue-0cfa!))))))
		     as)))
	     approxs))

;*---------------------------------------------------------------------*/
;*    wrong-number! ...                                                */
;*---------------------------------------------------------------------*/
(define (wrong-number! fun formals args-approx)
   (trace 0cfa
	  "!!! WARNING !!! WARNING !!! WARNING !!! WARNING !!! WARNING !!!"
	  #\Newline
	  "Probably wrong number of arguments in a call to `" (shape fun) "'"
	  #\Newline
	  "The formals arguments are: " (shape formals) #\Newline
	  "The actuals values are   : " (shape args-approx) #\Newline
	  "in function: " (current-function) #\Newline)
   (warning (current-function)
	    "Probably wrong number of arguments in a call to `"
	    (shape fun) "'"
	    #\Newline
	    "The formals arguments are: " (shape formals) #\Newline
	    "The actuals values are   : " (shape args-approx) #\Newline)
   (mark-bottom*! args-approx)
   (for-each (lambda (f) (mark-local-approx! f 'bottom)) formals)
   'bottom)
   
;*---------------------------------------------------------------------*/
;*    mark-site! ...                                                   */
;*---------------------------------------------------------------------*/
(define (mark-site! a local)
   (cond
      ((local? a)
       (cond
	  ((eq? local 'bottom)
	   (local-site*-set! a 'bottom))
	  ((eq? (local-site* a) 'bottom)
	   'nothing)
	  (else
	   (local-site*-set! a (cons local (local-site* a))))))
      ((global? a)
       (get-global-lambda! a)
       (cond
	  ((eq? local 'bottom)
	   (global-site*-set! a 'bottom))
	  ((eq? (global-site* a) 'bottom)
	   'nothing)
	  (else
	   (global-site*-set! a (cons local (global-site* a))))))))

;*---------------------------------------------------------------------*/
;*    mark-local-approx! ...                                           */
;*---------------------------------------------------------------------*/
(define (mark-local-approx! local approxs)
   (trace (loop 0cfa) "*** mark-local-approx!: " (shape local) " "
	  (shape approxs) #\Newline)
   (if (eq? (local-A* local) 'bottom)
       'done
       (cond
	  ((eq? approxs 'bottom)
	   (if (not (eq? (local-A* local) 'bottom))
	       (begin
		  (local-A*-set! local 'bottom)
		  (continue-0cfa!))))
	  ((eq? (local-A* local) 'bottom)
	   (mark-bottom*! approxs))
	  (else
	   (let ((not-memq (not-memq approxs (local-A* local))))
	      (if (null? not-memq)
		  'done
		  (begin
		     (local-A*-set! local (append not-memq (local-A* local)))
		     (for-each (lambda (a) (mark-site! a local)) not-memq)
		     (continue-0cfa!)
		     'done)))))))

;*---------------------------------------------------------------------*/
;*    approx-formals! ...                                              */
;*---------------------------------------------------------------------*/
(define (approx-formals! fun formals approxs)
   (trace (loop 0cfa)
	  "approx-formals!: " (shape fun) " " (shape approxs) #\Newline)
   (let loop ((formals formals)
	      (approxs approxs))
      (if (null? approxs)
	  'done
	  (let ((formal (car formals))
		(approx (car approxs)))
	     (mark-local-approx! formal approx)
	     (loop (cdr formals)
		   (cdr approxs))))))
					  
;*---------------------------------------------------------------------*/
;*    0cfa-lambda-body! ...                                            */
;*---------------------------------------------------------------------*/
(define (0cfa-lambda-body! fun)
   (trace (loop 0cfa)
	  "0cfa-lambda-body!: " (shape fun) #\Newline)
   (enter-function (shape fun))
   (let ((res (cond
		 ((local? fun)
		  (if (=fx (local-stamp fun) *time-stamp*)
		      (local-body-A* fun)
		      (begin
			 (local-stamp-set! fun *time-stamp*)
			 (let ((body (0cfa-exp!
				      (if (function? (local-value fun))
					  (function-body (local-value fun))
					  (return-body (local-value fun))))))
			    (local-body-A*-set! fun body)
			    body))))
		 (else
		  (get-global-lambda! fun)
		  (if (=fx (global-stamp fun) *time-stamp*)
		      (global-body-A* fun)
		      (begin
			 (global-stamp-set! fun *time-stamp*)
			 (let ((body (0cfa-exp!
				      (function-body (global-value fun)))))
			    (global-body-A*-set! fun body)
			    body)))))))
      (leave-function)
      res))

;*---------------------------------------------------------------------*/
;*    not-memq ...                                                     */
;*    -------------------------------------------------------------    */
;*    On retourne la liste des elements de `el*' qui ne sont pas dans  */
;*    `list'.                                                          */
;*---------------------------------------------------------------------*/
(define (not-memq el* list)
   (let loop ((el* el*)
	      (res '()))
      (cond
	 ((null? el*)
	  res)
	 ((null? (car el*))
	  (loop (cdr el*) res))
	 ((memq (car el*) list)
	  (loop (cdr el*) res))
	 (else
	  (loop (cdr el*) (cons (car el*) res))))))

;*---------------------------------------------------------------------*/
;*      union ...                                                      */
;*    -------------------------------------------------------------    */
;*    On fait une union bestial, peut-etre faudra-t-il ameliorer       */
;*    cette fonction. On verra plus tard.                              */
;*---------------------------------------------------------------------*/
(define (union exp1 exp2)
   (if (or (eq? exp1 'bottom) (eq? exp2 'bottom))
       'bottom
       (let loop ((exp1 exp1)
		  (res  exp2))
	  (cond
	     ((null? exp1)
	      res)
	     ((memq (car exp1) exp2)
	      (loop (cdr exp1) res))
	     (else
	      (loop (cdr exp1) (cons (car exp1) res)))))))

;*---------------------------------------------------------------------*/
;*    union* ...                                                       */
;*---------------------------------------------------------------------*/
(define (union* exps)
   (cond
      ((null? exps)
       '())
      ((null? (cdr exps))
       (car exps))
      (else
       (let loop ((exps (cdr exps))
		  (res  (car exps)))
	  (if (null? exps)
	      res
	      (loop (cdr exps)
		    (union (car exps) res)))))))
	      
;*---------------------------------------------------------------------*/
;*    get-global-A*! ...                                               */
;*---------------------------------------------------------------------*/
(define (get-global-A*! exp)
   (if (_variable? (global-info exp))
       (global-A* exp)
       (let ((new (create-variable! exp)))
	  (_variable-A*-set! new 'bottom)
	  'bottom)))

;*---------------------------------------------------------------------*/
;*    get-global-lambda! ...                                           */
;*---------------------------------------------------------------------*/
(define (get-global-lambda! exp)
   (if (_lambda? (global-info exp))
       (global-info exp)
       (let ((new (create-lambda! exp)))
	  (global-info-set! exp new)
	  (if (not (eq? (global-import exp) 'static))
	      (global-site*-set! exp 'bottom))
	  new))) 

;*---------------------------------------------------------------------*/
;*    external-function? ...                                           */
;*---------------------------------------------------------------------*/
(define (external-function? var)
   (and (global? var)
	(eq? (global-import var) 'import)))

;*---------------------------------------------------------------------*/
;*    foreign-function? ...                                            */
;*---------------------------------------------------------------------*/
(define (foreign-function? var)
   (and (global? var)
	(eq? (global-import var) 'foreign)))
 
;*---------------------------------------------------------------------*/
;*    Des variables pour pouvoir faire des traces ...                  */
;*---------------------------------------------------------------------*/
(define *all-local-variables*  '(dummy))
(define *all-global-variables* '())
(define *all-lambdas*          '(dummy))

;*---------------------------------------------------------------------*/
;*    create-variable! ...                                             */
;*---------------------------------------------------------------------*/
(define (create-variable! var)
   (let ((new (make-_variable)))
      (if (local? var)
	  (begin
	     (local-info-set! var new)
	     (insort! var *all-local-variables*))
	  (begin
	     (global-info-set! var new)
	     (set! *all-global-variables*
		   (cons var *all-global-variables*))))
      new))

;*---------------------------------------------------------------------*/
;*    create-lambda! ...                                               */
;*---------------------------------------------------------------------*/
(define (create-lambda! var)
   (let ((new (make-_lambda)))
      (_lambda-stamp-set! new -1)
      (if (local? var)
	  (begin
	     (local-info-set! var new)
	     (insort! var *all-lambdas*)))
      new))

;*---------------------------------------------------------------------*/
;*    insort! ...                                                      */
;*---------------------------------------------------------------------*/
(define (insort! var list)
   (let ((key (local-key var)))
      (if (null? (cdr list))
	  (set-cdr! list (cons var '()))
	  (let loop ((place list)
		     (tail  (cdr list)))
	     (cond
		((null? tail)
		 (set-cdr! place (cons var '())))
		((>fx (local-key (car tail)) key)
		 (set-cdr! place (cons var tail)))
		(else
		 (loop (cdr place) (cdr tail))))))))

;*---------------------------------------------------------------------*/
;*    *continue-0cfa* ...                                              */
;*---------------------------------------------------------------------*/
(define *continue-0cfa?* #t)
(define *time-stamp*     0)

;*---------------------------------------------------------------------*/
;*    continue-0cfa! ...                                               */
;*---------------------------------------------------------------------*/
(define (continue-0cfa!)
   (set! *continue-0cfa?* #t))

;*---------------------------------------------------------------------*/
;*    trace-0cfa ...                                                   */
;*---------------------------------------------------------------------*/
(define (trace-0cfa)
   (when-trace '(0cfa)
	       (lambda ()
		  (fprint *trace-port* "====== local variables =======")
		  (for-each (lambda (variable)
			       (fprint *trace-port*
				       (shape variable) ": " 
				       (shape (local-A* variable))
				       #\Newline))
			    (cdr *all-local-variables*))
		  (fprint *trace-port* "====== global variables ======")
		  (for-each (lambda (variable)
			       (fprint *trace-port*
				       (shape variable) ": " 
				       (shape (global-A* variable))
				       #\Newline))
			    *all-global-variables*)
		  (fprint *trace-port* "====== lambdas ===============")
		  (for-each (lambda (function)
			       (fprint *trace-port*
				       (shape function) ": " 
				       (if (local? function)
					   (shape (local-site* function))
					   (shape (global-site* function)))
				       #\Newline))
			    (cdr *all-lambdas*)))))
			       

;*---------------------------------------------------------------------*/
;*    cleanup-variables ...                                            */
;*    -------------------------------------------------------------    */
;*    On nettoie tous les champs `info'                                */
;*---------------------------------------------------------------------*/
(define (cleanup-variables!)
   (for-each (lambda (variable)
		(local-info-set! variable '()))
	     (cdr *all-local-variables*))
   (for-each (lambda (variable)
		(global-info-set! variable '()))
	     *all-global-variables*)
   (for-each (lambda (function)
		(if (local? function)
		    (local-info-set! function '())
		    (global-info-set! function '())))
	     (cdr *all-lambdas*)))
			      
