;;; 30 Dec 89, version 2.6
;;;===============================================
;;; The denotational semantics of pattern-matching
;;;    C. Queinnec   Ecole Polytechnique --- INRIA
;;;          queinnec@poly.polytechnique.fr
;;;===============================================
;;; These programs are not written to be efficient, they
;;; only provide canonical semantics and compilation
;;; for pattern matching. A small functional language is also
;;; given and two new forms of function with pattern
;;; matching. Errors are roughly handled. Standardization
;;; of patterns is weak.

;;;===========================================================
;;; Semantics of pattern matching

;;; Domains
;;; e, ee       : Exp
;;; r, rr       : Env   = Id -> Exp + {unbound-pattern}
;;; a           : Seg   = Id -> Exp * Env * Alt -> Ans
;;; m           : Rep   = Id -> Exp * Env * Alt -> Ans
;;; k           : MCont = Exp * Env * Alt -> Ans
;;; z, zz       : Alt   = Unit -> Ans
;;; n           : Id 
;;; f           : Pattern
;;; [[ ]]       : Match-Meaning = Exp * Env * Seg * Rep * MCont * Alt -> Ans

;*---------------------------------------------------------------------*/
;*    Le module                                                        */
;*---------------------------------------------------------------------*/
(herald interprete)

;*---------------------------------------------------------------------*/
;*    Les macros                                                       */
;*---------------------------------------------------------------------*/
(define-local-syntax (repeat n exp wanted)
   `(letrec ((_loop_ (lambda (n)
                        (if (fx= n 1)
                            (if (equal? ,wanted ,exp)
                                (mprint 'ok)
                                (mprint 'error))
                            (begin
                               ,exp
                               (_loop_ (fx- n 1)))))))
       (_loop_ ,n)))

(define-local-syntax (=fx x y)
                `(fx= ,x ,y))

(define-local-syntax (+fx x y)
                `(fx+ ,x ,y))

(define-local-syntax (-fx x y)
                `(fx- ,x ,y))

(define-local-syntax (<fx x y)
                `(fx< ,x ,y))

(define (mprint . v)
   (for-each display v)
   (newline))

;*---------------------------------------------------------------------*/
;*    L'interprete                                                     */
;*---------------------------------------------------------------------*/
(define (wrong m1 m2)
   (display m1)
   (display m2)
   (newline)
   m1)

(define (match f e)
  ((match-meaning f) e r.init a.init m.init 
		     (lambda (e r z) #t)
		     (lambda () #f)))

(define (match-meaning f)
  (case (car f)
    ((*sexp) (match-sexp-meaning))
    ((*quote) (match-quote-meaning (cadr f)))
    ((*or) (match-or-meaning (cadr f) (caddr f)))
    ((*and) (match-and-meaning (cadr f) (caddr f)))
    ((*not) (match-not-meaning (cadr f)))
    ((*setq) (match-setq-meaning (cadr f) (caddr f)))
    ((*eval) (match-eval-meaning (cadr f)))
    ((*cons) (match-cons-meaning (cadr f) (caddr f)))
    ((*ssetq-append) (match-ssetq-append-meaning (cadr f) (caddr f) (cadddr f)))
    ((*eval-append) (match-eval-append-meaning (cadr f) (caddr f)))
    ((*end-ssetq) (match-end-ssetq-meaning (cadr f)))
    ((*times) (match-times-meaning (cadr f) (caddr f) (cadddr f)))
    ((*end-times) (match-end-times-meaning (cadr f)))
    (else (wrong "Unrecognized pattern" f)) ) )

(define (match-sexp-meaning)
  (lambda (e r a m k z) (k e r z)) )

(define (match-quote-meaning ee)
  (lambda (e r a m k z) 
    (if (eq? e ee)
        (k e r z)
        (z) ) ) )

(define (match-or-meaning f1 f2)
  (lambda (e r a m k z)
    ((match-meaning f1)
     e r a m k (lambda () 
                 ((match-meaning f2) 
                  e r a m k z ) ) ) ) )

(define (match-and-meaning f1 f2)
  (lambda (e r a m k z)
    ((match-meaning f1)
     e r a m (lambda (ee rr zz)
               ((match-meaning f2)
                e rr a m k zz ) )
     z ) ) )

(define (match-not-meaning f)
  (lambda (e r a m k z)
    ((match-meaning f)
     e r a m (lambda (ee rr zz) (z)) 
             (lambda () (k e r z)) ) ) )

(define (match-setq-meaning n f)
  (lambda (e r a m k z)
    ((match-meaning f)
     e r a m (lambda (ee rr zz)
               (if (eq? (rr n) unbound-pattern)
                   (k e (extend rr n e) zz)
                   (wrong "Cannot rebind pattern" n) ) )
     z ) ) )

(define (match-eval-meaning n)
  (lambda (e r a m k z)
    (if (eq? (r n) unbound-pattern)
        (wrong "Unbound pattern" n)
        (if (eq? (r n) e)
            (k e r z)
            (z) ) ) ) )

(define (match-cons-meaning f1 f2)
  (lambda (e r a m k z)
    (if (pair? e)
        ((match-meaning f1)
         (car e) r a.init m.init 
         (lambda (ee rr zz)
           ((match-meaning f2)
            (cdr e) rr a m k zz ) )
         z )
        (z) ) ) )

(define (match-ssetq-append-meaning n f1 f2)
  (lambda (e r a m k z)
    ((match-meaning f1)
     e r (extend a.init n 
                 (lambda (ee rr zz)
                    (if (eq? (rr n) unbound-pattern)
                        ((match-meaning f2)
                         ee (extend rr n (cut e ee)) a m k zz ) 
                        (wrong "cannot rebind" n) ) ) )
     m.init (lambda (ee rr zz)
              (wrong "Ssetq not ended" f1) )
     z ) ) )

(define (match-eval-append-meaning n f)
  (lambda (e r a m k z)
    (if (eq? (r n) unbound-pattern)
        (wrong "Unbound segment" n)
        (check e (r n) (lambda (ee)
                         ((match-meaning f)
                          ee r a m k z ) )
                       z ) ) ) )

(define (match-end-ssetq-meaning n)
  (lambda (e r a m k z)
    ((a n) e r z) ) )

;;; corrected version thanks to ML
(define (match-times-meaning n f1 f2)
  (lambda (e r a m k z)
    (letrec ((try (lambda (e r z)
                    ((match-meaning f2)
                     e r a m k
                     (lambda () 
                       ((match-meaning f1)
                        e r a.init 
                            (extend m.init n try)
                            (lambda (ee rr zz)
                              (wrong "Times not ended" f1) )
                            z ) ) ) )))
      (try e r z) ) ) )

(define (match-end-times-meaning n)
  (lambda (e r a m k z)
    ((m n) e r z) ) )

(define (a.init n)
  (lambda (e r z)
    (wrong "No current ssetq for" n) ) )

(define (m.init n)
  (lambda (e r z)
    (wrong "No current repetition named" n) ) )

(define (r.init n)
  unbound-pattern )

(define unbound-pattern '**unbound-pattern**)

(define (check e ee fn z)
  (if (and (pair? e)
           (pair? ee)
           (eq? (car e) (car ee)) )
      (check (cdr e) (cdr ee) fn z)
      (if (null? ee) (fn e) (z)) ) )

(define (extend fn pt im)
  (lambda (x) (if (eq? pt x) im (fn x))) )

(define (cut e ee)
  (if (eq? e ee) '()
      (cons (car e) (cut (cdr e) ee)) ) )

;*---------------------------------------------------------------------*/
;*    Les jeux d'essai                                                 */
;*---------------------------------------------------------------------*/
;;; Le symbol a
(define f0 '(*quote a))

;;; La liste (a)
(define f1 '(*cons (*quote a) (*quote ())))

;;; Une liste de 0 ou + a.
(define f2 '(*times x 
		    (*cons (*quote a) (*end-times x))
		    (*quote ())))

;;; Une liste (a ... b)
(define f3 '(*times x
		    (*cons (*quote a) (*end-times x))
		    (*cons (*quote b) (*quote ()))))

;;; Une liste  de deux elements egaux
(define f4 '(*cons (*setq y (*sexp))
		  (*cons (*eval y) (*quote ()))))

(define (main argv)
   (define (mprint . l)
      (for-each display l)
      (newline))
   (repeat (string->number (cadr argv))
	   (let loop ((i 1000)
		      (l '()))
	      (if (= i 0)
		  (begin
		     (mprint "t run: " l)
		     0)
		  (loop (- i 1)
			(list (match f0 'a)   
			      (match f0 'b)  
			      (match f1 '(a)) 
			      (match f1 '(b))
			      (match f2 '(a a a a))
			      (match f2 '(a b a a))
			      (match f3 '(a a a b))
			      (match f3 '(a a a a))
			      (match f4 '(e e))
			      (match f4 '(e f))))))
	   '(#t #f #t #f #t #f #t #f #t #f)))




	
