/* -------------------------------------------------------------------------
   eval2.c:  Definitions and routines for runtime usage
             (includes routines from original eval.c and new routines)
   ------------------------------------------------------------------------- */

/*
 * Copyright (C) 1986, 1988, 1989, 1991-1995 the Free Software Foundation, Inc.
 *
 * This file is part of GAWK, the GNU implementation of the
 * AWK Progamming Language.
 *
 * GAWK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * GAWK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GAWK; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include "awk.h"

#if defined(MSDOS) && defined(_MSC_VER) && _MSC_VER <= 510
double _msc51bug;        /* to get around a bug in MSC 5.1 */
#endif

NODE *ret_node;

int OFSlen;
int ORSlen;
int OFMTidx;
int CONVFMTidx;

NODE **stack_ptr;

NODE *_t;

/*
 * This table is used by the regexp routines to do case independant
 * matching. Basically, every ascii character maps to itself, except
 * uppercase letters map to lower case ones. This table has 256
 * entries, which may be overkill. Note also that if the system this
 * is compiled on doesn't use 7-bit ascii, casetable[] should not be
 * defined to the linker, so gawk should not load.
 *
 * Do NOT make this array static, it is used in several spots, not
 * just in this file.
 */
#if 'a' == 97	/* it's ascii */
char casetable[] = {
	'\000', '\001', '\002', '\003', '\004', '\005', '\006', '\007',
	'\010', '\011', '\012', '\013', '\014', '\015', '\016', '\017',
	'\020', '\021', '\022', '\023', '\024', '\025', '\026', '\027',
	'\030', '\031', '\032', '\033', '\034', '\035', '\036', '\037',
	/* ' '     '!'     '"'     '#'     '$'     '%'     '&'     ''' */
	'\040', '\041', '\042', '\043', '\044', '\045', '\046', '\047',
	/* '('     ')'     '*'     '+'     ','     '-'     '.'     '/' */
	'\050', '\051', '\052', '\053', '\054', '\055', '\056', '\057',
	/* '0'     '1'     '2'     '3'     '4'     '5'     '6'     '7' */
	'\060', '\061', '\062', '\063', '\064', '\065', '\066', '\067',
	/* '8'     '9'     ':'     ';'     '<'     '='     '>'     '?' */
	'\070', '\071', '\072', '\073', '\074', '\075', '\076', '\077',
	/* '@'     'A'     'B'     'C'     'D'     'E'     'F'     'G' */
	'\100', '\141', '\142', '\143', '\144', '\145', '\146', '\147',
	/* 'H'     'I'     'J'     'K'     'L'     'M'     'N'     'O' */
	'\150', '\151', '\152', '\153', '\154', '\155', '\156', '\157',
	/* 'P'     'Q'     'R'     'S'     'T'     'U'     'V'     'W' */
	'\160', '\161', '\162', '\163', '\164', '\165', '\166', '\167',
	/* 'X'     'Y'     'Z'     '['     '\'     ']'     '^'     '_' */
	'\170', '\171', '\172', '\133', '\134', '\135', '\136', '\137',
	/* '`'     'a'     'b'     'c'     'd'     'e'     'f'     'g' */
	'\140', '\141', '\142', '\143', '\144', '\145', '\146', '\147',
	/* 'h'     'i'     'j'     'k'     'l'     'm'     'n'     'o' */
	'\150', '\151', '\152', '\153', '\154', '\155', '\156', '\157',
	/* 'p'     'q'     'r'     's'     't'     'u'     'v'     'w' */
	'\160', '\161', '\162', '\163', '\164', '\165', '\166', '\167',
	/* 'x'     'y'     'z'     '{'     '|'     '}'     '~' */
	'\170', '\171', '\172', '\173', '\174', '\175', '\176', '\177',
	'\200', '\201', '\202', '\203', '\204', '\205', '\206', '\207',
	'\210', '\211', '\212', '\213', '\214', '\215', '\216', '\217',
	'\220', '\221', '\222', '\223', '\224', '\225', '\226', '\227',
	'\230', '\231', '\232', '\233', '\234', '\235', '\236', '\237',
	'\240', '\241', '\242', '\243', '\244', '\245', '\246', '\247',
	'\250', '\251', '\252', '\253', '\254', '\255', '\256', '\257',
	'\260', '\261', '\262', '\263', '\264', '\265', '\266', '\267',
	'\270', '\271', '\272', '\273', '\274', '\275', '\276', '\277',
	'\300', '\301', '\302', '\303', '\304', '\305', '\306', '\307',
	'\310', '\311', '\312', '\313', '\314', '\315', '\316', '\317',
	'\320', '\321', '\322', '\323', '\324', '\325', '\326', '\327',
	'\330', '\331', '\332', '\333', '\334', '\335', '\336', '\337',
	'\340', '\341', '\342', '\343', '\344', '\345', '\346', '\347',
	'\350', '\351', '\352', '\353', '\354', '\355', '\356', '\357',
	'\360', '\361', '\362', '\363', '\364', '\365', '\366', '\367',
	'\370', '\371', '\372', '\373', '\374', '\375', '\376', '\377',
};
#else
#include "You lose. You will need a translation table for your character set."
#endif


/*
 * compare two nodes, returning negative, 0, positive
 */
int
cmp_nodes(t1, t2)
register NODE *t1, *t2;
{
	register int ret;
	register size_t len1, len2;

	if (t1 == t2)
		return 0;
	if (t1->flags & MAYBE_NUM)
		(void) force_number(t1);
	if (t2->flags & MAYBE_NUM)
		(void) force_number(t2);
	if ((t1->flags & NUMBER) && (t2->flags & NUMBER)) {
		if (t1->numbr == t2->numbr) return 0;
		else if (t1->numbr - t2->numbr < 0)  return -1;
		else return 1;
	}
	(void) force_string2(t1);
	(void) force_string2(t2);
	len1 = t1->stlen;
	len2 = t2->stlen;
	if (len1 == 0 || len2 == 0)
		return len1 - len2;
	ret = memcmp(t1->stptr, t2->stptr, len1 <= len2 ? len1 : len2);
	return ret == 0 ? len1-len2 : ret;
}


/*
 * This returns a POINTER to a node pointer. get_lhs(ptr) is the current
 * value of the var, or where to store the var's new value 
 */

NODE **
r_get_lhs(ptr, assign)
register NODE *ptr;
Func_ptr *assign;
{
	register NODE **aptr = NULL;
	register NODE *n;

	if (ptr->type == Node_param_list)
		ptr = stack_ptr[ptr->param_cnt];

	switch (ptr->type) {
	case Node_var_array:
		fatal("attempt to use array `%s' in a scalar context", ptr->vname);
	case Node_var:
		aptr = &(ptr->var_value);
#ifdef DEBUG
		if ((char)ptr->var_value->stref <= 0)
			cant_happen();
#endif
		break;

	case Node_FIELDWIDTHS:
		aptr = &(FIELDWIDTHS_node->var_value);
		if (assign)
			*assign = set_FIELDWIDTHS;
		break;

	case Node_RS:
		aptr = &(RS_node->var_value);
		if (assign)
			*assign = set_RS;
		break;

	case Node_FS:
		aptr = &(FS_node->var_value);
		if (assign)
			*assign = set_FS;
		break;

	case Node_FNR:
		unref(FNR_node->var_value);
		FNR_node->var_value = make_number((AWKNUM) FNR);
		aptr = &(FNR_node->var_value);
		if (assign)
			*assign = set_FNR;
		break;

	case Node_NR:
		unref(NR_node->var_value);
		NR_node->var_value = make_number((AWKNUM) NR);
		aptr = &(NR_node->var_value);
		if (assign)
			*assign = set_NR;
		break;

	case Node_NF:
		if (NF == -1)
			(void) get_field(HUGE-1, assign); /* parse record */
		unref(NF_node->var_value);
		NF_node->var_value = make_number((AWKNUM) NF);
		aptr = &(NF_node->var_value);
		if (assign)
			*assign = set_NF;
		break;

	case Node_IGNORECASE:
		unref(IGNORECASE_node->var_value);
		IGNORECASE_node->var_value = make_number((AWKNUM) IGNORECASE);
		aptr = &(IGNORECASE_node->var_value);
		if (assign)
			*assign = set_IGNORECASE;
		break;

	case Node_OFMT:
		aptr = &(OFMT_node->var_value);
		if (assign)
			*assign = set_OFMT;
		break;

	case Node_CONVFMT:
		aptr = &(CONVFMT_node->var_value);
		if (assign)
			*assign = set_CONVFMT;
		break;

	case Node_ORS:
		aptr = &(ORS_node->var_value);
		if (assign)
			*assign = set_ORS;
		break;

	case Node_OFS:
		aptr = &(OFS_node->var_value);
		if (assign)
			*assign = set_OFS;
		break;

	case Node_param_list:
		aptr = &(stack_ptr[ptr->param_cnt]->var_value);
		break;

	case Node_field_spec:
		{
		int field_num;

		n = (NODE *) tree_eval(ptr->lnode);
		field_num = (int) force_number(n);
		free_temp(n);
		if (field_num < 0)
			fatal("attempt to access field %d", field_num);
		if (field_num == 0 && field0_valid) {	/* short circuit */
			aptr = &fields_arr[0];
			if (assign)
				*assign = reset_record;
			break;
		}
		aptr = get_field(field_num, assign);
		break;
		}
	case Node_subscript:
		n = ptr->lnode;
		if (n->type == Node_param_list)
			n = stack_ptr[n->param_cnt];
		aptr = assoc_lookup(n, concat_exp(ptr->rnode));
		break;

	case Node_func:
		fatal ("`%s' is a function, assignment is not allowed",
			ptr->lnode->param);
	default:
		cant_happen();
	}
	return aptr;
}




/* A modfied version of r_get_lhs() with all dependencies on syntax tree
   removed.  Used in both assigning to, and accessing value node of various
   types of variables

   Parameters:  vartype   = Node type (variable, field, etc)
                regvar    = Pointer to regular variables 
                assign    = Function pointer to any pre/post assignment routines
                field_num = Field# for any field associated variable operations
                subs_node = String value node for array subscripts

   Returns:     Pointer to Pointer to a value node (of requested variable) 
   
   Usage:       At RUNTIME by converted C program
*/

NODE **
get_lhs2(vartype, regvar, assign, field_num, subs_node)
NODETYPE vartype; 
register NODE *regvar; 
Func_ptr *assign;
int field_num;
register NODE *subs_node;
{
	register NODE **aptr = NULL;

	switch (vartype) {

#ifdef DEBUG
        case Node_var:
                aptr = &(regvar->var_value);
                if ((char)regvar->var_value->stref <= 0)
                  cant_happen();
                break;
#endif                 

        case Node_field_spec:
                {
                if (field_num < 0)
                        fatal("attempt to access field %d", field_num);
                if (field_num == 0 && field0_valid) {   /* short circuit */
                        aptr = &fields_arr[0];
                        if (assign)
                                *assign = reset_record;
                        break;
                }
                aptr = get_field(field_num, assign);
                break;
                }
 
        case Node_subscript:
                aptr = assoc_lookup2(regvar->lnode, subs_node);
                break;
                
	case Node_FIELDWIDTHS:
		aptr = &(FIELDWIDTHS_node->var_value);
		if (assign)
			*assign = set_FIELDWIDTHS;
		break;

	case Node_RS:
		aptr = &(RS_node->var_value);
		if (assign)
			*assign = set_RS;
		break;

	case Node_FS:
		aptr = &(FS_node->var_value);
		if (assign)
			*assign = set_FS;
		break;

	case Node_FNR:
		unref(FNR_node->var_value);
		FNR_node->var_value = make_number((AWKNUM) FNR);
		aptr = &(FNR_node->var_value);
		if (assign)
			*assign = set_FNR;
		break;

	case Node_NR:
		unref(NR_node->var_value);
		NR_node->var_value = make_number((AWKNUM) NR);
		aptr = &(NR_node->var_value);
		if (assign)
			*assign = set_NR;
		break;

	case Node_NF:
		if (NF == -1)
			(void) get_field(HUGE-1, assign); /* parse record */
		unref(NF_node->var_value);
		NF_node->var_value = make_number((AWKNUM) NF);
		aptr = &(NF_node->var_value);
		if (assign)
			*assign = set_NF;
		break;

	case Node_IGNORECASE:
		unref(IGNORECASE_node->var_value);
		IGNORECASE_node->var_value = make_number((AWKNUM) IGNORECASE);
		aptr = &(IGNORECASE_node->var_value);
		if (assign)
			*assign = set_IGNORECASE;
		break;

	case Node_OFMT:
		aptr = &(OFMT_node->var_value);
		if (assign)
			*assign = set_OFMT;
		break;

	case Node_CONVFMT:
		aptr = &(CONVFMT_node->var_value);
		if (assign)
			*assign = set_CONVFMT;
		break;

	case Node_ORS:
		aptr = &(ORS_node->var_value);
		if (assign)
			*assign = set_ORS;
		break;

	case Node_OFS:
		aptr = &(OFS_node->var_value);
		if (assign)
			*assign = set_OFS;
		break;
		
        case Node_var_array:
                fatal("attempt to use array `%s' in a scalar context", regvar->vname);

	default:
		cant_happen();
	}

	return aptr;
}



/* --------------------------------------------------------------------------
   Modified match_op() for use at runtime.  
   (not local like original match_op) 

   Parameters:

        tree = Pointer to regexp node which the parser printed out C code
               for (to create at runtime).

        Following parameters represent the results of tree_eval which 
        match_op() and re_update() used to call

        t1 = String forced value node of left operand in match operators
             ( ~ , !~ )
             (t1 is NULL when parent node is Node_regex and will be assigned
              the value of field $0 here)

        t2 = String forced nodes for certain regexp node fields
             (required by tree_update2)  
        
        Note:  All free_temp's in original match_op and re_update are
               not necessary since tree_eval is not being used
   -------------------------------------------------------------------------- */

AWKNUM match_op(register NODE *tree, register NODE *t1, NODE *t2)
{
	register Regexp *rp;
	int i;
	int match = 1;

	if (tree->type == Node_nomatch)
		match = 0;

        /* Pattern for rule is a sole regular expression => compare with
           with field $0 */
	if (tree->type == Node_regex) 
		t1 = *get_field(0, (Func_ptr *) 0);

	rp = re_update2(tree, t2);
	i = research(rp, t1->stptr, 0, t1->stlen, 0);
	i = (i == -1) ^ (match == 1);
	return (AWKNUM) i;
}


void
set_IGNORECASE()
{
	static int warned = 0;

	if ((do_lint || do_unix) && ! warned) {
		warned = 1;
		warning("IGNORECASE not supported in compatibility mode");
	}
	IGNORECASE = (force_number(IGNORECASE_node->var_value) != 0.0);
	set_FS_if_not_FIELDWIDTHS();
}

void
set_OFS()
{
	OFS = force_string(OFS_node->var_value)->stptr;
	OFSlen = OFS_node->var_value->stlen;
	OFS[OFSlen] = '\0';
}

void
set_ORS()
{
	ORS = force_string(ORS_node->var_value)->stptr;
	ORSlen = ORS_node->var_value->stlen;
	ORS[ORSlen] = '\0';
}

NODE **fmt_list = NULL;
static int fmt_ok P((NODE *n));
static int fmt_index P((NODE *n));

static int
fmt_ok(n)
NODE *n;
{
	/* to be done later */
	return 1;
}

static int
fmt_index(n)
NODE *n;
{
	register int ix = 0;
	static int fmt_num = 4;
	static int fmt_hiwater = 0;

	if (fmt_list == NULL)
		emalloc(fmt_list, NODE **, fmt_num*sizeof(*fmt_list), "fmt_index");
	(void) force_string(n);
	while (ix < fmt_hiwater) {
		if (cmp_nodes(fmt_list[ix], n) == 0)
			return ix;
		ix++;
	}
	/* not found */
	n->stptr[n->stlen] = '\0';
	if (!fmt_ok(n))
		warning("bad FMT specification");
	if (fmt_hiwater >= fmt_num) {
		fmt_num *= 2;
		emalloc(fmt_list, NODE **, fmt_num, "fmt_index");
	}
	fmt_list[fmt_hiwater] = dupnode(n);
	return fmt_hiwater++;
}

void
set_OFMT()
{
	OFMTidx = fmt_index(OFMT_node->var_value);
	OFMT = fmt_list[OFMTidx]->stptr;
}

void
set_CONVFMT()
{
	CONVFMTidx = fmt_index(CONVFMT_node->var_value);
	CONVFMT = fmt_list[CONVFMTidx]->stptr;
}




/* ------------------------------------------------------------------------- 
   Added functions to original eval.c    
   ------------------------------------------------------------------------- */


/* For checking value nodes for boolean equivalent, if it has a current
   number or string value, and if that number is not 0 or if that string
   is not empty, then true 
   For strings, if the length field is 0 => empty */
AWKNUM bool_var( register NODE *var )  {
  if (var->flags & NUM)  
    if (var->numbr)  
      return (AWKNUM) 1;
  if (var->flags & STR)
    if (var->stlen)
      return (AWKNUM) 1;
}


/* --------------------------------------------------------------------------
   Assign value nodes to a variable.  Fashioned after the Node_assign
   handler code in original eval.c.
   Function used (optimal for each kind):
     assign_var_*  : Assigning to regular variables
     assign_var2_* : Assigning to non-regular variables
     *_num         : Assigning number value
     *_var         : Assigning value nodes

   Parameters:  vartype   = Type of variable
                varptr    = Pointer to pointer to regular variable value node
                regvar    = Pointer to regular variable node
                fieldnum  = Field# if vartype is a field variable specification
                subs_node = String value node for array subscripts
                rr        = Pointer to value node
                n         = Number value 
                
   Returns:     Pointer to just assigned value node

   Usage:       At runtime by converted C program
   ------------------------------------------------------------------------- */

NODE *assign_var_var( register NODE **varptr,                        
                      register NODE *rr 
                    )  {      

  /* If value is different, assign a copy of it to this variable 
     (dupnode will only up the reference counter on the string part
      of the src node) */ 
  if ( rr != *varptr )  {
    NODE *save;
    save = *varptr; 
    *varptr = dupnode(rr);
    unref(save); 
  }
    
  return *varptr;
}

NODE *assign_var_num( register NODE **varptr,                      
                      AWKNUM n 
                    )  { 

  /* Optimization: If already a number node, don't malloc, etc */
  if ( (*varptr)->flags == (MALLOC|NUM|NUMBER) ||
       (*varptr)->flags == (MALLOC|NUM|NUMBER|STR) )  {
    (*varptr)->numbr = n;
    if ( (*varptr)->flags & STR )  {
      (*varptr)->flags &= ~STR;
      if ((*varptr)->stref > 1)
        if ((*varptr)->stref != 255)
          (*varptr)->stref--;
      free((*varptr)->stptr);
    }
    return *varptr;
  }

  else  {
    NODE *save;
    save = *varptr;
    *varptr = make_number(n);
    unref(save);
    return *varptr;
  }
}

NODE *assign_var2_var( NODETYPE vartype, 
                       NODE *regvar, 
                       int fieldnum, 
                       NODE *subs_node,
                       register NODE *rr 
                     )  {

  Func_ptr after_assign = NULL;
  register NODE **lhs
    = get_lhs2(vartype, regvar, &after_assign, fieldnum, subs_node);
    
  if ( rr != *lhs )  {
    NODE *save;
    save = *lhs; 
    *lhs = dupnode(rr);
    unref(save); 
  }

  /* This is for calling post-assignment functions after special variables have 
     been assigned to. (i.e. if field $2 was modified, then $0 changes */
  if (after_assign)  (*after_assign)();
  
  return *lhs;
}


NODE *assign_var2_num( NODETYPE vartype, 
                       NODE *regvar, 
                       int fieldnum, 
                       NODE *subs_node,
                       AWKNUM n 
                     )  {

  Func_ptr after_assign = NULL;
  register NODE **tmp = 
    get_lhs2(vartype, regvar, &after_assign, fieldnum, subs_node);

  if ( (*tmp)->flags == (MALLOC|NUM|NUMBER) ||
       (*tmp)->flags == (MALLOC|NUM|NUMBER|STR) )  {
    (*tmp)->numbr = n;
    if ( (*tmp)->flags & STR )  {
      (*tmp)->flags &= ~STR;
      if ((*tmp)->stref > 1)
        if ((*tmp)->stref != 255)
          (*tmp)->stref--;
      free((*tmp)->stptr);
    }
    if (after_assign)  (*after_assign)();
    return *tmp;
  }
    
  else  {
    NODE *save;
    save = *tmp;
    *tmp = make_number(n);
    unref(save);
    if (after_assign)  (*after_assign)();
    return *tmp;
  }
}


/* Used at runtime to setup a regular variable node and it's value node 
   initialized to Nnull_string */
NODE *setup_varnode(void)  {
  register NODE *r;

  getnode(r);
  r->flags = MALLOC;
  r->type = Node_var;
  r->var_value = dupnode(Nnull_string);
  r->rnode = (NODE *) NULL;
  return r;
}


/* Used at runtime to setup a regular expression node
   (The routine is equivalent to the one the parser uses) */
NODE *setup_regnode (register char *regstr, size_t len)  {

  register NODE *n;

  getnode(n);
  n->type = Node_regex;
  n->re_exp = make_string(regstr, len);
  n->re_reg = make_regexp(regstr, len, 0, 1);
  n->re_text = NULL;
  n->re_flags = CONST;
  n->re_cnt = 1;

  return n;

}


/* Similar to above, but for creating regexp nodes on demand based
   on a node value */
NODE *new_regnode (register NODE *regstr, register NODE *n)  {

  size_t len;

  getnode(n);
  n->type = Node_regex;
  len = n->stlen;
  n->re_exp = dupnode(force_string(n));
  n->re_reg = make_regexp(n->stptr, len, 0, 1);
  n->re_text = NULL;
  n->re_flags = CONST;
  n->re_cnt = 1;

  return n;

}


