/*
 *   This program will call someone back on a modem. It requires no special
 *   extra programs (such as special gettys). The user just dials up the
 *   system and logs in as callback.
 *
 *   Account "callback" uses this callback program as a shell. This program
 *   prompts the user for an account name and then hangs up. If the user's
 *   name is found in /etc/callback.dat, the callback program calls the user
 *   back using the telephone number listed in the /etc/callback.dat file.
 * 
 *   Once the user is called back, the user should enter "ATA" to his modem
 *   program. Once the user receives a "CONNECT" message, the user should
 *   hit the "Enter" key. The logon prompt will be displayed.
 *
 *   File: callback.c
 *   Created 04-DEC-1995   Richard B. Johnson
 *   Email                 rjohnson@analogic.com
 *
 *   All rights waived.  There is no implied Copyright nor warranty.
 *   Please send any enhancements to the author at rjohnson@analogic.com.
 */


#include <stdio.h>
#include <unistd.h>
#include <malloc.h>
#include <signal.h>
#include <string.h>
#include <fcntl.h>
#include <time.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/wait.h>
#include <termios.h>

#define BUF_LEN   0x100
#define UNAME_LEN 0x100
#define UTELL_LEN 0x100
#define TTY_LEN   0x100
#define TIM_LEN   0x100
#define TOTAL_LEN (BUF_LEN + UNAME_LEN + UTELL_LEN + TTY_LEN + TIM_LEN)
#define WAIT_TIME 30
#define CONNECT_TIME 60
#define RESPONSE_TIME 5

const char pathname[]="/etc/callback.dat";
const char logf[]="/var/adm/callback.log";
const char crlf[]="\r\n";
const char atdt[]="ATDT";
const char hnow[]="HANGUP NOW";
const char name[]="NAMES";
const char prmt[]="Username: ";
const char welc[]="Welcome back!\n";
const char inst[]="Callback program not installed properly!\n";
char *login[]={ "/bin/login", NULL };
int clean(char *);

void catcher(int unused)
{
    signal(SIGALRM, catcher);
    alarm(1);
}

int main(void);
int main()
{
    struct termio io;
    char *username;
    char *phone;
    FILE *file;
    FILE *log;
    char *buffer;
    char *tty_name;
    char *tim_buf;
    char *probe;
    struct tm *tmnow;
    time_t tnow;
    int tt, speed;
    int i, len;

    tzset();
    if((setuid(0)) != 0)
        (void)fprintf(stderr, inst); 
    if((setgid(0)) != 0)
        (void)fprintf(stderr, inst);
    signal(SIGHUP,  SIG_IGN);                       /* Trap Hangup signal */
    if((file = fopen(pathname, "r")) == NULL)
    {
        (void)fprintf(stderr,"Can't open callback file!\n");
        return ENOENT;
    }
    if((buffer = (char *) malloc(TOTAL_LEN * sizeof(char))) == NULL)
    {
        (void)fprintf(stderr, "Can't allocate memory!\n");
        (void)fclose(file);                         /* Close callback.dat    */
        return ENOMEM;
    }
    username = buffer   + BUF_LEN;                  /* Partition the buffers */
    phone    = username + UNAME_LEN; 
    tty_name = phone    + UTELL_LEN;
    tim_buf  = tty_name + TTY_LEN;
    time(&tnow);                                    /* Get system time       */
    tmnow = localtime(&tnow);                       /* Convert to local time */
    (void)strcpy(tim_buf, (asctime(tmnow)));        /* Convert to ASCII      */
    (void)clean(tim_buf);                           /* Remove line-feed      */
    if((log = fopen(logf, "a+")) == NULL)           /* Open a log file       */
    {
        (void)fprintf(stderr, "Can't open log file!\n");
        free(buffer);
        (void)fclose(file);
        return ENOENT;
    }
    (void)strcpy(tty_name, (ttyname(0)));           /* Get treminal name     */
    (void)close(0);                                 /* Now close terminal    */
    (void)close(1);
    (void)close(2);
    (void)setsid();                                 /* Make top-level        */
    if((tt = open(tty_name, O_RDWR)) < 0)           /* Now reopen terminal   */
        exit(errno); 
    catcher(0);                                     /* Make line time-out    */
    (void)read(tt, buffer, BUF_LEN);                /* Clear any type-ahead  */
    (void)write(tt, prmt, sizeof(prmt));            /* Send prompt string    */
    for(i=0; i < WAIT_TIME; i++)
        if((len = read(tt, buffer, BUF_LEN)) > 0)   /* Wait for a response   */
            break;
    alarm(0);                                       /* Turn off timer ticks  */
    if(i == WAIT_TIME)  exit(0);                    /* Too much time expired */
    if(len > UNAME_LEN) exit(0);                    /* Name was too long     */
    buffer[len]=0x00;                               /* Terminate the string  */
    if((len = clean(buffer)) == 0)                  /* Remove spaces, etc.   */
        exit(0);                                    /* It was a null string  */
    (void)strcpy(username, buffer);                 /* Save username for now */
    (void)fprintf(log, "%s callback started for %s on %s\n",
                                 tim_buf, username, tty_name);
    (void)strcpy(buffer, hnow);                     /* Prepare output string */
    (void)strcat(buffer, crlf);
    (void)write(tt, buffer, strlen(buffer));        /* Send hangup command   */
    while((probe = fgets(buffer, BUF_LEN, file)) != NULL)
    {
        if((clean(buffer)) == 0) continue;          /* Filter comment text   */ 
        if((probe = strstr(buffer, name)) != NULL)  /* Got the 'NAME' token  */
            break;
    }
    if(probe == NULL) exit(0);                      /* It's the end-of-file  */
    while((probe = fgets(buffer, BUF_LEN, file)) != NULL)
    {
        if((clean(buffer)) == 0) continue;          /* Blank lines, comments */
        if((strncmp(buffer, username, len)) == 0)   /* Found the user name   */
        {
            probe = buffer;
            while(*probe++ > 0x20)                  /* Get past first space  */
                ;
            (void)strcpy(phone, probe);             /* Save the phone number */
            (void)strcat(phone, crlf); 
                break;
        }
    }
    if(probe == NULL) exit(0);                      /* It's the end-of-file  */
    rewind(file);
    (void)sleep(5);
    catcher(0);
    while((fgets(buffer, BUF_LEN, file)) != NULL)
    {
        if((clean(buffer)) == 0) continue;          /* Comments, blank lines */
        if((strstr(buffer, "WAIT")) != NULL)        /* Found 'WAIT' token    */
        {
            pause();
            pause();
            continue;
        }
        if((strstr(buffer, "HANGUP")) != NULL)      /* Found 'HANGUP' token  */
        {
            alarm(0);                               /* Stop the timer ticks  */
            ioctl(tt, TCGETS, &io);                 /* Get terminal params   */ 
            speed = io.c_cflag;                     /* Save what we modify   */
            io.c_cflag &= ~CBAUD;                   /* Set baud-rate to zero */ 
            io.c_cflag |= B0;
            ioctl(tt, TCSETSF, &io);                /* Set the terminal      */
            sleep(1);
            io.c_cflag = speed;                     /* Restore modified pars */
            ioctl(tt, TCSETSF, &io);                /* Set terminal back     */
            catcher(0);                             /* Restart the timer     */
            continue;
        }
        if((strstr(buffer, name)) != NULL)          /* End of modem commands */
            break; 
        (void)strcat(buffer, crlf);                 /* Append EOL for modem  */
        (void)write(tt, buffer, strlen(buffer));    /* Send string to modem  */
        (void)read(tt, buffer, BUF_LEN);            /* Read any echo         */
    }
    (void)strcpy(buffer, atdt);                     /* Build dial string     */
    (void)strcat(buffer, phone);
    (void)write(tt, buffer, strlen(buffer));        /* Send string to modem  */
    for(i=0;i < CONNECT_TIME; i++)                  /* Wait for connection   */
        if(((read(tt, buffer, BUF_LEN)) > 0) && (i > RESPONSE_TIME)) 
            break;
    time(&tnow);                                    /* Get new system time   */
    tmnow = localtime(&tnow);                       /* Convert to local time */
    (void)strcpy(tim_buf, (asctime(tmnow)));        /* Convert to ASCII time */
    (void)clean(tim_buf);                           /* Remove line-feed      */
    if(i < CONNECT_TIME)
        (void)fprintf(log, "%s connection succeeded.\n", tim_buf);
    else
        (void)fprintf(log, "%s connection failed.\n", tim_buf);
    alarm(0);                                       /* Turn off timer ticks  */
    (void)fclose(file);                             /* Close callback.dat    */
    (void)fclose(log);                              /* Close the log file    */
    free(buffer);                                   /* Free all memory here  */
    if(i < CONNECT_TIME)                            /* We got a connection   */
    {
        (void)write(tt, welc, sizeof(welc));        /* Send welcome string   */
        (void)dup2(tt, 0);                          /* Is probably 0 already */
        (void)dup2(tt, 1);                          /* Set other standard IO */
        (void)dup2(tt, 2);
        signal(SIGHUP, SIG_DFL);                    /* Fix Hangup signal     */
        (void)execve(login[0], login, environ);     /* Try to log-in user    */
    }
    (void)close(tt);                                /* Close terminal        */
    return 0; 
}
/*
 *  This cleans up the lines from the input file because some folks can't
 *  type or follow instructions very well.
 */
#define FALSE 0
#define TRUE -1

int clean(char *buffer)
{
    int len;
    char *tmp;
    char *outp;
    int space;
    int lead;
    int i;
 
    if((len = strlen(buffer)) == 0)
        return 0;
    if((tmp = (char *) malloc((len + 1) * sizeof(char))) == NULL)
        return 0;
    for(i=0, space = FALSE, lead = TRUE, outp = tmp; i < len; i++)
    {
        if(buffer[i] > 0x20)
        {
            *outp++ = buffer[i];
            space = FALSE;
            lead  = FALSE;
            continue;
        }
        else if((space == FALSE) && (lead == FALSE))
        {
            *outp++ = 0x20;
            space = TRUE;
        }
    }
    *outp = 0x00;                       /* Terminate new string       */
    if(outp > tmp)
    {
        while(*outp < 0x21) outp--;     /* Clear trailing white-space */
        outp++;
    }
    *outp = 0x00;
    len = (outp - tmp);
    (void)strcpy(buffer, tmp);
    free(tmp);
    if(*buffer == '#')
        return 0;
    return len;
}

