/***********************************************************************
 * ext2spice
 * Converts circuits from .ext form to .spice form, including area and
 * perimeter of source and drain.
 * Copyright (c) 1987 
 * Andrew Burstein
 * 8/28/89 - Monte Mar.  Modified to read 2 poly caps.  
 *************************************************************************/
#include "ext2spice.h"
#include <math.h>
#include <sys/types.h>
#include <sys/dir.h>
#include <pwd.h>
#include <stdio.h>


	  

/*  Print out transistor, capacitor, and subckt spice cards to the SpiceFile.
 */  
OutputSpice(SpiceFile, NodesFile, FirstFet, FirstCap, UNFirst, 
  MinCap, UFFirst, ThisCell, useIntNames)
  FILE  *SpiceFile, *NodesFile;
  FET   **FirstFet;
  CAP   *FirstCap;
  USEDNUM *UNFirst, *UFFirst;
  double  MinCap;
  CELL  *ThisCell;
  int	useIntNames;
{

  FET  *Fetp, *FetSort, *FetMove;
  CAP  *Capp;
  int  i, FetCount, NodeCount, CapNum, SubCktNum;
  int  LineLen;
  NODE  *Np;
  int    D, G, S, B;
  double  AD, AS, PD, PS;
  extern CELL *CellFirst;
  TERMINAL  *Term;
  INSTANCE  *Inst;
  char    NodeName[256], *EndNodeName;
  char    LineBuf[256];

  if(ThisCell == CellFirst)fprintf(SpiceFile,"****** top level cell is %s\n",
    ThisCell->PathName);
  FetCount = 1;
  NodeCount = 0;
  CapNum = 1;
  SubCktNum = 1;
  LineLen = 0;
  /* Give Node Numbers to the terminals in the current cell  
   * and print the subckt header
   */
  if (ThisCell != CellFirst)  {
    fprintf(SpiceFile,"\n*****  Subcircuit from file %s\n",ThisCell->PathName);

    sprintf(LineBuf,".SUBCKT %s ",ThisCell->CellName);
    LineLen = strlen(LineBuf);
    fputs(LineBuf, SpiceFile);

    for (Term = ThisCell->TermList; Term != NULL; Term = Term->Next)  {
      if (LineLen > 66){fprintf(SpiceFile,"\n+ "); LineLen = 0;  }
      FindNode(Term->NodeName,&(Term->Node));
      if (useIntNames)  {
	Term->Node->NodeNum = GiveNum(&Term->Node->NodeNum,UNFirst,&NodeCount);
        sprintf(LineBuf,"%d ",Term->Node->NodeNum);
      } 
      else  {
        sprintf(LineBuf,"%s ",Term->Node->Name);
      }
      LineLen += strlen(LineBuf);
      fputs(LineBuf, SpiceFile);
    }
    fprintf(SpiceFile,"\n\n");
  }
  /* print all of the subcircuit calls in this subcircuit
   * give node numbers to all of their terminals
   */
  for (Inst = ThisCell->InstList; Inst != NULL; Inst = Inst->Next)  {
    sprintf(LineBuf,"x%s ",Inst->InstID);
    LineLen = strlen(LineBuf);
    fputs(LineBuf, SpiceFile);
    strcpy(NodeName,Inst->InstID);
    strcat(NodeName,"/");
    EndNodeName = NodeName + strlen(NodeName);
    for (Term = Inst->Cellp->TermList; Term != NULL; Term = Term->Next)  {
      strcpy(EndNodeName,Term->NodeName);
      /*  Nodes that end in ! are global;  don't add instID to ! node name */
      if (*(EndNodeName + strlen(EndNodeName) -1) != '!')FindNode(NodeName,&Np);
      else  FindNode(EndNodeName,&Np);
      if (LineLen > 66){fprintf(SpiceFile,"\n+ "); LineLen = 0;}
      if (useIntNames)  {
        Np->NodeNum = GiveNum(&Np->NodeNum,UNFirst,&NodeCount);
        sprintf(LineBuf,"%d ",Np->NodeNum);
      }
      else {
        sprintf(LineBuf,"%s ",Np->Name);
      }
      LineLen += strlen(LineBuf);
      fputs(LineBuf, SpiceFile);
    }
    if ((LineLen + strlen(Inst->Cellp->CellName)) >78)fprintf(SpiceFile,"\n+ ");
    fprintf(SpiceFile,"%s\n\n",Inst->Cellp->CellName);
  }

  /* Assign numbers to all fets that are still unnumbered
   * Sort fets into numerical order
   */
  for (Fetp = *FirstFet; Fetp != NULL; Fetp = Fetp->Next)  {
    Fetp->FetNum = GiveNum(&Fetp->FetNum,UFFirst,&FetCount);
  }
  FetSort = *FirstFet;
  while ((FetSort != NULL)&&(FetSort->Next != NULL))  {
    if (FetSort->Next->FetNum < (*FirstFet)->FetNum)  {
      FetMove =FetSort->Next;
      FetSort->Next = FetMove->Next;
      FetMove->Next = *FirstFet;
      *FirstFet = FetMove;
    }
    else  {
      for (Fetp = *FirstFet; Fetp->Next->FetNum < FetSort->Next->FetNum; 
      
      Fetp=Fetp->Next);
      if (Fetp != FetSort)  {
        FetMove = FetSort->Next;
        FetSort->Next = FetMove->Next;
        FetMove->Next = Fetp->Next;
        Fetp->Next = FetMove;
      }
      else FetSort = FetSort->Next;
    }
  }      

  for (Fetp = *FirstFet; Fetp != NULL; Fetp = Fetp->Next)  {
    FindRootNode(&Fetp->Drain); FindRootNode(&Fetp->Gate);
    FindRootNode(&Fetp->Source); FindRootNode(&Fetp->Sub);
    if (useIntNames)  {
      D = GiveNum(&Fetp->Drain->NodeNum,UNFirst,&NodeCount);
      G = GiveNum(&Fetp->Gate->NodeNum,UNFirst,&NodeCount);
      S = GiveNum(&Fetp->Source->NodeNum,UNFirst,&NodeCount);
      B = GiveNum(&Fetp->Sub->NodeNum,UNFirst,&NodeCount);
    }

    if ( *Fetp->Model->Flavor == 'n')  {
      AD = (Fetp->Drain->NA * Fetp->W) / Fetp->Drain->NTotalW;
      PD = (Fetp->Drain->NP * Fetp->W) / Fetp->Drain->NTotalW;
      AS = (Fetp->Source->NA * Fetp->W) / Fetp->Source->NTotalW;
      PS = (Fetp->Source->NP * Fetp->W) / Fetp->Source->NTotalW;
    }
    else {  /*  Flavor is p   */
      AD = (Fetp->Drain->PA * Fetp->W) / Fetp->Drain->PTotalW;
      PD = (Fetp->Drain->PP * Fetp->W) / Fetp->Drain->PTotalW;
      AS = (Fetp->Source->PA * Fetp->W) / Fetp->Source->PTotalW;
      PS = (Fetp->Source->PP * Fetp->W) / Fetp->Source->PTotalW;
    }
    if (useIntNames)  {
      fprintf(SpiceFile,
      "M%d %d %d %d %d %s W=%.1lfU L=%.1lfU \n+ AD=%.1lfP PD=%.1lfU AS=%.1lfP PS=%.1lfU\n",
      Fetp->FetNum,D,G,S,B,Fetp->Model->SpiceName,Fetp->W,Fetp->L,AD,PD,AS,PS);
    } 
    else {
      fprintf(SpiceFile,
      "M%d %s %s %s %s %s W=%.1lfU L=%.1lfU \n+ AD=%.1lfP PD=%.1lfU AS=%.1lfP PS=%.1lfU \n",
      Fetp->FetNum,
      Fetp->Drain->Name,
      Fetp->Gate->Name,
      Fetp->Source->Name,
      Fetp->Sub->Name,
      Fetp->Model->SpiceName,Fetp->W,Fetp->L,AD,PD,AS,PS);
    }
  }
  fprintf(SpiceFile,"\n");
  for (Capp = FirstCap; Capp != NULL; Capp = Capp->Next)  {
    if (Capp->Value >= MinCap)  {
      FindRootNode(&Capp->Node1); FindRootNode(&Capp->Node2);
      if (useIntNames)  {
	fprintf(SpiceFile,"C%d %d %d %.1lfF \n"
	  ,CapNum++,GiveNum(&Capp->Node1->NodeNum, UNFirst,&NodeCount)
	  ,GiveNum(&Capp->Node2->NodeNum,UNFirst,&NodeCount),Capp->Value);
      }
      else {
        fprintf(SpiceFile,"C%d %s %s %.1lfF \n"
	  ,CapNum++,Capp->Node1->Name, Capp->Node2->Name,Capp->Value);
      }
    }
  }
  for (i=0; i < TBLSIZE; i++)  {
    Np = NodeTable[i];
    while (Np != NULL)  {
      if ((Np->Merge == NULL)&&(Np->Cap >= MinCap))  {
	if (useIntNames)  {
	  fprintf(SpiceFile,"C%d %d 0 %.1lfF \n"
	  ,CapNum++,GiveNum(&Np->NodeNum,UNFirst,&NodeCount), Np->Cap);
	}
	else {
	  fprintf(SpiceFile,"C%d %s 0 %.1lfF \n"
 	    ,CapNum++,Np->Name, Np->Cap);
	}
      }
      Np = Np->Next;
    }
  }
  if (NodesFile != NULL)  PrintNodes(ThisCell,NodesFile);
  if (ThisCell != CellFirst)  fprintf(SpiceFile,".ENDS\n\n");
}




PrintNodes(ThisCell,OutFile)
  CELL	*ThisCell;
  FILE	*OutFile;
{
  NODE	*Np, *NodeRoot, *InsertBefore, *InsertAfter;
  int	i;

  for (i=0; i < TBLSIZE; i++)  {
    Np = NodeTable[i];
    for (; Np != NULL; Np = Np->Next)  {
      NodeRoot = Np;
      FindRootNode(&NodeRoot);
      Np->NodeNum = NodeRoot->NodeNum;
    }
  }
  Np = NodeTable[0];
  NodeTable[0] = (NODE *) calloc(1,sizeof(NODE));
  NodeTable[0]->Name =(char *) calloc(1,64);
  strcpy(NodeTable[0]->Name,"Node 0 is the global ground node");
  NodeTable[0]->NodeNum = 0;
  NodeTable[0]->Next = Np;

  for (i=0; i < TBLSIZE; i++)  {
    Np = NodeTable[i];
    while (Np !=NULL)  {
      if (Np->NodeNum != -1)  {
        InsertBefore = NodeTable[0];
        InsertAfter = InsertBefore;
        while ((InsertBefore !=NULL)&&(Np->NodeNum >= InsertBefore->NodeNum))  {
	  InsertAfter = InsertBefore;
	  InsertBefore = InsertBefore->Next;
        }
        if (InsertAfter != Np)  {
          InsertAfter->Next = Np;
          Np = Np->Next;
          InsertAfter->Next->Next = InsertBefore;
        }
        else Np = Np->Next;
      }
      else Np = Np->Next;
    }
  }
  for (i=1; i < TBLSIZE; i++)  NodeTable[i] = NULL;

  fprintf(OutFile,"*** Node Listing for subckt:  %s\n",ThisCell->CellName);
  for (Np= NodeTable[0]; Np != NULL; Np = Np->Next)  {
    fprintf(OutFile,"**  %d \t%s\n",Np->NodeNum,Np->Name);
  }
}



	
GiveNum(Num, UNp, Count)
  int  *Num;
  USEDNUM  *UNp;
  int *Count;
{
      /* Return the next free number if Num == -1      */  
  if (*Num == -1)  {
    /* skip the set node numbers  */
    while  ((UNp != NULL) && (*Count >= UNp->NodeNum)) {
      if (*Count == UNp->NodeNum) (*Count)++;
      UNp = UNp->Next;
    }
    *Num  = (*Count)++;
  }
  return(*Num);
}

	  


