/***********************************************************************
 * ext2spice
 * Converts circuits from .ext form to .spice form, including area and
 * perimeter of source and drain.
 * Copyright (c) 1987 
 * Andrew Burstein
 * 8/28/89 - Monte Mar.  Modified to read 2 poly caps.  
 *************************************************************************/
#include "ext2spice.h"
#include <math.h>
#include <sys/types.h>
#include <sys/dir.h>
#include <pwd.h>
#include <stdio.h>


main(argc,argv)
  int argc;
  char *argv[];
{
  int i;
  FETTYPE  *TypeFirst;
  FILE   *NodesFile, *AlFile, *DefsFile, *ExtFile, *SpiceFile;
  char  NodesName[32], SpiceName[32], AlName[32], *DefsName;
  char  **Path;
  int Merge = 0;  /* Default is not to merge  */
  int NodesInNames = 0; /* Print node #'s and names in .names file */
			/* (was .nodes file) */
  int NodesInSpice = 0; /* Print node #'s and names in .spice file */
  int useIntNames = 1;  /* use integer node names, not character strings */
  double MinCap = 1.0;  /*Default min output cap is 1.0fF  */
  extern CELL *CellFirst;
  CELL   *RootCell;
  char  *FindPath();
  void  ReadExt();
  void  OutputSpice();

  if (argc == 1)  {
    fprintf(stderr,
      "usage: ext2spice [-d defs] [-c value] [-mnNs] ext_basename\n");
    exit(1);
  }
  DefsName = NULL;
  for (i=1; i<argc; i++)  {
    if (argv[i][0] == '-')  {
      switch (argv[i][1])  {
	case 'd':
	  DefsName = argv[++i];
	  break;

	case 'c':  /*Specify the smallest Cap that will be listed, in fF  */
	  MinCap = atof(argv[++i]);
	  break;
	
	case 'm' :  /* Merge parallel fets  */
	  Merge = 1;
	  break;

	case 'n' :  /* Print node #'s and names in .spice file */
	  NodesInSpice = 1;
	  break;

	case 'N' :  /* Print node #'s and names in .names file */
	  NodesInNames = 1;
	  break;

	case 's' :  /*  use integer node names, not character strings */
	  useIntNames = 0;
	  break;


	case 'a' :  /* Redefine area capacitance  */
	  areacap = atof(argv[++i]);
	  if(areacap <= 0){
		fprintf(stderr, "Specified area capacitance is not valid.  ");
		fprintf(stderr, "Default value set to 500 attofarads.\n");
		areacap = .5;
	  }
	  break;

	default:
	  fprintf(stderr,"ext2sim: %s, unknown option \n",argv[i]);
	  break;
      }
    }
  }
  strcpy(SpiceName,argv[--i]); strcat(SpiceName,".spice");
  strcpy(AlName,argv[i]); strcat(AlName,".al");
  OpenFile(&SpiceFile,SpiceName,"w");
  OpenFile(&AlFile,AlName,"w");
  if (NodesInSpice) NodesFile = SpiceFile;
  else if (NodesInNames)  {
    strcpy(NodesName,argv[i]); strcat(NodesName,".names");
    OpenFile(&NodesFile,NodesName,"w");
  }
  else NodesFile = NULL;

  Init(DefsName,argv[i],&TypeFirst,&Path);

  GetCells(Path);

  for (RootCell = CellFirst; RootCell->Next != NULL; RootCell= RootCell->Next){
    PushTerminals(RootCell);
  }

  /* Start from the end of the Cell list and work backwards  */
  for (; RootCell!= NULL; RootCell= RootCell->Last)  {
    OpenFile(&ExtFile,RootCell->PathName,"r");

    ReadExt(ExtFile,AlFile, &RootCell->FetList, &RootCell->CapList,
      TypeFirst,RootCell->UsedFetList);
    if (Merge)  MergeParallelFets(RootCell->FetList);
    MergeSubcktTerms(RootCell);
    FindMergedTerminals(RootCell);
    /*  Read the defs file, if there is one    */
    if (DefsName != NULL)  {
      OpenFile(&DefsFile,DefsName,"r");
      Define(DefsFile,RootCell->UsedNodeList);
    }
    OutputSpice(SpiceFile, NodesFile, &RootCell->FetList, RootCell->CapList,
      RootCell->UsedNodeList, MinCap, RootCell->UsedFetList,
      RootCell,useIntNames);
    FreeNodeTable();
    fclose(ExtFile);
    if (DefsName != NULL) fclose(DefsFile); 
  }
  fprintf(SpiceFile,".END\n");
  fclose(AlFile);
  fclose(SpiceFile);
  if (NodesInNames) fclose(NodesFile);
}

/*  Initialize the following lists:    
 *  CellFirst, NodeTable (a hash table)
 *  input from defs file:  prclass, nrclass, def, path
 *  nrclass num
 *  prclass num
 *   specifies which resistance class corresponds to ndiff and pdiff
 *   default: ndiff=1;  pdiff = 2;
 */
Init(DefsName,ExtName,TypeFirst,Path)
  char  *DefsName, *ExtName;
  FETTYPE  **TypeFirst;
  char  ***Path;
{
  FETTYPE *Typep;
  int     Count, i;
  char    *Word[256], buf[256], *PathBuf;
  char  *FindPath(), *cp_tildexpand();
  FILE  *DefsFile;


  *tech = '\0';
  *TypeFirst = NULL;
  /*  Initialize elements of NodeTable to zero    */
  for (i=0; i<TBLSIZE; i++) NodeTable[i] = NULL;

  /*  Default:  Ndiffusion is first resistance class;  Pdiffusion is second*/
  PRClass = 2;
  NRClass = 1;
  /*  default def commands:
   *  def pfet PMOS p
   *  def nfet NMOS n
   */
  Typep = (FETTYPE *)  calloc(1,sizeof(FETTYPE));
  strcpy(Typep->ExtName,"pfet");
  strcpy(Typep->SpiceName,"PMOS");
  strcpy(Typep->Flavor,"p");
  Typep->Next = *TypeFirst;
  *TypeFirst = Typep;
  Typep = (FETTYPE *)  calloc(1,sizeof(FETTYPE));
  strcpy(Typep->ExtName,"nfet");
  strcpy(Typep->SpiceName,"NMOS");
  strcpy(Typep->Flavor,"n");
  Typep->Next = *TypeFirst;
  *TypeFirst = Typep;

  *Path = NULL;
  if (DefsName != NULL)  {
    OpenFile(&DefsFile,DefsName,"r");
    while (fgets(buf,256,DefsFile) != NULL)  {
      Parse(buf, &Count, &(Word[0]));
      if (!strcmp(Word[0],"prclass"))  {
        PRClass = atoi(Word[1]);
      }
      else if (!strcmp(Word[0],"nrclass"))  {
        NRClass = atoi(Word[1]);
      }
      else if (!strcmp(Word[0],"def"))  {
        if (Count == 4)  {
	  Typep = (FETTYPE *)  calloc(1,sizeof(FETTYPE));
	  strcpy(Typep->ExtName,Word[1]);
	  strcpy(Typep->SpiceName,Word[2]);
	  strcpy(Typep->Flavor,Word[3]);
	  Typep->Next = *TypeFirst;
	  *TypeFirst = Typep;
        }
      }
      else if (!strcmp(Word[0],"path"))  {
        *Path = (char **) calloc((Count +1),sizeof(char **));
        for (i=1; i < Count; i++)  {
	  PathBuf = cp_tildexpand(Word[i]);
	  (*Path)[i] = (char *) calloc(1,(strlen(PathBuf) +2));
	  strcpy((*Path)[i],PathBuf);
	  strcat((*Path)[i],"/");
        }
        (*Path)[0] = (char *) calloc(1,(strlen("./")+1));
        strcpy((*Path)[0],"./");
        (*Path)[Count] = (char *) calloc(1,(strlen(" ")+1));
        strcpy((*Path)[Count],"") ;
      }
    }
  }
  if (*Path == NULL) {
    *Path = (char **) calloc(2,sizeof(char **));
    (*Path)[0] = (char *) calloc(1,(strlen("./")+1));
    strcpy((*Path)[0],"./");
        (*Path)[1] = (char *) calloc(1,(strlen(" ")+1));
        strcpy((*Path)[1],"") ;
  }
  CellFirst = (CELL *) calloc(1,sizeof(CELL));
  CellFirst->Next = NULL;
  CellFirst->Last = NULL;
  CellFirst->TermList = NULL;
  CellFirst->FetList = NULL;
  CellFirst->CapList = NULL;
  CellFirst->UsedFetList = (struct UsedNum *) calloc(1, sizeof(struct UsedNum));
  CellFirst->UsedFetList->NodeNum =0;
  CellFirst->UsedFetList->Next = NULL;
  CellFirst->UsedNodeList =(struct UsedNum *) calloc(1, sizeof(struct UsedNum));
  CellFirst->UsedNodeList->NodeNum =0;
  CellFirst->UsedNodeList->Next = NULL;
  CellFirst->CellName = ExtName;
  CellFirst->PathName = FindPath(ExtName,Path);
}
  


  
/*  Free the lists that will be reused by the next cell
 */
FreeNodeTable()
{
  NODE  *Np, *NpNext;
  int    i;

  /*  Free the NodeTable     */
  for (i=0; i<TBLSIZE;  i++)  {
    for ( Np = NodeTable[i];  Np != NULL; Np = NpNext)  {
      NpNext = Np->Next;
      free(Np);
    }
    NodeTable[i] = NULL;
  }
}


/*  read all of the "set" commands from the defs file.
 *  This is done for each cell since each cell redoes its own NodeTable
 */
Define(DefsFile,UNfirst)
  FILE  *DefsFile;
  USEDNUM  *UNfirst;
{
  char buf[256];
  USEDNUM  *UNtemp, *UNp;
  int  Count, n, hash(), NodeNum;
  char *Word[256];
  NODE  *Np;


  while (fgets(buf,256,DefsFile) != NULL)  {
    Parse(buf, &Count, &(Word[0]));
    if (!strcmp(Word[0],"set"))  {
      if  (Count == 3) {
	NodeNum = atof(Word[2]);
        n = hash(Word[1]);
        Np = NodeTable[n];
        while ((Np != NULL) && (strcmp(Word[1],Np->Name)))  {
          Np = Np->Next;
        }
        if (Np != NULL)  {  /*If the "set" node is present in current cell,
			     * give it the number from the defs file       */
	  FindRootNode(&Np);
	  Np->NodeNum = NodeNum;
	}
	UNp = UNfirst;
	while((UNp->Next != NULL) && (UNp->Next->NodeNum <= NodeNum)) { 
	  UNp = UNp->Next;
	}
	if (UNp->NodeNum != NodeNum)  {/*No need to make a new UsedNum
	      *if number is already used    */
	  UNtemp = (struct UsedNum *) calloc(1, sizeof(struct UsedNum));
	  UNtemp->NodeNum = NodeNum;
	  UNtemp->Next = UNp->Next;
	  UNp->Next = UNtemp;
	}
      }
    }
  }
}
	


/*  See if any of a cell's terminals are nodes in its subcells.  If so, 
 *  make them terminals of the subcells.
 */
PushTerminals(RootCell)
  CELL	*RootCell;
{
  INSTANCE  *Inst;
  TERMINAL *Term;
  char   InstID[256], *RestOfPath;

  for (Inst = RootCell->InstList; Inst!=NULL; Inst=Inst->Next)  {
    for (Term = RootCell->TermList; Term != NULL; Term = Term->Next)  {
      StripPath(Term->NodeName,InstID,&RestOfPath);
      if (!strcmp(InstID,Inst->InstID))  {
        AddTerm(Inst->Cellp,RestOfPath);
      }
    }
  }
}


/*  Find the Node with name 'Name'.
 *  If it doesn't exist, create it.
 *  If it is merged to another node(s), follow the mergers until the root
 *   is found.
 *  Set *Nodepp to the location of the Node in the hashtable, NodeTable.
 */
FindNode(Name, Nodepp)
  char *Name;
  NODE **Nodepp;
{
  int  n;
  NODE *Nodep;

  n = hash(Name);
  Nodep = NodeTable[n];
  while ((Nodep != NULL) && (strcmp(Name,Nodep->Name)))  {
    Nodep = Nodep->Next;
  }
  if (Nodep == NULL)  {  /*Create new node */
    Nodep = (NODE *) calloc(1,sizeof(NODE));
    Nodep->Name = (char *) calloc(1,(strlen(Name)+1));
    strcpy(Nodep->Name, Name);
    Nodep->NodeNum = -1;
    Nodep->PTotalW = Nodep->NTotalW = 0;
    Nodep->NA = Nodep->NP = Nodep->PA = Nodep->PP = 0;
    Nodep->Cap = 0;
    Nodep->Merge = NULL;
    Nodep->Next = NodeTable[n];
    NodeTable[n] = Nodep;
  }
  /* Go to the "root" merge node  */
  FindRootNode(&Nodep);
  *Nodepp = Nodep;
}

/*  Follow the merge pointers and set *Nodepp to the root
 */
FindRootNode(Nodepp)
  NODE  **Nodepp;
{
  while ((*Nodepp)->Merge != NULL)  (*Nodepp) = (*Nodepp)->Merge;
}


hash(name)
  char *name;
{
  int i,n;
  n = 0;
  i = 0;
  while (name[i] != '\0')  {
    n = n*5 + name[i++];
  }
  n = n%TBLSIZE;
  return(abs(n));
}

/*  Check each FET in list, FirstFet, against each other FET to see if they
 *  are parallel (they are still parallel if source and drain are flipped).
 *  Combine parallel FET's into one larger FET (if channel lengths are the
 *  same).
 *
 *  When two FET's are merged, one FET is eliminated and the other one
 *  is bigger than before.
 */
MergeParallelFets(FirstFet)
  FET  *FirstFet;
{
  FET  *F1, *F2;
  NODE  *Node;

  F1 = FirstFet;
  while ((F1 != NULL)&&(F1->Next != NULL))  {
    F2 = F1;
    while ((F2 != NULL)&&(F2->Next != NULL))  {
      if (((F1->Gate==F2->Next->Gate)&&(F1->Sub==F2->Next->Sub)) && 
	  (((F1->Drain==F2->Next->Drain)&&(F1->Source==F2->Next->Source))|| 
	    ((F1->Drain==F2->Next->Source)&&(F1->Source==F2->Next->Drain))) && 
	  (F1->L == F2->Next->L))  {
        F1->W += F2->Next->W;
	/* Save the FetNum FetName if one was specified in the Gate atribute  */
	if (F2->Next->FetNum != -1)  {
	  F1->FetNum = F2->Next->FetNum;
	  strcpy(F1->FetName,F2->Next->FetName);
	  /*  Keep the S D orientation of the fet with specified number  */
	  if (F1->Drain==F2->Next->Source)  {
	    Node = F1->Drain;
	    F1->Drain = F1->Source;
	    F1->Source = Node;
	  }
	}
        F2->Next = F2->Next->Next;
      }
      else  {
        F2 = F2->Next;
      }
    }
    F1 = F1->Next;
  }
}    

/*  For each terminal of Cell, Cellp, copy into Term->Merge the name of the
 *  root of the merger list for that node.
 *  This is useful when calling up an instance of this cell.  At that time,
 *  the parent cell can tell which terminals are merged together (ie, which
 *  teminals have the same name in Term->Merge.
 *  See MergeSubcktTerms
 */
FindMergedTerminals(Cellp)
  CELL  *Cellp;
{
  
  TERMINAL  *Term;
  NODE      *Nodep;

  for (Term = Cellp->TermList; Term != NULL; Term = Term->Next)  {
    FindNode(Term->NodeName,&Nodep);
    Term->Merge = (char *) calloc(1,(strlen(Nodep->Name)+1));
    strcpy(Term->Merge,Nodep->Name);
  }
}

/*  Merge all terminals of all subckts of ThisCell to their mergenodes, within
 *  their subckts (ie., prefix the merge names with the subckt instance name).
 *  This is necessary to prevent two different node (with different spice
 *  nodenumbers) in ThisCell from being (unknowingly) merged (ie. shorted) 
 *  in a subckt.  While this would be topologically correct, spice will not
 *  allow it.  Now, all nodes that are merged in any subckt will be merged
 *  in ThisCell.
 */
MergeSubcktTerms(ThisCell)
  CELL  *ThisCell;
{
  INSTANCE  *Inst;
  char      NodeName[128], MergeName[128];
  char      *EndNodeName, *EndMergeName;
  TERMINAL  *Term;
  NODE      *TermNode, *MergeNode;


  for (Inst = ThisCell->InstList; Inst != NULL; Inst = Inst->Next)  {
    strcpy(NodeName,Inst->InstID);
    strcat(NodeName,"/");
    EndNodeName = NodeName + strlen(NodeName);
    strcpy(MergeName,Inst->InstID);
    strcat(MergeName,"/");
    EndMergeName = MergeName + strlen(MergeName);
    for (Term = Inst->Cellp->TermList; Term != NULL; Term = Term->Next)  {
      if ( strcmp(Term->NodeName, Term->Merge)) {
        strcpy(EndNodeName,Term->NodeName);
        strcpy(EndMergeName,Term->Merge);
        /*  Nodes that end in ! are global;  don't add instID to ! node name */
        if (*(EndNodeName + strlen(EndNodeName) -1) != '!')
	  FindNode(NodeName,&TermNode);
        else  FindNode(EndNodeName,&TermNode);
        if (*(EndMergeName + strlen(EndMergeName) -1) != '!')
	  FindNode(MergeName,&MergeNode);
        else  FindNode(EndMergeName,&MergeNode);
        MergeNodes(TermNode,MergeNode);
      }
    }
  }
}
 

/*  When nodes are merged, no nodes are actually destroyed.  For all 
 *  equivalent nodes, ie. for all nodes that have been merged together
 *  one of them is a special "root" node.  If you follow the path of
 *  Merge pointers in any of these nodes, you will wind up at the "root"
 *  node.  It is the "root" node that contains all of the information
 *  concerning Areas, Perimeters, Capacitances, and Node Numbers.
 *  Thus, when merging a new node to an existing one, set its merge
 *  node to the old one's root, and add in its Area, Perimeter, and
 *  Capacitance values.
 */
MergeNodes(TermNode,MergeNode) 
  NODE  *TermNode, *MergeNode;
{
  if ( TermNode != MergeNode)  { /* Merge if they aren't merged already */
    MergeNode->Merge = TermNode;
    if (TermNode->NodeNum == -1) TermNode->NodeNum = MergeNode->NodeNum;
    TermNode->PTotalW += MergeNode->PTotalW; MergeNode->PTotalW = 0;
    TermNode->NTotalW += MergeNode->NTotalW; MergeNode->NTotalW = 0;
    TermNode->NA += MergeNode->NA; MergeNode->NA = 0;
    TermNode->NP += MergeNode->NP; MergeNode->NP = 0;
    TermNode->PA += MergeNode->PA; MergeNode->PA = 0;
    TermNode->PP += MergeNode->PP; MergeNode->PP = 0;
    TermNode->Cap += MergeNode->Cap; MergeNode->Cap = 0;
  }
}
	  

	  
Parse(buf,Count,Word)
  char *buf;
  int *Count;
  char  **Word;
{

  *Word = buf;
  *Count = 0;
  while((*buf != '\n') && (*buf != '\0'))  {
    while ((*buf==' ') || (*buf=='\t') || (*buf=='"')) *(buf++)='\0';
    if ((*buf != '\n') && (*buf != '\0'))  {
      Word[(*Count)++] = buf;
    }
    while ((*buf!=' ')&&(*buf!='\0')&&(*buf!='\n')&&(*buf!='\t')&&(*buf!='"')) {
      buf++;
    }
  }
  *buf = '\0';
}



/* Expand tildes. */

char *
cp_tildexpand(string)
	char *string;
{
	struct passwd *pw;
	char *rest, buf[256];

	if (*string != '~' )
		return (string);
	

	if (string[1] == '/') {
		pw = getpwuid(getuid());
		if (pw == NULL) {
			fprintf(stderr, "Hey, neat trick...\n");
			return ("error");
		}
		rest = string + 2;
	} else if (string[1] == '\0') {
		pw = getpwuid(getuid());
		rest = NULL;
	} else {
		rest = strchr(string, '/');
		if (rest)
			*rest++ = '\0';
		if (*string == '\0')
			pw = getpwuid(getuid());
		else
			pw = getpwnam(++string);
		if (!pw) {
				fprintf(stderr, "Error: no such user %s\n",
							string);
				return (NULL);
		}
	}
	(void) strcpy(buf, pw->pw_dir);
	if (rest) {
		(void) strcat(buf, "/");
		(void) strcat(buf, rest);
	}
	return (buf);
}

OpenFile(Filep, FileName, Mode)
  FILE **Filep;
  char *FileName, *Mode;
{
  if ((*Filep = fopen(FileName,Mode))== NULL)  {
    fprintf(stderr,"Couldn't open file %s \n",FileName);
    exit(1);
  }
}

