/******************************************************************************
 * The XJNSlib X library (XJNSlib.c and XJNSlib.h) are Copyright (c) 1995 by  *
 * Jon N. Steiger, and is licensed for non-commercial use *only*.  If you     *
 * use any of this code in your own programs, please give credit where credit *
 * is due.  Thanks.                                           -Jon-           *
 ******************************************************************************/

#include "XJNSlib.h"

/******************************************************************************
FUNCTION CenterWidth; return pixel offset of a text string
******************************************************************************/
int CenterWidth(XFontStruct *TheFont, char *string, int width)
{
    return(width - XTextWidth(TheFont, string, strlen(string)))/2;
}

/******************************************************************************
FUNCTION CenterHeight; return pixel offset of text string
******************************************************************************/
int CenterHeight(XFontStruct *TheFont, int height)
{
    return height - ((height - TheFont->ascent)/2 + 1);
}


/******************************************************************************
FUNCTION 3DDraw; draws text in "3D"
******************************************************************************/
void Draw3D(Display *TheDisplay, Window TheWindow, GC TheGC, 
            int xpos, int ypos, char *string,
            int maincolor, int highcolor, int lowcolor)
{

    XSetForeground(TheDisplay, TheGC, highcolor);
    XDrawString(TheDisplay, TheWindow, TheGC, xpos-1, ypos-1,
                string, strlen(string));

    XDrawString(TheDisplay, TheWindow, TheGC, xpos-1, ypos,
                string, strlen(string));

    XDrawString(TheDisplay, TheWindow, TheGC, xpos, ypos-1,
                string, strlen(string));

    XSetForeground(TheDisplay, TheGC, lowcolor);
    XDrawString(TheDisplay, TheWindow, TheGC, xpos+1, ypos+1,
                string, strlen(string));

    XDrawString(TheDisplay, TheWindow, TheGC, xpos, ypos+1,
                string, strlen(string));

    XDrawString(TheDisplay, TheWindow, TheGC, xpos, ypos,
                string, strlen(string));

    XSetForeground(TheDisplay, TheGC, maincolor);
    XDrawString(TheDisplay, TheWindow, TheGC, xpos, ypos,
                string, strlen(string));

}

/******************************************************************************
FUNCTION Draw2D; draws text in "2D"
******************************************************************************/
void Draw2D(Display *TheDisplay, Window TheWindow, GC TheGC, 
            int xpos, int ypos, int depth, char *string,
            int maincolor, int lowcolor)
{

    XSetForeground(TheDisplay, TheGC, lowcolor);
    XDrawString(TheDisplay, TheWindow, TheGC, xpos+depth, ypos+depth,
                string, strlen(string));

    XDrawString(TheDisplay, TheWindow, TheGC, xpos+depth-1, ypos+depth,
                string, strlen(string));

    XDrawString(TheDisplay, TheWindow, TheGC, xpos+depth-1, ypos+depth,
                string, strlen(string));

    XSetForeground(TheDisplay, TheGC, maincolor);
    XDrawString(TheDisplay, TheWindow, TheGC, xpos, ypos,
                string, strlen(string));

}

/******************************************************************************
FUNCTION: ColorAdjust; returns the 0-65535 value for 0-256 RGB values.
******************************************************************************/
int ColorAdjust(int color)
{
    return color * 256 + color;
}

/******************************************************************************
FUNCTION: BevelWindow; draws lines in a window to create a beveled effect.
******************************************************************************/
void BevelWindow(Display *TheDisplay, Drawable TheDrawable, GC TheGC,
                 int DirectionFlag,   /* 0 = lowered, 1 = raised */
                 int Xpos, int Ypos,   /* upper left coordinate of bevel */
                 int BevelWidth, int BevelHeight, int BevelThickness,  
                 int lightcolor, int darkcolor,  /* colors to use for bevel */
                 int BackgroundFlag /* color (if any) to use for background */ )
{
    int highcolor = ((DirectionFlag) ? lightcolor : darkcolor),
    lowcolor = ((DirectionFlag) ? darkcolor : lightcolor),
    bevelheight = BevelHeight,
    bevelwidth = BevelWidth,
    xpos = Xpos,
    ypos = Ypos,
    x;


    if (BackgroundFlag >= 1000) {       /* if the background is to be filled */
       XSetForeground(TheDisplay, TheGC, (BackgroundFlag - 1000));
       XFillRectangle(TheDisplay, TheDrawable, TheGC, Xpos, Ypos, 
                      BevelWidth, BevelHeight);
    }
        

     XSetForeground(TheDisplay, TheGC, highcolor);
    
/*
 * draw top lines
 */

    for (x = BevelThickness; x; x--)  {
        XDrawLine( TheDisplay, TheDrawable, TheGC,
                   xpos+1, ypos,
                   xpos +1 + bevelwidth - 1, ypos);
        ypos++;
        xpos++;
        bevelwidth -= 2;
    } 

    xpos = Xpos;
    ypos = Ypos;
    bevelheight = BevelHeight;
    bevelwidth = BevelWidth;

/*
 * draw left lines
 */

    for (x = BevelThickness; x; x--)  {
        XDrawLine( TheDisplay, TheDrawable, TheGC,
                   xpos, ypos+1,
                   xpos, ypos+1 + bevelheight -2);
        ypos++;
        xpos++;
        bevelheight -= 2;
    } 
                   
    xpos = Xpos;
    ypos = Ypos;
    bevelheight = BevelHeight;
    bevelwidth = BevelWidth;

    XSetForeground(TheDisplay, TheGC, lowcolor);

/*
 * draw bottom lines
 */


    for (x = BevelThickness; x; x--)  {
        XDrawLine( TheDisplay, TheDrawable, TheGC,
                   xpos, ypos + bevelheight - 1,
                   xpos -1 + bevelwidth - 1, ypos + bevelheight - 1);
        ypos--;
        xpos++;
        bevelwidth -= 2;
    } 

    xpos = Xpos;
    ypos = Ypos;
    bevelheight = BevelHeight;
    bevelwidth = BevelWidth;
/*
 * draw right lines
 */

    for (x = BevelThickness; x; x--)  {
        XDrawLine( TheDisplay, TheDrawable, TheGC,
                   xpos + bevelwidth -1, ypos+1,
                   xpos + bevelwidth -1, ypos+ 1 + bevelheight -3);
        ypos++;
        xpos--;
        bevelheight -= 2;
    }



}
        


/******************************************************************************
FUNCTION DigitalPrint;  prints an integer in a "digital" "font".
******************************************************************************/
void DigitalPrint (Display *TheDisplay, Drawable TheDrawable, GC TheGC,
                   int XPos, int YPos,  /* x and y to start draw at */
                   int DrawableWidth, int DrawableHeight, /* for background */
                   int PixelHeight,     /* height of "font" in pixels */
                   int ForeColor,       /* foregound color */
                   int FlagBackColor,   /* background color > 1000 = custom */
                   int FlagShadeColor,  /* unused portion of number flag */
                   int number)          /* integer to print */
{
#define LONGSCALAR 0.4210526316
#define SHORTSCALAR 0.3157894737
#define THICKSCALAR 0.1152631579 

    int longlength = LONGSCALAR * PixelHeight;
    int shortlength = SHORTSCALAR * PixelHeight;
    int thickness = THICKSCALAR * PixelHeight;
    int step = 2;
    int sidespace = thickness / 2;
    int backcolor = FlagBackColor; 
    int spacing = thickness * 6;
    int x;

   
    if (FlagBackColor >= 1000) { /* if the background is to be filled */
        backcolor = FlagBackColor - 1000;
        XSetForeground(TheDisplay, TheGC, (backcolor));
        XFillRectangle(TheDisplay, TheDrawable, TheGC, 0, 0, 
                       DrawableWidth, DrawableHeight);
    }

    if (FlagShadeColor >= 1000)  /* if the "unused" portions are to be filled */
        backcolor = FlagShadeColor - 1000;
    else 
        backcolor = backcolor;

    if (step < 2)
        step = 2;    
    
/* 
 *  Now we interpret and print out the integer
 */

   XPos += (CharsIn(number) * spacing);  
        
    if(!number)
        DigitalInterpret(TheDisplay, TheDrawable, TheGC,
                       XPos,  YPos,  
                       longlength,  thickness,  step,
                       sidespace, 
                       ForeColor,  backcolor,
                       number);    

    
    DigitalIntegerPrint(TheDisplay, TheDrawable, TheGC,
                        XPos, YPos, longlength, thickness, step, 
                        sidespace, spacing, ForeColor, backcolor, 
                        (number));
    XFlush(TheDisplay);
     

}

/******************************************************************************
FUNCTION DigitalPrintRJ; Same as DigitalPrint but right justified. 
******************************************************************************/
void DigitalPrintRJ (Display *TheDisplay, Drawable TheDrawable, GC TheGC,
                   int XPos, int YPos,  /* x and y to start draw at */
                   int DrawableWidth, int DrawableHeight, /* for background */
                   int PixelHeight,     /* height of "font" in pixels */
                   int ForeColor,       /* foregound color */
                   int FlagBackColor,   /* background color > 1000 = custom */
                   int FlagShadeColor,  /* unused portion of number flag */
                   int columns,         /* number of columns to print # in */
                   int number)          /* integer to print */
{

    int longlength = LONGSCALAR * PixelHeight;
    int shortlength = SHORTSCALAR * PixelHeight;
    int thickness = THICKSCALAR * PixelHeight;
    int step = 2;
    int sidespace = thickness / 2;
    int backcolor = FlagBackColor; 
    int spacing = thickness * 6;
    int x;

   
    if (FlagBackColor >= 1000) { /* if the background is to be filled */
        backcolor = FlagBackColor - 1000;
        XSetForeground(TheDisplay, TheGC, (backcolor));
        XFillRectangle(TheDisplay, TheDrawable, TheGC, 0, 0, 
                       DrawableWidth, DrawableHeight);
    }

    if (FlagShadeColor >= 1000)  /* if the "unused" portions are to be filled */
        backcolor = FlagShadeColor - 1000;
    else 
        backcolor = backcolor;

    if (step < 2)
        step = 2;    
    
/* 
 *  Now we interpret and print out the integer
 */

   ClearDigitalField(TheDisplay, TheDrawable, TheGC,
                                  XPos, YPos, PixelHeight , backcolor, 3);


   XPos += (columns * spacing) - spacing;  
        
    if(!number)
        DigitalInterpret(TheDisplay, TheDrawable, TheGC,
                       XPos,  YPos,  
                       longlength,  thickness,  step,
                       sidespace, 
                       ForeColor,  backcolor,
                       number);    

    
    DigitalIntegerPrint(TheDisplay, TheDrawable, TheGC,
                        XPos, YPos, longlength, thickness, step, 
                        sidespace, spacing, ForeColor, backcolor, 
                        (number));
    XFlush(TheDisplay);
     

}
         
/******************************************************************************
             THE FOLLOWING ARE ALL DIGITAL SUPPORT FUNCTIONS
******************************************************************************/

/******************************************************************************
FUNCTION: ClearDigitalField;  clears a certain number of columns.
******************************************************************************/
void ClearDigitalField(Display *TheDisplay, Drawable TheDrawable, GC TheGC,
                      int XPos, int YPos, int PixelHeight, 
                      int backcolor,
                      int columns)
{


    int x;
    int longlength = LONGSCALAR * PixelHeight;
    int shortlength = SHORTSCALAR * PixelHeight;
    int thickness = THICKSCALAR * PixelHeight;
    int step = 2;
    int sidespace = thickness / 2;
    int spacing = thickness * 6;


    for(x = 1; x <= columns; x++) {
        DigitalInterpret(TheDisplay, TheDrawable, TheGC,
                           XPos ,  YPos,
                           longlength,  thickness,  step,
                           sidespace,
                           backcolor,  backcolor,
                           8);
        XPos += spacing;
    }




}

/******************************************************************************
FUNCTION: CharsIn;  tells how many "characters" are in an integer
******************************************************************************/
int CharsIn (int number)
{
    int x;

    for (x = 0; number > 0; x++)
        number /= 10;

    return x;
}

/******************************************************************************
FUNCTION: DigitalIntegerPrint;  recursive for integers greater than 9
******************************************************************************/
void DigitalIntegerPrint (Display *TheDisplay, Drawable TheDrawable, GC TheGC,
                     int XPos, int YPos, /* for the whole number */ 
                      int longlength, int thickness, int step,
                      int sidespace, int spacing, 
                      int forecolor, int backcolor,
                      int number)
{
    if (number) {
        DigitalIntegerPrint (TheDisplay, TheDrawable, TheGC,
                             (XPos - spacing), YPos,
                             longlength, thickness, step, sidespace, spacing,  
                             forecolor, backcolor, (number / 10));
        DigitalInterpret(TheDisplay, TheDrawable, TheGC, XPos, YPos, 
                         longlength, thickness, step, sidespace,
                         forecolor, backcolor, (number % 10));
    }
}

/******************************************************************************
FUNCTION: DigitalSectionPrint; used by DigitalPrint to print each section.
   
          - <--- 1
 6 ---> |   | <--- 2     (These are the numbers to use for SectionNum)
          - <--------- 7
 5 ---> |   | <--- 3
          - <--- 4

******************************************************************************/
void DigitalSectionPrint(Display *TheDisplay, Drawable TheDrawable, GC TheGC,
                         int XPos, int YPos,/* of one digit                  */
                         int longlength, int thickness, int step,
                         int sidespace,     /* space on the side of the digit*/
                         int forecolor,     /* foreground color              */
                         int SectionNum)    /* section to print (see above)  */
{
    int i, x1, y1, x2, y2;
    double middlefactor = (((thickness / 1.5) > 0) ?  1.5 : 1); 
    
    XSetForeground(TheDisplay, TheGC, forecolor);

    if (SectionNum == 1) {
        x1 = XPos + thickness - 1;
        y1 = YPos ;
        x2 = XPos + thickness + longlength - 1;
        y2 = y1;

        for (i = 1; i <= (.75 * thickness); i++) {
            XDrawLine(TheDisplay, TheDrawable, TheGC,
                x1, y1, x2, y2);
            x1 += step/2;
            x2 -= step/2;
            y1++;
            y2++; 
        }

    
        x1 = XPos + thickness - 1;
        y1 = YPos ;
        x2 = XPos + thickness + longlength - 1;
        y2 = y1;

        for (i = 1; i <= (.25 * thickness) - 1; i++) {
            XDrawLine(TheDisplay, TheDrawable, TheGC,
                x1, y1, x2, y2);
            x1 += step/2;
            x2 -= step/2;
            y1--;
            y2--; 
        }

    }

    else if (SectionNum == 2) {
        x1 = XPos + thickness + longlength - 1 + (.75 * thickness);
        y1 = YPos;
        x2 = x1;
        y2 = YPos + longlength - 1;

        for (i = 1; i <= (.75 * thickness); i++) {
            XDrawLine(TheDisplay, TheDrawable, TheGC,
                x1, y1, x2, y2);
            x1--; 
            x2--;
            y1 += step/2;
            y2 -= step/2;
        }
        x1 = XPos + thickness + longlength - 1  + (.75 * thickness);
        y1 = YPos;
        x2 = x1;
        y2 = YPos + longlength - 1;

        for (i = 1; i <= (.25 * thickness); i++) {
            XDrawLine(TheDisplay, TheDrawable, TheGC,
                x1, y1, x2, y2);
            x1++; 
            x2++;

            y1 += step/2;
            y2 -= step/2;
        }
   
   
    }
    else if (SectionNum == 3) {
        x1 = XPos + thickness + longlength - 1 + (.75 * thickness);
        y1 = YPos + longlength + sidespace - 1;
        x2 = x1;
        y2 = YPos + (2 * longlength) + sidespace - 1;

        for (i = 1; i <= (.75 * thickness); i++) {
            XDrawLine(TheDisplay, TheDrawable, TheGC,
                x1, y1, x2, y2);
            x1--; 
            x2--;
            y1 += step/2;
            y2 -= step/2;
        }
        x1 = XPos + thickness + longlength - 1  + (.75 * thickness);
        y1 = YPos + longlength + sidespace - 1;
        x2 = x1;
        y2 = YPos + (2 * longlength) + sidespace  - 1;

        for (i = 1; i <= (.25 * thickness); i++) {
            XDrawLine(TheDisplay, TheDrawable, TheGC,
                x1, y1, x2, y2);
            x1++; 
            x2++;

            y1 += step/2;
            y2 -= step/2;
        }
   
    }
    else if (SectionNum == 4) {
        x1 = XPos + thickness - 1;
        y1 = YPos + (2 * longlength) + sidespace; 
        x2 = XPos + thickness + longlength - 1;
        y2 = y1;

        for (i = 1; i <= (.75 * thickness); i++) {
            XDrawLine(TheDisplay, TheDrawable, TheGC,
                x1, y1, x2, y2);
            x1 += step/2;
            x2 -= step/2;
            y1--;
            y2--; 
        }

    
        x1 = XPos + thickness - 1;
        y1 = YPos + (2 * longlength) + sidespace - 1;
        x2 = XPos + thickness + longlength - 1;
        y2 = y1;

        for (i = 1; i <= (.25 * thickness) - 1; i++) {
            XDrawLine(TheDisplay, TheDrawable, TheGC,
                x1, y1, x2, y2);
            x1 += step/2;
            x2 -= step/2;
            y1++;
            y2++; 
        }
   
    }
    else if (SectionNum == 5) {
        x1 = XPos + (.25 * thickness) - 1;
        y1 = YPos + longlength + sidespace - 1;
        x2 = x1;
        y2 = YPos + (2 * longlength) + sidespace - 1;

        for (i = 1; i <= (.25 * thickness); i++) {
            XDrawLine(TheDisplay, TheDrawable, TheGC,
                x1, y1, x2, y2);
            x1--; 
            x2--;
            y1 += step/2;
            y2 -= step/2;
        }
        x1 = XPos + (.25 * thickness) - 1;
        y1 = YPos + longlength + sidespace - 1;
        x2 = x1;
        y2 = YPos + (2 * longlength) + sidespace  - 1;

        for (i = 1; i <= (.75 * thickness); i++) {
            XDrawLine(TheDisplay, TheDrawable, TheGC,
                x1, y1, x2, y2);
            x1++; 
            x2++;

            y1 += step/2;
            y2 -= step/2;
        }
   
    }
    else if (SectionNum == 6) {
        x1 = XPos + (.25 * thickness) - 1;
        y1 = YPos;
        x2 = x1;
        y2 = YPos + longlength - 1;

        for (i = 1; i <= (.25 * thickness); i++) {
            XDrawLine(TheDisplay, TheDrawable, TheGC,
                x1, y1, x2, y2);
            x1--; 
            x2--;
            y1 += step/2;
            y2 -= step/2;
        }
        x1 = XPos + (.25 * thickness) - 1;
        y1 = YPos;
        x2 = x1;
        y2 = YPos + longlength - 1;

        for (i = 1; i <= (.75 * thickness); i++) {
            XDrawLine(TheDisplay, TheDrawable, TheGC,
                x1, y1, x2, y2);
            x1++; 
            x2++;

            y1 += step/2;
            y2 -= step/2;
        }
    }
    else if (SectionNum == 7) {
        x1 = XPos + thickness - 1;
        y1 = YPos + longlength + (sidespace / 2) - 1; 
        x2 = XPos + thickness + longlength - 1;
        y2 = y1;

        for (i = 1; i <= (thickness / middlefactor); i++) {
            XDrawLine(TheDisplay, TheDrawable, TheGC,
                x1, y1, x2, y2);
            x1 += step/2;
            x2 -= step/2;
            y1++;
            y2++; 
        }

        x1 = XPos + thickness - 1;
        y1 = YPos + longlength + (sidespace / 2) - 1;
        x2 = XPos + thickness + longlength -1;
        y2 = y1;

        for (i = 1; i <= (thickness / middlefactor); i++) {
            XDrawLine(TheDisplay, TheDrawable, TheGC,
                x1, y1, x2, y2);
            x1 += step/2;
            x2 -= step/2;
            y1--;
            y2--; 
        }
    }



}


/******************************************************************************
FUNCTION: DigitalInterpret;  calls the proper sections to be printed.
******************************************************************************/
void DigitalInterpret(Display *TheDisplay, Drawable TheDrawable, GC TheGC,
                      int XPos, int YPos,  /* x and y to draw at (1 digit)*/
                      int longlength, int thickness, int step,
                      int sidespace, /* space on the side of the digit */
                      int forecolor, int shadecolor, 
                      int number)    /* to be printed */
{
    int x;

    switch(number) {
        case 0:
            for(x = 1; x <= 6; x++)
                DigitalSectionPrint(TheDisplay, TheDrawable, TheGC, XPos, YPos,
                                    longlength, thickness, step, sidespace,
                                    forecolor, x);
            DigitalSectionPrint(TheDisplay, TheDrawable, TheGC, XPos, YPos,
                                longlength, thickness, step, sidespace,
                                shadecolor, 7);
        break;
        case 1:
            DigitalSectionPrint(TheDisplay, TheDrawable, TheGC, XPos, YPos,
                                longlength, thickness, step, sidespace,
                                shadecolor, 1);
            for(x = 2; x <= 3; x++)
                DigitalSectionPrint(TheDisplay, TheDrawable, TheGC, XPos, YPos,
                                    longlength, thickness, step, sidespace,
                                    forecolor, x);
            for(x = 4; x <= 7; x++)
                DigitalSectionPrint(TheDisplay, TheDrawable, TheGC, XPos, YPos,
                                    longlength, thickness, step, sidespace,
                                    shadecolor, x);
        break;
        case 2:
            for(x = 1; x <= 2; x++)
                DigitalSectionPrint(TheDisplay, TheDrawable, TheGC, XPos, YPos,
                                    longlength, thickness, step, sidespace,
                                    forecolor, x);
            DigitalSectionPrint(TheDisplay, TheDrawable, TheGC, XPos, YPos,
                                longlength, thickness, step, sidespace,
                                forecolor, 7);
            for(x = 4; x <= 5; x++)
                DigitalSectionPrint(TheDisplay, TheDrawable, TheGC, XPos, YPos,
                                    longlength, thickness, step, sidespace,
                                    forecolor, x);
            for(x = 3; x <= 6; x += 3)
                DigitalSectionPrint(TheDisplay, TheDrawable, TheGC, XPos, YPos,
                                    longlength, thickness, step, sidespace,
                                    shadecolor, x);
        break;
        case 3:
            for(x = 1; x <= 4; x++)
                DigitalSectionPrint(TheDisplay, TheDrawable, TheGC, XPos, YPos,
                                    longlength, thickness, step, sidespace,
                                    forecolor, x);
            DigitalSectionPrint(TheDisplay, TheDrawable, TheGC, XPos, YPos,
                                longlength, thickness, step, sidespace,
                                forecolor, 7);
            for(x = 5; x <= 6; x++)
                DigitalSectionPrint(TheDisplay, TheDrawable, TheGC, XPos, YPos,
                                    longlength, thickness, step, sidespace,
                                    shadecolor, x);
        break;
        case 4:
            for(x = 2; x <= 3; x++)
                DigitalSectionPrint(TheDisplay, TheDrawable, TheGC, XPos, YPos,
                                    longlength, thickness, step, sidespace,
                                    forecolor, x);
            for(x = 6; x <= 7; x++)
                DigitalSectionPrint(TheDisplay, TheDrawable, TheGC, XPos, YPos,
                                    longlength, thickness, step, sidespace,
                                    forecolor, x);
            DigitalSectionPrint(TheDisplay, TheDrawable, TheGC, XPos, YPos,
                                longlength, thickness, step, sidespace,
                                shadecolor, 1);
            for(x = 4; x <= 5; x++)
                DigitalSectionPrint(TheDisplay, TheDrawable, TheGC, XPos, YPos,
                                    longlength, thickness, step, sidespace,
                                    shadecolor, x);
        break;
        case 5:
            DigitalSectionPrint(TheDisplay, TheDrawable, TheGC, XPos, YPos,
                                longlength, thickness, step, sidespace,
                                forecolor, 1);
            for(x = 6; x <= 7; x++)
                DigitalSectionPrint(TheDisplay, TheDrawable, TheGC, XPos, YPos,
                                    longlength, thickness, step, sidespace,
                                    forecolor, x);
            for(x = 3; x <= 4; x++)
                DigitalSectionPrint(TheDisplay, TheDrawable, TheGC, XPos, YPos,
                                    longlength, thickness, step, sidespace,
                                    forecolor, x);
            for(x = 2; x <= 5; x += 3)
                DigitalSectionPrint(TheDisplay, TheDrawable, TheGC, XPos, YPos,
                                    longlength, thickness, step, sidespace,
                                    shadecolor, x);
        break;
        case 6:
            for(x = 3; x <= 7; x++)
                DigitalSectionPrint(TheDisplay, TheDrawable, TheGC, XPos, YPos,
                                    longlength, thickness, step, sidespace,
                                    forecolor, x);
                DigitalSectionPrint(TheDisplay, TheDrawable, TheGC, XPos, YPos,
                                    longlength, thickness, step, sidespace,
                                    forecolor, 1);
                DigitalSectionPrint(TheDisplay, TheDrawable, TheGC, XPos, YPos,
                                    longlength, thickness, step, sidespace,
                                    shadecolor, 2);
        break;
        case 7:
            for(x = 1; x <= 3; x++)
                DigitalSectionPrint(TheDisplay, TheDrawable, TheGC, XPos, YPos,
                                    longlength, thickness, step, sidespace,
                                    forecolor, x);
            for(x = 4; x <= 7; x++)
                DigitalSectionPrint(TheDisplay, TheDrawable, TheGC, XPos, YPos,
                                    longlength, thickness, step, sidespace,
                                    shadecolor, x);
        break;
        case 8:
            for(x = 1; x <= 7; x++)
                DigitalSectionPrint(TheDisplay, TheDrawable, TheGC, XPos, YPos,
                                    longlength, thickness, step, sidespace,
                                    forecolor, x);
        break;
        case 9:
            for(x = 1; x <= 3; x++)
                DigitalSectionPrint(TheDisplay, TheDrawable, TheGC, XPos, YPos,
                                    longlength, thickness, step, sidespace,
                                    forecolor, x);
            for(x = 6; x <= 7; x++)
                DigitalSectionPrint(TheDisplay, TheDrawable, TheGC, XPos, YPos,
                                    longlength, thickness, step, sidespace,
                                    forecolor, x);
            for(x = 4; x <= 5; x++)
                DigitalSectionPrint(TheDisplay, TheDrawable, TheGC, XPos, YPos,
                                    longlength, thickness, step, sidespace,
                                    shadecolor, x);
        break;
    }
}
        

/******************************************************************************
                       END OF DIGITAL SUPPORT FUNCTIONS
******************************************************************************/


/******************************************************************************
FUNCTION: CursorSpin; Spins the cursor!  :-) returns the next array position
******************************************************************************/
int CursorSpin (Display *TheDisplay, Drawable TheDrawable, 
                 Cursor TheCursorArray[], int numcursor, 
                 int direction) /* 1 = right, 0 = left  (in array) */
{
    int x;
    
    if (direction)
            XDefineCursor(TheDisplay, TheDrawable, TheCursorArray[numcursor++]);
    else 
        XDefineCursor(TheDisplay, TheDrawable, TheCursorArray[numcursor--]);

    XFlush(TheDisplay);

    return numcursor;
}

/******************************************************************************
FUNCTION: DrawTriangle;  draws a triangle, with a 2-D option.
******************************************************************************/
void DrawTriangle(Display *TheDisplay, Drawable TheDrawable, GC TheGC,
                         int XPos, int YPos, /* of center of triangle */ 
                         int Size, /* in pixels, heighth of the triangle */
                         int Direction, /* clockwise, 1 = up -> 4 = left */
                         int Height,  /* in pixels, 0 = no 2-D */
                         int DarkColor, /* for 2-D */
                         int MainColor) /* main triangle color */
{
    int x1, x2, y1, y2, i;

    if (Height) {
        DrawTriangle(TheDisplay, TheDrawable, TheGC, 
                     XPos , YPos , 
                     Size, Direction,
                     0, 
                     DarkColor, DarkColor);
        }


    XSetForeground(TheDisplay, TheGC, MainColor);


    x1 = x2 = XPos - ((MainColor == DarkColor) ? 0  : (Height * .65));
    y1 = y2 = YPos - ((MainColor == DarkColor) ? 0 : (Height));

    switch (Direction) {

    case 1:
        y1 = y2 = y1 - Size / 2;
        for (i = 1; i <= Size; i++) {
            XDrawLine(TheDisplay, TheDrawable, TheGC,
                      x1--, y1,
                      x2++, y1); 
            y1++;
        }
        XFlush(TheDisplay);
        break;

    case 2:

        x1 = x2 = x1 + Size / 2;
        for (i = 1; i <= Size; i++) {
            XDrawLine(TheDisplay, TheDrawable, TheGC,
                      x1, y1--,
                      x1, y2++);
            x1--;
        }
        XFlush(TheDisplay);
        break;

    case 3:

    y1 = y2 = y1 + Size / 2;
    
    for (i = 1; i <= Size; i++) {
            XDrawLine(TheDisplay, TheDrawable, TheGC,
                      x1--, y1,
                      x2++, y1);
            y1--;
        }
        XFlush(TheDisplay);
        break;
	 
    case 4:

        x1 = x2 = x1 - Size / 2;
        for (i = 1; i <= Size; i++) {
            XDrawLine(TheDisplay, TheDrawable, TheGC,
                      x1, y1--,
                      x1, y2++);
            x1++;
        }
        XFlush(TheDisplay);
        break;
    }
}
