/*
 * Serial line (modem) status program
 * ncurses code based (sort of) on gdc distributed with ncurses 1.9.7
 *
 * Copyright
 * =========
 * This program is copyright P.J.H.Fox (fox@roestock.demon.co.uk)
 * and distributed under the GPL (see COPYING).
 * 
 * This copyright notice is not to be removed or altered.
 *
 * If mailing patches, bug reports or comments, please put 'serialmon'
 * somewhere in the subject line, and mail to serialmon@roestock.demon.co.uk.
 *
 */

#define USECURSES
#define CHECKLOCKFILE
#define LOOKUPPROCESS

#include <time.h>
#include <signal.h>
#ifdef USECURSES
#include <ncurses/curses.h>
#endif
#include <stdlib.h>
#include <string.h>
#ifdef CHECKLOCKFILE
#include <sys/stat.h>
#endif
#ifndef NONPOSIX
#include <unistd.h>
#endif
#include <termios.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/time.h>
#include <fcntl.h>
#include <stdio.h>

#ifndef UPDATEINTERVAL
#define UPDATEINTERVAL (50000) /* microseconds */
#endif

#ifndef NICEVAL
#define NICEVAL (-10)
#endif

/*
 * Define LEDS if you want LED simulation
 */
#define LEDS

#ifndef USECURSES
#undef LEDS
#endif
/*
 * Line up/down detection:
 * LINEUPDTR   LINEUPDTRANDCAR   Meaning
 *   UNDEF           -           Line is up if another process has the line open (info.count > 1)
 *  DEFINED        UNDEF         Line is up if DTR set (probably won't work now hangup code changed)
 *  DEFINED       DEFINED        Line is up if DTR and CAR is set
 *
 * If LOGCAR defined, then Carrier detect status logged, not
 * very useful if LINEUPDTRANDCAR defined.
 * Similarly LOGDTR for DTR status, but note this is not much
 * use if LINEUPDTR defined.
 *
 * CHECKLOCKFILE specifies an additional test, before the line is
 * deemed to be up.
 */

#undef LINEUPDTR

#ifndef LINEUPDTR
#undef LINEUPDTRANDCAR	/* Can't be defined unless LINEUPDTR is too */
#endif

#define LOGDTR
#define LOGCAR

#ifndef LINEUPDTR
#define __KERNEL__
#include <linux/serial.h>	/* For the definition of async_struct */
#endif

static char rcsid[] = "$Id: serialmon.c,v 0.20 1997/06/08 22:53:18 fox Exp $";

/*
 * Display
 * +---------------------+
 * | RI  DCD RXD TXD DTR |
 * | ### ### ### ### ### |
 * +---------------------+
 * RI  = Ring indicator
 * DCD = Data carrier detect
 * RXD = Receive data
 * TXD = Transmit data
 * DTR = Data terminal Ready is on
 *
 * Stats Table
 * +-----+------------+------------+-----------+------------+-----------+
 * |     |    Chars   |   Latest   | Peak (1s) | Peak (10s) |   Average |
 * |     |    Bytes   | Bytes/sec  | Bytes/sec | Bytes/sec  | Bytes/sec |
 * +-----+------------+------------+-----------+------------+-----------+
 * | Rxd |            |            |           |            |           |
 * +-----+------------+------------+-----------+------------+-----------+
 * | Txd |            |            |           |            |           |
 * +-----+------------+------------+-----------+------------+-----------+
 *
 * Times Table
 * 
 * +-----------+---------------------+
 * | Line      | /dev/ttyS2          |
 * +-----------+---------------------+
 * | Line Up   | 1996/04/10 04:20:28 |
 * +-----------+---------------------+
 * | Line Down | 1996/04/10 04:20:28 |
 * +-----------+---------------------+
 * | Duration  |                     |
 * +-----------+---------------------+
 * | Idle  Rxd |                     |
 * |       Txd |                     |
 * +-----------+---------------------+
 */
#ifdef USECURSES
#ifdef LEDS
#define YBASE	9
#define XBASE	8
#define XLENGTH	24
#define YDEPTH	 2
#define YLEDS	 (YBASE + 1)
#endif

#define NOCOLS 6
unsigned char tablewidths[NOCOLS] = { 5, 12, 12, 12, 12, 12};
unsigned char tablestarts[NOCOLS];
unsigned char *tablehead1[NOCOLS] = { "", "Chars", "Latest", "Peak (1s)", "Peak (10s)", "Average"};
unsigned char *tablehead2[NOCOLS] = { "", "Bytes", "Bytes/sec", "Bytes/sec", "Bytes/sec",  "Bytes/sec"};

#define STATSYBASE	18
#define STATSXBASE	2
#define RXDY		(STATSYBASE + 3)
#define TXDY		(STATSYBASE + 5)
#define STATSCHARS	tablestarts[1]
#define STATSLATEST	tablestarts[2]
#define STATSPEAK		tablestarts[3]
#define STATSPEAK10	tablestarts[4]
#define STATSAVERAGE	tablestarts[5]

#define TIMESYBASE	5
#define TIMESXBASE	39
#define TIMESCOLS	2
#define TIMESROWS	10	/* Excluding the top and bottom */
unsigned char timeswidths[TIMESCOLS] = { 11, 21};
unsigned char *timesrows[TIMESROWS] = { "Line", 0, "Line   up", 0, "Line down", 0, "Duration ", 0, "Idle  Rxd", "      Txd"};
#define TIMESX		(TIMESXBASE + 12)
#define TIMESDEV	(TIMESYBASE + 0)
#define UPY			(TIMESYBASE + 2)
#define DOWNY		(TIMESYBASE + 4)
#define TIMESTIME	(TIMESYBASE + 6)
#define TIMESIDLER	(TIMESYBASE + 8)
#define TIMESIDLET	(TIMESYBASE + 9)
#define COUNTX      (TIMESXBASE + 30)

#define PROCESSY    16
#define PROCESSX    3
#endif

#define SPEEDUPDATE 1.0

#define LINSIZ 256
char reportline[LINSIZ];
#ifdef LOOKUPPROCESS
char processline[LINSIZ];
#endif /* LOOKUPPROCESS */

/*
 * Now and previous times to work out data rates
 * Need the microseconds
 */
struct timeval start;	/* When line became active */
struct timeval end;		/* When line became inactive */
struct timeval prev;	/* Time of previous ioctls */
struct timeval prevtime;	/* Time of previous ioctls, which updated the display */
struct timeval now;		/* Latest ioctls */
struct timeval reopen;	/* When sighup received, or get I/O error */
struct timezone tz;		/* Same for all */
struct tm *tm;

int sigtermed = 0;
char errbuf[LINSIZ];	/* For the error message for the logfile */
char * sigreopen = 0;

int hascolor = 0;

int fd = STDIN_FILENO;	/* File descriptor of the open serial line */
char *devname = "<stdin>";

#ifdef CHECKLOCKFILE
#define LOCKFILE "/var/lock/LCK.."
char *lockfile = "";
static struct stat statbuf;
static int lockingpid;
static int lockfd = -1;
#endif

#ifdef LEDS
void standt(int);
#endif

static
void sighndl(int signo)
{
	signal(signo, sighndl);
	sprintf(errbuf, "%s", strsignal(signo));
	fprintf(stderr, "Got signal %d (%s)\n", signo, errbuf);
	fflush(stderr);
	sigtermed=signo;
}

void
gettime(struct timeval *tv)
{
	if( gettimeofday(tv, &tz) < 0)
	{
		perror("gettime failed");
	}
}

static
int reopenfd(char * str)
{
	int arg;
	/* Close the old (now useless) file descriptor */
	close(fd);
	/* And reopen it */
	fd = open(devname, O_NONBLOCK | O_RDONLY);
	if(fd < 0)
	{
		sprintf(errbuf, "Failed to reopen device %s - %s", devname, strerror(errno));
		fprintf(stderr, "%s\n", errbuf);
		return(-2);
	}
	gettime(&reopen);
	sigreopen = str;
	/* Remove DTR which was set by opening the line */
	if(ioctl(fd, TIOCMGET, &arg) < 0)
	{
		sprintf(errbuf, "TIOCMGET ioctl failed after reopening device %s - %s", devname, strerror(errno));
		fprintf(stderr, "%s\n", errbuf);
		return(-2);
	}
	/* But only if Carrier not detected, don't care if this works */
	if(!(arg & TIOCM_CAR))
	{
		arg &= ~TIOCM_DTR;
		(void)ioctl(fd, TIOCMSET, &arg);
	}
	return(0);
}

static
void sighuphndl(int signo)
{
	signal(signo, sighuphndl);
	fprintf(stderr, "Got signal %d (SIGHUP)\n", signo);
	fflush(stderr);
	if(reopenfd("Signal HUP") < 0)
	{
		sigtermed = signo;
	}
}

double
usecdiff(struct timeval *old, struct timeval *new)
{
	return (new->tv_sec - old->tv_sec) + (new->tv_usec - old->tv_usec)/1000000.0;
}

/*
 * Not used by TIOCMGET
 */
#define TIORX (1<<16)
#define TIOTX (1<<17)
#define HISTSIZE 10

int
main(int argc, char *argv[])
{
	int arg, prevarg;
#ifdef USECURSES
	int i, j, tmp;
#endif
	unsigned long dataresidue[2] = {0, 0};
	unsigned long data[2];
	unsigned long prevdata[2] = {0, 0};
	unsigned long prevtimedata[2] = {0, 0};
	unsigned long rxdchars, txdchars;
	unsigned long rxdcnthist[HISTSIZE], rxd10tot = 0;
	unsigned long txdcnthist[HISTSIZE], txd10tot = 0;
	struct async_struct info;
#ifndef LINEUPDTR
	int prevcount = 1000; /* Unlikely -- ensure we start with the line down */
#endif
#ifdef LOGCAR
	int hadcarrier = 0;
#endif
	double timeshist[HISTSIZE], time10tot = 0.0, rxd10max = 0.0, txd10max = 0.0;
	int histsize = 0;
	int histwrptr = 0;
	double tdiff, tfloat;
	double rxrate, maxrxrate = 0.0, rxidle = 0.0;
	double txrate, maxtxrate = 0.0, txidle = 0.0;
	int lineup = 0, linewasup = 0;
	FILE *logfile = (FILE *)0;
	signal(SIGINT,sighndl);
	signal(SIGTERM,sighndl);
	signal(SIGKILL,sighndl);
	if((argc > 3)||(argc < 2))
	{
		fprintf(stderr, "Usage: %s device [logfile]", argv[0]);
		exit(2);
	}
	if(argc >= 2)
	{
		char *sptr;
		fd = open(argv[1], O_NONBLOCK | O_RDONLY);
		if(fd < 0)
		{
			fprintf(stderr, "Failed to open device %s - ", argv[1]);
			perror("");
			exit(1);
		}
		devname = argv[1];
#ifdef CHECKLOCKFILE
		/*
		 * Work out the  lock file name - 
		 */
		/* Make sure have enough room */
		lockfile = (char *)malloc(strlen(LOCKFILE) + strlen(argv[1]));
		/* Base path */
		strcpy(lockfile, LOCKFILE);
		/* Now add the device name, strip leading /dev/ if nec */
		sptr = argv[1];
		if(*sptr == '/')
			sptr++;
		if(!strncmp(sptr, "dev/", 4))
			sptr += 4;
		strcat(lockfile, sptr);
#endif /* CHECKLOCKFILE */
		/* Remove DTR which was set by opening the line */
        if(ioctl(fd, TIOCMGET, &arg) < 0)
        {
            perror("TIOCMGET ioctl failed");
            exit(1);
        }
        /* But only if Carrier not detected, don't care if this works */
        if(!(arg & TIOCM_CAR))
        {
        	arg &= ~TIOCM_DTR;
        	(void)ioctl(fd, TIOCMSET, &arg);
		}
	}
	if(argc >= 3)
	{
		logfile = fopen(argv[2], "a");
		if(!logfile)
		{
			fprintf(stderr, "Failed to open logfile %s - ", argv[1]);
			perror("");
			exit(1);
		}
	}
#ifndef USECURSES
	else
	{
		logfile = stdout;
	}
#endif
#ifdef USECURSES
	/*
	 * Get the screen up
	 */
	initscr();
	cbreak();
	noecho();
	nodelay(stdscr, 1);
	hascolor = has_colors();
	if(hascolor) {	
		start_color();
		init_pair(1, COLOR_BLACK, COLOR_RED);
		init_pair(2, COLOR_RED, COLOR_BLACK);
		init_pair(3, COLOR_WHITE, COLOR_BLACK);
		init_pair(4, COLOR_GREEN, COLOR_BLACK);
		init_pair(5, COLOR_YELLOW, COLOR_BLACK);
		attrset(COLOR_PAIR(3));
	}
	clear();
	if(hascolor)
		attrset(COLOR_PAIR(3));
#ifdef LEDS
	/* Draw LED display */
	mvaddch(YBASE - 1,  XBASE - 1, ACS_ULCORNER);
	hline(ACS_HLINE, XLENGTH);
	mvaddch(YBASE - 1,  XBASE + XLENGTH, ACS_URCORNER);
	mvaddch(YBASE + YDEPTH,  XBASE - 1, ACS_LLCORNER);
	hline(ACS_HLINE, XLENGTH);
	mvaddch(YBASE + YDEPTH,  XBASE + XLENGTH, ACS_LRCORNER);
	move(YBASE,  XBASE - 1);
	vline(ACS_VLINE, YDEPTH);
	move(YBASE,  XBASE + XLENGTH);
	vline(ACS_VLINE, YDEPTH);
	mvaddstr(YBASE, XBASE + 2, " RI  DCD RXD TXD DTR");
#endif
	/* Now stats table */
	mvaddch(STATSYBASE - 1,  STATSXBASE - 1, ACS_ULCORNER);
	hline(ACS_HLINE, tablewidths[0]);
	tablestarts[0] = STATSXBASE;
	for(i = 1; i < NOCOLS; i++)
	{
		tablestarts[i] = tablestarts[i-1] + tablewidths[i-1] + 1;
		mvaddch(STATSYBASE - 1,  tablestarts[i] - 1, ACS_TTEE);
		if(tablewidths[i])
			hline(ACS_HLINE, tablewidths[i]);
	}
	mvaddch(STATSYBASE - 1,  tablestarts[NOCOLS-1] + tablewidths[NOCOLS-1], ACS_URCORNER);
	mvaddch(STATSYBASE,  STATSXBASE - 1, ACS_VLINE);
	for(i = 1; i < NOCOLS; i++)
	{
		mvaddch(STATSYBASE,  tablestarts[i] - 1, ACS_VLINE);
		if(*tablehead1[i])
			mvaddstr(STATSYBASE,  tablestarts[i] + (tablewidths[i] + 1 - strlen(tablehead1[i]))/2, tablehead1[i]);
	}
	mvaddch(STATSYBASE,  tablestarts[NOCOLS-1] + tablewidths[NOCOLS-1], ACS_VLINE);
	mvaddch(STATSYBASE+1,  STATSXBASE - 1, ACS_VLINE);
	for(i = 1; i < NOCOLS; i++)
	{
		mvaddch(STATSYBASE+1,  tablestarts[i] - 1, ACS_VLINE);
		if(*tablehead2[i])
			mvaddstr(STATSYBASE+1,  tablestarts[i] + (tablewidths[i] + 1 - strlen(tablehead2[i]))/2, tablehead2[i]);
	}
	mvaddch(STATSYBASE+1,  tablestarts[NOCOLS-1] + tablewidths[NOCOLS-1], ACS_VLINE);
	for(j = 2; j < 6; j += 1) /* Really += 2, but one is in the loop */
	{
		mvaddch(STATSYBASE+j,  STATSXBASE - 1, ACS_LTEE);
		hline(ACS_HLINE, tablewidths[0]);
		for(i = 1; i < NOCOLS; i++)
		{
			mvaddch(STATSYBASE+j,  tablestarts[i] - 1, ACS_PLUS);
			if(tablewidths[i])
				hline(ACS_HLINE, tablewidths[i]);
		}
		mvaddch(STATSYBASE+j,  tablestarts[NOCOLS-1] + tablewidths[NOCOLS-1], ACS_RTEE);
		j++;
		mvaddch(STATSYBASE+j,  STATSXBASE - 1, ACS_VLINE);
		for(i = 1; i < NOCOLS; i++)
		{
			mvaddch(STATSYBASE+j,  tablestarts[i] - 1, ACS_VLINE);
		}
		mvaddch(STATSYBASE+j,  tablestarts[NOCOLS-1] + tablewidths[NOCOLS-1], ACS_VLINE);
	}
	mvaddch(STATSYBASE+j,  STATSXBASE - 1, ACS_LLCORNER);
	hline(ACS_HLINE, tablewidths[0]);
	for(i = 1; i < NOCOLS; i++)
	{
		mvaddch(STATSYBASE+j,  tablestarts[i] - 1, ACS_BTEE);
		if(tablewidths[i])
			hline(ACS_HLINE, tablewidths[i]);
	}
	mvaddch(STATSYBASE+j,  tablestarts[NOCOLS-1] + tablewidths[NOCOLS-1], ACS_LRCORNER);
	mvaddstr(RXDY, tablestarts[0]+1, "Rxd");
	mvaddstr(TXDY, tablestarts[0]+1, "Txd");
	/* Now the times table */
	mvaddch(TIMESYBASE - 1,  TIMESXBASE - 1, ACS_ULCORNER);
	hline(ACS_HLINE, timeswidths[0]);
	tmp = TIMESXBASE;
	for(i = 1; i < TIMESCOLS; i++)
	{
		tmp += timeswidths[i-1] + 1;
		mvaddch(TIMESYBASE - 1,  tmp - 1, ACS_TTEE);
		if(timeswidths[i])
			hline(ACS_HLINE, timeswidths[i]);
	}
	mvaddch(TIMESYBASE - 1,  tmp + timeswidths[TIMESCOLS-1], ACS_URCORNER);
	for(j = 0; j < TIMESROWS; j++)
	{
		if(timesrows[j])
		{
			mvaddch(TIMESYBASE + j,  TIMESXBASE - 1, ACS_VLINE);
			mvaddstr(TIMESYBASE + j, TIMESXBASE + 1, timesrows[j]);
			tmp = TIMESXBASE;
			for(i = 1; i < TIMESCOLS; i++)
			{
				tmp += timeswidths[i-1] + 1;
				mvaddch(TIMESYBASE + j,  tmp - 1, ACS_VLINE);
			}
			mvaddch(TIMESYBASE + j,  tmp + timeswidths[TIMESCOLS-1], ACS_VLINE);
		}
		else
		{
			mvaddch(TIMESYBASE + j,  TIMESXBASE - 1, ACS_LTEE);
			hline(ACS_HLINE, timeswidths[0]);
			tmp = TIMESXBASE;
			for(i = 1; i < TIMESCOLS; i++)
			{
				tmp += timeswidths[i-1] + 1;
				mvaddch(TIMESYBASE + j,  tmp - 1, ACS_PLUS);
				if(timeswidths[i])
					hline(ACS_HLINE, timeswidths[i]);
			}
			mvaddch(TIMESYBASE + j,  tmp + timeswidths[TIMESCOLS-1], ACS_RTEE);
		}
	}
	mvaddch(TIMESYBASE + j,  TIMESXBASE - 1, ACS_LLCORNER);
	hline(ACS_HLINE, timeswidths[0]);
	tmp = TIMESXBASE;
	for(i = 1; i < TIMESCOLS; i++)
	{
		tmp += timeswidths[i-1] + 1;
		mvaddch(TIMESYBASE + j,  tmp - 1, ACS_BTEE);
		if(timeswidths[i])
			hline(ACS_HLINE, timeswidths[i]);
	}
	mvaddch(TIMESYBASE + j,  tmp + timeswidths[TIMESCOLS-1], ACS_LRCORNER);
	if(hascolor)
		attrset(COLOR_PAIR(5));
	mvaddstr(TIMESDEV,  TIMESX+1, devname);
	if(hascolor)
		attrset(COLOR_PAIR(2));
	move(1, 1); /* Incase of error */
	refresh();
#endif
	/* Don't want sig hup ! */
	if( signal(SIGHUP, sighuphndl) < 0)
	{
		perror("Couldn't catch signal HUP");
	}
	/* Initialise the time variables, in case */
	gettime(&start);
	gettime(&end);
	gettime(&prev);
	gettime(&prevtime);
	prevarg = 0;
	if(logfile)
	{
		gettime(&now);
		tm = localtime((time_t *)&now.tv_sec);
		sprintf(reportline, " %04d/%02d/%02d %02d:%02d:%02d",
			tm->tm_year+1900, tm->tm_mon+1, tm->tm_mday,
			tm->tm_hour, tm->tm_min, tm->tm_sec);
		fprintf(logfile, "\n%s\nLogging for line %s started at %s\n", rcsid, devname, reportline);
		fflush(logfile);
	}
	nice(NICEVAL);     /* Up our priority to minimise time errors */
	while(!sigtermed)
	{
		/*
		 * Quickly do the IOCTLs
		 */
#ifdef USECURSES
		int updated = 0;
#endif
		if(ioctl(fd, TIOCSERGSTRUCT, &info) < 0)
		{
			/* Oh, line must have hupped, reopen it */
			sigtermed = reopenfd("ioctl TIOCSERGSTRUCT");
			continue;
		}
		/*
		 * Really want the next two functions to have
		 * the same interval every time
		 */
		gettime(&now);
		if(ioctl(fd, TIOCMGET, &arg) < 0)
		{
			/* Oh, line must have hupped, reopen it */
			sigtermed = reopenfd("ioctl TIOCMGET");
			continue;
		}
		/* Don't update the counters if we're the only process */
		if(info.count > 1)
		{
			data[0] = info.rx_char_count - dataresidue[0];
			data[1] = info.tx_char_count - dataresidue[1];
			/*
			 * Now update the display
			 */
			/* Someone's just reset the byte counters */
			if((data[0] < prevdata[0])||(data[1] < prevdata[1]))
			{
				data[0] += dataresidue[0];
				data[1] += dataresidue[1];
				dataresidue[0] = prevdata[0] = prevtimedata[0] = 0;
				dataresidue[1] = prevdata[1] = prevtimedata[1] = 0;
			}
		}
#ifdef LEDS
		/*
		 * generate pseudo bits
		 */
		tdiff = usecdiff(&prev, &now);
		if(data[0] > prevdata[0])
			arg |= TIORX;
		else
			rxidle += tdiff;
		if(data[1] > prevdata[1])
			arg |= TIOTX;
		else
			txidle += tdiff;
#endif
		/* Check each bit for changed status */
		if((arg ^ prevarg) & TIOCM_DTR)
		{
#ifdef LEDS
			standt(arg & TIOCM_DTR);
			mvaddstr(YLEDS, XBASE + 19, "   ");
			updated++;
#endif
#ifdef LINEUPDTR
			if(arg & TIOCM_DTR)
			{
#ifdef LINEUPDTRANDCAR
				if(arg & TIOCM_CAR)
					lineup = 1;
#else
#ifdef LINEUPDTR
				lineup = 1;
#endif
#endif
			}
			else
				lineup = 0;
#endif
#ifdef LOGDTR
			if(logfile)
			{
				tm = localtime((time_t *)&now.tv_sec);
				sprintf(reportline, " %04d/%02d/%02d %02d:%02d:%02d",
					tm->tm_year+1900, tm->tm_mon+1, tm->tm_mday,
					tm->tm_hour, tm->tm_min, tm->tm_sec);
				if(arg & TIOCM_DTR)
					fprintf(logfile, "DTR   set: %s\n", reportline);
				else
					fprintf(logfile, "DTR clear: %s\n", reportline);
				fflush(logfile);
			}
#endif
		}
		if((arg ^ prevarg) & TIOCM_CAR)
		{
#ifdef LEDS
			standt(arg & TIOCM_CAR);
			mvaddstr(YLEDS, XBASE + 7, "   ");
			updated++;
#endif
#ifdef LINEUPDTRANDCAR
			if(arg & TIOCM_CAR)
			{
				if(arg & TIOCM_DTR)
					lineup = 1;
			}
			else
				lineup = 0;
#endif
#ifdef LOGCAR
			if(logfile)
			{
				tm = localtime((time_t *)&now.tv_sec);
				sprintf(reportline, " %04d/%02d/%02d %02d:%02d:%02d",
					tm->tm_year+1900, tm->tm_mon+1, tm->tm_mday,
					tm->tm_hour, tm->tm_min, tm->tm_sec);
				if(arg & TIOCM_CAR)
				{
					hadcarrier = 1;
					fprintf(logfile, "DCD found: %s\n", reportline);
				}
				else
					fprintf(logfile, "DCD  lost: %s\n", reportline);
				fflush(logfile);
			}
#endif
		}
#ifndef LINEUPDTR
		if(info.count != prevcount)
		{
#ifdef USECURSES
			if(hascolor)
				attrset(COLOR_PAIR(4));
			sprintf(reportline, "%2d", info.count);
			mvaddstr(TIMESDEV, COUNTX, reportline);
			updated++;
#endif
#ifndef CHECKLOCKFILE
			if(info.count > 1)
				lineup = 1;
			else
				lineup = 0;
#endif
			prevcount = info.count;
		}
#endif
#ifdef CHECKLOCKFILE
		if(info.count > 1)
		{
			if(lockfd == -1)
				lockfd = open(lockfile, O_RDONLY);
			if((lockfd != -1)&&(fstat(lockfd, &statbuf), statbuf.st_nlink))
			{
				lineup = 1;
				if(!lockingpid)
				{
					FILE * fp;
#ifdef LOOKUPPROCESS
					FILE * procfp;
#endif /* LOOKUPPROCESS */
					fp = fdopen(lockfd, "r");
					if(fp)
					{
						fscanf(fp, "%d", &lockingpid);
						if(logfile)
						{
							fprintf(logfile, "\nProcess  : %d\n", lockingpid);
							fflush(logfile);
						}
#ifdef LOOKUPPROCESS
						sprintf(reportline, "/proc/%d/cmdline", lockingpid);
						procfp = fopen(reportline, "r");
						if(procfp)
						{
							int count = fread(reportline, sizeof(char), LINSIZ-1, procfp);
							fclose(procfp);
							reportline[count] = 0;
							while(--count)
								if(!reportline[count])
									reportline[count] = ' ';
							fprintf(logfile, "Command  : %s\n", reportline);
						}
#ifdef USECURSES
						sprintf(processline, "%5d: %-*.*s", lockingpid, LINSIZ-8, LINSIZ-8, reportline);
						processline[COLS-PROCESSX] = 0;
						if(hascolor)
							attrset(COLOR_PAIR(4));
						mvaddstr(PROCESSY, PROCESSX, processline);
#endif
#endif /* LOOKUPPROCESS */
					}
					else
						lockingpid = -1;	/* Prevent futile efforts */
				}
			}
			else
			{
				if(lockfd != -1)
					close(lockfd);
				lockfd = -1;
				lockingpid = lineup = 0;
			}
		}
		else
			lineup = 0;
#endif
		/* Initialise stuff when the line goes up */
		if(!linewasup && lineup)
		{
#ifdef LOGCAR
			if(arg & TIOCM_CAR)
				hadcarrier = 1;
			else
				hadcarrier = 0;
#endif
			/* Line up */
			start.tv_sec = now.tv_sec;
			start.tv_usec = now.tv_usec;
			/* Initialise data rate stuff */
			prevtime.tv_sec = prev.tv_sec = now.tv_sec;
			prevtime.tv_usec = prev.tv_usec = now.tv_usec;
			prevtimedata[0] = 0;
			prevtimedata[1] = 0;
			dataresidue[0] += data[0];
			dataresidue[1] += data[1];
			data[0] = 0;
			data[1] = 0;
			maxrxrate = maxtxrate = 0.0;
			histsize = histwrptr = 0;
			rxd10tot = rxd10tot = 0;
			time10tot = 0.0;
			rxidle = txidle = 0.0;
#ifdef USECURSES
			if(hascolor)
				attrset(COLOR_PAIR(4));
			/* Clear out table */
			for(j = RXDY; j <= TXDY; j += (TXDY - RXDY))
			{
				for(i = 1; i < NOCOLS; i++)
				{
					sprintf(reportline,  "%*s", tablewidths[i], "");
					mvaddstr(j, tablestarts[i], reportline);
				}
			}
			/* Clear out times, but not the device ! */
			for(j = 2; j < TIMESROWS; j++)
			{
				if(!timesrows[j])
					continue;
				tmp = TIMESXBASE;
				for(i = 1; i < TIMESCOLS; i++)
				{
					tmp += timeswidths[i-1] + 1;
					sprintf(reportline,  "%*s", timeswidths[i], "");
					mvaddstr(TIMESYBASE + j, tmp, reportline);
				}
			}
#endif
			/* Show up time */
			tm = localtime((time_t *)&now.tv_sec);
			sprintf(reportline, " %04d/%02d/%02d %02d:%02d:%02d",
				tm->tm_year+1900, tm->tm_mon+1, tm->tm_mday,
				tm->tm_hour, tm->tm_min, tm->tm_sec);
#ifdef USECURSES
			mvaddstr(UPY, TIMESX, reportline);
#endif
			if(logfile)
			{
#ifdef CHECKLOCKFILE
				fprintf(logfile, "Line   up: %s\n", reportline);
#else
				fprintf(logfile, "\nLine   up: %s\n", reportline);
#endif
				fflush(logfile);
			}
		}
#ifdef LEDS
		if((arg ^ prevarg) & TIOCM_RNG)
		{
			standt(arg & TIOCM_RNG);
			mvaddstr(YLEDS, XBASE + 3, "   ");
			updated++;
		}
		if((arg ^ prevarg) & TIORX)
		{
			standt(arg & TIORX);
			mvaddstr(YLEDS, XBASE + 11, "   ");
			updated++;
		}
		if((arg ^ prevarg) & TIOTX)
		{
			standt(arg & TIOTX);
			mvaddstr(YLEDS, XBASE + 15, "   ");
			updated++;
		}
#endif
		prevarg = arg;
		prevdata[0] = data[0];
		prevdata[1] = data[1];
		/*
		printf("%3.3s ", arg & TIOCM_RTS ? "RTS" : "");
		printf("%3.3s ", arg & TIOCM_DTR ? "DTR" : "");
		printf("%3.3s ", arg & TIOCM_CAR ? "DCD" : "");
		printf("%3.3s ", arg & TIOCM_RNG ? "RI" : "");
		printf("%3.3s ", arg & TIOCM_DSR ? "DSR" : "");
		printf("%3.3s ", arg & TIOCM_CTS ? "CTS" : "");
		*/
		/* Line just gone down */
		if(!lineup && linewasup)
		{
			/* Line down */
			end.tv_sec = now.tv_sec;
			end.tv_usec = now.tv_usec;
			tm = localtime((time_t *)&now.tv_sec);
			sprintf(reportline, " %04d/%02d/%02d %02d:%02d:%02d",
				tm->tm_year+1900, tm->tm_mon+1, tm->tm_mday,
				tm->tm_hour, tm->tm_min, tm->tm_sec);
#ifdef USECURSES
			if(hascolor)
				attrset(COLOR_PAIR(4));
			mvaddstr(DOWNY, TIMESX, reportline);
#ifdef LOOKUPPROCESS
			if(hascolor)
				attrset(COLOR_PAIR(2));
			mvaddstr(PROCESSY, PROCESSX, processline);
#endif /* LOOKUPPROCESS */
#endif
			if(logfile)
			{
				fprintf(logfile, "Line down: %s\n", reportline);
				fflush(logfile);
			}
		}
		/* Need to accumulate transient byte rates per sec here */
		if((linewasup || lineup) && (((tdiff = usecdiff(&prevtime, &now)) > SPEEDUPDATE) || !lineup))
		{
			/* Update transient speed */
			rxdchars = data[0] - prevtimedata[0];
			txdchars = data[1] - prevtimedata[1];
			rxrate = rxdchars/tdiff;
			txrate = txdchars/tdiff;
#ifdef USECURSES
			if(hascolor)
				attrset(COLOR_PAIR(4));
#endif
			if(rxrate > maxrxrate)
			{
				maxrxrate = rxrate;
#ifdef USECURSES
				sprintf(reportline, " %9.1f", maxrxrate);
				mvaddstr(RXDY, STATSPEAK, reportline);
#endif
			}
			if(txrate > maxtxrate)
			{
				maxtxrate = txrate;
#ifdef USECURSES
				sprintf(reportline, " %9.1f", maxtxrate);
				mvaddstr(TXDY, STATSPEAK, reportline);
#endif
			}
#ifdef USECURSES
			/* Update idletimes */
			sprintf(reportline, " %7d:%02d", (int)rxidle/60, (int)rxidle % 60);
			mvaddstr(TIMESIDLER, TIMESX, reportline);
			sprintf(reportline, " %7d:%02d", (int)txidle/60, (int)txidle % 60);
			mvaddstr(TIMESIDLET, TIMESX, reportline);
#endif
			/* Update average of 10 speeds */
			if(histsize < HISTSIZE)
			{
				/* Still starting up */
				rxdcnthist[histwrptr] = rxdchars;
				rxd10tot += rxdcnthist[histwrptr];
				txdcnthist[histwrptr] = txdchars;
				txd10tot += txdcnthist[histwrptr];
				timeshist[histwrptr] = tdiff;
				time10tot += tdiff;
				histwrptr++;
				histsize++;
				if(histwrptr >= HISTSIZE)
				{
					rxd10max = rxd10tot/time10tot;
					txd10max = txd10tot/time10tot;
#ifdef USECURSES
					sprintf(reportline, " %9.1f", rxd10max);
					mvaddstr(RXDY, STATSPEAK10, reportline);
					sprintf(reportline, " %9.1f", txd10max);
					mvaddstr(TXDY, STATSPEAK10, reportline);
#endif
					histwrptr = 0;
				}
			}
			else
			{
				/* First subtract oldest numbers */
				rxd10tot -= rxdcnthist[histwrptr];
				txd10tot -= txdcnthist[histwrptr];
				time10tot -= timeshist[histwrptr];
				/* Then add in the new */
				rxdcnthist[histwrptr] = rxdchars;
				rxd10tot += rxdcnthist[histwrptr];
				txdcnthist[histwrptr] = txdchars;
				txd10tot += txdcnthist[histwrptr];
				timeshist[histwrptr] = tdiff;
				time10tot += tdiff;
				histwrptr++;
				if(histwrptr >= HISTSIZE)
					histwrptr = 0;
				/* Use tdiff as temporary */
				tdiff = rxd10tot/time10tot;
				if(tdiff > rxd10max)
				{
					rxd10max = tdiff;
#ifdef USECURSES
					sprintf(reportline, " %9.1f", rxd10max);
					mvaddstr(RXDY, STATSPEAK10, reportline);
#endif
				}
				tdiff = txd10tot/time10tot;
				if(tdiff > txd10max)
				{
					txd10max = tdiff;
#ifdef USECURSES
					sprintf(reportline, " %9.1f", txd10max);
					mvaddstr(TXDY, STATSPEAK10, reportline);
#endif
				}
			}
			tdiff = usecdiff(&start, &now);
#ifdef USECURSES
			sprintf(reportline, " %9.1f", rxrate);
			mvaddstr(RXDY, STATSLATEST, reportline);
			sprintf(reportline, " %9.1f", txrate);
			mvaddstr(TXDY, STATSLATEST, reportline);
			sprintf(reportline, " %7d:%02d", (int)tdiff/60, (int)tdiff % 60);
			mvaddstr(TIMESTIME, TIMESX, reportline);
			sprintf(reportline, " %9ld", data[0]);
			mvaddstr(RXDY, STATSCHARS, reportline);
			sprintf(reportline, " %9.1f", data[0]/tdiff);
			mvaddstr(RXDY, STATSAVERAGE, reportline);
			sprintf(reportline, " %9ld", data[1]);
			mvaddstr(TXDY, STATSCHARS, reportline);
			sprintf(reportline, " %9.1f", data[1]/tdiff);
			mvaddstr(TXDY, STATSAVERAGE, reportline);
			updated++;
#endif
			prevtime.tv_sec = now.tv_sec;
			prevtime.tv_usec = now.tv_usec;
			prevtimedata[0] = data[0];
			prevtimedata[1] = data[1];
			/* if(!lineup) can write final report */
			if((!lineup) && (logfile))
			{
				fprintf(logfile, "Duration : %14d:%02d.%02d", (int)tdiff/60, ((int)(tdiff * 100) % 6000)/100, ((int)(tdiff * 100) % 6000)%100);
#ifndef LINEUPDTRANDCAR
				if(hadcarrier)
					fprintf(logfile, " (Data connection achieved)");
				else
					fprintf(logfile, " (No data connection, or FAX call)");
#endif
				fprintf(logfile, "\n         Chars    Activetime      Idletime    Peak(1s) Peak(10s) ActiveAvg   Average\n");
				tfloat = tdiff - rxidle;
				fprintf(logfile, "Rxd: %9ld %7d:%02d.%02d", data[0], (int)tfloat/60, ((int)(tfloat * 100) % 6000)/100, ((int)(tfloat * 100) % 6000)%100);
				fprintf(logfile, " %7d:%02d.%02d  %9.1f", (int)rxidle/60, ((int)(rxidle * 100) % 6000)/100, ((int)(rxidle * 100) % 6000)%100, maxrxrate);
				if(histsize >= HISTSIZE)
					fprintf(logfile, " %9.1f", rxd10max);
				else
					fprintf(logfile, " %9s", "-");
				if(tfloat > 0.1)
					fprintf(logfile, " %9.1f", data[0]/tfloat);
				else
					fprintf(logfile, " %9s", "-");
				fprintf(logfile, " %9.1f\n", data[0]/tdiff);
				tfloat = tdiff - txidle;
				fprintf(logfile, "Txd: %9ld %7d:%02d.%02d", data[1], (int)tfloat/60, ((int)(tfloat * 100) % 6000)/100, ((int)(tfloat * 100) % 6000)%100);
				fprintf(logfile, " %7d:%02d.%02d  %9.1f", (int)txidle/60, ((int)(txidle * 100) % 6000)/100, ((int)(txidle * 100) % 6000)%100, maxtxrate);
				if(histsize >= HISTSIZE)
					fprintf(logfile, " %9.1f", txd10max);
				else
					fprintf(logfile, " %9s", "-");
				if(tfloat > 0.1)
					fprintf(logfile, " %9.1f", data[1]/tfloat);
				else
					fprintf(logfile, " %9s", "-");
				fprintf(logfile, " %9.1f\n", data[1]/tdiff);
				fflush(logfile);
			}
		}
		linewasup = lineup;
#ifdef USECURSES
		if(updated)
		{
			move(1,1);
			refresh();
		}
#endif
		prev.tv_sec = now.tv_sec;
		prev.tv_usec = now.tv_usec;
		usleep(UPDATEINTERVAL);
		/* Do this after the sleep, as that's when it's most likely to happen */
		if(sigreopen && logfile)
		{
			tm = localtime((time_t *)&reopen.tv_sec);
			sprintf(reportline, " %04d/%02d/%02d %02d:%02d:%02d",
				tm->tm_year+1900, tm->tm_mon+1, tm->tm_mday,
				tm->tm_hour, tm->tm_min, tm->tm_sec);
			fprintf(logfile, "Reopen at: %s (due to %s)\n", reportline, sigreopen);
			fflush(logfile);
			sigreopen = 0;
		}
	}
	if(logfile)
	{
		gettime(&now);
		tm = localtime((time_t *)&now.tv_sec);
		sprintf(reportline, " %04d/%02d/%02d %02d:%02d:%02d",
			tm->tm_year+1900, tm->tm_mon+1, tm->tm_mday,
			tm->tm_hour, tm->tm_min, tm->tm_sec);
		fprintf(logfile, "\nLogging for line %s ended due to signal %d ", devname, sigtermed);
		fprintf(logfile, "(%s) ", errbuf);
		fprintf(logfile, "at %s\n", reportline);
		fclose(logfile);
	}
#ifdef USECURSES
	standend();
	/*
	clear();
	*/
	refresh();
	endwin();
#endif
	close(fd);
	exit(0);
}

#ifdef LEDS
void
standt(int on)
{
	if (on) {
		if(hascolor) {
			attron(COLOR_PAIR(1));
		} else {
			attron(A_STANDOUT);	
		}	
	} else {
		if(hascolor) {
			attron(COLOR_PAIR(2));
		} else {
			attroff(A_STANDOUT);
		}
	}
}
#endif

