/*
 * @(#)$Id: record.c,v 1.3 1993/01/22 15:14:24 wcp Exp $
 *
 * Copyright (C) 1993	Walter Pelissero
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * $Log: record.c,v $
 * Revision 1.3  1993/01/22  15:14:24  wcp
 * Removed magic string.
 *
 * Revision 1.2  1993/01/13  21:33:46  wcp
 * Forced default to EXTMIC in case of -c option.
 *
 * Revision 1.1  1993/01/06  18:12:32  wcp
 * Initial revision
 *
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <fcntl.h>
#include <termio.h>
#include <getopt.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <signal.h>
#include <utmp.h>
#include <stdlib.h>
#include "common.h"
#include "modemio.h"
#include "zyxel.h"
#include "voice.h"

static char RcsId[] = "@(#)$Id: record.c,v 1.3 1993/01/22 15:14:24 wcp Exp $";
static Port port;
static bool already_connected = TRUE;
const char *myname;
unsigned allow_digits = 0;
bool allow_fax = TRUE;
#ifdef DEBUG
static int verbosity = 9;
#else
static int verbosity = 0;
#endif
static bool timedout = TRUE;

static SIGTYPE terminate(int sig)
{
  if (verbosity)
    fprintf(stderr, "%s: got signal %d\n", myname, sig);
  if (timedout)
    {
      if (already_connected)
	resetLine(port);
      else
	{
	  mdhangup(port.outfd);
	  mdwrite("ATZ\r", port.outfd);
	  mdflushin(port.infd);
	  closeLine(port);
	}
      exit(0);
    }
  else
    {
      timedout = TRUE;
      signal(sig, terminate);
    }
#ifndef SIGH_VOID
  return 0;
#endif
}

static SIGTYPE timeout(int sig)
{
  timedout = TRUE;
  signal(sig, timeout);
#ifndef SIGH_VOID
  return 0;
#endif
}

typedef enum {
  REC_ERROR = -1,
  REC_OK = 0,
  REC_SILENCE,
  REC_FAX,
  REC_0 = 10,
  REC_1,
  REC_2,
  REC_3,
  REC_4,
  REC_5,
  REC_6,
  REC_7,
  REC_8,
  REC_9,
  REC_D,
  REC_A
  } ReceiveResult;

inline int safe_write(int fd, const char *p, unsigned len)
{
  if (fd < 0)
    return 0;
  return write(fd, p, len);
}

#define ELAPSED		(time(0) - timerStart)

static int receiveMessage(int infd, int outfd,
			  unsigned min_secs, unsigned max_secs)
{
  bool dle = FALSE;
  unsigned phase = 0;
  unsigned long timerStart;

  timedout = FALSE;
  alarm(max_secs);
  timerStart = time(0);
  do
    {
      char buf[1024];
      int received;

      if ((received = read(infd, buf, sizeof(buf))) < 0)
	{
	  alarm(0);
	  timedout = TRUE;
	  return ELAPSED >= max_secs ? REC_OK : REC_ERROR;
	}
      else
	{
	  int mark, i;

	  for (i = mark = 0; i < received; ++i)
	    {
	      if (dle)
		{
		  switch (buf[i])
		    {
		    case DLE:
		      safe_write(outfd, &buf[i], 1);
		      break;
		    case '0':
		    case '1':
		    case '2':
		    case '3':
		    case '4':
		    case '5':
		    case '6':
		    case '7':
		    case '8':
		    case '9':
		      if (verbosity > 6)
			fprintf(stderr, "%s: got DTMF %c\n", myname, buf[i]);
		      if (allow_digits && ELAPSED < min_secs)
			{
			  /* caller typed a digit */
			  timedout = TRUE;
			  return REC_0 + buf[i] - '0';
			}
		      break;
		    case '#':
		    case '*':
		      if (verbosity > 6)
			fprintf(stderr, "%s: got DTMF %c\n", myname, buf[i]);
		      if (allow_digits && ELAPSED < min_secs)
			{
			  /* caller typed '#' or '*' */
			  timedout = TRUE;
			  return buf[i] == '#' ? REC_D : REC_A;
			}
		      break;
		    case 'c':	/* T.30 fax calling tone */
		      if (verbosity > 6)
			fprintf(stderr, "%s: got T.30 fax tone\n", myname);
		      if (allow_fax)
			{
			  /* it was a fax machine */
			  timedout = TRUE;
			  return REC_FAX;
			}
		      break;
		    case 'b':	/* Busy tone */
		      if (verbosity > 6)
			fprintf(stderr, "%s: got busy tone\n", myname);
		      timedout = TRUE;
		      return (ELAPSED < min_secs) ? REC_ERROR : REC_OK;
		    case 'q':	/* quiet after talking */
		      if (verbosity > 6)
			fprintf(stderr, "%s: got quiet\n", myname);
		      timedout = TRUE;
		      /* 5 is a magic number to be parametrized
			 later -wcp1/3/93. */
		      return (ELAPSED - 5 < min_secs) ? REC_SILENCE : REC_OK;
		    case 's':	/* silence (no response) */
		      if (verbosity > 6)
			fprintf(stderr, "%s: got silence\n", myname);
		      timedout = TRUE;
		      return REC_SILENCE; /* not a human */
		    case ETX:
		      alarm(0);
		      timedout = TRUE;
		      return REC_OK;
		    default:
		      break;
		    }
		  dle = FALSE;
		}
	      else
		{
		  if (buf[i] == DLE)
		    {
		      dle = TRUE;
		      safe_write(outfd, &buf[mark], i - mark);
		      mark = i + 2;
		    }
		}
	    }
	  if (mark < i)
	    safe_write(outfd, &buf[mark], received - mark);
	}
      if (timedout)
	{
	  if (phase++)
	    return REC_ERROR;
	  mdwrite("AT\r", 1);	/* close the connection */
	  timedout = FALSE;
	  alarm(30);		/* should be enough to drain input */
	}
    }
  while (!timedout);
  return REC_OK;
}

#undef ELAPSED

static int recordVoice(int outfd, unsigned min_secs, unsigned max_secs,
		       Speaker mic, Port port, VoiceEncoding encoding)
{
  int ret;

  if (verbosity > 3)
    fprintf(stderr, "%s: starting receiving\n", myname);
  if (already_connected && outfd >= 0)
    {
      FILE *outfp = fdopen(outfd, "w");
      long t = time(0);

      fprintf(outfp, "DATE: %s", ctime(&t));
#ifdef NEVER
      fprintf(outfp, "CALLER: unknown\n"); /* currently not implemented */
#endif /* NEVER */
      fprintf(outfp, "ENCODING: %u\n", encoding);
      fprintf(outfp, "DATA:\n");
      fflush(outfp);
    }
  else
    {
      char buf[16];

      mdwrite("AT+FCLASS=8\r", port.outfd);
      if (receive(2, port.infd) != MDOK)
	return REC_ERROR;
      sprintf(buf, "AT+VSM=%u\r", encoding);
      mdwrite(buf, port.outfd);
      if (receive(2, port.infd) != MDOK)
	return REC_ERROR;
      sprintf(buf, "AT+VLS=%u\r", mic);
      mdwrite(buf, port.outfd);
      if (receive(2, port.infd) != MDVCON)
	return REC_ERROR;
    }
  mdwrite("AT+VRX\r", port.outfd);
  if (receive(10, port.infd) != MDCONNECT)
    return REC_ERROR;
  if (verbosity > 4)
    fprintf(stderr, "%s: recording...\n", myname);
  ret = receiveMessage(port.infd, outfd, min_secs, max_secs);
  /* the following line may be a repetition butit is necessary
     to be sure to break any recording */
  mdwrite("AT\r", port.outfd);
  if (verbosity > 7)
    fprintf(stderr, "%s: flushing input...\n", myname);
  mdflushin(port.infd);
  if (!already_connected)
    {
      mdwrite("AT+VLS=0\r", port.outfd);
      if (receive(2, port.infd) != MDOK)
	return REC_ERROR;
      mdwrite("AT+FCLASS=0\r", port.outfd);
      if (receive(2, port.infd) != MDOK)
	return REC_ERROR;
      mdwrite("ATZ\r", port.outfd);
      sleep(2);
      mdflushin(port.infd);
    }
  return ret;
}

int main(unsigned argc, char *argv[])
{
  unsigned max_rec_time = 5 * 60, min_rec_time = 10;
  const char *base = 0;
  int i, ret;
  bool error = FALSE;
  const char *device;
  Speaker mic = TELCOLINE;
  VoiceEncoding encoding = ADPCM2;

  myname = basename(argv[0]);
  while ((i = getopt(argc, argv, "fdt:x:c:m:e:")) != EOF)
    switch (i)
      {
      case 'e':
	if ((encoding = atoi(optarg)) > LAST_VOICEEN ||
	    encoding < FIRST_VOICEEN)
	  {
	    fprintf(stderr, "%s: encoding not in allowable range (%u-%u)\n",
		    myname, FIRST_VOICEEN, LAST_VOICEEN);
	    error = TRUE;
	  }
	break;
      case 'm':
	switch (atoi(optarg))
	  {
	  case 0:
	    mic = TELCOLINE;
	    break;
	  case 1:
	    mic = EXTMIC;
	    break;
	  case 2:
	    mic = INTSPKR;
	    break;
	  default:
	    fprintf(stderr, "%s: invalid speaker (0=telco, 1=mic, 2=int)\n",
		    myname);
	    error = TRUE;
	    break;
	  }
	break;
      case 'c':
	device = optarg;
	already_connected = FALSE;
	mic = EXTMIC;
	break;
      case 'f':
	allow_fax = FALSE;
	break;
      case 'd':
	++allow_digits;
	break;
      case 'x':
	verbosity = atoi(optarg);
	break;
      case 't':
	{
	  char *p = strchr(optarg, ':');

	  if (p)
	    {
	      *p++ = '\0';
	      max_rec_time = atoi(p);
	    }
	  min_rec_time = atoi(optarg);
	}
	break;
      default:
	error = TRUE;
	break;
      }
  if (argc > optind)
    base = argv[optind++];
  if (!base)
    if (already_connected)
      {
	error = TRUE;
	fprintf(stderr, "%s: at least -c or output must be specified\n",
		myname);
      }
    else if (isatty(1))
      {
	error = TRUE;
	fprintf(stderr, "%s: are you sure you want do \"display\" sound?\n",
		myname);
      }
  if (optind < argc)		/* too many arguments */
    error = TRUE;
  if (error)
    {
      fprintf(stderr, "%s\nusage: %s [-x level][-f][-d][-m microphone][-e encoding]
\t[-t min_seconds[:max_seconds][-c device][out{file|dir}]\n", RcsId, myname);
      exit(-1);
    }
  mdverbosity = verbosity;
  signal(SIGALRM, timeout);
  signal(SIGQUIT, terminate);
  signal(SIGTERM, terminate);
  signal(SIGHUP, timeout);
  signal(SIGINT, terminate);
  if (already_connected)
    {
      struct termio tty;

      port.infd = 0;
      port.outfd = 1;
      ioctl(port.outfd, TCGETA, &tty);
      port.orig_tty_settings = tty;
      setLine(port);
    }
  else
    port = openLine(device);
  if (base)
    {
      unsigned nmsg = 1;
      char msg_name[1024];
      struct stat st;

      if (stat(base, &st) == 0 && (st.st_mode & S_IFDIR))
	{
	  /* Base is a directory.
	   */
	  for (; nmsg > 0; ++nmsg)
	    {
	      sprintf(msg_name, "%s/in%08u", base, nmsg);
	      /* We avoid to overwrite existing messages */
	      if (access(msg_name, F_OK))
		break;
	    }
	}
      else
	/* Base is a probably inexistent file name.
	 */
	strcpy(msg_name, base);
      if (nmsg == 0)
	{
	  fprintf(stderr, "%s: spool area is full.\n", myname);
	  if (already_connected)
	    resetLine(port);
	  else
	    {
	      mdhangup(port.outfd);
	      closeLine(port);
	    }
	  exit(1);
	}
      else
	{
	  int outfd;

	  if (allow_digits > 1)
	    outfd = -1;
	  else
	    if ((outfd = open(msg_name, O_WRONLY | O_CREAT, 0644)) < 0)
	      {
		fprintf(stderr, "%s: cannot create %s (%s)\n",
			myname, msg_name, sys_errlist[errno]);
		if (!already_connected)
		  {
		    mdhangup(port.outfd);
		    closeLine(port);
		  }
		exit(2);
	      }
	  ret = recordVoice(outfd, min_rec_time, max_rec_time, mic, port,
			    encoding);
	  if (outfd >= 0)
	    close(outfd);
	  if (ret != REC_OK)
	    unlink(msg_name);
	}
    }
  else
    ret = recordVoice(1, min_rec_time, max_rec_time, mic, port, encoding);
  if (!already_connected)
    {
      mdhangup(port.outfd);
      closeLine(port);
    }
  if (verbosity > 8)
    fprintf(stderr, "%s: returning %d\n", myname, ret);
  return exit(ret), -1;
}
