/*
 * @(#)$Id: play.c,v 1.2 1993/01/13 21:31:54 wcp Exp $
 *
 * Copyright (C) 1993	Walter Pelissero
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * $Log: play.c,v $
 * Revision 1.2  1993/01/13  21:31:54  wcp
 * Changed speaker default to TELCOLINE, and forced default change to
 * INTSPKR if -c is specified.
 *
 * Revision 1.1  1993/01/06  18:10:45  wcp
 * Initial revision
 *
 */

#include <stdio.h>
#include <termio.h>
#include <fcntl.h>
#include <stdlib.h>
#include <errno.h>
#include <getopt.h>
#include <signal.h>
#include "common.h"
#include "modemio.h"
#include "zyxel.h"
#include "voice.h"

static char RcsId[] = "@(#)$Id: play.c,v 1.2 1993/01/13 21:31:54 wcp Exp $";
#ifdef DEBUG
static int verbosity = 9;
#else
static int verbosity = 0;
#endif
const char *myname;
static bool terminated = TRUE;
static Port port;
static bool already_connected = TRUE;

static SIGTYPE terminate(int sig)
{
  if (verbosity)
    fprintf(stderr, "%s: got signal %d\n", myname, sig);
  if (terminated)
    {
      if (already_connected)
	resetLine(port);
      else
	{
	  mdhangup(port.outfd);
	  mdwrite("ATZ\r", port.outfd);
	  mdflushin(port.infd);
	  closeLine(port);
	}
      exit(0);
    }
  else
    {
      terminated = TRUE;
      signal(sig, terminate);
    }
#ifndef SIGH_VOID
  return 0;
#endif
}

static bool playFile(int infd, int outfd)
{
  int nread;
  static char eof[] = { DLE, ETX };
  
  terminated = FALSE;
  do
    {
      char buf[1024];
      int i;

      nread = read(infd, buf, sizeof(buf));
      for (i = 0; i < nread; ++i)
	{
	  if (write(outfd, &buf[i], 1) < 1)
	    nread = -1;
	  else
	    if (buf[i] == DLE)
	      write(outfd, &buf[i], 1);
	}
    }
  while (nread > 0 && !terminated);
  if (write(outfd, eof, sizeof(eof)) < sizeof(eof) || terminated)
    return FALSE;
  else
    {
      terminated = TRUE;
      return TRUE;
    }
}

static bool play(int fd, Port port, Speaker spkr, VoiceEncoding encoding,
		 unsigned volume)
{
  if (!already_connected)
    {
      char buf[32];
      
      if (verbosity > 2)
	fprintf(stderr, "%s: initializing modem\n", myname);
      sprintf(buf, "ATL%u+FCLASS=8\r", volume);
      mdwrite(buf, port.outfd);
      if (receive(2, port.infd) != MDOK)
	return FALSE;
      sprintf(buf, "AT+VSM=%u\r", encoding);
      mdwrite(buf, port.outfd);
      if (receive(2, port.infd) != MDOK)
	return FALSE;
      sprintf(buf, "AT+VLS=%u\r", spkr);
      mdwrite(buf, port.outfd);
      if (receive(2, port.infd) != MDVCON)
	return FALSE;
    }
  if (verbosity > 1)
    fprintf(stderr, "%s: starting playing\n", myname);
  mdwrite("AT+VTX\r", port.outfd);
  if (receive(10, port.infd) != MDCONNECT)
    return FALSE;
  if (verbosity > 4)
    fprintf(stderr, "%s: connected: playing...\n", myname);
  playFile(fd, port.outfd);
  if (verbosity > 4)
    fprintf(stderr, "%s: expecting VCON at end of playing\n", myname);
  if (receive(30, port.infd) != MDVCON)
    return FALSE;
  if (!already_connected)
    {
      if (verbosity > 2)
	fprintf(stderr, "%s: resetting modem\n", myname);
      mdwrite("AT+VLS=0\r", port.outfd);
      if (receive(2, port.infd) != MDOK)
	return FALSE;
      mdwrite("AT+FCLASS=0\r", port.outfd);
      if (receive(2, port.infd) != MDOK)
	return FALSE;
      mdwrite("ATZ\r", port.outfd);
      sleep(2);
      mdflushin(port.infd);
    }
  return TRUE;
}

int main(unsigned argc, char *argv[])
{
  unsigned i;
  VoiceEncoding encoding = ADPCM2;
  unsigned volume = 5;	/* I don't know if it is relevant */
  Speaker spkr = TELCOLINE;
  bool error = FALSE;
  const char *device;

  myname = basename(argv[0]);
  while ((i = getopt(argc, argv, "s:x:c:v:e:")) != EOF)
    switch (i)
      {
      case 'e':
	if ((encoding = atoi(optarg)) > LAST_VOICEEN ||
	    encoding < FIRST_VOICEEN)
	  {
	    fprintf(stderr, "%s: encoding not in allowable range (1-3)\n",
		    myname);
	    error = TRUE;
	  }
	break;
      case 'v':
	if ((volume = atoi(optarg)) > 7 || volume < 1)
	  {
	    fprintf(stderr, "%s: volume not in allowable range (1-7)\n",
		    myname);
	    error = TRUE;
	  }
	break;
      case 'c':
	device = optarg;
	already_connected = FALSE;
	spkr = INTSPKR;
	break;
      case 'x':
	verbosity = atoi(optarg);
	break;
      case 's':
	switch (atoi(optarg))
	  {
	  case 0:
	    spkr = TELCOLINE;
	    break;
	  case 1:
	    spkr = EXTMIC;
	    break;
	  case 2:
	    spkr = INTSPKR;
	    break;
	  default:
	    fprintf(stderr, "%s: invalid speaker (0=telco, 1=mic, 2=int)\n",
		    myname);
	    error = TRUE;
	    break;
	  }
	break;
      default:
	error = TRUE;
	break;
      }
  if (argc - optind < 1)
    if (already_connected)
      {
	fprintf(stderr, "%s: cannot get data data from stdin while connected to stdin/out\n", myname);
	error = TRUE;
      }
    else if (isatty(0))
      {
	fprintf(stderr, "%s: are you sure you want to play from your keyboard?\n", myname);
	error = TRUE;
      }
  if (error)
    {
      fprintf(stderr, "%s\nusage: %s [-s speaker][-x debug_level][-c device][-v volume]
 \t[-e encoding][voice_file ...]\n",
	      RcsId, myname);
      exit(-1);
    }
  mdverbosity = verbosity;
  signal(SIGQUIT, terminate);
  signal(SIGTERM, terminate);
  signal(SIGHUP, terminate);
  signal(SIGINT, terminate);
  if (already_connected)
    {
      struct termio tty;

      port.infd = 0;
      port.outfd = 1;
      ioctl(port.outfd, TCGETA, &tty);
      port.orig_tty_settings = tty;
      setLine(port);
    }
  else
    port = openLine(device);
  if (argc - optind)
    for (i = optind; i < argc; ++i)
      {
	int fd;
      
	if ((fd = open(argv[i], O_RDONLY)) >= 0)
	  {
	    play(fd, port, spkr, encoding, volume);
	    close(fd);
	  }
	else
	  fprintf(stderr, "%s: cannot open %s: skipped\n", myname, argv[i]);
      }
  else
    /* If no files specified in command line, get data from stdin */
    play(0, port, spkr, encoding, volume);
  if (already_connected)
    resetLine(port);
  else
    {
      mdhangup(port.outfd);
      closeLine(port);
    }
  return exit(0), 0;
}
