/*
 * $Id: modemio.c,v 1.1 1993/01/06 18:09:44 wcp Exp $
 *
 * Copyright (C) 1992	Walter Pelissero
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 1, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * $Log: modemio.c,v $
 * Revision 1.1  1993/01/06  18:09:44  wcp
 * Initial revision
 *
 */

#include <sys/types.h>
#include <string.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <termio.h>
#include "common.h"

static char RcsId[] = "$Id: modemio.c,v 1.1 1993/01/06 18:09:44 wcp Exp $";
bool mdtimedout;
unsigned mdverbosity = 0;
bool mdslowrite = FALSE;
extern const char *myname;
char mdread_buffer[1024];
static SIGTYPE (*exsa)(int), (*exsh)(int);

static SIGTYPE timeout(int sig)
{
  mdtimedout = TRUE;
#ifndef SIGH_VOID
  return 0;
#endif
}

static void getSignals()
{
  exsa = signal(SIGALRM, timeout);
  exsh = signal(SIGHUP, timeout);
  mdtimedout = FALSE;
}

static void relSignals()
{
  alarm(0);
  signal(SIGALRM, exsa);
  signal(SIGHUP, exsh);
}

/*
 *  substr(s, l)
 *
 *  Function:	Checks for the presence of the string pointed to by s
 *		somewhere within the string pointed to by l.
 *
 *  Returns:	0 if found.
 *		-1 if not found.
 */
static int substr(const char *s, const char *l)
{
  int len;
  
  len = strlen(s);
  while ((l = strchr(l, *s)) != NULL)
    {
      if (!strncmp(s, l, len))
	return(0);
      l++;
    }
  return(-1);
}

/*
 *  mdwrite(c)
 *
 *  Function:	Outputs the string pointed to by c to the ACU device.
 *
 *  Returns:	0 on completion.
 *		-1 on write errors.
 *
 */
int mdwrite(const char *c, int fd)
{
  getSignals();
  if (mdverbosity > 6)
    fprintf(stderr, "%s: sent modem <<", myname);
  while (*c)
    {
      alarm(1);
      if (mdtimedout || write(fd, c, 1) != 1)
	{
	  relSignals();
	  if (mdverbosity > 6)
	    fprintf(stderr, ">>-FAIL\n");
	  fprintf(stderr, "%s: acu write error (%s)\n", myname,
		  sys_errlist[errno]);
	  return(-1);
	}
      if (mdverbosity > 6)
	fprintf(stderr, "%s", vgets(*c));
      if (mdslowrite)
	NAP(100);
      c++;
    }
  relSignals();
  if (mdverbosity > 6)
    fprintf(stderr, ">>-OK\n");
  return(0);
}

/*
 *
 *  Function:	Reads from the ACU until it finds a valid response (found
 *		in ModemAnswers) or times out after rtime seconds.
 *
 *  Returns:	The index in ModemAnswers of the modem response found.
 *		-1 on timeout.
 *
 */
char *mdread(unsigned rtime, int fd, const char **tokens, unsigned *ret)
{
  char *bp = mdread_buffer, *rp = bp;
  unsigned char c;
  
  getSignals();
  if (mdverbosity > 6)
    fprintf(stderr, "%s: modem returned <<", myname);
  errno = 0;
  alarm(rtime);
  while (!mdtimedout && read(fd, &c, 1) == 1)
    {
      c &= 0177;
      if ((*bp = c) != '\0')
	*++bp = '\0';
      if (mdverbosity > 6)
	fprintf(stderr, "%s", vgets(c));
      if (bp >= mdread_buffer + sizeof(mdread_buffer))
	{
	  relSignals();
	  if (mdverbosity > 6)
	    fprintf(stderr, ">>-FAIL\n");
	  return 0;
	}
      if (c == '\r' || c == '\n')
	{
	  if (tokens)
	    {
	      const char **mp;
	      
	      for (mp = tokens; *mp; ++mp)
		if (substr(*mp, mdread_buffer) == 0)
		  {
		    relSignals();
		    if (mdverbosity > 6)
		      fprintf(stderr, ">>-OK\n");
		    if (mdverbosity > 4)
		      fprintf(stderr, "%s: got %s\n", myname, *mp);
		    if (ret)
		      *ret = mp - tokens;
		    return (char *)*mp;
		  }
	    }
	  else
	    if (strlen(rp) > 1)
	      {
		relSignals();
		bp[-1] = '\0';
		if (mdverbosity > 6)
		  fprintf(stderr, ">>-OK\n");
		if (mdverbosity > 4)
		  fprintf(stderr, "%s: got %s\n", myname, rp);
		return rp;
	      }
	    else
	      rp = bp;
	}
    }
  relSignals();
  if (mdverbosity > 6)
    fprintf(stderr, ">>-FAIL");
  if (!mdtimedout)
    fprintf(stderr, " %s\n", sys_errlist[errno]);
  else
    if (mdverbosity > 4)
      fprintf(stderr, " no response\n");
  return 0;
}

bool mdhangup(int fd)
{
  struct termio tty_settings, old_tty;
  bool ret = FAILED;
  SIGTYPE (*exsh)(int);

  exsh = signal(SIGHUP, SIG_IGN);	/* just in case fd is the
					   controlling tty */
  if (ioctl(fd, TCGETA, &tty_settings) >= 0)
    {
      old_tty = tty_settings;
      tty_settings.c_cflag &= ~CBAUD;
      tty_settings.c_cflag |= B0;
      if (mdverbosity > 5)
	fprintf(stderr, "%s: hanging\n", myname);
      if (ioctl(fd, TCSETA, &tty_settings) >= 0)
	{
	  sleep(1);
	  if (ioctl(fd, TCSETA, &old_tty) >= 0)
	    ret = SUCCEEDED;
	}
    }
  signal(SIGHUP, exsh);
  return ret;
}

void mdflushin(int fd)
{
  getSignals();
  while (!mdtimedout)
    {
      char c;

      alarm(2);
      read(fd, &c, 1);
    }
  relSignals();
}
