/*
 * $Id: mgetty.c,v 1.4 1993/01/15 23:06:14 wcp Exp $
 *
 * Copyright (C) 1992	Walter Pelissero
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 1, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * $Log: mgetty.c,v $
 * Revision 1.4  1993/01/15  23:06:14  wcp
 * Modified CMD_receive to accept multiple tokens instead of a list.
 *
 * Revision 1.3  1993/01/13  21:28:37  wcp
 * spawn() is substituted by invoke().
 * Added CMD_slow_write(), to leave this choice at run time.
 * Reenabled the old exec.
 *
 * Revision 1.2  1993/01/07  23:17:13  wcp
 * Modified open() flags to tty for compatibility with tset.
 *
 * Revision 1.1  1993/01/06  18:07:40  wcp
 * Initial revision
 *
 */

#include <sys/types.h>
#include <stdio.h>
#include <fcntl.h>
#include <termio.h>
#include <getopt.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <signal.h>
#include <utmp.h>
#include <stdlib.h>
#include <tcl.h>
#include <sys/wait.h>
#include "common.h"
#include "modemio.h"

const unsigned long DEFAULT_BAUDRATE = 9600;

typedef enum {
  MD_RAW = 1,
  MD_COOKED = 2,
  MD_XONXOFF = 4,
  MD_CTSRTS = 8,
  MD_NOFLOW = 16,
  MD_LOCAL = 32,
  MD_REMOTE = 64
  } TtyMode;

typedef enum { UTMP_GETTY, UTMP_LOGIN, UTMP_EXIT, UTMP_PROCESS } UtmpStatus;

#ifdef DEBUG
static int verbosity = 9;
#else
static int verbosity = 0;
#endif

const char *myname;

static char RcsId[] = "$Id: mgetty.c,v 1.4 1993/01/15 23:06:14 wcp Exp $";
static bool suspended = FALSE;
static unsigned long baudrate;
static const char *line, *site;
static bool timedout;
static Tcl_Interp *interp;
static bool set_utmp(UtmpStatus, const char *, int);


inline bool setUtmp(UtmpStatus status, const char *line)
{
  return set_utmp(status, line, getpid());
}

static bool closeLine()
{
  close(0);
  close(1);
  close(2);
  return TRUE;
}

static SIGTYPE suspend(int sig)
{
  if (!suspended)
    {
      suspended = TRUE;
      signal(sig, suspend);
      setUtmp(UTMP_PROCESS, line);
      closeLine();
      while(suspended)
	sleep(1024);
    }
#ifndef SIGH_VOID
  return 0;
#endif
}

static SIGTYPE restart(int sig)
{
  suspended = FALSE;
  signal(sig, restart);
  setUtmp(UTMP_LOGIN, line);
#ifndef SIGH_VOID
  return 0;
#endif
}

static bool setMode(int mode)
{
  struct termio	tty_settings;

  if (ioctl(1, TCGETA, &tty_settings) < 0)
    {
      fprintf(stderr, "%s: can't ioctl\n", myname);
      return FAILED;
    }
  if (mode & MD_RAW)
    {
      if (verbosity > 8)
	fprintf(stderr, "%s: setting MD_RAW\n", myname);
      tty_settings.c_iflag &= ~(BRKINT | ICRNL | ISTRIP);
      tty_settings.c_cflag &= ~PARENB;
      tty_settings.c_cflag |= (HUPCL | CSIZE | CS8 | CREAD);
      tty_settings.c_lflag &= ~(ISIG | ECHO | ICANON);
      tty_settings.c_cc[VMIN] = 1;
      tty_settings.c_cc[VTIME] = 0;
      tty_settings.c_oflag &= ~OPOST;
    }
  if (mode & MD_COOKED)
    {
      if (verbosity > 8)
	fprintf(stderr, "%s: setting MD_COOKED\n", myname);
      tty_settings.c_lflag |= ICANON | ECHO | ISIG | ECHOE | ECHOK;
      tty_settings.c_iflag |= ICRNL | BRKINT | IGNPAR;
      tty_settings.c_oflag |= ONLCR | OPOST | TAB3;
      tty_settings.c_cc[VMIN] = 4;
      tty_settings.c_cc[VTIME] = 0;
    }
  if (mode & MD_XONXOFF)
    {
      if (verbosity > 8)
	fprintf(stderr, "%s: setting MD_XONXOFF\n", myname);
      tty_settings.c_iflag |= IXON | IXANY;
    }
  if (mode & MD_CTSRTS)
    {
      if (verbosity > 8)
	fprintf(stderr, "%s: setting MD_CTSRTS\n", myname);
      tty_settings.c_cflag |= CRTSCTS;
    }
  if (mode & MD_NOFLOW)
    {
      if (verbosity > 8)
	fprintf(stderr, "%s: setting MD_NOFLOW\n", myname);
      tty_settings.c_cflag &= ~(CRTSCTS);
      tty_settings.c_iflag &= ~(IXON | IXANY);
    }
  if (mode & MD_LOCAL)
    {
      if (verbosity > 8)
	fprintf(stderr, "%s: setting MD_LOCAL\n", myname);
      tty_settings.c_cflag |= CLOCAL;
    }
  if (mode & MD_REMOTE)
    {
      if (verbosity > 8)
	fprintf(stderr, "%s: setting MD_REMOTE\n", myname);
      tty_settings.c_cflag &= ~CLOCAL;
    }
  if (ioctl(1, TCSETAF, &tty_settings) < 0)
    {
      fprintf(stderr, "%s: can't set baud\n", myname);
      return FAILED;
    }
  return SUCCEEDED;
}

static bool setBaudRate(unsigned speed)
{
  struct termio tty_settings;
  int baud = B38400;

  baudrate = speed;
  if (verbosity > 5)
    fprintf(stderr, "%s: setting baud rate to about %u\n", myname, speed);
  if (ioctl(1, TCGETA, &tty_settings) < 0)
    return FAILED;
  if (speed <= 19200)	baud = B19200;
  if (speed <= 9600)	baud = B9600;
  if (speed <= 4800)	baud = B4800;
  if (speed <= 2400)	baud = B2400;
  if (speed <= 1800)	baud = B1800;
  if (speed <= 1200)	baud = B1200;
  if (speed <= 600)	baud = B600;
  if (speed <= 300)	baud = B300;
  if (speed <= 200)	baud = B200;
  if (speed <= 150)	baud = B150;
  if (speed <= 134)	baud = B134;
  if (speed <= 110)	baud = B110;
  if (speed <= 75)	baud = B75;
  if (speed <= 50)	baud = B50;
  tty_settings.c_cflag &= ~CBAUD;
  tty_settings.c_cflag |= baud;
  if (ioctl(1, TCSETA, &tty_settings) < 0)
    return FAILED;
  return SUCCEEDED;
}

static bool openLine(const char *portname)
{
  /* If spawned from init stdin/stdout/stderr may (should?) point
     to unopened file descriptors. */
  close(0); close(1);
#ifndef DEBUG
  close(2);
#endif
  setpgrp();
  errno = 0;
  chdir("/dev");
  /* The following two open()s should have O_RDONLY and O_WRONLY
     respectively but tset (and other programs?) cannot work properly
     if we do so. */
  if (open(portname, (O_RDWR | O_NDELAY)) != 0 ||
      open(portname, (O_RDWR | O_NDELAY)) != 1)  {
#ifdef DEBUG
      if (errno == EBUSY)
	fprintf(stderr, "%s: port %s BUSY\n", myname, portname);
      else
	fprintf(stderr, "%s: can't connect into %s (%s)\n", myname,
		portname, sys_errlist[errno]);
#endif
      return FAILED;
    }
  setMode(MD_RAW | MD_NOFLOW | MD_LOCAL);
  setBaudRate(DEFAULT_BAUDRATE);
  if (fcntl(0, F_SETFL, O_RDONLY) == ERROR ||
      fcntl(1, F_SETFL, O_WRONLY) == ERROR)
    {
#ifdef DEBUG
      fprintf(stderr, "%s: can't unset O_NDELAY for %s\n", myname, portname);
#endif
      return FAILED;
    }
  if (verbosity > 5)
    fprintf(stderr, "%s: line parameters set for %s\n", myname, portname);
#ifndef DEBUG
  dup(1);
#endif
  return SUCCEEDED;
}

static bool set_utmp(UtmpStatus status, const char *line, int pid)
{
  int ufd;
  bool ret = FAILED;
  const char *id;
  struct utmp ut;

  id = line + strlen(line) - 2;
  if((ufd = open(UTMP_FILE, O_RDWR | O_CREAT, 0644)) < 0)
    return FAILED;
  while(read(ufd, (char *)&ut, sizeof(ut)) == sizeof(ut))
    {
      char buf[sizeof(ut.ut_line) + 1];

      strncpy(buf, ut.ut_line, sizeof(ut.ut_line));
      buf[sizeof(buf) - 1] = 0;
      if(!strcmp(buf, line) ||
         ((ut.ut_type == INIT_PROCESS) && pid == ut.ut_pid))  /* -ot */
	{
	  strncpy(ut.ut_line, line, sizeof(ut.ut_line));
	  strncpy(ut.ut_id, id, sizeof(ut.ut_id));
	  ut.ut_time = time(0);
	  switch (status)
	    {
	    case UTMP_GETTY:
	      strcpy(ut.ut_name, "LOGIN");
	      ut.ut_type = LOGIN_PROCESS;
	      break;
	    case UTMP_LOGIN:
	      strcpy(ut.ut_name, "LOGIN2");
	      ut.ut_type = LOGIN_PROCESS;
	      break;
	    case UTMP_PROCESS:
	      strncpy(ut.ut_name, "DIALOUT", sizeof(ut.ut_name));
	      ut.ut_type = USER_PROCESS;
	      break;
	    case UTMP_EXIT:
	      strncpy(ut.ut_name, "LOGIN", sizeof(ut.ut_name));
	      ut.ut_type = DEAD_PROCESS;
	      break;
	    }
	  lseek(ufd, -sizeof(ut), 1);
	  write(ufd, (char *)&ut, sizeof(ut));
	  ret = SUCCEEDED;
	  if (verbosity > 8)
	    fprintf(stderr, "%s: utmp file set\n", myname);
	  break;
	}
    }
  close(ufd);
  return ret;
}

static SIGTYPE terminate(int sig)
{
  mdhangup(1);
  closeLine();
  setUtmp(UTMP_EXIT, line);
  exit(0);
#ifndef SIGH_VOID
  return 0;
#endif
}

static char *getLoginName()
{
  static char buf[80];
  int len;

  timedout = FALSE;
  alarm(180);
  if ((len = read(0, buf, sizeof(buf))) > 0)
    {
      buf[len - 1] = 0;
      mdwrite("\r", 1);
    }
  alarm(0);
  if (timedout || len < 1)
    {
      mdhangup(1);
      return 0;
    }
  return buf;
}

static void spawnGetty(char **getty_args)
{
  struct termio tty_settings;
  char getty[strlen(getty_args[0] + 1)];
  
  if (ioctl(1, TCGETA, &tty_settings) < 0)
    return;
  /* Here we have to reset HUPCL because getty tries to hang up
     before continuing on its job.
     Resetting CLOCAL is necessary because we want getty to be
     aware of line losing. */
  tty_settings.c_cflag &= ~(HUPCL | CLOCAL);
  if (ioctl(1, TCSETAF, &tty_settings) < 0)
    return;
  strcpy(getty, getty_args[0]);
  getty_args[0] = "-";
  execv(getty, getty_args);
  return;
}

static void resetSignals()
{
  alarm(0);
  signal(SIGALRM, SIG_DFL);
#ifdef SIGUSR1
  signal(SIGUSR1, SIG_DFL);
#endif
#ifdef SIGUSR2
  signal(SIGUSR2, SIG_DFL);
#endif
#ifdef SIGTSTP
  signal(SIGTSTP, SIG_DFL);
#endif
#ifdef SIGCONT
  signal(SIGCONT, SIG_DFL);
#endif
  signal(SIGQUIT, SIG_DFL);
  signal(SIGTERM, SIG_DFL);
  signal(SIGHUP, SIG_DFL);
  signal(SIGINT, SIG_DFL);
}

static void doLogin()
{
  char *name;
  
  mdread(2, 0, 0, 0);
  setMode(MD_REMOTE | MD_NOFLOW | MD_COOKED);
  do
    {
      char buf[256];
      
      sprintf(buf, "\r\n\n[%u on %s]\r\n%s!login: ", baudrate, line, site);
      if (write(1, buf, strlen(buf)) < 0)
	{
	  name = 0;
	  break;
	}
    }
  while (!(name = getLoginName()) || !strlen(name));
  if (name)
    {
      if (verbosity > 4)
	fprintf(stderr, "%s: executing login\n", myname);
      setUtmp(UTMP_LOGIN, line);
      if (!lockLine(line, 30))
	return;
#ifdef DEBUG
      close(2);
      dup(1);
#endif
      setMode(MD_XONXOFF);
      resetSignals();
      execlp("/etc/login", "login", name, (char *)0);
      execlp("/bin/login", "login", name, (char *)0);
      {
	char buf[128];
	
	sprintf(buf, "\r\n%s: login not executable.\r\n", myname);
	mdwrite(buf, 1);
      }
    }
  sleep(1);
}

static SIGTYPE timeout(int sig)
{
  timedout = TRUE;
  signal(sig, timeout);
#ifndef SIGH_VOID
  return 0;
#endif
}


static SIGTYPE catcher(int sig)
{
  fprintf(stderr, "GOT SIGNAL %u\n", sig);
  signal(sig, catcher);
#ifndef SIGH_VOID
  return 0;
#endif
}

static void catchSignals()
{
  unsigned i;

  for (i = 0; i < NSIG; ++i)	/*  -wp9/20/92. */
    if (i != SIGCLD)
      signal(i, catcher);
  signal(SIGALRM, timeout);
#ifdef SIGUSR1
  signal(SIGUSR1, suspend);
#endif
#ifdef SIGTSTP
  signal(SIGTSTP, suspend);
#endif
#ifdef SIGUSR2
  signal(SIGUSR2, terminate);	/* when restarted we exit to redo all the
				   settings we may have already done. */
#endif
#ifdef SIGCONT
  signal(SIGCONT, terminate);
#endif
  signal(SIGQUIT, terminate);
  signal(SIGTERM, terminate);
  signal(SIGHUP, terminate);
  signal(SIGINT, terminate);
}

static int CMD_send(ClientData clientData, Tcl_Interp *interp, int argc,
		    char *argv[])
{
  int i;

  if (argc < 2)
    {
      Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		       " string ?string ...?\"", 0);
      return TCL_ERROR;
    }
  for (i = 1; i < argc; ++i)
    {
      if (mdwrite(argv[i], 1) < 0)
	{
	  Tcl_AppendResult(interp, "cannot send \"", argv[i], "\" (",
			   sys_errlist[errno], ")", 0);
	  return TCL_ERROR;
	}
    }
  return TCL_OK;
}

static int CMD_listen(ClientData clientData, Tcl_Interp *interp, int argc,
		      char *argv[])
{
  char *response;

  if (argc > 1)
    {
      Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], "\"", 0);
      return TCL_ERROR;
    }
  while (!(response = mdread(1000, 0, 0, 0)) || strlen(response) == 0);
  Tcl_SetResult(interp, response, TCL_STATIC);
  return TCL_OK;
}

static int CMD_receive(ClientData clientData, Tcl_Interp *interp, int argc,
		       char *argv[])
{
  unsigned t = 30;

  if (argc > 1)
    t = atoi(argv[1]);
  if (argc > 2)
    {
      const char *p;
      
      if ((p = mdread(t, 0, (const char **)argv + 2, 0)))
	Tcl_SetResult(interp, (char *)p, TCL_VOLATILE);
      else
	Tcl_SetResult(interp, "", TCL_STATIC);
    }
  else
    {
      char *p = mdread(t, 0, 0, 0);
      
      if (p)
	Tcl_SetResult(interp, p, TCL_STATIC);
      else
	Tcl_SetResult(interp, "", TCL_STATIC);
    }
return TCL_OK;
}

static int CMD_flush_input(ClientData clientData, Tcl_Interp *interp,
			   int argc, char *argv[])
{
  if (argc > 1)
    {
      Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		       "\"", 0);
      return TCL_ERROR;
    }
  mdflushin(0);
  return TCL_OK;
}

static int CMD_hangup(ClientData clientData, Tcl_Interp *interp, int argc,
		      char *argv[])
{
  if (argc > 1)
    {
      Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		       "\"", 0);
      return TCL_ERROR;
    }
  mdhangup(1);
  return TCL_OK;
}

/* This is mainly an exec (Tcl command) reedition to better follow
   some need related to DCE handling. Input, output streams are
   redirected to tty so invoked programs can deal with it. Output
   is not collected in any way. */
static int CMD_invoke(ClientData clientData, Tcl_Interp *interp, int argc,
		     char *argv[])
{
  int *pidPtr;

  /* See if the command is to be run in background;  if so, create
     the command, detach it, and return. */
  if ((argv[argc-1][0] == '&') && (argv[argc-1][1] == 0))
    {
      int numPids;
      
      argc--;
      argv[argc] = NULL;
      numPids = Tcl_CreatePipeline(interp, argc-1, argv+1, &pidPtr,
				   (int *) NULL, (int *) NULL, (int *) NULL);
      if (numPids < 0)
	return TCL_ERROR;
      Tcl_DetachPids(numPids, pidPtr);
    }
  else
    {
      int numPids, i, rvalue;
      char buf[32];

      lockLine(line, 30);
      setUtmp(UTMP_PROCESS, line);
      numPids = Tcl_CreatePipeline(interp, argc-1, argv+1, &pidPtr,
				   (int *) NULL, (int *) NULL, (int *) NULL);
      if (numPids < 0)
	return TCL_ERROR;
      /* This ensure last process on pipe has value saved
	 in rvalue, bacuse we must return it. */
      for (i = 0; i < numPids; i++)
	{
	  int pid;

	  pid = Tcl_WaitPids(1, &pidPtr[i], (int *) &rvalue);
	  if (pid == -1)
	    {
	      Tcl_AppendResult(interp, "error waiting for process to exit: ",
			       Tcl_UnixError(interp), (char *) NULL);
	      continue;
	    }
	}
      rvalue = (WIFEXITED(rvalue)) ? WEXITSTATUS(rvalue) : -1;
      if (verbosity > 6)
	fprintf(stderr, "%s: invoked program returned %d\n", myname, rvalue);
      sprintf(buf, "%d", rvalue);
      Tcl_SetResult(interp, buf, TCL_VOLATILE);
      unlockLine(line);
      setUtmp(UTMP_GETTY, line);
    }
  ckfree((char *) pidPtr);
  return TCL_OK;
}

static int CMD_mode(ClientData clientData, Tcl_Interp *interp, int argc,
		    char *argv[])
{
  int i;

  if (argc < 2)
    {
      Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		       " mode ?mode ...?\"", 0);
      return TCL_ERROR;
    }
  for (i = 1; i < argc; ++i)
    {
      if (!strcomp(argv[i], "raw"))
	setMode(MD_RAW);
      else if (!strcomp(argv[i], "cooked"))
	setMode(MD_COOKED);
      else if (!strcomp(argv[i], "xon_xoff"))
	setMode(MD_XONXOFF);
      else if (!strcomp(argv[i], "cts_rts"))
	setMode(MD_CTSRTS);
      else if (!strcomp(argv[i], "no_flow"))
	setMode(MD_NOFLOW);
      else if (!strcomp(argv[i], "local"))
	setMode(MD_LOCAL);
      else if (!strcomp(argv[i], "remote"))
	setMode(MD_REMOTE);
      else
	{
	  Tcl_AppendResult(interp, "unrecognized flag \"", argv[i], "\"", 0);
	  return TCL_ERROR;
	}
    }
  return TCL_OK;
}

static int CMD_baudrate(ClientData clientData, Tcl_Interp *interp, int argc,
			char *argv[])
{
  char buf[16];

  if (argc > 2)
    {
      Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		       " baud_rate\"", 0);
      return TCL_ERROR;
    }
  if (argc > 1)
    setBaudRate(atoi(argv[1]));
  sprintf(buf, "%u", baudrate);
  Tcl_SetResult(interp, buf, TCL_VOLATILE);
  return TCL_OK;
}

static int CMD_login(ClientData clientData, Tcl_Interp *interp, int argc,
		     char *argv[])
{
  if (argc > 1)
    {
      Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		       "\"", 0);
      return TCL_ERROR;
    }
  doLogin();
  /* Should not return. */
  Tcl_SetResult(interp, "error in login process", TCL_STATIC);
  return TCL_ERROR;
}

static int CMD_sleep(ClientData clientData, Tcl_Interp *interp, int argc,
		     char *argv[])
{
  if (argc != 2)
    {
      Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		       " seconds\"", 0);
      return TCL_ERROR;
    }
  sleep(atoi(argv[1]));
  return TCL_OK;
}

#if DEBUG > 7
void Tcl_DEBUG(ClientData clientData, Tcl_Interp *interp, int level,
	       char *command, Tcl_CmdProc *cmdProc, ClientData cmdClientData,
	       int argc, char *argv[])
{
  int i;

  fprintf(stderr, "%s: TRACE: [%u] %s --", myname, level, command);
  for (i = 0; i < argc; ++i)
    fprintf(stderr, " [%s]", argv[i]);
  putc('\n', stderr);
}
#endif	/* DEBUG */

int CMD_exit(ClientData dummy, Tcl_Interp *interp, int argc, char *argv[])
{
  if ((argc != 1) && (argc != 2))
    {
      Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		       " ?returnCode?\"", (char *) NULL);
      return TCL_ERROR;
    }
  mdhangup(1);
  closeLine();
  setUtmp(UTMP_EXIT, line);
  if (argc == 1)
    exit(0);
  exit(atoi(argv[1]));
  return TCL_ERROR;
}


static void setBits(int flag, bool on, unsigned short *field)
{
  if (on)
    *field |= flag;
  else
    *field &= ~flag;
}

static int CMD_slow_write(ClientData clientData, Tcl_Interp *interp, int argc,
			  char *argv[])
{
  if (argc > 2)
    {
      Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		       " ?ON|OFF|1|0|TRUE|FALSE|TOGGLE?\"", 0);
      return TCL_ERROR;
    }
  if (argc > 1)
    {
      char *p = argv[1];

      if (!strcomp(p, "ON") || !strcomp(p, "1") || !strcomp(p, "TRUE"))
	mdslowrite = TRUE;
      else if (!strcomp(p, "OFF") || !strcomp(p, "0") || !strcomp(p, "FALSE"))
	mdslowrite = FALSE;
      else if (!strcomp(p, "TOGGLE"))
	mdslowrite = !mdslowrite;
      else
	{
	  Tcl_AppendResult(interp, "unrecognized flag \"", p, "\"", 0);
	  return TCL_ERROR;
	}
    }
  Tcl_SetResult(interp, mdslowrite ? "ON" : "OFF", TCL_STATIC);
  return TCL_OK;
}

/* THIS FUNCTION IS NOT COMPLETE AT ALL!! -wp9/20/92. */

static int CMD_stty(ClientData clientData, Tcl_Interp *interp, int argc,
		    char *argv[])
{
  int i;
  struct termio tty;

  if (argc < 2)
    {
      Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		       " mode ?mode ...?\"", 0);
      return TCL_ERROR;
    }
  if (ioctl(1, TCGETA, &tty) < 0)
    {
      Tcl_SetResult(interp, "cannot get current line setting", TCL_STATIC);
      return TCL_ERROR;
    }
  for (i = 1; i < argc; ++i)
    {
      char *p = argv[i];

      if (*p == '-')
	++p;
      /* Input flags */
      if (!strcomp(p, "IGNBRK"))
	setBits(IGNBRK, p == argv[i], &tty.c_iflag);
      else if (!strcomp(p, "BRKINT"))
	setBits(BRKINT, p == argv[i], &tty.c_iflag);
      else if (!strcomp(p, "IGNPAR"))
	setBits(IGNPAR, p == argv[i], &tty.c_iflag);
      else if (!strcomp(p, "PARMRK"))
	setBits(PARMRK, p == argv[i], &tty.c_iflag);
      else if (!strcomp(p, "INPCK"))
	setBits(INPCK, p == argv[i], &tty.c_iflag);
      else if (!strcomp(p, "ISTRIP"))
	setBits(ISTRIP, p == argv[i], &tty.c_iflag);
      else if (!strcomp(p, "INLCR"))
	setBits(INLCR, p == argv[i], &tty.c_iflag);
      else if (!strcomp(p, "IGNCR"))
	setBits(IGNCR, p == argv[i], &tty.c_iflag);
      else if (!strcomp(p, "ICRNL"))
	setBits(ICRNL, p == argv[i], &tty.c_iflag);
      else if (!strcomp(p, "IUCLC"))
	setBits(IUCLC, p == argv[i], &tty.c_iflag);
      else if (!strcomp(p, "IXON"))
	setBits(IXON, p == argv[i], &tty.c_iflag);
      else if (!strcomp(p, "IXANY"))
	setBits(IXANY, p == argv[i], &tty.c_iflag);
      else if (!strcomp(p, "IXOFF"))
	setBits(IXOFF, p == argv[i], &tty.c_iflag);
#ifdef DOSMODE
      else if (!strcomp(p, "DOSMODE"))
	setBits(DOSMODE, p == argv[i], &tty.c_iflag);
#endif
      /* Output flags */
      else if (!strcomp(p, "OPOST"))
	setBits(OPOST, p == argv[i], &tty.c_oflag);
      else if (!strcomp(p, "OLCUC"))
	setBits(OLCUC, p == argv[i], &tty.c_oflag);
      else if (!strcomp(p, "ONLCR"))
	setBits(ONLCR, p == argv[i], &tty.c_oflag);
      else if (!strcomp(p, "OCRNL"))
	setBits(OCRNL, p == argv[i], &tty.c_oflag);
      else if (!strcomp(p, "ONOCR"))
	setBits(ONOCR, p == argv[i], &tty.c_oflag);
      else if (!strcomp(p, "ONLRET"))
	setBits(ONLRET, p == argv[i], &tty.c_oflag);
      else if (!strcomp(p, "OFILL"))
	setBits(OFILL, p == argv[i], &tty.c_oflag);
      else if (!strcomp(p, "OFDEL"))
	setBits(OFDEL, p == argv[i], &tty.c_oflag);
      else if (!strcomp(p, "NL0"))
	if (p != argv[i])
	  break;
	else
	  setBits(NL1, FALSE, &tty.c_oflag);
      else if (!strcomp(p, "NL1"))
	if (p != argv[i])
	  break;
	else
	  setBits(NL1, TRUE, &tty.c_oflag);
      else if (!strcomp(p, "CR0"))
	if (p != argv[i])
	  break;
	else
	  setBits(CRDLY, FALSE, &tty.c_oflag);
      else if (!strcomp(p, "CR1"))
	if (p != argv[i])
	  break;
	else setBits(CR1, TRUE, &tty.c_oflag);
      else if (!strcomp(p, "CR2"))
	if (p != argv[i])
	  break;
	else setBits(CR2, TRUE, &tty.c_oflag);
      else if (!strcomp(p, "CR3"))
	if (p != argv[i])
	  break;
	else setBits(CR3, TRUE, &tty.c_oflag);
      else if (!strcomp(p, "TAB0"))
	if (p != argv[i])
	  break;
	else setBits(TAB0, p == argv[i], &tty.c_oflag);
      else if (!strcomp(p, "TAB1"))
	if (p != argv[i])
	  break;
	else setBits(TAB1, p == argv[i], &tty.c_oflag);
      else if (!strcomp(p, "TAB2"))
	if (p != argv[i])
	  break;
	else setBits(TAB2, p == argv[i], &tty.c_oflag);
      else if (!strcomp(p, "TAB3"))
	if (p != argv[i])
	  break;
	else setBits(TAB3, p == argv[i], &tty.c_oflag);
      else if (!strcomp(p, "BS0"))
	if (p != argv[i])
	  break;
	else setBits(BS0, p == argv[i], &tty.c_oflag);
      else if (!strcomp(p, "BS1"))
	if (p != argv[i])
	  break;
	else setBits(BS1, p == argv[i], &tty.c_oflag);
      else if (!strcomp(p, "VT0"))
	if (p != argv[i])
	  break;
	else setBits(VT0, p == argv[i], &tty.c_oflag);
      else if (!strcomp(p, "VT1"))
	if (p != argv[i])
	  break;
	else setBits(VT1, p == argv[i], &tty.c_oflag);
      else if (!strcomp(p, "FF0"))
	if (p != argv[i])
	  break;
	else setBits(FF0, p == argv[i], &tty.c_oflag);
      else if (!strcomp(p, "FF1"))
	if (p != argv[i])
	  break;
	else setBits(FF1, p == argv[i], &tty.c_oflag);
      /* Control flags */
      else if (!strcomp(p, "B0"))
	if (p != argv[i])
	  break;
	else setBits(B0, p == argv[i], &tty.c_cflag);
      else if (!strcomp(p, "B50"))
	if (p != argv[i])
	  break;
	else setBits(B50, p == argv[i], &tty.c_cflag);
      else if (!strcomp(p, "B75"))
	if (p != argv[i])
	  break;
	else setBits(B75, p == argv[i], &tty.c_cflag);
      else if (!strcomp(p, "B110"))
	if (p != argv[i])
	  break;
	else setBits(B110, p == argv[i], &tty.c_cflag);
      else if (!strcomp(p, "B134"))
	if (p != argv[i])
	  break;
	else setBits(B134, p == argv[i], &tty.c_cflag);
      else if (!strcomp(p, "B150"))
	if (p != argv[i])
	  break;
	else setBits(B150, p == argv[i], &tty.c_cflag);
      else if (!strcomp(p, "B200"))
	if (p != argv[i])
	  break;
	else setBits(B200, p == argv[i], &tty.c_cflag);
      else if (!strcomp(p, "B300"))
	if (p != argv[i])
	  break;
	else setBits(B300, p == argv[i], &tty.c_cflag);
      else if (!strcomp(p, "B600"))
	if (p != argv[i])
	  break;
	else setBits(B600, p == argv[i], &tty.c_cflag);
      else if (!strcomp(p, "B1200"))
	if (p != argv[i])
	  break;
	else setBits(B1200, p == argv[i], &tty.c_cflag);
      else if (!strcomp(p, "B1800"))
	if (p != argv[i])
	  break;
	else setBits(B1800, p == argv[i], &tty.c_cflag);
      else if (!strcomp(p, "B2400"))
	if (p != argv[i])
	  break;
	else setBits(B2400, p == argv[i], &tty.c_cflag);
      else if (!strcomp(p, "B4800"))
	if (p != argv[i])
	  break;
	else setBits(B4800, p == argv[i], &tty.c_cflag);
      else if (!strcomp(p, "B9600"))
	if (p != argv[i])
	  break;
	else setBits(B9600, p == argv[i], &tty.c_cflag);
      else if (!strcomp(p, "B19200") || !strcomp(p, "EXTA"))
	if (p != argv[i])
	  break;
	else setBits(B19200, p == argv[i], &tty.c_cflag);
      else if (!strcomp(p, "B38400") || !strcomp(p, "EXTB"))
	if (p != argv[i])
	  break;
	else setBits(B38400, p == argv[i], &tty.c_cflag);
      else if (!strcomp(p, "CS5"))
	if (p != argv[i])
	  break;
	else setBits(CS5, p == argv[i], &tty.c_cflag);
      else if (!strcomp(p, "CS6"))
	if (p != argv[i])
	  break;
	else setBits(CS6, p == argv[i], &tty.c_cflag);
      else if (!strcomp(p, "CS7"))
	if (p != argv[i])
	  break;
	else setBits(CS7, p == argv[i], &tty.c_cflag);
      else if (!strcomp(p, "CS8"))
	if (p != argv[i])
	  break;
	else setBits(CS8, p == argv[i], &tty.c_cflag);
      else if (!strcomp(p, "CSTOPB"))
	setBits(CSTOPB, p == argv[i], &tty.c_cflag);
      else if (!strcomp(p, "CREAD"))
	setBits(CREAD, p == argv[i], &tty.c_cflag);
      else if (!strcomp(p, "PARENB"))
	setBits(PARENB, p == argv[i], &tty.c_cflag);
      else if (!strcomp(p, "PARODD"))
	setBits(PARODD, p == argv[i], &tty.c_cflag);
      else if (!strcomp(p, "HUPCL"))
	setBits(HUPCL, p == argv[i], &tty.c_cflag);
      else if (!strcomp(p, "CLOCAL"))
	setBits(CLOCAL, p == argv[i], &tty.c_cflag);
#ifdef LOBLK
      else if (!strcomp(p, "LOBLK"))
	setBits(LOBLK, p == argv[i], &tty.c_cflag);
#endif
#ifdef CTSFLOW
      else if (!strcomp(p, "CTSFLOW"))
	setBits(CTSFLOW, p == argv[i], &tty.c_cflag);
#endif
#ifdef RTSFLOW
      else if (!strcomp(p, "RTSFLOW"))
	setBits(RTSFLOW, p == argv[i], &tty.c_cflag);
#endif
      /* Line discipline flags */
      else if (!strcomp(p, "ISIG"))
	setBits(ISIG, p == argv[i], &tty.c_lflag);
      else if (!strcomp(p, "ICANON"))
	setBits(ICANON, p == argv[i], &tty.c_lflag);
      else if (!strcomp(p, "XCASE"))
	setBits(XCASE, p == argv[i], &tty.c_lflag);
      else if (!strcomp(p, "ECHO"))
	setBits(ECHO, p == argv[i], &tty.c_lflag);
      else if (!strcomp(p, "ECHOE"))
	setBits(ECHOE, p == argv[i], &tty.c_lflag);
      else if (!strcomp(p, "ECHOK"))
	setBits(ECHOK, p == argv[i], &tty.c_lflag);
      else if (!strcomp(p, "ECHONL"))
	setBits(ECHONL, p == argv[i], &tty.c_lflag);
      else if (!strcomp(p, "NOFLSH"))
	setBits(NOFLSH, p == argv[i], &tty.c_lflag);
#ifdef XCLUDE
      else if (!strcomp(p, "XCLUDE"))
	setBits(XCLUDE, p == argv[i], &tty.c_lflag);
#endif
      else
	break;
    }
  if (i < argc)
    {
      Tcl_AppendResult(interp, "unrecognized flag \"", argv[i], "\"", 0);
      return TCL_ERROR;
    }
  if (ioctl(1, TCSETAF, &tty) < 0)
    {
      Tcl_SetResult(interp, "cannot set line parameters", TCL_STATIC);
      return TCL_ERROR;
    }
  return TCL_OK;
}

static void createTclEnvironment()
{
  interp = Tcl_CreateInterp ();
  Tcl_CreateCommand(interp, "listen", CMD_listen, 0, 0);
  Tcl_CreateCommand(interp, "send", CMD_send, 0, 0);
  Tcl_CreateCommand(interp, "receive", CMD_receive, 0, 0);
  Tcl_CreateCommand(interp, "flush_input", CMD_flush_input, 0, 0);
  Tcl_CreateCommand(interp, "hangup", CMD_hangup, 0, 0);
  Tcl_CreateCommand(interp, "login", CMD_login, 0, 0);
  Tcl_CreateCommand(interp, "invoke", CMD_invoke, 0, 0);
  Tcl_CreateCommand(interp, "mode", CMD_mode, 0, 0);
  Tcl_CreateCommand(interp, "baudrate", CMD_baudrate, 0, 0);
  Tcl_CreateCommand(interp, "stty", CMD_stty, 0, 0);
  Tcl_CreateCommand(interp, "sleep", CMD_sleep, 0, 0);
  Tcl_CreateCommand(interp, "slow_write", CMD_slow_write, 0, 0);
  Tcl_DeleteCommand(interp, "exit");
  Tcl_CreateCommand(interp, "exit", CMD_exit, 0, 0);
#if DEBUG > 7
  Tcl_CreateTrace(interp, 5, Tcl_DEBUG, 0);
#endif
}

static bool processInitFile()
{
  char pathname[256];

  sprintf(pathname, "%s/ttys/%s", LIB_DIR, line);
  if (access(pathname, R_OK) == 0)
    {
      if (Tcl_EvalFile(interp, pathname) == TCL_OK)
	return SUCCEEDED;
      else
	fprintf(stderr, "%s: error in tcl script (\"%s\" at line %u)\n",
		myname, interp->result, interp->errorLine);
    }
  return FALSE;
}

int main(unsigned argc, char *argv[])
{
#ifdef notdef
  mdslowrite = TRUE;
#endif
  myname = basename(argv[0]);
  site = siteName();
#ifdef DEBUG
  if (!isatty(fileno(stderr)))
    {
      /* If spawned from init stdin/stdout/stderr may (should?) point
	 to unopened file descriptors. */
      close(0); close(1); close(2);
      open("/dev/null", O_RDONLY);
      dup(open("/tmp/getty-stderr", O_WRONLY | O_CREAT | O_APPEND, 0644));
    }
#endif
  optind = 1;	/* the first argument is the real getty -wp9/20/92. */
  mdverbosity = verbosity;
#if defined(M_XENIX) || defined(BSD)
  line = argv[optind + 1];
#else
  line = argv[argc - 1];
#endif
  if (isLocked(line))
    exit(1);
  createTclEnvironment();
  if (!openLine(line))
    {
      fprintf(stderr, "%s: cannot open assigned tty\n", myname);
      exit(6);
    }
  catchSignals();
  if (!setUtmp(UTMP_GETTY, line))
    {
      fprintf(stderr, "%s: no slot in utmp\n", myname);
      exit(5);
    }
  if (!processInitFile())
    spawnGetty(argv + optind);
  mdhangup(1);
  closeLine();
  setUtmp(UTMP_EXIT, line);
  return exit(0), 0;
}
