/*
 * $Id: class2rec.c,v 1.1 1992/07/17 16:44:41 wally Exp $
 *
 * Copyright (C) 1992	Walter Pelissero
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 1, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * $Log: class2rec.c,v $
 * Revision 1.1  1992/07/17  16:44:41  wally
 * Initial revision
 *
 */

#include <stdio.h>
#include <fcntl.h>
#include <termio.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <signal.h>
#include <stdlib.h>
#include <getopt.h>
#include "/u/wally/c/faxem/src/common.h"

extern const char *TMP_DIR;
extern bool loadDefaults(void);
extern char *makeTemp(const char *);

typedef enum {
  MDERROR,
  MDOK,
  MDRING,
  MDCONNECT,
  MDNOCARRIER,
  MDNODIALTONE,
  MDBUSY,
  MDNOANSWER,
  MDFCON,
  MDFDCS,
  MDFDIS,
  MDFCFR,
  MDFTSI,
  MDFCSI,
  MDFPTS,
  MDFET,
  MDFHNG,
  MDLAST_ANS = MDFHNG
} ModemReturnCode;

char *ModemAnswers[] = {
  "ERROR",
  "OK",
  "RING",
  "CONNECT",
  "NO CARRIER",
  "NO DIAL TONE",
  "BUSY",
  "NO ANSWER",
  "+FCON",
  "+FDCS:",
  "+FDIS:",
  "+FCFR",
  "+FTSI:",
  "+FCSI:",
  "+FPTS:",
  "+FET:",
  "+FHNG:",
  0
};

typedef struct {
  int infd, outfd;
  struct termio orig_tty_settings;
} Port;

static SIGTYPE	catcher();

#define DLE	'\020'
#define ETX	'\003'
#define DC1	'\021'
#define DC3	'\023'
#define XON	DC1
#define XOFF	DC3
/* #define SLOWRITE */

#ifdef DEBUG
static int verbosity = 9;
#else
static int verbosity = 0;
#endif

char *myname;
static char *local_phone = "";
static char RcsId[] = "$Id: class2rec.c,v 1.1 1992/07/17 16:44:41 wally Exp $";
static char read_buffer[1024];

#define	toprint(x)	((x)<' '?((x)+'@'):'?')

/* vgets - Format one character in "always printable" format (like cat -v)
 */
static char *vgets(unsigned char c)
{
  static char buffer[10];
  char *pnt;
  
  pnt = buffer;
  if (iscntrl(c) || !isprint(c))
    {
      if (!isascii(c))
	{			/* Top bit is set */
	  *pnt++ = 'M';
	  *pnt++ = '-';
	  c = toascii(c);
	}
      if (iscntrl(c))
	{			/* Not printable */
	  *pnt++ = '^';
	  c = toprint(c);
	}
    }
  *pnt++ = c;
  *pnt = '\0';
  return(buffer);
}

static SIGTYPE catcher(int sig)
{
  signal(sig, catcher);
#ifndef SIGH_VOID
  return 0;
#endif
}

/* Flush file input.
   Here we hog all the bytes from fd until alarm() expires. */
static void flush_input(int fd)
{
  char c;

  alarm(2);
  errno = 0;
  while (errno != EINTR)
    read(fd, &c, 1);
}

/*
 *  mdwrite(c)
 *
 *  Function:	Outputs the string pointed to by c to the ACU device.
 *
 *  Returns:	0 on completion.
 *		-1 on write errors.
 *
 */
static int mdwrite(const char *c, int fd)
{
  int err;
  
  if (verbosity > 6)
    fprintf(stderr, "Sent MODEM <<");
  while (*c)
    {
      if ((err = write(fd, c, 1)) != 1)
	{
	  if (verbosity > 6)
	    fprintf(stderr, ">>-FAIL\n");
	  fprintf(stderr, "ACU write error (%s)\n", sys_errlist[errno]);
	  return(-1);
	}
      if (verbosity > 6)
	fprintf(stderr, "%s", vgets(*c));
#ifdef SLOWRITE
      nap(100);
#endif
      c++;
    }
  if (verbosity > 6)
    fprintf(stderr, ">>-OK\n");
  return(0);
}


/*
 *  substr(s, l)
 *
 *  Function:	Checks for the presence of the string pointed to by s
 *		somewhere within the string pointed to by l.
 *
 *  Returns:	0 if found.
 *		-1 if not found.
 */
static int substr(const char *s, const char *l)
{
  int len;
  
  len = strlen(s);
  while ((l = strchr(l, *s)) != NULL)
    {
      if (!strncmp(s, l, len))
	return(0);
      l++;
    }
  return(-1);
}

/*
 *  mdread(rtime, fd)
 *
 *  Function:	Reads from the ACU until it finds a valid response (found
 *		in ModemAnswers) or times out after rtime seconds.
 *
 *  Returns:	The index in ModemAnswers of the modem response found.
 *		-1 on timeout.
 *
 */
static ModemReturnCode mdread(unsigned rtime, int fd)
{
  char **mp, *bp;
  unsigned char c;
  
  bp = read_buffer;
  alarm(rtime);
  if (verbosity > 6)
    fprintf(stderr, "MODEM returned <<");
  while (read(fd, &c, 1) == 1)
    {
      c &= 0177;
      if ((*bp = c) != '\0')
	*++bp = '\0';
      if (verbosity > 6)
	fprintf(stderr, "%s", vgets(c));
      if (bp >= read_buffer + sizeof(read_buffer))
	{
	  alarm(0);
	  if (verbosity > 6)
	    fprintf(stderr, ">>-FAIL\n");
	  return -1;
	}
      if (c == '\r')
	{
	  for (mp = ModemAnswers; *mp; ++mp)
	    if (substr(*mp, read_buffer) == 0)
	      {
		alarm(0);
		if (verbosity > 6)
		  fprintf(stderr, ">>-OK\n");
		if (verbosity > 4)
		  fprintf(stderr, "got %s\n", ModemAnswers[mp - ModemAnswers]);
		return mp - ModemAnswers;
	      }
	}
    }
  alarm(0);
  if (verbosity > 6)
    fprintf(stderr, ">>-FAIL");
  if (verbosity > 4)
    fprintf(stderr, " no response\n");
  return -1;
}

static bool hangup(Port port)
{
  struct termio tty_settings, old_tty;

  if (ioctl(port.outfd, TCGETA, &tty_settings) < 0)
    return FAILED;
  old_tty = tty_settings;
  tty_settings.c_cflag = B0;
  if (verbosity > 5)
    fprintf(stderr, "%s: hanging\n", myname);
  if (ioctl(port.outfd, TCSETA, &tty_settings) < 0)
    return FAILED;
  sleep(1);
  if (ioctl(port.outfd, TCSETA, &old_tty) < 0)
    return FAILED;
  return SUCCEEDED;
}

bool initModem(Port port)
{
  char outbuf[128];

  mdwrite("ATZ\r", port.outfd);
  sleep(1);			/* to let modem recover */
  mdwrite("ATPE0V1Q0X6S0=3S7=60&B1&C1&D3\r", port.outfd);
  flush_input(port.infd);	/* to avoid OK propagation */
  
  sprintf(outbuf, "ATM%u+FCLASS=2\r", !!verbosity);
  if (mdwrite(outbuf, port.outfd) < 0)
    return FAILED;
  if (mdread(2, port.infd) != MDOK)
    return FAILED;
  /* I don't know if the following is necessary, but I saw it in
     an example session so .... */
  if (mdwrite("AT+FCR=1\r", port.outfd) < 0)
    return FAILED;
  if (mdread(2, port.infd) != MDOK)
    return FAILED;
  /* Setting Local IDentification (actually,
     it is the local fax number). */
  sprintf(outbuf, "AT+FLID=\"%s\"\r", local_phone);
  if (mdwrite(outbuf, port.outfd) < 0)
    return FAILED;
  if (mdread(2, port.infd) != MDOK)
    return FAILED;
  /* Setting Byte ORder (0 = normal, 1 = reverse). */
  if (mdwrite("AT+FBOR=0\r", port.outfd) < 0)
    return FAILED;
  if (mdread(2, port.infd) != MDOK)
    return FAILED;
  /* Setting DCE capabilities. */
  if (mdwrite("AT+FDCC=1\r", port.outfd) < 0)
    return FAILED;
  if (mdread(2, port.infd) != MDOK)
    return FAILED;
  return SUCCEEDED;
}

bool waitConnection(Port port)
{
  int ret = -1;

  do
    {
      switch (ret = mdread(1000, port.infd))
	{
	  unsigned i;

	case MDFCON:
	  for (i = 0; i < 10; ++i)
	    if ((ret = mdread(10, port.infd)) == MDOK)
	      break;
	  if (ret != MDOK)
	    {
	      initModem(port);
	      ret = -1;
	    }
	  break;
	case MDRING:
	  mdwrite("ATA\r", port.outfd);
	  ret = -1;
	  break;
	default:	/* Nothing read */
	  ret = -1;
	  break;
	}
    }
  while (ret < 0);
  return SUCCEEDED;
}
    
bool tryRead(ModemReturnCode answer, int fd)
{
  unsigned i;
  ModemReturnCode ret;

  for (i = 0; i < 10; ++i)
    if ((ret = mdread(10, fd)) == answer)
      return SUCCEEDED;
  return FAILED;
}

static void receivePage(int infd, int outfd)
{
  register bool dle_read = FALSE;

  do
    {
      char c;

      if (read(infd, &c, 1) < 1)
	return;
      if (c == DLE)
	{
	  if (dle_read = !dle_read)
	    continue;
	}
      else
	{
	  if (c == ETX && dle_read)
	    return;
	  dle_read = FALSE;
	}
      write(outfd, &c, 1);
    }
  while (TRUE);
}

void receiveFax(Port port, const char *base_name)
{
  bool eot;
  unsigned page = 0;

  do
    {
      mdwrite("AT+FDR\r", port.outfd);
      if (tryRead(MDCONNECT, port.infd))
	{
	  char pathname[1024];
	  int outfd;

	  sprintf(pathname, "%s.%u", base_name, ++page);
	  if ((outfd = open(pathname, O_WRONLY | O_TRUNC | O_CREAT, 0640)) < 0)
	    {
	      fprintf(stderr, "%s: cannot open for writing %s (%s)\n",
		      myname, pathname, sys_errlist[errno]);
	      return;
	    }
	  else
	    {
	      if (verbosity > 7)
		fprintf(stderr, "%s: sending initial DC2\n", myname);
	      sleep(1);
	      mdwrite("\022", port.outfd);
	      receivePage(port.infd, outfd);
	      close(outfd);
	      if (tryRead(MDFET, port.infd))
		{
		  char *fet = strstr(read_buffer, "+FET:") + 5;
		  
		  *strpbrk(fet, "\r,;") = '\0';
		  eot = (atoi(fet) == 2);
		  tryRead(MDOK, port.infd);
		}
	      else
		{
		  initModem(port);
		  return;
		}
	    }
	}
      else
	return;
    }
  while (!eot);
  mdwrite("AT+FDR\r", port.outfd);
  if (tryRead(MDFHNG, port.infd))
    tryRead(MDOK, port.infd);
}



static bool closeLine(Port port)
{
  bool errflag = FALSE;
  
  sleep(1);
  errflag |= !hangup(port);
  if (ioctl(port.outfd, TCSETA, &port.orig_tty_settings) < 0)
    return FAILED;
  close(port.infd);
  close(port.outfd);
  return errflag;
}

static Port openLine(const char *portname)
{
  Port port;
  struct termio	tty_settings;

  errno = 0;
  if (!strcmp(portname, "-"))
    {
      port.infd = fileno(stdin);
      port.outfd = fileno(stdout);
    }
  else
    {
      if ((port.infd = open(portname, (O_RDONLY | O_NDELAY), 0600)) < 0)
	{
	  if (errno == EBUSY)
	    fprintf(stderr, "%s: port %s BUSY\n", myname, portname);
	  else
	    fprintf(stderr, "%s: can't connect into %s\n", myname, portname);
	  port.infd = ERROR;
	  return port;
	}
      if ((port.outfd = open(portname, (O_WRONLY | O_NDELAY), 0600)) < 0)
	{
	  if (errno == EBUSY)
	    fprintf(stderr, "%s: port %s BUSY\n", myname, portname);
	  else
	    fprintf(stderr, "%s: can't connect into %s\n", myname, portname);
	  port.infd = ERROR;
	  return port;
	}
    }
  if (ioctl(port.outfd, TCGETA, &port.orig_tty_settings) < 0)
    {
      fprintf(stderr, "%s: can't ioctl on %s\n", myname, portname);
      port.infd = ERROR;
      return port;
    }
  tty_settings = port.orig_tty_settings;
  
  tty_settings.c_iflag &= ~(BRKINT | ICRNL | ISTRIP | IXON | IXANY);
  tty_settings.c_cflag &= ~(CBAUD | PARENB);
  /* ZyXEL U-1496E ROM 4.09 can receive faxes at 19200 baud only. */
  tty_settings.c_cflag |= (CSIZE | CS8 | HUPCL | CREAD | B19200);
  tty_settings.c_cflag |= (CTSFLOW | RTSFLOW);
  tty_settings.c_lflag &= ~(ISIG | ECHO | ICANON);
  tty_settings.c_lflag |= XCLUDE;
  tty_settings.c_cc[VMIN] = 1;
  tty_settings.c_cc[VTIME] = 0;
  tty_settings.c_oflag &= ~OPOST;
  
  if (ioctl(port.outfd, TCSETAF, &tty_settings) < 0)
    {
      fprintf(stderr, "%s: can't set baud for %s\n", myname, portname);
      port.infd = ERROR;
      return port;
    }
  
  if (fcntl(port.infd, F_SETFL, O_RDONLY) == ERROR)
    {
      fprintf(stderr, "%s: can't unset O_NDELAY for %s\n", myname, portname);
      port.infd = ERROR;
      return port;
    }
  if (fcntl(port.outfd, F_SETFL, O_WRONLY) == ERROR)
    {
      fprintf(stderr, "%s: can't unset O_NDELAY for %s\n", myname, portname);
      port.infd = ERROR;
      return port;
    }
  if (verbosity > 5)
    fprintf(stderr, "%s: baud set for %s\n", myname, portname);
  return port;
}

int main(unsigned argc, char *argv[])
{
  Port port;
  int i;
  bool error = FALSE;
  struct option options[] = {
    { "debug", 2, 0, 'x' },
    { 0, 0, 0, 0 }
  };
  
  myname = argv[0];
  while ((i = getopt_long(argc, argv, "x::", options, (int *)0))
	 != EOF)
    switch (i)
      {
      case 'x':
	if (optarg)
	  verbosity = atoi(optarg);
	break;
      case '?':
	++error;
	break;
      }
  if (argc - optind < 2 || error)
    {
      fprintf(stderr, "usage: %s [-x[debug_level]] [+debug[=level]] device fax_pages_base_name\n", myname);
      exit(-1);
    }
  signal(SIGALRM, catcher);
  port = openLine(argv[1]);
  if (port.infd < 0)
    exit(3);
  if (!initModem(port))
    exit(1);
  if (!waitConnection(port))
    exit(2);
  receiveFax(port, argv[2]);
  closeLine(port);
  return exit(0), 1;
}
