/*
 * $Id: c2rec.c,v 4.3 1993/01/22 15:14:24 wcp Exp $
 *
 * Copyright (C) 1992	Walter Pelissero
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 1, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * $Log: c2rec.c,v $
 * Revision 4.3  1993/01/22  15:14:24  wcp
 * Removed magic string.
 *
 * Revision 4.2  1993/01/06  17:47:40  wcp
 * More robust handling of inter-page negotiation.
 * Already_connected is no longer needed.
 * Added SPEED as header entry in output files.
 * Converted debug messages to dprint() syntax.
 * Modified output file name generation (base name may be a directory
 * or a file). Removed -b option: now basename is mandatory.
 *
 * Revision 4.1  1993/01/02  18:55:30  wcp
 * Rewritten to make it work under mgetty just as fax receiver.
 *
 * Revision 3.2  1992/09/04  13:30:18  wally
 * Deleted unnecessary array reference in mdread().
 * Corrected some misspelling.
 * Fix in getLoginName(): returned value had to point to static area.
 *
 * Revision 3.1  1992/08/29  18:21:51  wally
 * This is now a getty program (no need of an external getty).
 *
 * Revision 2.1  1992/08/27  19:30:11  wally
 * Modified to be a real replacement for getty.
 * Better baud rates handling.
 * /etc/utmp management.
 * New signal handling.
 * Many bug fixes.
 *
 * Revision 1.1  1992/07/17  16:43:57  wally
 * Initial revision
 *
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <fcntl.h>
#include <termio.h>
#include <getopt.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <signal.h>
#include <utmp.h>
#include <stdlib.h>
#include "common.h"
#include "modemio.h"
#include "zyxel.h"

#ifdef DEBUG
static int verbosity = 9;
#else
static int verbosity = 0;
#endif

const char *myname;
static char RcsId[] = "$Id: c2rec.c,v 4.3 1993/01/22 15:14:24 wcp Exp $";
static bool timedout;

static SIGTYPE timeout(int sig)
{
  timedout = TRUE;
  signal(sig, timeout);
#ifndef SIGH_VOID
  return 0;
#endif
}

static bool tryRead(ModemReturnCode answer, int fd)
{
  unsigned i;
  ModemReturnCode ret;

  dprint(5, (stderr, "%s: trying to read \"%s\"\n", myname,
	     ModemAnswers[answer]));
  for (i = 0; i < 10; ++i)
    {
      if ((ret = receive(10, fd)) == answer)
	return SUCCEEDED;
      if (ret < 0)
	break;
    }
  return FAILED;
}

static bool receivePage(int infd, int outfd)
{
  register unsigned bytes = 0;

  timedout = FALSE;
  for (bytes = 0; TRUE; ++bytes)
    {
      char c;

      if ((bytes % 1000) == 0)
	alarm(30);
      if (timedout || read(infd, &c, 1) < 1)
	{
	  alarm(0);
	  return FAILED;
	}
      if (c == DLE)
	{
	  if (timedout || read(infd, &c, 1) < 1)
	    {
	      alarm(0);
	      return FAILED;
	    }
	  switch (c)
	    {
	    case DLE:
	      break;
	    case ETX:
	      alarm(0);
	      return SUCCEEDED;
	    default:
	      continue;
	    }
	}
      write(outfd, &c, 1);
    }
}

static bool receiveFax(const char *base_name)
{
  unsigned page = 0;

  do
    {
      char pathname[1024];
      int outfd, ret;
      FILE *outfp;

      sprintf(pathname, "%s.%03u", base_name, ++page);
      if ((outfd = open(pathname, O_WRONLY | O_TRUNC | O_CREAT, 0644)) < 0)
	{
	  fprintf(stderr, "%s: cannot open for writing %s (%s)\n",
		  myname, pathname, sys_errlist[errno]);
	  return FAILED;
	}
      else
	{
	  long t = time(0);

	  outfp = fdopen(outfd, "w");
	  fprintf(outfp, "DATE: %s", ctime(&t));
	}
      do
	{
	  switch (ret = receive(10, 0))
	    {
	    case MDOK:
	    case MDFCFR:
	      mdwrite("AT+FDR\r", 1);
	      break;
	    case MDFDCS:
	    case MDFTSI:
	    case MDCONNECT:
	      /* allowed answers */
	      break;
	    default:
	      /* all the other ones are errors */
	      ret = MDERROR;
	      break;
	    }
	}
      while (ret != MDCONNECT && ret != MDERROR);
      if (ret == MDERROR)
	{
	  fclose(outfp);
	  unlink(pathname);
	  return FAILED;
	}
      else
	{
	  fprintf(outfp, "RESOLUTION: %d\n", C2_resolution);
	  fprintf(outfp, "CALLER: %s\n", C2_caller);
	  fprintf(outfp, "ENCODING: 0\n"); /* 0=plain, 1=reversed */
	  fprintf(outfp, "SPEED: %d\n", (C2_baud + 1) * 2400);
	  fprintf(outfp, "DATA:");	/* no \n because there is one still
					   outstanding since last response */
	  fflush(outfp);
	  dprint(7, (stderr, "%s: sending initial DC2\n", myname));
	  sleep(1);
	  mdwrite("\022", 1);
	  dprint(2, (stderr, "%s: receiving page %u\n", myname, page));
	  if (!receivePage(0, outfd))
	    {
	      dprint(1, (stderr, "%s: page %u discarded\n", myname, page));
	      unlink(pathname);
	    }
	  fclose(outfp);
	  do
	    {
	      switch (ret = receive(10, 0))
		{
		case MDFPTS:	/* Is it a ZyXEL bug? */
		  mdwrite("AT+FDR\r", 1);
		case MDFET:
		  break;
		default:
		  ret = MDERROR;
		  break;
		}
	    }
	  while (ret != MDERROR && ret != MDFET);
	  if (ret == MDERROR)
	    return FAILED;
	}
    }
  while (C2_eop == 0);
  mdwrite("AT+FDR\r", 1);
  if (tryRead(MDFHNG, 0))
    tryRead(MDOK, 0);
  sleep(1);	/* let modem recover */
  return SUCCEEDED;
}

static bool printPages(const char *base_name)
{
  char command[1024];

  sprintf(command, "%s/printfax %s", LIB_DIR, base_name);
  return !system(command);
}

static void deletePages(const char *base_name)
{
  char buf[strlen(base_name) + 10];
  unsigned i = 0;

  do sprintf(buf, "%s.%03u", base_name, ++i);
  while (unlink(buf) == 0);
}

static SIGTYPE terminate(int sig)
{
  dprint(0, (stderr, "%s: got HANGUP signal\n", myname));
  mdhangup(1);
  exit(0);
#ifndef SIGH_VOID
  return 0;
#endif
}
      
int main(unsigned argc, char *argv[])
{
  bool error = FALSE;
  unsigned print = 0;
  int i;
  const char *base = 0;
  char fax_base_name[1024];

  myname = basename(argv[0]);
  while ((i = getopt(argc, argv, "px:")) != EOF)
    switch (i)
      {
      case 'p':
	++print;
	break;
      case 'x':
	verbosity = atoi(optarg);
	break;
      default:
	error = TRUE;
	break;
      }
  if (optind == argc)
    error = TRUE;
  base = argv[optind++];
  if (optind < argc)		/* too many arguments */
    error = TRUE;
  if (error)
    {
      fprintf(stderr, "%s\nusage: %s [-x level][-p ...] out{file|dir}\n",
	      RcsId, myname);
      exit(-1);
    }
  mdverbosity = verbosity;
  signal(SIGALRM, timeout);
  signal(SIGQUIT, terminate);
  signal(SIGTERM, terminate);
  signal(SIGHUP, timeout);
  signal(SIGINT, terminate);
  {
    unsigned nfax = 1;
    struct stat st;

    if (stat(base, &st) == 0 && (st.st_mode & S_IFDIR))
      {
	/* Base is a directory.
	 */
	for (; nfax > 0; ++nfax)
	  {
	    sprintf(fax_base_name, "%s/in%08u.%03u", base, nfax, 1);
	    /* We avoid to overwrite existing faxes */
	    if (access(fax_base_name, F_OK))
	      {
		*strrchr(fax_base_name, '.') = '\0'; /* chop ".001" */
		break;
	      }
	  }
      }
    else
      /* Base is a probably inexistent file name.
       */
      strcpy(fax_base_name, base);
    if (nfax == 0)
      {
	fprintf(stderr, "%s: spool area is full.\n", myname);
	exit(1);
      }
  }
  receiveFax(fax_base_name);
  mdhangup(1);
  if (print-- > 0)
    {
      if (fork() == 0)
	{
	  close(0); close(1); close(2);
	  setpgrp();
	  dup(dup(open("/dev/null", O_RDWR)));
	  if (printPages(fax_base_name) && print > 0)
	    deletePages(fax_base_name);
	}
      signal(SIGCLD, SIG_IGN);
    }
  return exit(0), 0;
}
