/*
 * This software is copyrighted as noted below.  It may be freely copied,
 * modified, and redistributed, provided that the copyright notice is
 * preserved on all copies.
 *
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely "as is".  Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 */
/*
 * zoomlug.c - performs image scaling, file format conversion, progressive
 *             JPEGs and transparent GIFs.
 *
 * Author:      Raul Rivero
 *              Vice-Rectorado de Estudiantes y Extension Universitaria 
 *              University of Oviedo
 * Date:        Sun 3 Dec 1995
 * Copyright (c) 1995, Raul Rivero
 *
 */

#include <lug.h>
#include <lugfnts.h>

#define CALCULATE_HALF				1
#define CALCULATE_DOUBLE			2

extern int LUGverbose;
extern char *MY_NAME;

main( argc, argv )
int argc;
char **argv;
{
  char c;
  bitmap_hdr in, out;
  int xsize, ysize;
  int half_double;
  int quality;
  int progressive;
  int transparent;
  char *inputfile, *outputfile;
  extern int optind;
  extern char *optarg;

  MY_NAME = argv[0];

  /* Reset to defaults */
  xsize = ysize = -1;
  quality = 75;
  progressive = transparent = half_double = FALSE;
  

  while ( (c = getopt( argc, argv, "x:y:q:tdphv!" )) != EOF ) {
    switch(c) {
      case 'x':
                xsize = atoi( optarg );
                break;
      case 'y':
                ysize = atoi( optarg );
                break;
      case 'q':
                quality = atoi( optarg );
                break;
      case 'p':
                progressive = TRUE;
                break;
      case 't':
                transparent = TRUE;
                break;
      case 'd':
		half_double = CALCULATE_DOUBLE;
                break;
      case 'h':
		half_double = CALCULATE_HALF;
                break;
      case 'v':
                LUGverbose++;
                break;
      case '!':
                print_copyright();
                break;
      default :
                usage();
                break;
    }
  }

  /*
   * We need two file names.
   */
  if ( (argc-optind) < 2 )
    usage();

  /*
   * The input/output file.
   */
  inputfile = argv[optind];
  outputfile = argv[optind+1];

  /*
   * Fix the input options.
   */ 
  lug_set_jpeg_quality( quality );
  lug_set_jpeg_progressive( progressive );
  if ( transparent )
    set_write_gif89_flag();

  /*
   * Ok... go!.
   */ 
  read_lug_file( inputfile, &in );

  if ( xsize < 1 && ysize < 1 && !half_double ) {
    VPRINTF( stderr, "Only copping the image.\n" );
    write_lug_file( outputfile, &in );
  }else {
    /*
     * If we had the 'd' or 'h' switches, then calculate the
     * new sizes.
     */
    if ( half_double == CALCULATE_HALF ) {
      xsize = in.xsize >> 1;
      ysize = in.ysize >> 1;
    }else if ( half_double == CALCULATE_DOUBLE ) {
      xsize = in.xsize << 1;
      ysize = in.ysize << 1;
    }

    /*
     * Ok, now we have the original image. Now, the new Xsize and Ysize
     * was arguments ... fine!. Else, calculate the other.
     */
    if ( xsize < 1 ) {
      /* We have the ysize parameter */
      xsize = ( ((double)ysize)/(double)in.ysize ) * (double)in.xsize;
    }else if ( ysize < 1 ) {
            /* We have the xsize parameter */
            ysize = ( ((double)xsize)/(double)in.xsize ) * (double)in.ysize;
          }

    VPRINTF( stderr, "Resampling %s (%dx%d) to %s (%dx%d)\n", 
             inputfile, in.xsize, in.ysize, outputfile, xsize, ysize );

    slow_adjust_bitmap( &in, &out, xsize, ysize );
    write_lug_file( outputfile, &out );
  }

  return 0;
}

usage()
{
  char *msg = "\n\
%s: Usage: %s [-dhptv!] [-x <xsize>] [-y<ysize>] [-q <quality>] <input_file> <output_file>\n\n\
Flags:\n\
\t-d: double, duplicate the size of the image\n\
\t-h: half, scale down (/2) the size of the image\n\
\t-q: quality of the image (if the output format is JPEG)\n\
\t-p: get a progressive image (if the output format is JPEG)\n\
\t-t: get a transparent image (if the output format is GIF)\n\
\t-x -y: the new size of the image (proportional if only one)\n\
\t-v: verbose\n\
\t-!: hey!, what about this program ?!\n\n\
The file type is got using its suffix:\n\n\
\t* .gif\t\t\t* .hf\t\t\t* .pbm/.pgm/.ppm\n\
\t* .pcx\t\t\t* .raw\t\t\t* .rgb\n\
\t* .rla\t\t\t* .rle\t\t\t* .sgi\n\
\t* .sun\t\t\t* .tga\t\t\t* .tif/.tiff\n\
\t* .ps\t\t\t* .jpeg/.jpg\t\t* .pix (** default **)\n\n\
The Alias 'pix' format will be used by default.\n\n\
If required, the quantization method is the default process to reduce\n\
the number of colors.\n\n";

  fprintf( stderr, msg, MY_NAME, MY_NAME );
  exit( 1 );
}


print_copyright()
{
  char *msg = "\
zoomlug ( %s ) - resamples images\n\n\
This program - (c) 1992, Raul Rivero\n\
LUG  library - (c) 1992, Raul Rivero && Math Dept. ( U. of Oviedo )\n\n\
This software is free and you can get a full copy of original LUG library\n\
via E-mail to rivero@pinon.ccu.uniovi.es, via anonymous ftp to\n\
ftp://ftp.uniovi.es/pub/uniovi/mathdept/src/ or using LUG's WWW pages,\n\
http://www.uniovi.es/~rivero/LUG/.\n\n\
The LUG library includes support for several file formats, viewers on\n\
different architectures and digital image processing.\n\n\
Supported input formats:\n\n\
\t* Pix ( Alias )  *** default ***\n\
\t* TIFF ( needs Sam Leffler's TIFF library )\n\
\t* RLE ( needs Utah Raster Toolkit library )\n\
\t* RLA ( Wavefront )\n\
\t* SGI ( internal Silicon Graphics file format )\n\
\t* Targa ( Truevision )\n\
\t* GIF ( Compuserve )\n\
\t* PCX ( ZSoft )\n\
\t* PBM/PGM/PPM\n\
\t* Postscript\n\
\t* JPEG ( needs Thomas G. Lane's JPEG library )\n";

  fprintf( stderr, msg, get_lug_version() );
  exit( 1 );
}
