/*
 *  Project   : tin - a Usenet reader
 *  Module    : xref.c
 *  Author    : I. Lea & H. Brugge
 *  Created   : 1993-07-01
 *  Updated   : 1998-20-24
 *  Notes     :
 *  Copyright : (c) Copyright 1991-98 by Iain Lea & Herman ten Brugge
 *              You may  freely  copy or  redistribute  this software,
 *              so  long as there is no profit made from its use, sale
 *              trade or  reproduction.  You may not change this copy-
 *              right notice, and it must be included in any copy made
 */

#include	"tin.h"

/* dbmalloc checks memset() parameters, so we'll use it to check the assignments */
#ifdef USE_DBMALLOC
#	define BIT_OR(n,b,mask)	memset(n+NOFFSET(b), n[NOFFSET(b)] | (mask), 1)
#	define BIT_AND(n,b,mask)	memset(n+NOFFSET(b), n[NOFFSET(b)] & (mask), 1)
#else
#	define BIT_OR(n,b,mask)	n[NOFFSET(b)] |= mask
#	define BIT_AND(n,b,mask)	n[NOFFSET(b)] &= mask
#endif /* USE_DBMALLOC */

static void read_xref_header (struct t_article *art);

/*
 *  Read NEWSLIBDIR/overview.fmt file to check if Xref:full is enabled/disabled
 *  Note: if the file cannot be found set the default Xref supported to TRUE
 */

int
overview_xref_support (void)
{
	char buf[HEADER_LEN];
	FILE *fp;
	int supported;

	supported = FALSE;

	if ((fp = open_overview_fmt_fp ()) != (FILE *) 0) {
		while ((tin_fgets (buf, sizeof (buf), fp)) != (char *) 0) {
			if (STRCMPEQ(buf, "Xref:full")) {
				supported = TRUE;
				drain_buffer(fp);
				break;
			}
		}
		TIN_FCLOSE (fp);
		/* If user aborted with 'q', then we continue regardless. If Xref was
		 * found, then fair enough. If not, tough. No real harm done
		 */
	}

	if (!supported)
		wait_message (2, txt_warn_xref_not_supported);

	return supported;
}

/*
 * read xref reference for current article
 * This enables crosspost marking even if the xref records are not
 * part of the xover record.
 */
static void
read_xref_header (struct t_article *art)
{
#ifdef NNTP_ABLE
	/* xref_supported means already supported in xover record */
	if (!xref_supported && read_news_via_nntp && art && !art->xref) {
		char buf[HEADER_LEN];
		char *ptr, *q;
		long artnum = 0;

		sprintf(buf, "xhdr xref %ld", art->artnum);
		put_server (buf);
		if (get_respcode (NULL) != OK_HEAD)
			return;
		while (tin_fgets (buf, sizeof (buf)-1, (FILE*)nntp_rd_fp)) {
			ptr = buf;
			while (*ptr && isspace(*ptr))
				ptr++;
			if (*ptr == '.')
				break;
			/*
			 * read the article number
			 */
			artnum = atol (ptr);
			if ((artnum == art->artnum) && !art->xref && !strstr (ptr, "(none)")) {
				q = strchr (ptr, ' ');	/* skip artikel number */
				if (q == NULL)
					continue;
				ptr = q;
				while (*ptr && isspace(*ptr))
					ptr++;
				q = strchr (ptr, '\n');
				if (q)
					*q = '\0';
				art->xref = my_strdup (ptr);
			}
		}

	}
#endif
	return;
}

/*
 *  mark all other Xref: crossposted articles as read when one article read
 *  Xref: sitename newsgroup:artnum newsgroup:artnum [newsgroup:artnum ...]
 */

void
art_mark_xref_read (
	struct t_article *art)
{
	char *xref_ptr;
	char *group;
	char *ptr, c;
	int artread;
	long artnum;
	struct t_group *psGrp;

	read_xref_header (art);

	if (art->xref == '\0')
		return;
	xref_ptr = art->xref;

	/*
	 *  check sitename matches nodename of current machine (ignore for now!)
	 */
	while (*xref_ptr != ' ' && *xref_ptr)
		xref_ptr++;

	/*
	 *  tokenize each pair and update that newsgroup if it is in my_group[].
	 */
	forever {
		while (*xref_ptr == ' ')
			xref_ptr++;

		group = xref_ptr;
		while (*xref_ptr != ':' && *xref_ptr)
			xref_ptr++;

		if (*xref_ptr != ':')
			break;

		ptr = xref_ptr++;
		artnum = atol (xref_ptr);
		while (*xref_ptr >= '0' && *xref_ptr <= '9')
			xref_ptr++;

		if (&ptr[1] == xref_ptr)
			break;

		c = *ptr;
		*ptr = 0;

		psGrp = psGrpFind (group);

#ifdef DEBUG
		if (debug == 3) {
			sprintf (msg, "LOOKUP Xref: [%s:%ld] active=[%s] num_unread=[%ld]",
				group, artnum,
				(psGrp ? psGrp->name : ""),
				(psGrp ? psGrp->newsrc.num_unread : 0));
#ifdef DEBUG_NEWSRC
			debug_print_comment (msg);
			debug_print_bitmap (psGrp, NULL);
#endif
			error_message (msg);
		}
#endif /* DEBUG */

		if (psGrp && psGrp->newsrc.xbitmap) {
			if (artnum >= psGrp->newsrc.xmin && artnum <= psGrp->xmax) {
				artread = ((NTEST(psGrp->newsrc.xbitmap, artnum - psGrp->newsrc.xmin) == ART_READ) ? TRUE : FALSE);
				if (!artread) {
					NSET0(psGrp->newsrc.xbitmap, artnum - psGrp->newsrc.xmin);
					if (psGrp->newsrc.num_unread > 0)
						psGrp->newsrc.num_unread--;
#ifdef DEBUG
					if (debug == 3) {
						sprintf (msg, "FOUND!Xref: [%s:%ld] marked READ num_unread=[%ld]",
							group, artnum, psGrp->newsrc.num_unread);
#ifdef DEBUG_NEWSRC
						debug_print_comment (msg);
						debug_print_bitmap (psGrp, NULL);
#endif
						wait_message (2, msg);
					}
#endif /* DEBUG */
				}
			}
		}
		*ptr = c;
	}
}

/*
 * Set bits [low..high] of 'bitmap' to 1's
 */
void
NSETRNG1 (
	t_bitmap *bitmap,
	long low,
	long high)
{
	register long i;

	if (bitmap == (t_bitmap *) 0) {
		error_message ("NSETRNG1() failed. Bitmap == NULL");
		return;
	}

	if (high >= low) {
		if (NOFFSET(high) == NOFFSET(low)) {
			for (i=low; i <= high; i++) {
				NSET1(bitmap, i);
			}
		} else {
			BIT_OR(bitmap, low, (NBITSON << NBITIDX(low)));
			if (NOFFSET(high) > NOFFSET(low) + 1) {
				memset (&bitmap[NOFFSET(low) + 1], NBITSON,
					(size_t) (NOFFSET(high)-NOFFSET(low)-1));
			}
			BIT_OR(bitmap, high, ~ (NBITNEG1 << NBITIDX(high)));
		}
	}
}

/*
 * Set bits [low..high] of 'bitmap' to 0's
 */
void
NSETRNG0 (
	t_bitmap *bitmap,
	long low,
	long high)
{
	register long i;

	if (bitmap == (t_bitmap *) 0) {
		error_message ("NSETRNG0() failed. Bitmap == NULL");
		return;
	}

	if (high >= low) {
		if (NOFFSET(high) == NOFFSET(low)) {
			for (i=low; i <= high; i++) {
				NSET0(bitmap, i);
			}
		} else {
			BIT_AND(bitmap, low, ~(NBITSON << NBITIDX(low)));

			if (NOFFSET(high) > NOFFSET(low) + 1) {
				memset (&bitmap[NOFFSET(low) + 1], 0,
					(size_t) (NOFFSET(high)-NOFFSET(low)-1));
			}
			BIT_AND(bitmap, high, NBITNEG1 << NBITIDX(high));
		}
	}
}
