#include <unistd.h>
#include <stdio.h>
#include "file2.h"
#include "internal.h"
#include <trace.h>

/****************************************************************************
 *
 * Function:    _stbuf
 *
 * Description: Complete any pending buffering for the stdout/stderr files.
 *
 * Input:       stream   - A pointer to the file to be initialized.
 *
 * Result:      TRUE to indicate the file is buffered, FALSE otherwise.
 */

int _stbuf (FILE *stream)
    {
    unsigned char *buff;
    FUNC_ENTRY ("_stbuf");
/*
 *  Attempt to find the buffer for this file.
 */
    if (stream == stdout)
        {
	buff = _bufout;
	_cflush++;
	_libc__cleanup = _cleanup;
        }
    else
        {
	if (stream == stderr)
	    {
	    buff = _buferr;
	    }
	else
	    {
	    buff = NULL;
	    }
        }
/*
 *  If the file is currently buffered then reject the operation.
 */
    if (anybuf (stream))
        {
	buff = NULL;
	}
/*
 *  Initialize the buffer pointers
 */
    if (buff != NULL)
        {
	stream->_ptr     =
        stream->_base    = buff;
	stream->_cnt     = BUFSIZ;
        set_buf_size(stream, BUFSIZ);
	stream->_flag   |= _IOWRT;
	stream->_flag   &= ~_IOFREEBF;
/*
 *  Indicate a pending flush is required
 */
	_cflush++;
	_libc__cleanup = _cleanup;
        }
/*
 *  Return the status to the caller. If the buffer is non-null then the
 *  operation was completed.
 */
    FUNC_EXIT ("_stbuf");
    return (buff != NULL);
    }

/**************************************************************************
 *
 * Function:    _ftbuf
 *
 * Description:
 *    flush and dismantle the buffer for the file.
 *
 * Entry:
 *    flag     - TRUE if discard temp buffering on the stream
 *    stream   - The stream to process
 *
 * Result:
 *     none.
 */

void _ftbuf (int flag, FILE *stream)
    {
    unsigned char tty_flag;
    FUNC_ENTRY ("_ftbuf");
/*
 *  If the file is invalid then assume that this is not a tty device.
 *  (This is probably a stream file. -- Hey, I had to give up a flag bit, Ok?)
 */
    if ((unsigned int) fileno (stream) >= (unsigned int) _NFILE)
        {
	tty_flag = _IONOTTY;
        }
/*
 *  If the file has not been classified, then classify the file as to wether
 *  or not it is a tty device.
 */
    else
        {
	tty_flag = tty_flag_list [fileno (stream)];
	switch (tty_flag)
	    {
	case _IONOTTY:
	case _IOISTTY:
	    break;

	default:
	    tty_flag = isatty (fileno (stream)) ? _IOISTTY : _IONOTTY;
	    tty_flag_list [fileno (stream)] = tty_flag;
	    break;
	    }
        }
/*
 *  If this is a tty then flush the table.
 */
    if (tty_flag == _IOISTTY)
        {
	if (flag)
	    {
	    if (stream == stdout || stream == stderr)
	        {
		fflush(stream);
		stream->_base    =
		stream->_ptr     = NULL;
		set_buf_size(stream, 0);
	        }
	    }
/*
 *  If the output buffer is local for stdout or stderr then flush the
 *  stream.
 */
	else
            {
	    if (stream->_base == _bufout || stream->_base == _buferr)
	        {
                fflush (stream);
	        }
	    }
        }

    FUNC_EXIT ("_ftbuf");
    }

