/*
 *
 *	default.c
 *
 *	Support for object and variable defaults.
 *
 *	HNMS User Interface
 *	HNMS 2.0
 *
 *	February 1994
 *
 *	Leslie Schlecht
 *	Computer Sciences Corporation
 *	Numerical Aerodynamic Simulation Systems Division
 *	NASA Ames Research Center
 *
 *	Copyright (c) 1994 Leslie Schlecht
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 1, or (at your option)
 *	any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include	<stdio.h>
#include	<string.h>
#include	<time.h>

#include	<X11/StringDefs.h>
#include	<X11/Intrinsic.h>
#include	<X11/Shell.h>
#include	<X11/Core.h>
#include	<Xm/Xm.h>
#include	<Xm/Text.h>
#include	<Xm/Form.h>
#include	<Xm/ScrolledW.h>
#include	<Xm/List.h>
#include	<Xm/PanedW.h>

#include	"defines.h"
#include	"externs.h"
#include	"object.h"
#include	"viewpanel.h"
#include	"xsupport.h"
#include	"variable.h"
#include	"view.h"
#include	"default.h"

typedef		struct deftext	{
		Widget		defnam;
		Widget		defval;
		Widget		deftime;
		} DT;

static int	defaultinterval = 60;

static VIEW	*defaultview;


/*
 *	Create default view.
 */
void
CreateDefaultViewPanel(obj)
OBJECT	*obj;
	{
	Widget		f, w, l;
	XmString	xs;
	VIEWPANEL	*v;
	Widget		defvar, pw;
	DT		*d;

	if (!obj) return;
	if (obj->def) {
		OpenViewPanel(NULL, obj->def, NULL);
		return;
		}
	v = CreateViewPanel("(DEFAULT)", obj->name, 500, 0, "default", 0, 0, 0);
	obj->def = (caddr_t)v;
	XmTextSetString(v->titletext, obj->uname);
	XtVaSetValues(v->titletext,
		XmNsensitive, False,
		NULL);
	v->wa = (caddr_t)myalloc(NULL, 1, sizeof(DT));
	d = (DT*)(v->wa);
	XtRemoveAllCallbacks(v->closebutton, XmNactivateCallback);
	XtAddCallback(v->closebutton, XmNactivateCallback,
		RemoveDefaultViewPanel, obj);
	XtVaSetValues(v->closebutton,
		XmNleftAttachment, XmATTACH_FORM,
		NULL);
	XtUnmanageChild(v->helpbutton);

	defvar = XtVaCreateManagedWidget("defaultview",
		xmScrolledWindowWidgetClass,
		v->viewform,
		XmNscrollingPolicy, XmAUTOMATIC,
		XmNhighlightThickness, 0,
		XmNshadowThickness, 0,
		XmNwidth, 475,
		XmNheight, 100,
		XmNtopAttachment, XmATTACH_FORM,
		XmNleftAttachment, XmATTACH_FORM,
		XmNbottomAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		NULL);
	f = XtVaCreateManagedWidget("defaultview",
		xmFormWidgetClass,
		defvar,
		NULL);
	d->defnam = XtVaCreateManagedWidget("defaultview",
		xmTextWidgetClass,
		f,
		XmNtopAttachment, XmATTACH_FORM,
		XmNleftAttachment, XmATTACH_FORM,
		XmNbottomAttachment, XmATTACH_FORM,
		XmNcolumns, 16,
		XmNhighlightThickness, 0,
		XmNshadowThickness, 0,
		XmNeditable, False,
		XmNeditMode, XmMULTI_LINE_EDIT,
		XmNcursorPositionVisible, False,
		XmNresizable, True,
		XmNresizeHeight, True,
		XmNresizeWidth, True,
		NULL);
	d->defval = XtVaCreateManagedWidget("defaultview",
		xmTextWidgetClass,
		f,
		XmNtopAttachment, XmATTACH_FORM,
		XmNleftAttachment, XmATTACH_WIDGET,
		XmNleftWidget, d->defnam,
		XmNleftOffset, 4,
		XmNcolumns, 8,
		XmNbottomAttachment, XmATTACH_FORM,
		XmNhighlightThickness, 0,
		XmNshadowThickness, 0,
		XmNeditable, False,
		XmNeditMode, XmMULTI_LINE_EDIT,
		XmNcursorPositionVisible, False,
		XmNresizable, True,
		XmNresizeHeight, True,
		XmNresizeWidth, True,
		NULL);
	d->deftime = XtVaCreateManagedWidget("defaultview",
		xmTextWidgetClass,
		f,
		XmNtopAttachment, XmATTACH_FORM,
		XmNleftAttachment, XmATTACH_WIDGET,
		XmNleftWidget, d->defval,
		XmNleftOffset, 4,
		XmNbottomAttachment, XmATTACH_FORM,
		XmNhighlightThickness, 0,
		XmNshadowThickness, 0,
		XmNeditable, False,
		XmNeditMode, XmMULTI_LINE_EDIT,
		XmNcursorPositionVisible, False,
		XmNresizable, True,
		XmNresizeHeight, True,
		XmNresizeWidth, True,
		NULL);
	LoadDefaultVariables(obj);

	OpenViewPanel(NULL, v, NULL);
	}


/*
 *	Add the default variables to an object's default view if it exists.
 */
void
AddDefaultObject(view, obj, class, name)
VIEW	*view;
OBJECT	*obj;
int	class;
char	*name;
	{
	if (obj->def)
		LoadDefaultVariables(obj);
	}


/*
 *	Determine if the UI cares about keeping an object just announced.
 */
int
AllowObject(obj)
OBJECT	*obj;
	{
	if (!defaultview->objstr[obj->class]) return(0);
	return(InStrList(defaultview->objstr[obj->class], obj->name));
	}


/*
 *	Edit the user interface defaults.
 */
void
EditObjectDefaults(widg, a1, a2)
Widget	widg;
caddr_t	a1, a2;
	{
	EditDefaults(defaultview);
	}


/*
 *	Get the default update interval.
 */
int
GetDefaultInterval()
	{
	return(defaultinterval);
	}


/*
 *	Set defaults.
 */
void
LoadDefault(view)
VIEW	*view;
	{
	register	i;
	
	defaultview = view;
	SetDefaultInterval(defaultview->interval);
	SetDefaultBackground(defaultview->background);
	for (i=0; i<NUM_OBJCLASSES; i++) {
		if (defaultview->varcnt[i]) {
			free(defaultview->varlist[i]);
			defaultview->varlist[i] = NULL;
			}
		StrToVariable(defaultview->varstr[i],
			&(defaultview->varlist[i]),
			defaultview->varcnt[i]);
		}
	}


/*
 *	Load the current object variables.
 */
void
LoadDefaultVariables(obj)
OBJECT	*obj;
	{
	OBJVARENTRY	*ove;
	int	l;
	char	bufname[BUFSIZ], bufvalue[BUFSIZ], buftime[BUFSIZ];
	char	s[256];
	DT	*d;
	caddr_t	p, *var;
	register	i;

	d = (DT*)((VIEWPANEL*)(obj->def))->wa;
	bufname[0] = '\0';
	bufvalue[0] = '\0';
	buftime[0] = '\0';
	/* list all variables */
	var = defaultview->varlist[obj->class];
	for (i=0; i<defaultview->varcnt[obj->class]; i++) {
		strcat(bufname, VarOde(var[i]));
		p = obj->varlist;
		while (NextEntry(&p, &ove)) {
			if (var[i] != ove->var) continue;
			/* determine if variable is string or integer */
			if (IsVarString(ove->var)) {
				if (ove->value) {
					strcat(bufvalue, ove->value);
					l = strlen(bufvalue)-1;
					if (bufvalue[l] == '\n')
						bufvalue[l] = '\0';
					}
				}
			else {
				sprintf(s, "%d", (unsigned int)ove->value);
				strcat(bufvalue, s);
				}
			strcat(buftime, get_time(ove->tsc));
			break;
			}
		strcat(bufname, "\n");
		strcat(bufvalue, "\n");
		strcat(buftime, "\n");
		}
	XmTextSetString(d->defnam, bufname);
	XmTextSetString(d->defval, bufvalue);
	XmTextSetString(d->deftime, buftime);
	}



/*
 *	Remove a default view panel and its work area.
 */
void
RemoveDefaultViewPanel(widg, obj, a)
Widget	widg;
OBJECT	*obj;
caddr_t	a;
	{
	VIEWPANEL	*v;

	v = (VIEWPANEL*)obj->def;
	free(v->wa);
	DestroyViewPanel(obj->def);
	obj->def = NULL;
	}


/*
 *	Designate the button for editing the defaults.
 */
void
SetDefaultButton(b)
Widget	b;
	{
	XtAddCallback(b, XmNactivateCallback, EditObjectDefaults, 0);
	}


/*
 *	Set the default update interval.
 */
void
SetDefaultInterval(i)
	{
	defaultinterval = i;
	}


/*
 *	Update the value of a default variable.
 */
void
UpdateDefaultVariable(obj)
OBJECT		*obj;
	{
	if (!obj->def) return;
	LoadDefaultVariables(obj);
	}
