/*\
 *	DISTRIBUTION: HNMS v2.0
 *	FILE: hnmslib/param.c
 *
 *	Parameter code for HNMS modules.
 *
 *	Jude George
 *	NAS Facility, NASA Ames Research Center
 *
 *	Copyright (c) 1994 Jude George
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 1, or (at your option)
 *	any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\*/

#include <string.h>
#include <netdb.h>

#include "stdhnms.h"

static VarBindList	my_params = NULL;

/*\
 *  Read in values of "bootstrap" parameters from environment variables
 *  and system calls.  Set default values for all other parameters.
\*/
int PARAM_init()
{
    unsigned int	debug_flags;
    char		*cp;
    char		my_hostname[STRBUFLEN];

    bzero(my_hostname, STRBUFLEN);
    gethostname(my_hostname, STRBUFLEN);

    /*
     * "Bootstrap" parameters.
     */
    cp = getenv("HNMS_HOME");
    if (cp)
	PARAM_set_str(oid_hnmsModuleHnmsHome, cp);
    else
	PARAM_set_str(oid_hnmsModuleHnmsHome, def_hnmsModuleHnmsHome);
    
    cp = getenv("HNMS_COMMUNITY");
    if (cp)
	PARAM_set_str(oid_hnmsModuleHnmsCommunity, cp);
    else
	PARAM_set_str(oid_hnmsModuleHnmsCommunity,
		      def_hnmsModuleHnmsCommunity);

    cp = getenv("HNMS_SERVER");
    if (cp)
	PARAM_set_str(oid_hnmsModuleCurrentServerHost, cp);
    else
	PARAM_set_str(oid_hnmsModuleCurrentServerHost, my_hostname);
    
    /*
     * Default values for Server module operational parameters.
     */
    PARAM_set_int(oid_hnmsModuleServerSyncInterval,
		  def_hnmsModuleServerSyncInterval);
    PARAM_set_int(oid_hnmsModuleServerCreateInterval,
		  def_hnmsModuleServerCreateInterval);
    PARAM_set_int(oid_hnmsModuleServerAssignInterval,
		  def_hnmsModuleServerAssignInterval);

    /*
     * Default values for RBIP module operational parameters.
     */
    PARAM_set_int(oid_hnmsModuleRBIPCalcInterval,
		  def_hnmsModuleRBIPCalcInterval);

    /*
     * Default values for IO module operational parameters.
     */
    cp = getenv("HNMS_FILTER");
    if (cp)
	PARAM_set_str(oid_hnmsModuleIoAgentFilter, cp);
    else
	PARAM_set_str(oid_hnmsModuleIoAgentFilter,
		      def_hnmsModuleIoAgentFilter);
    PARAM_set_str(oid_hnmsModuleIoDefSnmpCommunity,
		  def_hnmsModuleIoDefSnmpCommunity);
    PARAM_set_int(oid_hnmsModuleIoNonresponsive,
		  def_hnmsModuleIoNonresponsive);
    PARAM_set_int(oid_hnmsModuleIoUnreachable,
		  def_hnmsModuleIoUnreachable);
    PARAM_set_int(oid_hnmsModuleIoReachInterval,
		  def_hnmsModuleIoReachInterval);
    PARAM_set_int(oid_hnmsModuleIoDiscoverInterval,
		  def_hnmsModuleIoDiscoverInterval);
    PARAM_set_int(oid_hnmsModuleIoCacheCheckInterval,
		  def_hnmsModuleIoCacheCheckInterval);
    PARAM_set_int(oid_hnmsModuleIoRampTime,
		  def_hnmsModuleIoRampTime);

    /*
     * Default values for HNMP operational parameters -- for any module.
     */
    PARAM_set_int(oid_hnmsModuleHnmpWinSize,
		  def_hnmsModuleHnmpWinSize);
    PARAM_set_int(oid_hnmsModuleHnmpRetransInterval,
		  def_hnmsModuleHnmpRetransInterval);
    PARAM_set_int(oid_hnmsModuleHnmpConnTimeout,
		  def_hnmsModuleHnmpConnTimeout);
    PARAM_set_int(oid_hnmsModuleHnmpAckInterval,
		  def_hnmsModuleHnmpAckInterval);
    return 0;
}

/*\
 *  Set the value of a parameter, identifying it by it's oid.
 *  Pass the value in as a PE.
\*/
void PARAM_set_pe(oid, value)
    const OID		oid;
    const PE		value;
{
    if (!oid)
	return;
    VarBindList_set(&my_params, oid, value, 0, 0, 0, 0, 0);
}

/*\
 *  Set the value of a parameter, identifying it by it's oid.
 *  Pass the value in as an integer.
\*/
void PARAM_set_int(oid, int_val)
    const OID		oid;
    const int		int_val;
{
    PE			value;
    
    if (!oid)
	return;
    value = int2prim(int_val);
    PARAM_set_pe(oid, value);
    pe_free(value);
}

/*\
 *  Set the value of a parameter, identifying it by it's oid.
 *  Pass the value in as a NULL-terminated string.
\*/
void PARAM_set_str(oid, cp)
    const OID		oid;
    const char		*cp;
{
    PE			value;
    
    if (!oid)
	return;
    value = oct2prim(cp, strlen(cp));
    PARAM_set_pe(oid, value);
    pe_free(value);
}

/*\
 *  Get the value of a parameter, identifying it by it's oid.
 *  Return the value as a PE.
\*/
PE PARAM_get_pe(oid)
    const OID		oid;
{
    PE			value;
    
    if (!oid) {
	hnms_errno = HNMS_err_var_not_found;
	return NULL;
    }
    value = VarBindList_get(my_params, oid, 0, 0, 0, 0, 0, 0);
    if (value)
	return value;
    else {
	hnms_errno = HNMS_err_var_not_found;
	return NULL;
    }
}

/*\
 *  Get the value of a parameter, identifying it by it's oid.
 *  Return the value as an integer.  Warning: -1 is returned on an error.
\*/
int PARAM_get_int(oid)
    const OID		oid;
{
    PE			value;
    int			i;
    
    value = PARAM_get_pe(oid);
    if (!value)
	return -1;
    i = prim2num(value);
    return i;
}

/*\
 *  Get the value of a parameter, identifying it by it's oid.
 *  Return the value as a static NULL-terminated string.
\*/
char *PARAM_get_str(oid)
    const OID		oid;
{
    static char		buf[STRBUFLEN];
    PE			value;
    int			len;
    char		*cp;
    
    value = PARAM_get_pe(oid);
    if (!value)
	return NULL;
    cp = prim2str(value, &len);
    bzero(buf, STRBUFLEN);
    bcopy(cp, buf, len);
    free(cp);
    return buf;
}

/*\
 *  Clear a parameter, identifying it by it's oid.
\*/
void PARAM_unset(oid)
    const OID		oid;
{
    if (!oid)
	return;
    VarBindList_unset(&my_params, oid);
}

/*\
 *  Iterate a nondestructive function over all parameters.  The
 *  function should take an OID and a PE as the first two arguments.
 *  If magic exists, it will be passed to the function as the third
 *  argument.  The function will be expected to know what to do with it.
\*/
void PARAM_iterate(func, magic)
    void		(*func)();
    caddr_t		magic;
{
    VarBindList_iterate(my_params, func, magic);
}

/*\
 *  Print my local parameters.
\*/
void PARAM_print()
{
    VarBindList_print(my_params);
}
